<?php

declare( strict_types=1 );

namespace Packetery\Module\Views;

use Packetery\Core\CoreHelper;
use Packetery\Module\Carrier;
use Packetery\Module\Checkout\CheckoutService;
use Packetery\Module\Checkout\CheckoutSettings;
use Packetery\Module\ContextResolver;
use Packetery\Module\Dashboard\DashboardPage;
use Packetery\Module\Framework\WcAdapter;
use Packetery\Module\Framework\WpAdapter;
use Packetery\Module\Log;
use Packetery\Module\ModuleHelper;
use Packetery\Module\Options;
use Packetery\Module\Order;
use Packetery\Module\Order\Metabox;
use Packetery\Module\Plugin;
use Packetery\Module\WidgetUrlResolver;
use Packetery\Nette\Http\Request;

class AssetManager {

	/**
	 * @var ContextResolver
	 */
	private $contextResolver;

	/**
	 * @var Metabox
	 */
	private $orderMetabox;

	/**
	 * @var Request
	 */
	private $request;

	/**
	 * @var CheckoutSettings
	 */
	private $checkoutSettings;

	/**
	 * @var WpAdapter
	 */
	private $wpAdapter;

	/**
	 * @var WcAdapter
	 */
	private $wcAdapter;

	/**
	 * @var CheckoutService
	 */
	private $checkoutService;

	/**
	 * @var WidgetUrlResolver
	 */
	private $widgetUrlResolver;

	public function __construct(
		ContextResolver $contextResolver,
		Metabox $orderMetabox,
		Request $request,
		CheckoutSettings $checkoutSettings,
		WpAdapter $wpAdapter,
		WcAdapter $wcAdapter,
		CheckoutService $checkoutService,
		WidgetUrlResolver $widgetUrlResolver
	) {
		$this->contextResolver   = $contextResolver;
		$this->orderMetabox      = $orderMetabox;
		$this->request           = $request;
		$this->checkoutSettings  = $checkoutSettings;
		$this->wpAdapter         = $wpAdapter;
		$this->wcAdapter         = $wcAdapter;
		$this->checkoutService   = $checkoutService;
		$this->widgetUrlResolver = $widgetUrlResolver;
	}

	/**
	 * Enqueues admin JS file.
	 *
	 * @param string $name Name of script.
	 * @param string $file Relative file path.
	 * @param bool   $inFooter Tells where to include script.
	 * @param array  $deps Script dependencies.
	 */
	public function enqueueScript( string $name, string $file, bool $inFooter, array $deps = [] ): void {
		$this->wpAdapter->enqueueScript(
			$name,
			$this->wpAdapter->pluginDirUrl( ModuleHelper::getPluginMainFilePath() ) . $file,
			$deps,
			md5( (string) filemtime( PACKETERY_PLUGIN_DIR . '/' . $file ) ),
			$inFooter
		);
	}

	/**
	 * Enqueues CSS file.
	 *
	 * @param string $name Name of script.
	 * @param string $file Relative file path.
	 */
	public function enqueueStyle( string $name, string $file ): void {
		$this->wpAdapter->enqueueStyle(
			$name,
			$this->wpAdapter->pluginDirUrl( ModuleHelper::getPluginMainFilePath() ) . $file,
			[],
			md5( (string) filemtime( PACKETERY_PLUGIN_DIR . '/' . $file ) )
		);
	}

	/**
	 * Enqueues javascript files and stylesheets for checkout.
	 */
	public function enqueueFrontAssets(): void {
		if ( ! $this->wcAdapter->isCheckout() ) {
			return;
		}
		if ( $this->wpAdapter->doingAjax() === false ) {
			$this->enqueueStyle( 'packetery-front-styles', 'public/css/front.css' );

			$customFrontCssFilename = 'packeta-custom-front.css';
			$customFrontCssPath     = WP_CONTENT_DIR . '/' . $customFrontCssFilename;
			if ( file_exists( $customFrontCssPath ) ) {
				$this->wpAdapter->enqueueStyle(
					'packetery-custom-front-styles',
					$this->wpAdapter->contentUrl( $customFrontCssFilename ),
					[],
					md5( (string) filemtime( $customFrontCssPath ) )
				);
			}
		}
		if ( $this->checkoutService->areBlocksUsedInCheckout() ) {
			$this->wpAdapter->enqueueScript(
				'packetery-widget-library',
				$this->widgetUrlResolver->getUrl(),
				[],
				Plugin::VERSION,
				false
			);
		} elseif ( $this->wpAdapter->doingAjax() === false ) {
			$this->enqueueScript( 'packetery-checkout', 'public/js/checkout.js', true, [ 'jquery' ] );
			$this->wpAdapter->localizeScript( 'packetery-checkout', 'packeteryCheckoutSettings', $this->checkoutSettings->createSettings() );
		}
	}

	/**
	 * Enqueues javascript files and stylesheets for administration.
	 */
	public function enqueueAdminAssets(): void {
		$page                  = $this->request->getQuery( 'page' );
		$isOrderGridPage       = $this->contextResolver->isOrderGridPage();
		$isOrderDetailPage     = $this->contextResolver->isOrderDetailPage();
		$isProductCategoryPage = $this->contextResolver->isProductCategoryDetailPage() || $this->contextResolver->isProductCategoryGridPage();
		$datePickerSettings    = [
			'deliverOnMinDate' => $this->wpAdapter->date( CoreHelper::DATEPICKER_FORMAT, strtotime( 'tomorrow' ) ),
			'dateFormat'       => CoreHelper::DATEPICKER_FORMAT_JS,
		];

		if ( $isOrderGridPage || $isOrderDetailPage || in_array(
			$page,
			[
				Carrier\OptionsPage::SLUG,
				Options\Page::SLUG,
			],
			true
		) ) {
			$this->enqueueScript( 'live-form-validation-options', 'public/js/live-form-validation-options.js', false );
			$this->enqueueScript( 'live-form-validation', 'public/libs/live-form-validation/live-form-validation.js', false, [ 'live-form-validation-options' ] );
			$this->enqueueScript( 'live-form-validation-extension', 'public/js/live-form-validation-extension.js', false, [ 'live-form-validation' ] );
		}

		if ( in_array( $page, [ Carrier\OptionsPage::SLUG, Options\Page::SLUG ], true ) ) {
			$this->enqueueStyle( 'packetery-select2-css', 'public/libs/select2-4.0.13/dist.min.css' );
			$this->enqueueScript( 'packetery-select2', 'public/libs/select2-4.0.13/dist.min.js', true, [ 'jquery' ] );
		}

		if ( $page === Carrier\OptionsPage::SLUG ) {
			$this->enqueueScript(
				'packetery-multiplier',
				'public/js/multiplier.js',
				true,
				[
					'jquery',
					'live-form-validation-extension',
				]
			);
			$this->enqueueScript(
				'packetery-admin-country-carrier',
				'public/js/admin-country-carrier.js',
				true,
				[
					'jquery',
					'packetery-multiplier',
					'packetery-select2',
				]
			);
		}

		if ( $page === Options\Page::SLUG ) {
			$this->enqueueScript(
				'packetery-admin-options',
				'public/js/admin-options.js',
				true,
				[
					'jquery',
					'packetery-select2',
				]
			);

			$this->wpAdapter->localizeScript(
				'packetery-admin-options',
				'translationsAdminOptions',
				[
					'confirmLogDeletion' => $this->wpAdapter->__(
						'Are you sure you want to delete the log file?',
						'packeta'
					),
				]
			);

			$this->wpAdapter->enqueueEditor();
			$this->wpAdapter->enqueueScript( 'editor' );
			$this->enqueueScript(
				'packetery-bug-report-editor',
				'public/js/bug-report-editor.js',
				true,
				[
					'jquery',
					'editor',
				]
			);
		}

		$isProductPage = $this->contextResolver->isProductPage();
		$isPageDetail  = $this->contextResolver->isPageDetail();

		$screen      = $this->wpAdapter->getCurrentScree();
		$isDashboard = ( $screen !== null && $screen->id === 'dashboard' );

		if (
			$isOrderGridPage || $isOrderDetailPage || $isProductPage || $isProductCategoryPage || $isDashboard || $isPageDetail ||
			in_array(
				$page,
				[
					Options\Page::SLUG,
					Carrier\OptionsPage::SLUG,
					Log\Page::SLUG,
					Order\LabelPrint::MENU_SLUG,
					DashboardPage::SLUG,
				],
				true
			)
		) {
			$this->enqueueStyle( 'packetery-admin-styles', 'public/css/admin.css' );
		}

		if ( $isOrderGridPage ) {
			$this->enqueueScript(
				'packetery-admin-grid-order-edit-js',
				'public/js/admin-grid-order-edit.js',
				true,
				[
					'jquery',
					'wp-util',
					'backbone',
				]
			);

			$this->wpAdapter->localizeScript( 'packetery-admin-grid-order-edit-js', 'datePickerSettings', $datePickerSettings );

			$this->enqueueScript(
				'packetery-admin-stored-until-modal-js',
				'public/js/admin-stored-until-modal.js',
				true,
				[
					'jquery',
					'wp-util',
					'backbone',
				]
			);
			$this->wpAdapter->localizeScript( 'packetery-admin-stored-until-modal-js', 'datePickerSettings', $datePickerSettings );
		}

		$pickupPointPickerSettings = null;
		$addressPickerSettings     = null;

		if ( $isOrderDetailPage ) {
			$this->enqueueScript(
				'packetery-multiplier',
				'public/js/multiplier.js',
				true,
				[
					'jquery',
					'live-form-validation-extension',
				]
			);
			$this->enqueueScript(
				'admin-order-detail',
				'public/js/admin-order-detail.js',
				true,
				[
					'jquery',
					'packetery-multiplier',
					'live-form-validation-extension',
				]
			);

			$this->wpAdapter->localizeScript( 'admin-order-detail', 'datePickerSettings', $datePickerSettings );
			$pickupPointPickerSettings = $this->orderMetabox->getPickupPointWidgetSettings();
			$addressPickerSettings     = $this->orderMetabox->getAddressWidgetSettings();

			$this->enqueueScript(
				'packetery-admin-stored-until-modal-js',
				'public/js/admin-stored-until-modal.js',
				true,
				[
					'jquery',
					'wp-util',
					'backbone',
				]
			);

		}

		if ( $pickupPointPickerSettings !== null || $addressPickerSettings !== null ) {
			$this->wpAdapter->enqueueScript(
				'packetery-widget-library',
				$this->widgetUrlResolver->getUrl(),
				[],
				Plugin::VERSION,
				true
			);
		}

		if ( $pickupPointPickerSettings !== null ) {
			$this->enqueueScript(
				'packetery-admin-pickup-point-picker',
				'public/js/admin-pickup-point-picker.js',
				true,
				[
					'jquery',
					'packetery-widget-library',
				]
			);

			$this->wpAdapter->localizeScript(
				'packetery-admin-pickup-point-picker',
				'packeteryPickupPointPickerSettings',
				$pickupPointPickerSettings
			);
		}

		if ( $addressPickerSettings !== null ) {
			$this->enqueueScript(
				'packetery-admin-address-picker',
				'public/js/admin-address-picker.js',
				true,
				[
					'jquery',
					'packetery-widget-library',
				]
			);
			$this->wpAdapter->localizeScript(
				'packetery-admin-address-picker',
				'packeteryAddressPickerSettings',
				$addressPickerSettings
			);
		}

		if ( $this->contextResolver->isConfirmModalPage() ) {
			$this->enqueueScript( 'packetery-confirm', 'public/js/confirm.js', true, [ 'jquery', 'backbone' ] );
		}
		if ( $this->contextResolver->isPluginsOverviewPage() ) {
			$this->enqueueScript(
				'packetery-admin-confirm-deactivation',
				'public/js/admin-confirm-deactivation.js',
				true
			);

			$this->wpAdapter->localizeScript(
				'packetery-admin-confirm-deactivation',
				'translationsDeactivation',
				[
					'confirmDeactivation' => $this->wpAdapter->__(
						'When uninstalling the plugin, ALL data and settings will be removed only if the PACKETERY_REMOVE_ALL_DATA constant is defined in wp-config.php and is set to true. This is to prevent data loss when removing the plugin from the backend and to ensure that only the site owner can perform this action.',
						'packeta'
					),
				]
			);
		}
	}
}
