<?php

namespace WPOAuth2\Storage;

use WPOAuth2\Encryption\EncryptionInterface;
use WPOAuth2\Encryption\Jwt;

/**
 *
 * @author Brent Shaffer <bshafs at gmail dot com>
 */
class CryptoToken implements CryptoTokenInterface {

	protected $publicKeyStorage;
	protected $tokenStorage;
	protected $encryptionUtil;

	/**
	 * @param OAuth2\Encryption\PublicKeyInterface  $publicKeyStorage the public key encryption to use
	 * @param OAuth2\Storage\AccessTokenInterface   $tokenStorage     OPTIONAL persist the access token to another storage. This is useful if
	 *                                                                you want to retain access token grant information somewhere, but is not
	 *                                                                necessary when using this grant type.
	 * @param OAuth2\Encryption\EncryptionInterface $encryptionUtil   OPTIONAL class to use for "encode" and "decode" functions.
	 */
	public function __construct( PublicKeyInterface $publicKeyStorage, AccessTokenInterface $tokenStorage = null, EncryptionInterface $encryptionUtil = null ) {
		$this->publicKeyStorage = $publicKeyStorage;
		$this->tokenStorage     = $tokenStorage;
		if ( is_null( $encryptionUtil ) ) {
			$encryptionUtil = new Jwt();
		}
		$this->encryptionUtil = $encryptionUtil;
	}

	public function getAccessToken( $oauth_token ) {
		// just decode the token, don't verify
		if ( ! $tokenData = $this->encryptionUtil->decode( $oauth_token, null, false ) ) {
			return false;
		}

		$client_id   = isset( $tokenData['client_id'] ) ? $tokenData['client_id'] : null;
		$private_key = wpoauth_get_private_server_key();
		$algorithm   = 'RS256';

		// now that we have the client_id, verify the token
		if ( false === $this->encryptionUtil->decode( $oauth_token, $public_key, true ) ) {
			return false;
		}

		return $tokenData;
	}

	public function setAccessToken( $oauth_token, $client_id, $user_id, $expires, $scope = null ) {
		if ( $this->tokenStorage ) {
			return $this->tokenStorage->setAccessToken( $oauth_token, $client_id, $user_id, $expires, $scope );
		}
	}
}
