/**
 * MetaSlider Lightbox Admin JavaScript
 *
 * Handles the color picker functionality in the admin settings
 */
(function ($) {
    'use strict';

    $(document).ready(function () {
        // Initialize WordPress color picker
        if (typeof $.wp !== 'undefined' && typeof $.wp.wpColorPicker !== 'undefined') {
            $('.ml-color-picker').wpColorPicker({
                defaultColor: false,
                hide: true,
                change: function(event, ui) {
                    // Trigger change event for any listeners
                }
            });
        }

        // Initialize tooltips after a brief delay to ensure color pickers are ready
        // Only initialize if color pickers exist
        if ($('.ml-color-inline').length > 0) {
            setTimeout(initializeTooltips, 100);
        }

        // Enhanced range slider for opacity
        $('input[type="range"]').on('input', function () {
            var $this = $(this);
            var value = $this.val();
            var $output = $this.next('output');
            if ($output.length) {
                $output.text(value);
            }
        });

        // Initialize Select2 function
        function initializeSelect2() {
            if (typeof $.fn.select2 !== 'undefined') {
                var isIncludeMode = $('input[name="metaslider_lightbox_content_options[content_processing_mode]"]:checked').val() === 'include';
                var action = isIncludeMode ? 'include' : 'exclude';

                // Destroy existing Select2 instances if they exist
                if ($('.ml-select2-pages').hasClass('select2-hidden-accessible')) {
                    $('.ml-select2-pages').select2('destroy');
                }
                if ($('.ml-select2-posts').hasClass('select2-hidden-accessible')) {
                    $('.ml-select2-posts').select2('destroy');
                }
                if ($('.ml-select2-post-types').hasClass('select2-hidden-accessible')) {
                    $('.ml-select2-post-types').select2('destroy');
                }

                // Initialize with current mode placeholders
                $('.ml-select2-pages').select2({
                    placeholder: 'Select pages to ' + action + '...',
                    allowClear: true,
                    width: '100%',
                    theme: 'default'
                });

                $('.ml-select2-posts').select2({
                    placeholder: 'Select posts to ' + action + '...',
                    allowClear: true,
                    width: '100%',
                    theme: 'default'
                });

                $('.ml-select2-post-types').select2({
                    placeholder: 'Select post types to ' + action + '...',
                    allowClear: true,
                    width: '100%',
                    theme: 'default'
                });

                // Initialize custom post type selects
                $('.ml-select2-cpt').each(function() {
                    var $select = $(this);
                    var cptLabel = $select.data('cpt-label') || 'items';

                    if ($select.hasClass('select2-hidden-accessible')) {
                        $select.select2('destroy');
                    }

                    $select.select2({
                        placeholder: 'Select ' + cptLabel.toLowerCase() + ' to ' + action + '...',
                        allowClear: true,
                        width: '100%',
                        theme: 'default'
                    });
                });

                // Initialize Manual tab post types select (OPTION 3 - HTML only, no processing yet)
                if ($('.ml-select2-manual-post-types').length > 0) {
                    if ($('.ml-select2-manual-post-types').hasClass('select2-hidden-accessible')) {
                        $('.ml-select2-manual-post-types').select2('destroy');
                    }

                    $('.ml-select2-manual-post-types').select2({
                        placeholder: 'Select post types to exclude',
                        allowClear: true,
                        width: '100%',
                        theme: 'default'
                    });
                }
            }
        }

        // Initialize Select2 on page load
        initializeSelect2();

        // Content processing mode toggle functionality
        function updateInclusionLabels() {
            var isIncludeMode = $('input[name="metaslider_lightbox_content_options[content_processing_mode]"]:checked').val() === 'include';
            var prefix = isIncludeMode ? 'Include' : 'Exclude';

            // Update field titles (case-insensitive search)
            var $pagesTh = $('th').filter(function() { return $(this).text().toLowerCase().indexOf('specific pages') !== -1; });
            if ($pagesTh.length) $pagesTh.text(prefix + ' specific Pages');

            var $postsTh = $('th').filter(function() { return $(this).text().toLowerCase().indexOf('specific posts') !== -1; });
            if ($postsTh.length) $postsTh.text(prefix + ' specific Posts');

            var $postTypesTh = $('th').filter(function() { return $(this).text().toLowerCase().indexOf('specific post types') !== -1; });
            if ($postTypesTh.length) $postTypesTh.text(prefix + ' specific post types');

            // Update custom post type labels
            $('.ml-cpt-label').each(function() {
                var $label = $(this);
                var cptName = $label.data('cpt-name');
                var $select = $('.ml-select2-cpt-' + cptName);
                if ($select.length) {
                    var cptLabel = $select.data('cpt-label');
                    $label.text(prefix + ' specific ' + cptLabel);
                }
            });

            // CSS selector field stays as "Exclude" regardless of mode (makes more sense conceptually)
            var $cssTh = $('th').filter(function() { return $(this).text().indexOf('by CSS selector') !== -1; });
            if ($cssTh.length) $cssTh.text('Exclude by CSS selector');

            // Toggle descriptions
            $('.ml-mode-description').hide();
            $('#' + (isIncludeMode ? 'include' : 'exclude') + '-description').show();

            // Update field descriptions
            $('.ml-filter-description').each(function() {
                var $desc = $(this);
                var excludeText = $desc.data('exclude-text');
                var includeText = $desc.data('include-text');

                if (excludeText && includeText) {
                    $desc.text(isIncludeMode ? includeText : excludeText);
                }
            });

            // Reinitialize Select2 with updated placeholders
            initializeSelect2();
        }

        // Handle toggle button clicks (only if they exist)
        if ($('.ml-toggle-button').length > 0) {
            $('.ml-toggle-button').click(function() {
                var mode = $(this).data('mode');
                $(this).find('input[type=radio]').prop('checked', true);

                // Update button styles
                $('.ml-toggle-button').removeClass('active').css({
                    'background': '#ddd',
                    'color': '#646970'
                });

                $(this).addClass('active').css({
                    'background': '#dd6923',
                    'color': '#fff'
                });

                updateInclusionLabels();
            });

            // Handle hover effects
            $('.ml-toggle-button').hover(
                function() {
                    if (!$(this).hasClass('active')) {
                        $(this).css('background', '#c9c9c9');
                    } else {
                        $(this).css('background', '#c55a1e');
                    }
                },
                function() {
                    if (!$(this).hasClass('active')) {
                        $(this).css('background', '#ddd');
                    } else {
                        $(this).css('background', '#dd6923');
                    }
                }
            );

            // Initialize labels on page load
            updateInclusionLabels();
        }

        // Handle toggle switch clicks (event delegation for dynamically created elements)
        $(document).on('click', '.ml-toggle-switch', function() {
            var targetId = $(this).data('target');
            if (targetId) {
                document.getElementById(targetId).click();
            }
        });

        // Tooltip functionality for color pickers
        var $tooltip = null;

        function createTooltip() {
            if (!$tooltip) {
                $tooltip = $('<div class="ml-tooltip"></div>').appendTo('body');
            }
            return $tooltip;
        }

        function showTooltip($element) {
            var tooltipText = $element.attr('data-tooltip');
            if (!tooltipText) return;

            var $tip = createTooltip();
            $tip.text(tooltipText);

            // Position tooltip above the color picker button
            var $button = $element.siblings('.wp-color-result');
            if ($button.length) {
                var offset = $button.offset();
                var buttonWidth = $button.outerWidth();
                var tipWidth = $tip.outerWidth();

                $tip.css({
                    top: offset.top - $tip.outerHeight() - 10,
                    left: offset.left + (buttonWidth / 2) - (tipWidth / 2)
                }).addClass('show');
            }
        }

        function hideTooltip() {
            if ($tooltip) {
                $tooltip.removeClass('show');
            }
        }

        function initializeTooltips() {
            // Show tooltip on hover over color picker button
            $(document).on('mouseenter', '.ml-color-inline .wp-color-result', function() {
                var $input = $(this).siblings('.ml-has-tooltip');
                if ($input.length) {
                    showTooltip($input);
                }
            });

            $(document).on('mouseleave', '.ml-color-inline .wp-color-result', function() {
                hideTooltip();
            });

            // Hide tooltip when color picker opens
            $(document).on('click', '.ml-color-inline .wp-color-result', function() {
                hideTooltip();
            });

            console.log('Tooltips initialized. Found color pickers:', $('.ml-color-inline .wp-color-result').length);
        }
    });

})(jQuery);