<?php
/**
 * The sitemap generator functionality of the plugin.
 *
 * @link       https://searchatlas.com
 * @since      1.0.0
 *
 * @package    Metasync
 * @subpackage Metasync/sitemap
 */

/**
 * The sitemap generator class.
 *
 * Handles XML sitemap generation and management.
 *
 * @package    Metasync
 * @subpackage Metasync/sitemap
 * @author     Engineering Team <support@searchatlas.com>
 */
class Metasync_Sitemap_Generator
{
    /**
     * The sitemap file path
     *
     * @var string
     */
    private $sitemap_path;

    /**
     * The sitemap URL
     *
     * @var string
     */
    private $sitemap_url;

    /**
     * Initialize the class and set its properties.
     */
    public function __construct()
    {
        $this->sitemap_path = ABSPATH . 'sitemap.xml';
        $this->sitemap_url = home_url('/sitemap.xml');

        // Disable WordPress core sitemap if option is set
        if (get_option('metasync_disable_wp_sitemap', false)) {
            add_filter('wp_sitemaps_enabled', '__return_false', 10);
        }
    }

    /**
     * Generate the XML sitemap
     *
     * @return bool|WP_Error True on success, WP_Error on failure
     */
    public function generate_sitemap()
    {
        try {
            // Get all published posts, pages, and custom post types
            $post_types = get_post_types(['public' => true], 'names');

            // Exclude certain post types if needed
            $excluded_types = [
                'attachment',
                'revision',
                'nav_menu_item',
                'elementor_library',      // Elementor templates/sections
                'ct_template',            // Oxygen Builder templates
                'oxy_user_library',       // Oxygen library items
                'brizy-template',         // Brizy templates
                'fusion_template',        // Avada templates
                'fusion_tb_section',      // Avada theme builder sections
                'ae_global_templates',    // Anywhere Elementor templates
                'custom_css',             // Custom CSS
                'customize_changeset',    // Customizer changesets
                'oembed_cache',          // oEmbed cache
                'user_request',          // Privacy/GDPR requests
                'wp_block',              // Reusable blocks (might want to keep this)
                'wp_template',           // FSE templates
                'wp_template_part',      // FSE template parts
                'wp_global_styles',      // FSE global styles
                'wp_navigation',         // FSE navigation
                'acf-field-group',       // ACF field groups
                'acf-field',             // ACF fields
            ];
            $post_types = array_diff($post_types, $excluded_types);

            $args = [
                'post_type' => $post_types,
                'post_status' => 'publish',
                'posts_per_page' => -1,
                'orderby' => 'modified',
                'order' => 'DESC',
            ];

            $posts = get_posts($args);

            // Start XML generation
            $xml = new DOMDocument('1.0', 'UTF-8');
            $xml->formatOutput = true;

            // Create urlset element
            $urlset = $xml->createElement('urlset');
            $urlset->setAttribute('xmlns', 'http://www.sitemaps.org/schemas/sitemap/0.9');
            $urlset->setAttribute('xmlns:xsi', 'http://www.w3.org/2001/XMLSchema-instance');
            $urlset->setAttribute('xsi:schemaLocation', 'http://www.sitemaps.org/schemas/sitemap/0.9 http://www.sitemaps.org/schemas/sitemap/0.9/sitemap.xsd');
            $xml->appendChild($urlset);

            // Add homepage
            $this->add_url_to_sitemap($xml, $urlset, home_url('/'), get_lastpostmodified('gmt'), '1.0', 'daily');

            // Add posts, pages, and custom post types
            foreach ($posts as $post) {
                $permalink = get_permalink($post->ID);
                $modified = $post->post_modified_gmt;

                // Determine priority based on post type
                $priority = '0.8';
                if ($post->post_type === 'page') {
                    $priority = '0.9';
                } elseif ($post->post_type === 'post') {
                    $priority = '0.7';
                }

                // Determine change frequency
                $changefreq = 'weekly';
                if ($post->post_type === 'post') {
                    $changefreq = 'weekly';
                } elseif ($post->post_type === 'page') {
                    $changefreq = 'monthly';
                }

                $this->add_url_to_sitemap($xml, $urlset, $permalink, $modified, $priority, $changefreq);
            }

            // Add taxonomies (categories, tags, etc.)
            $taxonomies = get_taxonomies(['public' => true], 'names');
            foreach ($taxonomies as $taxonomy) {
                $terms = get_terms([
                    'taxonomy' => $taxonomy,
                    'hide_empty' => true,
                ]);

                if (!is_wp_error($terms) && !empty($terms)) {
                    foreach ($terms as $term) {
                        $term_link = get_term_link($term);
                        if (!is_wp_error($term_link)) {
                            // Get the last modified date of the most recent post in this term
                            $lastmod = $this->get_term_last_modified($term->term_id, $taxonomy);
                            $this->add_url_to_sitemap($xml, $urlset, $term_link, $lastmod, '0.6', 'weekly');
                        }
                    }
                }
            }

            // Save the XML file
            $saved = $xml->save($this->sitemap_path);

            if ($saved === false) {
                return new WP_Error('sitemap_save_failed', 'Failed to save sitemap file. Check file permissions.');
            }

            // Update last generation time
            update_option('metasync_sitemap_last_generated', current_time('mysql'));

            return true;

        } catch (Exception $e) {
            return new WP_Error('sitemap_generation_failed', $e->getMessage());
        }
    }

    /**
     * Get the last modified date for a taxonomy term
     *
     * @param int $term_id The term ID
     * @param string $taxonomy The taxonomy name
     * @return string The last modified date in MySQL format (GMT)
     */
    private function get_term_last_modified($term_id, $taxonomy)
    {
        // Get the most recently modified post in this term
        $recent_post = get_posts([
            'tax_query' => [
                [
                    'taxonomy' => $taxonomy,
                    'field' => 'term_id',
                    'terms' => $term_id,
                ],
            ],
            'post_status' => 'publish',
            'posts_per_page' => 1,
            'orderby' => 'modified',
            'order' => 'DESC',
        ]);

        if (!empty($recent_post)) {
            return $recent_post[0]->post_modified_gmt;
        }

        // Fallback to current time if no posts found
        return current_time('mysql', 1);
    }

    /**
     * Add URL to sitemap
     *
     * @param DOMDocument $xml The XML document
     * @param DOMElement $urlset The urlset element
     * @param string $loc The URL location
     * @param string $lastmod The last modified date
     * @param string $priority The priority
     * @param string $changefreq The change frequency
     */
    private function add_url_to_sitemap($xml, $urlset, $loc, $lastmod, $priority, $changefreq)
    {
        $url = $xml->createElement('url');

        $loc_element = $xml->createElement('loc', esc_url($loc));
        $url->appendChild($loc_element);

        if (!empty($lastmod)) {
            $lastmod_formatted = date('Y-m-d\TH:i:s+00:00', strtotime($lastmod));
            $lastmod_element = $xml->createElement('lastmod', $lastmod_formatted);
            $url->appendChild($lastmod_element);
        }

        $changefreq_element = $xml->createElement('changefreq', $changefreq);
        $url->appendChild($changefreq_element);

        $priority_element = $xml->createElement('priority', $priority);
        $url->appendChild($priority_element);

        $urlset->appendChild($url);
    }

    /**
     * Check if sitemap exists
     *
     * @return bool
     */
    public function sitemap_exists()
    {
        return file_exists($this->sitemap_path);
    }

    /**
     * Get sitemap content
     *
     * @return string|false
     */
    public function get_sitemap_content()
    {
        if ($this->sitemap_exists()) {
            return file_get_contents($this->sitemap_path);
        }
        return false;
    }

    /**
     * Get sitemap URL
     *
     * @return string
     */
    public function get_sitemap_url()
    {
        return $this->sitemap_url;
    }

    /**
     * Get sitemap file size
     *
     * @return int|false
     */
    public function get_sitemap_size()
    {
        if ($this->sitemap_exists()) {
            return filesize($this->sitemap_path);
        }
        return false;
    }

    /**
     * Get last generation time
     *
     * @return string|false
     */
    public function get_last_generated_time()
    {
        return get_option('metasync_sitemap_last_generated', false);
    }

    /**
     * Count URLs in sitemap
     *
     * @return int
     */
    public function count_urls()
    {
        if (!$this->sitemap_exists()) {
            return 0;
        }

        $content = $this->get_sitemap_content();
        if (!$content) {
            return 0;
        }

        return substr_count($content, '<url>');
    }

    /**
     * Delete sitemap file
     *
     * @return bool
     */
    public function delete_sitemap()
    {
        if ($this->sitemap_exists()) {
            return unlink($this->sitemap_path);
        }
        return false;
    }

    /**
     * Check if other sitemap plugins are active
     *
     * @return array Array of active sitemap plugins
     */
    public function check_active_sitemap_plugins()
    {
        // Ensure plugin.php is loaded for is_plugin_active()
        if (!function_exists('is_plugin_active')) {
            require_once ABSPATH . 'wp-admin/includes/plugin.php';
        }

        $sitemap_plugins = [
            'wordpress-seo/wp-seo.php' => 'Yoast SEO',
            'all-in-one-seo-pack/all_in_one_seo_pack.php' => 'All in One SEO',
            'google-sitemap-generator/sitemap.php' => 'Google XML Sitemaps',
            'better-wp-google-xml-sitemaps/bwp-simple-gxs.php' => 'Better WordPress Google XML Sitemaps',
            'xml-sitemap-feed/xml-sitemap.php' => 'XML Sitemap & Google News',
            'sitemap/sitemap.php' => 'Google Sitemap Plugin',
            'rank-math/rank-math.php' => 'Rank Math SEO',
            'seo-by-rank-math/rank-math.php' => 'Rank Math SEO (Free)',
            'wordpress-seo-premium/wp-seo-premium.php' => 'Yoast SEO Premium',
        ];

        $active_plugins = [];
        foreach ($sitemap_plugins as $plugin_path => $plugin_name) {
            if (is_plugin_active($plugin_path)) {
                $active_plugins[$plugin_path] = $plugin_name;
            }
        }

        return $active_plugins;
    }

    /**
     * Disable sitemap generation from other plugins
     *
     * @return bool True if any changes were made
     */
    public function disable_other_sitemap_generators()
    {
        $changes_made = false;

        // Disable Yoast SEO sitemap
        if (class_exists('WPSEO_Options')) {
            $yoast_options = get_option('wpseo');
            if (isset($yoast_options['enable_xml_sitemap']) && $yoast_options['enable_xml_sitemap'] === true) {
                $yoast_options['enable_xml_sitemap'] = false;
                update_option('wpseo', $yoast_options);
                $changes_made = true;
            }
        }

        // Disable All in One SEO sitemap
        if (function_exists('aioseo')) {
            $aioseo_options = get_option('aioseo_options');
            if ($aioseo_options && isset($aioseo_options['sitemap']['general']['enable']) && $aioseo_options['sitemap']['general']['enable'] === true) {
                $aioseo_options['sitemap']['general']['enable'] = false;
                update_option('aioseo_options', $aioseo_options);
                $changes_made = true;
            }
        }

        // Disable Rank Math sitemap
        if (class_exists('RankMath')) {
            $rank_math_options = get_option('rank-math-options-sitemap');
            if ($rank_math_options && isset($rank_math_options['sitemap_disable']) && $rank_math_options['sitemap_disable'] !== 'on') {
                $rank_math_options['sitemap_disable'] = 'on';
                update_option('rank-math-options-sitemap', $rank_math_options);
                $changes_made = true;
            }
        }

        // Disable WordPress core sitemap (WP 5.5+) - Store in option for persistence
        $wp_core_disabled = get_option('metasync_disable_wp_sitemap', false);
        if (!$wp_core_disabled) {
            update_option('metasync_disable_wp_sitemap', true);
            $changes_made = true;
        }

        return $changes_made;
    }

    /**
     * Re-enable sitemap generation from other plugins
     *
     * @return bool True if any changes were made
     */
    public function enable_other_sitemap_generators()
    {
        $changes_made = false;

        // Re-enable Yoast SEO sitemap
        if (class_exists('WPSEO_Options')) {
            $yoast_options = get_option('wpseo');
            if (isset($yoast_options['enable_xml_sitemap']) && $yoast_options['enable_xml_sitemap'] === false) {
                $yoast_options['enable_xml_sitemap'] = true;
                update_option('wpseo', $yoast_options);
                $changes_made = true;
            }
        }

        // Re-enable All in One SEO sitemap
        if (function_exists('aioseo')) {
            $aioseo_options = get_option('aioseo_options');
            if ($aioseo_options && isset($aioseo_options['sitemap']['general']['enable']) && $aioseo_options['sitemap']['general']['enable'] === false) {
                $aioseo_options['sitemap']['general']['enable'] = true;
                update_option('aioseo_options', $aioseo_options);
                $changes_made = true;
            }
        }

        // Re-enable Rank Math sitemap
        if (class_exists('RankMath')) {
            $rank_math_options = get_option('rank-math-options-sitemap');
            if ($rank_math_options && isset($rank_math_options['sitemap_disable']) && $rank_math_options['sitemap_disable'] === 'on') {
                $rank_math_options['sitemap_disable'] = 'off';
                update_option('rank-math-options-sitemap', $rank_math_options);
                $changes_made = true;
            }
        }

        // Re-enable WordPress core sitemap (WP 5.5+)
        $wp_core_disabled = get_option('metasync_disable_wp_sitemap', false);
        if ($wp_core_disabled) {
            delete_option('metasync_disable_wp_sitemap');
            $changes_made = true;
        }

        return $changes_made;
    }

    /**
     * Setup automatic sitemap updates on post changes
     */
    public function setup_auto_update_hooks()
    {
        // Post hooks
        add_action('save_post', [$this, 'auto_update_sitemap'], 10, 2);
        add_action('delete_post', [$this, 'auto_update_sitemap_simple']);
        add_action('trashed_post', [$this, 'auto_update_sitemap_simple']);
        add_action('untrashed_post', [$this, 'auto_update_sitemap_simple']);

        // Term hooks
        add_action('created_term', [$this, 'auto_update_sitemap_simple']);
        add_action('edited_term', [$this, 'auto_update_sitemap_simple']);
        add_action('delete_term', [$this, 'auto_update_sitemap_simple']);
    }

    /**
     * Auto update sitemap on post save
     *
     * @param int $post_id The post ID
     * @param WP_Post $post The post object
     */
    public function auto_update_sitemap($post_id, $post)
    {
        // Check if auto-update is enabled
        if (!get_option('metasync_sitemap_auto_update', false)) {
            return;
        }

        // Avoid auto-updates during autosave, revisions, or drafts
        if (wp_is_post_autosave($post_id) || wp_is_post_revision($post_id)) {
            return;
        }

        // Only update for published posts
        if ($post->post_status !== 'publish') {
            return;
        }

        // Regenerate sitemap
        $this->generate_sitemap();
    }

    /**
     * Auto update sitemap (simple version for hooks without post object)
     */
    public function auto_update_sitemap_simple()
    {
        // Check if auto-update is enabled
        if (!get_option('metasync_sitemap_auto_update', false)) {
            return;
        }

        // Regenerate sitemap
        $this->generate_sitemap();
    }
}
