<?php

/**
 * The site error logs for the plugin.
 *
 * @since      1.0.0
 * @package    Metasync
 * @subpackage Metasync/site-error-logs
 * @author     Engineering Team <support@searchatlas.com>
 */
class Metasync_Error_Logs
{
	/**
	 * The log file path
	 * @var string
	 */
	private $log_file_path = null;

	/**
	 * The error message if log cannot be displayed
	 * @var string
	 */
	private $error_message = '';

	/**
	 * Show copy button.
	 */
	public function show_copy_button()
	{
?>
		<div>
			<button type="button" class="button" id="copy-clipboard-btn">
				Logs Copy to Clipboard
			</button>
			<span class="hide">Copied!</span>
		</div>
	<?php
	}

	/**
	 * Show information about the error log file.
	 */
	public function show_info()
	{
		// Use the stored log file path
		$log_file = $this->get_actual_log_path();
		$wp_filesystem = $this->wp_filesystem();
		$size = $wp_filesystem->size($log_file);
	?>
		<div>
			<code><?php echo esc_html(basename($log_file)); ?></code>
			<em>(<?php echo esc_html($this->get_human_number($size)); ?>)</em>
		</div>
	<?php
	}

	/**
	 * Show strong message.
	 */
	public function show_strong_message($message)
	{
	?>
		<strong class="error-log-cannot-display"><?php echo esc_html($message) ?></strong><br>
	<?php
	}

	/**
	 * Show strong message.
	 */
	public function show_code_html($code)
	{
	?>
		<code><?php echo esc_html($code); ?></code>
	<?php
	}

	/**
	 * Show error log in text area.
	 */
	public function show_logs()
	{
	?>
		<div>
			<textarea rows="20" class="code-box" id="error-code-box" disabled><?php echo esc_textarea($this->get_error_logs(50)); ?></textarea>
		</div>
<?php
	}

	/**
	 * Show message if the log cannot be established.
	 */
	public function can_show_error_logs()
	{
		// Reset error message
		$this->error_message = '';
		
		// Create the error log file first
		$file_path = $this->createErrorLog();
		
		// Store the log file path for use in other methods
		$this->log_file_path = $file_path;
		
		// Use the created log file path instead of ini_get('error_log')
		$log_file = $file_path;

		if (empty($file_path)) {
			$this->error_message = 'Unable to create error log file.';
			return false;
		}

		$wp_filesystem = $this->wp_filesystem();
		
		if (is_null($wp_filesystem)) {
			$this->error_message = 'WordPress filesystem not available.';
			return false;
		}

		if (!$wp_filesystem->exists($log_file)) {
			$this->error_message = 'Error log file does not exist: ' . $log_file;
			return false;
		}

		if (!$wp_filesystem->is_readable($log_file)) {
			$this->error_message = 'Error log file is not readable.';
			return false;
		}

		// Error log must be smaller than 100 MB.
		$size = $wp_filesystem->size($log_file);
		if ($size > 100000000) {
			$wp_filesystem->delete($log_file);
			$this->error_message = 'The error log cannot be retrieved: Error log file is too large.';
			return false;
		}

		return true;
	}

	/**
	 * Get the error message if log cannot be displayed.
	 * @return string
	 */
	public function get_error_message()
	{
		return $this->error_message;
	}

	/**
	 * WordPress filesystem for use.
	 *
	 * @return string
	 */
	public function get_error_logs($limit = -1)
	{
		$wp_filesystem = $this->wp_filesystem();
		
		// Use the actual log file path
		$log_file = $this->get_actual_log_path();
		
		if (empty($log_file) || !$wp_filesystem->exists($log_file)) {
			return '';
		}
		
		$contents = $wp_filesystem->get_contents_array($log_file);

		if (-1 === $limit) {
			return join('', $contents);
		}

		return join('', array_slice($contents, -$limit));
	}

	/**
	 * WordPress filesystem for use.
	 *
	 * @return object
	 */
	private function wp_filesystem()
	{
		global $wp_filesystem;

		if (empty($wp_filesystem)) {
			require_once ABSPATH . '/wp-admin/includes/file.php';
			WP_Filesystem();
		}

		return $wp_filesystem;
	}

	/**
	 * Get error log file location.
	 *
	 * @return string
	 */
	private function get_log_path()
	{
		return ini_get('error_log') != '' ? ini_get('error_log') : '';
	}

	/**
	 * Get the actual log file path (either stored or create a new one).
	 *
	 * @return string
	 */
	private function get_actual_log_path()
	{
		// If we have a stored log file path, use it
		if (!empty($this->log_file_path)) {
			return $this->log_file_path;
		}
		
		// Otherwise, try to get the default log path
		$log_file = $this->get_log_path();
		
		// If default path is empty or doesn't exist, create one
		if (empty($log_file)) {
			$log_file = $this->createErrorLog();
			$this->log_file_path = $log_file;
		}
		
		return $log_file;
	}

	/**
	 * Clear the log.
	 * @return void
	 */
	public static function clear()
	{
		if (ini_get('error_log') === '') return;
		$handle = fopen(ini_get('error_log'), 'w');
		fclose($handle);
	}

	/**
	 * Get human read number of units.
	 *
	 * @param string $bytes
	 * @return string
	 */
	public function get_human_number(string $bytes)
	{
		if ($bytes >= 1073741824) {
			$bytes = number_format($bytes / 1073741824, 2) . ' GB';
		} elseif ($bytes >= 1048576) {
			$bytes = number_format($bytes / 1048576, 2) . ' MB';
		} elseif ($bytes >= 1024) {
			$bytes = number_format($bytes / 1024, 2) . ' KB';
		} elseif ($bytes > 1) {
			$bytes = $bytes . ' bytes';
		} elseif ($bytes == 1) {
			$bytes = $bytes . ' byte';
		} else {
			$bytes = '0 bytes';
		}

		return $bytes;
	}

	public function createErrorLog()
	{
		$directories = array(
			WP_CONTENT_DIR,
		);

		$log_directory = null;
		foreach ($directories as $log_directory) {
			$dir = wp_normalize_path($log_directory . DIRECTORY_SEPARATOR . 'metasync-error-logs');
			if (is_dir($dir)) {
				$log_directory = $dir;
				break;
			} else {
				if (@mkdir($dir, 0770)) {
					$log_directory = $dir;
					break;
				}
			}
		}

		$log_file = wp_normalize_path($log_directory . DIRECTORY_SEPARATOR . 'php-errors.log');
		if (!file_exists($log_file)) {
			@file_put_contents($log_file, '');
			@chmod($log_file, 0660);
		}

		return $log_file;
	}
}
