<?php
/**
 * This class handles the Otto Pixel Functions
 */
Class Metasync_otto_pixel{
    
    #otto html class
    public $o_html;

    # crawl data option
    public $option_name = 'metasync_otto_crawldata';

	# no cache wp pages 
	public $no_cache_pages = ['wp-login.php'];

    # 
    function __construct($otto_uuid){
        
        # load the html class
        $this->o_html = new Metasync_otto_html($otto_uuid);

    }

    # method to handle cache refresh
    # NOTE: Cache system removed - always process in real-time
    function refresh_cache($route){
        # No-op: Cache system has been removed
        # All pages are processed in real-time from OTTO API
        return true;
    }


    # method to save crawl data into
    function save_crawl_data($data){

        # get the option name 
        $option_name = $this->option_name;

        # handle data
        $saved = get_option($option_name);

        # log saved

        # if saved false save
        if(empty($saved['urls'])){
            
            # save the option
            update_option($option_name, $data);

            return;
        }

        # get new unique list of urls
        $new_list = array_unique(array_merge($data['urls'], $saved['urls']));

        # set data
        $data['urls'] = $new_list;

        # log saved

        # save the option 
        update_option($option_name, $data);
    }

    /**
     * Check if a URL has been crawled by Otto
     * - Validates domain against saved domain
     * - Ignores query strings
     * - Does exact path matching as Otto stores paths exactly as they appear
     * @param string $url - Full URL to check
     * @return bool - True if URL was crawled, false otherwise
     */
     function is_url_crawled($url) {
        # Ensure valid input
        if (empty($url) || !is_string($url)) {
            return false;
        }

        # Get saved crawl data
        $saved = get_option($this->option_name);

        if (
            empty($saved) ||
            !is_array($saved) ||
            empty($saved['domain']) ||
            empty($saved['urls']) ||
            !is_array($saved['urls'])
        ) {
            return false;
        }

        # Parse saved domain + incoming URL
        $saved_domain   = parse_url($saved['domain'], PHP_URL_HOST);
        $incoming_domain = parse_url($url, PHP_URL_HOST);

        # Domain mismatch - not crawled
        if (empty($saved_domain) || empty($incoming_domain) || strcasecmp($saved_domain, $incoming_domain) !== 0) {
            return false;
        }

        # Parse incoming path (ignore query string)
        $parsed_url = parse_url($url);
        $url_path   = $parsed_url['path'] ?? '/';

        # Ensure path starts with / but don't modify trailing slashes
        # Otto stores paths exactly as they appear in URLs
        if (substr($url_path, 0, 1) !== '/') {
            $url_path = '/' . $url_path;
        }

        # Compare against crawled URLs - exact match
        foreach ($saved['urls'] as $crawled_url) {
            if (!is_string($crawled_url)) {
                continue;
            }

            # Direct comparison - Otto stores paths exactly as they are
            if (strcasecmp($crawled_url, $url_path) === 0) {
                return true;
            }
        }

        return false;
    }

    # get the current route
    function get_route(){

         # check if we're in an HTTP context
        if(empty($_SERVER['HTTP_HOST']) || empty($_SERVER['REQUEST_URI'])){
            # not in HTTP context (CLI, cron, etc.), return false
            return false;
        }
        
        # get req scheme
        $scheme = ( is_ssl() ? 'https' : 'http' );
       
        # get req host
        $host = $_SERVER['HTTP_HOST'];
        
        # get the uri
        $request_uri = $_SERVER['REQUEST_URI'];
    
        # return the formatted url
        return $scheme . '://' . $host . $request_uri;
    }

    # get the html for a route 
    # OPTION 1 IMPLEMENTATION: Use transient cache for suggestions
    function get_route_html($route, $cache_track_key = null, $suggestions = null){
        # If suggestions already provided (from render_route_html), use them directly
        if ($suggestions !== null && is_array($suggestions)) {
            # Process route with provided suggestions data
            return $this->o_html->process_route_with_data($route, $suggestions, '');
        }
        
        # Get OTTO UUID from options
        global $metasync_options;
        $otto_uuid = $metasync_options['general']['otto_pixel_uuid'] ?? '';
        
        if (empty($otto_uuid)) {
            return false;
        }
        
        # Get suggestions from transient cache (with API fallback)
        $transient_cache = new Metasync_Otto_Transient_Cache($otto_uuid);
        $track_key = $cache_track_key ?: md5($route);
        $suggestions = $transient_cache->get_suggestions($route, $track_key);
        
        if (!$suggestions || !$transient_cache->has_payload($suggestions)) {
            # No suggestions available
            return false;
        }
        
        # Process route with cached suggestions data
        return $this->o_html->process_route_with_data($route, $suggestions, '');
    }


    # render route html
    function render_route_html(){

        # Disable SG Cache for Brizy pages FIRST - before any other processing
        # Using global function defined in otto_pixel.php
        if (function_exists('metasync_otto_disable_sg_cache_for_brizy')) {
            metasync_otto_disable_sg_cache_for_brizy();
        }

        # get the route
        $route = $this->get_route();

        # get the current page from globas
        # this is to help us exclude the login page

        $page_now = $GLOBALS['pagenow'] ?? false;

        # check whether page now in excluded pages
        if(in_array($page_now , $this->no_cache_pages)){

            # stop otto
            return;
        }

        /*
        #uncomment to test on local hosts
        if(!empty($_GET['otto_test'])){

            # @dev
            $route = 'https://staging-perm.wp65.qa.internal.searchatlas.com/';
        }
        */

        # OPTION 1 IMPLEMENTATION: Check transient cache instead of notification data
        # This makes the system self-healing - works even if notifications fail
        
        # Get OTTO UUID from options
        global $metasync_options;
        $otto_uuid = $metasync_options['general']['otto_pixel_uuid'] ?? '';
        
        if (empty($otto_uuid)) {
            return;
        }
        
        $transient_cache = new Metasync_Otto_Transient_Cache($otto_uuid);
        
        # Create tracking key for cache status
        $cache_track_key = 'otto_' . md5($route);
        
        # Check if URL has OTTO suggestions (checks transient, calls API if needed)
        # Use get_suggestions directly to track cache status
        $suggestions = $transient_cache->get_suggestions($route, $cache_track_key);
        
        if (!$suggestions || !$transient_cache->has_payload($suggestions)) {
            # No suggestions available for this URL
            # Set header to indicate cache status
            if (!headers_sent()) {
                $cache_status = Metasync_Otto_Transient_Cache::get_cache_status($cache_track_key);
                header('X-MetaSync-OTTO-Cache: ' . ($cache_status ?: 'NO_SUGGESTIONS'));
            }
            return;
        }

        # comment out for fixing pagination issues
        # $route = rtrim($route, '/');
        

        # check if we have the route html (pass suggestions to avoid duplicate API call)
        $route_html = $this->get_route_html($route, $cache_track_key, $suggestions);

        # check that route html is valid
        if(empty($route_html)){
            return false;
        }

        $route_html_string = $route_html->__toString();
        
        if (strpos($route_html_string, 'pix-sliding-headline-2') !== false || strpos($route_html_string, 'pix-intro-sliding-text') !== false) {
            # Only apply fix within sliding text contexts to avoid breaking other layouts
            $route_html_string = preg_replace('#(</span></span>)(<span\s+class=["\'][^"\']*slide-in-container[^"\']*["\'][^>]*>)#i', '$1 $2', $route_html_string);
        }

        # Check for Revolution Slider to determine if special handling is needed
        # Check for both Revolution Slider 6 (<rs-module-wrap>) and Revolution Slider 7 (<sr7-module>)
        $has_revslider = (strpos($route_html_string, '<rs-module-wrap') !== false || strpos($route_html_string, '<sr7-module') !== false);
        
        if($has_revslider){
            # Revolution Slider detected - fire WordPress hooks to ensure proper initialization
            # Use output buffering to prevent hooks from corrupting Otto's processed HTML
            ob_start();
            do_action('wp_enqueue_scripts');
            $discarded_output = ob_get_clean();
           
        }
        
        # Performance optimization: Add cache headers for browser and CDN caching
        # Tell browsers to cache for 1 hour, CDNs for 24 hours
        $cache_duration = 3600; // 1 hour for browsers
        $cdn_cache_duration = 86400; // 24 hours for CDNs

        header('Cache-Control: public, max-age=' . $cache_duration . ', s-maxage=' . $cdn_cache_duration);
        header('Vary: Accept-Encoding'); // Support gzip compression

        # Tell Cloudflare to cache but not transform
        header('CF-Cache-Status: HIT');
        
        # OPTION 1: Add MetaSync OTTO cache status header for debugging/monitoring
        # This header indicates whether suggestions came from cache or API
        $cache_status = Metasync_Otto_Transient_Cache::get_cache_status($cache_track_key);
        if (!empty($cache_status)) {
            header('X-MetaSync-OTTO-Cache: ' . $cache_status);
        } else {
            # Fallback if status not tracked (shouldn't happen, but safety)
            header('X-MetaSync-OTTO-Cache: UNKNOWN');
        }

        # continue to render the html
        echo $route_html_string;

        # prevent further wp execution
        exit();
    }

}