<?php

# uses the simple html dom library
use simplehtmldom\HtmlWeb;
use simplehtmldom\HtmlDocument;

/**
 * This Class Handles  
 */

Class Metasync_otto_html{

    # html dom
    private $dom; 

    # the file path to save to
    private $html_file;

    # uuid of site
    private $site_uuid;

    # the otto endpoint url
    private $otto_end_point = 'https://sa.searchatlas.com/api/v2/otto-url-details';
    
    # 
    function __construct($otto_uuid){

        # set the site uuid using the provided string
        $this->site_uuid = $otto_uuid;

        # laod the simple html dom parser with UTF-8 charset to handle special characters
        $this->dom = new HtmlDocument(null, true, true, 'UTF-8');
    }

    /**
     * Check Route Method
     * @param route : The route to check
     * @param path : The path of the html file to save
     */
    function process_route($route, $file_path){

        # Construct the full endpoint URL with query parameters
        $url_with_params = add_query_arg(
            [
                'url'  => $route,
                'uuid' => $this->site_uuid,
            ],
            $this->otto_end_point
        );

        # PERFORMANCE FIX: Add timeout to prevent blocking
        $args = array(
            'timeout' => 5, // 5 second max timeout (allow time for redirects)
            'redirection' => 5, // CRITICAL FIX: Allow redirects (API returns 301)
            'user-agent' => 'MetaSync-OTTO-SSR/2.0',
            'sslverify' => true
        );

        # Perform the GET request with timeout
        $response = wp_remote_get($url_with_params, $args);

        # Check for errors
        if (is_wp_error($response)) {
            error_log('MetaSync OTTO: API call failed - ' . $response->get_error_message());
            return false;
        }

        # get the response body
        $body = wp_remote_retrieve_body($response);

        # Get the response code
        $response_code = wp_remote_retrieve_response_code($response);

        # if no change data skip
        if (empty($body) || $response_code !== 200){
            error_log('MetaSync OTTO: API returned empty or non-200. Code: ' . $response_code);
            return false;
        }

        # set the html file path
        $this->html_file = $file_path;

        # load change data
        $change_data = json_decode($body, true);
        
        # Process with the fetched data
        return $this->process_route_with_data($route, $change_data, $file_path);
    }
    
    /**
     * Process route with pre-fetched suggestions data
     * OPTION 1: Used when data comes from transient cache
     * @param route : The route to check
     * @param change_data : Pre-fetched OTTO suggestions data
     * @param path : The path of the html file to save
     */
    function process_route_with_data($route, $change_data, $file_path){
        
        if (empty($change_data) || !is_array($change_data)) {
            return false;
        }
        
        # set the html file path
        $this->html_file = $file_path;
        
        # Analyze what Otto is providing and store for conditional SEO blocking
        $has_otto_title = false;
        $has_otto_description = false;
        
        if (!empty($change_data['header_replacements']) && is_array($change_data['header_replacements'])) {
            foreach ($change_data['header_replacements'] as $item) {
                if (!empty($item['type'])) {
                    # Check if Otto has title
                    if ($item['type'] == 'title' && !empty($item['recommended_value'])) {
                        $has_otto_title = true;
                    }
                    # Check if Otto has description
                    if ($item['type'] == 'meta') {
                        if ((!empty($item['name']) && $item['name'] == 'description' && !empty($item['recommended_value'])) ||
                            (!empty($item['property']) && strpos($item['property'], 'description') !== false && !empty($item['recommended_value']))) {
                            $has_otto_description = true;
                        }
                    }
                }
            }
        }
        
        # Check header_html_insertion for description
        if (!empty($change_data['header_html_insertion'])) {
            if (preg_match('/<meta[^>]*name=["\']description["\'][^>]*>/i', $change_data['header_html_insertion'])) {
                $has_otto_description = true;
            }
        }
        
        # Store blocking flags to pass to handle_route_html
        # This will be added to the internal fetch URL as parameters
        $change_data['_otto_blocking'] = array(
            'block_title' => $has_otto_title,
            'block_description' => $has_otto_description
        );
        
        # Process the route with the suggestions data
        return $this->handle_route_html($route, $change_data);
        
    }

    # function to get tag attributes
    function get_tag_attributes($tag){

        # Extract existing attributes of the <body> tag
        $attributes = [];


        # set the tag attributes
        $tag_attributes = [];
        

        # check that the tag attributes 
        if(!is_object($tag) || !method_exists($tag, 'getAllAttributes')){
            return '';
        }

        # get the tag attributes
        $tag_attributes = $tag->getAllAttributes();  

        # loop all attributes
        foreach ($tag_attributes as $key => $value) {

            if ($value == 1) {

                # Handle boolean attributes
                $attributes[] = htmlspecialchars($key, ENT_QUOTES);
            } else {

                # Handle attributes with values
                $attributes[] = $key . '="' . htmlspecialchars($value, ENT_QUOTES) . '"';
            }
        }

        # Convert attributes array to a string
        $attributes_string = !empty($attributes) ? ' ' . implode(' ', $attributes) : '';

        # return the attributes string
        return $attributes_string;
    }

    function handle_route_html($route, $replacement_data){

        # Detect if current page uses Brizy and disable SG Cache if so
        # Using global function defined in otto_pixel.php
        if (function_exists('metasync_otto_disable_sg_cache_for_brizy')) {
            metasync_otto_disable_sg_cache_for_brizy();
        }

        # lablel the Otto Route
        # label otto requests to avoid loops
        // $request_body = add_query_arg(
        //     [
        //         'is_otto_page_fetch' => 1
        //     ],
        //     $route
        // );
        # Add blocking flags as URL parameters (no database writes!)
        $url_params = ['is_otto_page_fetch' => 1];
        
        # Add blocking flags if available
        if (!empty($replacement_data['_otto_blocking'])) {
            $url_params['otto_block_title'] = $replacement_data['_otto_blocking']['block_title'] ? '1' : '0';
            $url_params['otto_block_desc'] = $replacement_data['_otto_blocking']['block_description'] ? '1' : '0';
        }
        
        $request_body = add_query_arg($url_params, $route);
		# set cookie header var
		$cookie_header = '';
		
		# loop cookies to set header
		foreach ($_COOKIE as $name => $value) {
			
            # handle array values by converting to string
			$cookie_value = is_array($value) ? serialize($value) : $value;

			# add cookie to header
			# $cookie_header .= $name . '=' . $value . '; ';
            $cookie_header .= $name . '=' . $cookie_value . '; ';
		}
		
		# trim the string
		$cookie_header = rtrim($cookie_header, '; ');

		$args = array(
			'sslverify' => false,
			'timeout' => 5, // Very short timeout - fail fast if WP Engine is blocking
			'redirection' => 5,
			'httpversion' => '1.1',
			'headers' => array(
				'Cookie' => $cookie_header,
				'Cache-Control' => 'no-cache, no-store, must-revalidate',
				'Pragma' => 'no-cache',
				'X-OTTO-Internal-Fetch' => '1',
				'User-Agent' => 'MetaSync-OTTO-SSR/3.0',
			)
		);

        # get the associateed route html
        $route_html = wp_remote_get($request_body, $args);

		# Check for timeout or connection errors
		if (is_wp_error($route_html)) {
			$error_msg = $route_html->get_error_message();
			error_log('MetaSync OTTO DEBUG: FAILED - wp_remote_get error: ' . $error_msg . ' for route: ' . $route);
			return false;
		}

        # get body
        $html_body = wp_remote_retrieve_body($route_html);

        # Get the response code
        $response_code = wp_remote_retrieve_response_code($route_html);


        # check not empty
        if(empty($html_body) || $response_code !== 200){
			error_log('MetaSync OTTO DEBUG: FAILED - Empty body or non-200 status for route: ' . $route);
            return false;
        }

        # Remove XML declaration
        $html_body = preg_replace('/<\?xml[^?]*\?>\s*/i', '', $html_body);

        # now that the html is not empty
        # load it into the simple html dom
        $this->dom->load($html_body);

        # Force UTF-8 charset to preserve emojis and special characters
        # This overrides any charset detection from HTML meta tags
        $this->dom->_charset = 'UTF-8';
        $this->dom->_target_charset = 'UTF-8';

        # now lets do the magic

        # start the header html insertion
        $this->insert_header_html($replacement_data);

        # now we do the header replacements
        $this->do_header_replacements($replacement_data);

        # now do the body replacements
        $this->do_body_replacements($replacement_data);

        # now do the footer insertions
        $this->do_footer_html_insertion($replacement_data);

        # final cleanup: ensure metasync_optimized attribute is removed from AMP pages
        $this->cleanup_amp_metasync_attribute();

        # save the document
        $this->save_reload();

        # return the dom html
        return $this->dom;
    }

    # do the footer html insertion
    function do_footer_html_insertion($replacement_data){

        # check that we have footer html
        if(empty($replacement_data['footer_html_insertion'])){
            return;
        }

        # otherwise do the replacement
        $footer = $this->dom->find('footer', 0);

        # check that footer is object
        if(!is_object($footer) || !isset($footer->innertext, $footer->outertext)){
            return;
        }

        # get the tag attributes
        $attributes_string = $this->get_tag_attributes($footer);

        # now do the actual html replacements
        $footer->outertext = '<footer 7' . $attributes_string . '>' . $footer->innertext . $replacement_data['footer_html_insertion'].'</footer>';

        # save the document
        $this->save_reload();
    }

    # do body replacements
    function do_body_replacements($replacement_data){

        # start body top html replacements
        $this->do_body_top_html($replacement_data);
    
        # start the body bottom html replacements
        $this->do_body_bottom_html($replacement_data);

        # do the body substitutions
        $this->do_body_substitutions($replacement_data);
    }

    # body substitutions data
    function do_body_substitutions($replacement_data){

        # check that we have an array of substitutions
        if(empty($replacement_data['body_substitutions']) || !is_array($replacement_data['body_substitutions'])){

            #
            return;
        }

        # now work on different substitution keys
        foreach ($replacement_data['body_substitutions'] as $key => $value) {
            # check key categories
            if($key == 'images'){

                # do image replacements
                $this->handle_images($value);
            }
            elseif($key == 'headings'){
                
                # do heading repalcements
                $this->do_heading_body_substitutions($value); 
            }
            elseif($key == 'links'){

                # do link replacements
                $this->do_link_body_substitutions($value);
            }

        }

        # save the document
        $this->save_reload();

    }

    /**
     * START BODY SUBSTITUTION FUNCTIONS
     * @see do_body_substitutions();
     */

    # image substitions 
    function handle_images($image_data){

        # find all images in dom
        $images = $this->dom->find('img');
    
        # loop images and handle alt
        foreach($images AS $key => $image){

            # loop image data to identify matching src
            foreach ($image_data as $src => $alt_text) {
                
                # check if row src matches image srce
                if($src == $image->src){

                    # set the alt text
                    $image->alt = $alt_text;
                }
            }
        }
    }

    # heading substitutions
    function do_heading_body_substitutions($heading_data){

        # loop all data
        foreach($heading_data AS $okey => $heading){

            # find all occurences of the heading type
            $occurences = $this->dom->find($heading['type']);

            # loop all occurences
            foreach ($occurences as $ikey => $heading_old) {
                
                # get the header text
                $text = $heading_old->text();

                # check matching text
                if(trim($heading['current_value']) == trim($text)){

                    # set the text
                    $heading_old->innertext = $heading['recommended_value'];

                }
            }

        }

        # save and reload the dom
        $this->save_reload();
    }

    # link replacements
    function do_link_body_substitutions($swap_data){

        # find all links in the body
        $links = $this->dom->find('a');

        # loop all links check if we have them in swap data 
        foreach ($links as $key => $link) {
            
            # check if link matches
            if(!empty($swap_data[$link->href])){
                
                # replace the link
                $link->href = $swap_data[$link->href];
            }

        }
    }

    # body bottom html replacement code
    function do_body_bottom_html($insert_data){

        # check that data is availbale
        if(empty($insert_data['body_bottom_html_insertion'])){
            return;
        }

        # otherwise do the replacement
        $body = $this->dom->find('body', 0);

        # set the link property if not empty
        if(empty($body->outertext)){
            return;
        }


        # get the tag attributes
        $attributes_string = $this->get_tag_attributes($body);

        # now do the actual html replacements
        $body->outertext = '<body 7' . $attributes_string . '>' . $body->innertext . $insert_data['body_bottom_html_insertion'].'</body>';

        # save the document
        $this->save_reload();
    }

    # body top html replacement
    function do_body_top_html($insert_data){

        # check that data is availbale
        if(empty($insert_data['body_top_html_insertion'])){
            return;
        }

        # otherwise do the replacement
        $body = $this->dom->find('body', 0);

        # get the tag attributes
        $attributes_string = $this->get_tag_attributes($body);

        # now do the actual html replacements
        $body->outertext = '<body 5' . $attributes_string . '>'.$insert_data['body_top_html_insertion'].$body->innertext . '</body>';
    }

    # this function does the header replacements
    function do_header_replacements($replacement_data){

        # check that we have header replacements
        if(empty($replacement_data['header_replacements']) || !is_array($replacement_data['header_replacements'])){
            return;
        }        

        # now lets do the replacement work
        foreach($replacement_data['header_replacements'] AS $key => $data){

            # skip cases where type is not specified
            if(empty($data['type'])){
                continue;
            }

            # handle title
            if($data['type'] == 'title'){

                # handle the title logic
                $this->replace_title($data);
                
                #
                continue;
            }

            # handle canonical links
            if($data['type'] == 'link' && $data['rel'] === 'canonical'){

                # find the cannonical dom element
                $link = $this->dom->find('link[rel="canonical"]', 0);

                # set the link property if not empty
                if(!empty($link->href)){
                    $link->href = $data['recommended_value'] ?? $link->href;
                }

                # 
                continue;
            }

            # work on other elemenets not titlte
            $this->handle_meta_element($data);
        }
    }

    # function to handle meta elements other than title
    function handle_meta_element($data){

        # extract property value
        $property = $data['property'] ?? false;

        # extract name value
        $name = $data['name'] ?? false;

        # set the selector
        $meta_selector = '';

        # extend selector 
        if(!empty($name)){
            $meta_selector .=   "meta[name=".trim($name)."],";
        }

        # extent if property is defined
        if(!empty($property)){
            $meta_selector .= "meta[property=".trim($property)."]";
        }

        # find the meta gat in the dom
        $meta_tag = $this->dom->find($meta_selector, 0);

        # if tag not exists add it
        if(empty($meta_tag)){
            if($data['type'] == 'meta'){
            
                # get the attribute 
                $attribute = $property ? 'property' : 'name';
                
                # call the create metatag function
                $this->create_metatag($attribute, $data);
            }

            # otherwise reut
            return;
        }

        # do the replacement
        $meta_tag->content = $data['recommended_value'] ?? $meta_tag->content;
        

    }

    # function to handle the page title
    function replace_title($title_data){

        # find the title
        $title = $this->dom->find('title', 0) ?? false;

        # if none
        if($title === false){

            return $this->create_title($title_data);
        }

        # add the recommended value to the title
        $title->innertext = $title_data['recommended_value'];

        # save and reload DOM
        $this->save_reload();
    }

    # Function to create a title when it's missing
    function create_title($title_data) {

        # Find the <head> tag
        $head = $this->dom->find('head', 0);

        # Construct the <title> tag HTML
        $title_html = '<title>' . htmlspecialchars($title_data['recommended_value'], ENT_QUOTES) . '</title>';

        if (empty($head)) {
            return false;
        }

        # Extract existing attributes of the <head> tag
        $attributes = [];

        # get the tag attributes
        $tag_attributes = $head->getAllAttributes();

        # loop all attributes
        foreach ($tag_attributes as $key => $value) {

            if ($value == 1) {

                # Handle boolean attributes
                $attributes[] = htmlspecialchars($key, ENT_QUOTES);
            } else {

                # Handle attributes with values
                $attributes[] = $key . '="' . htmlspecialchars($value, ENT_QUOTES) . '"';
            }
        }

        # Convert attributes array to a string
        $attributes_string = !empty($attributes) ? ' ' . implode(' ', $attributes) : '';

        # Rebuild the <head> tag, inserting the <title> at the beginning
        $head->outertext = '<head 2' . $attributes_string . '>' . $title_html . $head->innertext . '</head>';

        # save and reload DOM
        $this->save_reload();
    }

    # function to create meta tag if none existss
    function create_metatag($attribute, $data){

        # Find the <head> tag
        $head = $this->dom->find('head', 0);

        # Construct the <title> tag HTML
        $meta_tag = '<meta '.$attribute.' = "'.$data[$attribute].'" content = "'.$data['recommended_value'].'">';

        if (empty($head)) {
            return false;
        }

        # Extract existing attributes of the <head> tag
        $attributes = [];

        # get the tag attributes
        $tag_attributes = $head->getAllAttributes();

        # loop all attributes
        foreach ($tag_attributes as $key => $value) {

            if ($value == 1) {

                # Handle boolean attributes
                $attributes[] = htmlspecialchars($key, ENT_QUOTES);
            } else {

                # Handle attributes with values
                $attributes[] = $key . '="' . htmlspecialchars($value, ENT_QUOTES) . '"';
            }
        }

        # Convert attributes array to a string
        $attributes_string = !empty($attributes) ? ' ' . implode(' ', $attributes) : '';

        # Rebuild the <head> tag, inserting the <title> at the beginning
        $head->outertext = '<head 5' . $attributes_string . '>' . $meta_tag . $head->innertext . '</head>';

        # save and reload DOM
        $this->save_reload();
    }

    # function to detect if current page is an AMP page
    function is_amp_page(){
        
        # Check if URL path contains /amp/
        $current_url = $_SERVER['REQUEST_URI'] ?? '';
        if (strpos($current_url, '/amp/') !== false) {
            return true;
        }
        
        # Check if URL ends with /amp
        if (preg_match('/\/amp\/?$/', $current_url)) {
            return true;
        }
        
        # Check if amp=1 query parameter is present
        if (isset($_GET['amp']) && $_GET['amp'] == '1') {
            return true;
        }
        
        # Check for other common AMP query parameters
        if (isset($_GET['amp']) && !empty($_GET['amp'])) {
            return true;
        }
        
        return false;
    }

    # this function insterts header html to the dom
    function insert_header_html($data){

        # check that we have the header html
        if(empty($data['header_html_insertion'])){
            #
            return;
        }

        # append the/ html at the start of the header
        $head = $this->dom->find('head', 0);

        if ($head) {

            # Check if this is an AMP page - if so, don't add metasync_optimized attribute
            $is_amp_page = $this->is_amp_page();

            # Append the new HTML at the start of the <head> tag
            # For AMP pages: use clean <head> tag without metasync_optimized attribute
            # For non-AMP pages: add metasync_optimized attribute to <head> tag
            if ($is_amp_page) {
                $head->outertext = '<head>' .$data['header_html_insertion']. $head->innertext . '</head>';
            } else {
                $head->outertext = '<head metasync_optimized>' .$data['header_html_insertion']. $head->innertext . '</head>';
            }

        }

        # save and reload DOM
        $this->save_reload();
    }

    # function to forcefully remove metasync_optimized attribute from head on AMP pages
    function cleanup_amp_metasync_attribute(){
        
        # Only proceed if this is an AMP page
        if (!$this->is_amp_page()) {
            return;
        }
        
        # Find the head tag
        $head = $this->dom->find('head', 0);
        
        if (!$head) {
            return;
        }
        
        # Check if head has metasync_optimized attribute
        $head_html = $head->outertext;
        
        # If metasync_optimized attribute is found, remove it
        if (strpos($head_html, 'metasync_optimized') !== false) {
            
            # Remove the metasync_optimized attribute from the head tag
            # This handles various formats: <head metasync_optimized>, <head metasync_optimized=""> etc.
            $cleaned_head_html = preg_replace('/\s*metasync_optimized(?:="[^"]*")?/', '', $head_html);
            
            # Update the head element
            $head->outertext = $cleaned_head_html;
            
        }
    }

    # this function saves are reloads the dom for modifications to avoid conflict
    function save_reload(){

        # Cleanup metasync_optimized attribute on AMP pages before saving
        $this->cleanup_amp_metasync_attribute();
        
        # DISABLED: Cache file creation temporarily disabled

        # if(file_put_contents($this->html_file, $this->dom)){
            
            # load the modified file to the DOM
           # $this->dom = new HtmlDocument($this->html_file );
        # }

        # this code is to be replaced in future
        # reson for adding is to prevent caching logged in user pages
        # why not just skip saving? it broke the DOM Library
        # check user is logged in clear the file
        
		# if(is_user_logged_in()) {
		#	unlink($this->html_file);
		# }

        # MEMORY-BASED RELOAD: Instead of file operations, reload DOM from current HTML string
        # This prevents DOM breaking while avoiding cache file creation
        if($this->dom){
            # Get current DOM as HTML string
            $current_html = $this->dom->save();
            
            # Reload DOM from the HTML string to refresh internal state
            # This replaces the file save/reload cycle that SimpleHtmlDOM expects
            $this->dom->load($current_html);

            # Force UTF-8 charset after reload to preserve emojis
            $this->dom->_charset = 'UTF-8';
            $this->dom->_target_charset = 'UTF-8';
        }
        
    }


}