<?php
/**
 * Log_Manager Class
 *
 * Manages local log extraction and processing for the Metasync plugin.
 * This class extracts and filters WordPress debug logs for local storage.
 *
 * Features:
 *  - Extract logs from WordPress debug.log
 *  - Filter plugin-specific or general site logs
 *  - Batch processing to prevent timeouts on large files
 *  - Resumable processing with file position tracking
 *  - Automatic log rotation detection
 *
 * Note: Remote log monitoring is handled by Sentry integration.
 * ZIP creation and upload functionality has been removed in favor of Sentry.
 *
 * Settings Storage:
 * - Option name: metasync_logging_data
 * - Stored in wp_options table
 *
 * @package    Metasync
 * @subpackage Log_Sync
 */

Class Log_Manager{
    # wordpress log path
    protected $wp_log_path;

    # plugin logs path
    protected $metasync_logs_path;

    # current debug log file for data
    protected $plugin_error_log;

    # the logging option wp
    protected $logging_options;

    # the logging option name
    protected $logging_option_name = 'metasync_logging_data';

    # log settings
    protected $log_settings;

    # log sort chron name
    protected $log_sort_chron;

    # log upload chron name
    protected $log_upload_chron;

    # batch processing configuration
    protected $batch_size = 1000; // Process 1000 lines per batch
    protected $max_execution_time = 30; // Max execution time in seconds (safety margin)
    protected $file_position_option = 'metasync_log_file_position';

    function __construct(){
        # set the wp debug path
        $this->load_wp_debug_path();

        # Enforce error logging dir
        $this->enforce_errorlog_dir();

        # Set the logging optoin 
        $options = get_option($this->logging_option_name);

        #set the options
        $this->logging_options = $options;

        #ensure array
        if(!is_array($options)){
            $this->logging_options = [];
        }
    }

    # app log function
    function app_log($message){

        #log the message
        error_log($message);
    }

    # Enforce plugin log dir
    function enforce_errorlog_dir(){

        # Check we have the Content Dir Const
        if( !defined('WP_CONTENT_DIR')){
            
            return;
        }

        # now set the desired folder path
        $metasync_logs_path = WP_CONTENT_DIR. '/metasync_data/';

        # if not dir make dir recursive
        if(!is_dir($metasync_logs_path)){

            # create the dir with appropriate permissions
            if(!mkdir($metasync_logs_path, 0755, true)){

                #failed create log message
                $this->app_log('Metasync : Failed to Create Log Dir - Permission denied or path issues');
                error_log('MetaSync Log Manager: Failed to create directory: ' . $metasync_logs_path . ' - Check parent directory permissions');

                return;
            } else {
                # Successfully created, log it
            }
        }

        # Verify directory is writable
        if(!is_writable($metasync_logs_path)){
            error_log('MetaSync Log Manager: Directory not writable: ' . $metasync_logs_path . ' - Check permissions');
            
            # Try to fix permissions
            if(!chmod($metasync_logs_path, 0755)){
                error_log('MetaSync Log Manager: Failed to change directory permissions: ' . $metasync_logs_path);
            }
        }

        # set the metasync logs path
        $this->metasync_logs_path = $metasync_logs_path;

        # create the log file name 
        $this->plugin_error_log = $this->metasync_logs_path . '/plugin_errors.log';

    }

    # store the loging option in db
    function logging_options($save_ = false){
        
        # if we are not saving sth retrieve the option
        if($save_ == false){
            return $this->logging_options;
        }  

        # otherwise save the option data provided
        update_option($this->logging_option_name, $this->logging_options);
    }

    /**
     * Tracking the last checked log line's time
     * This is to help prevent appending duplicate lines to the log
     */
    function track_last_logline($line_time = false){
        # if the line time is false return the last log line time
        if($line_time == false){

            #update this to actual last line time
            return $this->logging_options()['last_debug_line'] ?? 0;
        }

        # save the last line
        $this->logging_options['last_debug_line'] = $line_time;

        #save
        $this->logging_options(true);


        $this->app_log('Metasync Last Log Time' . date('d-m-Y H:i:s', $line_time));
    }

    /**
     * Track file processing position for resumable processing
     * @param int|bool $position File position to save, or false to retrieve
     * @return int Current file position
     */
    function track_file_position($position = false){
        if($position === false){
            return (int) get_option($this->file_position_option, 0);
        }

        update_option($this->file_position_option, $position);
        return $position;
    }

    /**
     * Reset file position tracking (e.g., when log file is rotated)
     */
    function reset_file_position(){
        delete_option($this->file_position_option);
    }

    /**
     * Check if we're approaching max execution time
     * @param int $start_time Timestamp when processing started
     * @return bool True if we should stop processing
     */
    function should_stop_processing($start_time){
        $elapsed = time() - $start_time;
        return $elapsed >= $this->max_execution_time;
    }

    /**
     * Parse date from log line efficiently
     * @param string $line Log line content
     * @return int|false Timestamp or false if no valid date found
     */
    function parse_log_date($line){
        # Quick check: does the line start with a date pattern [DD-MMM-YYYY]?
        if(empty($line) || strlen($line) < 20 || $line[0] !== '['){
            return false;
        }

        # Extract date pattern [DD-MMM-YYYY HH:MM:SS]
        $pattern = '/\[(.*?)\]/';

        if(!preg_match($pattern, $line, $matches)){
            return false;
        }

        $date_string = $matches[1] ?? '';

        # Validate we have a non-empty date string
        if(empty($date_string) || strlen($date_string) < 10){
            return false;
        }

        # Convert to timestamp
        $time = strtotime($date_string);

        # Validate the timestamp is reasonable (not false or negative)
        return ($time !== false && $time > 0) ? $time : false;
    }

    # write log line method
    # this will help us in case we need to determine what lines should be written or not
    function write_error_log_line($write_log, $line){
        
        # log line conditin let's defaul to on
        $log_plugin_specific = True;

        # 
        if(!empty($this->log_settings['only_plugin_specific'])){

            # 
            $log_plugin_specific = $this->log_settings['only_plugin_specific'];
        }

        # Define regex pattern to match "metasync" with or without backslashes
        $pattern = '/metasync/i';

        # Check if the line contains "metasync"
        if ($log_plugin_specific == true && preg_match($pattern, $line)) {
            fwrite($write_log, $line);
            return;
        }

        # if we need to log every line
        if($log_plugin_specific == False){

            # otherwise write any lines
            fwrite($write_log, $line);
            return;
        }
    }

    # fetch the debug log path
    function load_wp_debug_path(){
        #
        $extracted_path = False;

        #check content dir
        if( defined('WP_CONTENT_DIR')){

            # extracted path
            $extracted_path = WP_CONTENT_DIR . '/debug.log';
        }

        #check if the debug is enabled 
        if (defined('WP_DEBUG_LOG') && WP_DEBUG_LOG){
            
            #check if it is a string 
            if(is_string(WP_DEBUG_LOG)){

                #
                $extracted_path = WP_DEBUG_LOG;
            }

        }

        #if no debug file path return
        if($extracted_path == False || !is_file($extracted_path)){

            #
            return False;
        }

        #set the wp_log_path prop to path
        $this->wp_log_path = $extracted_path;
    }

    /**
     * Method : Process_debug
     *
     * Optimized log processing with batch processing, execution time monitoring,
     * and resumable file position tracking to prevent timeouts on large log files.
     *
     * @return array Status information about the processing
     *
     */
    function process_debug(){
        # Track execution start time
        $start_time = time();

        # Get the debug log path
        if(!is_file($this->wp_log_path)){
            $this->app_log('Metasync: Debug file does not exist at: ' . $this->wp_log_path);
            return [
                'success' => false,
                'error' => 'Debug file not found',
                'lines_processed' => 0
            ];
        }

        # Check if log file has been rotated (file is smaller than last position)
        $file_size = filesize($this->wp_log_path);
        $last_position = $this->track_file_position();

        if($file_size < $last_position){
            # Log file was rotated, reset position
            $this->app_log('Metasync: Log file rotated, resetting position');
            $this->reset_file_position();
            $last_position = 0;
        }

        # Open debug file for reading
        $debug_file = fopen($this->wp_log_path, 'r');

        if(!$debug_file){
            $this->app_log('Metasync: Error reading debug file');
            return [
                'success' => false,
                'error' => 'Cannot open debug file',
                'lines_processed' => 0
            ];
        }

        # Open write log file
        $write_log = fopen($this->plugin_error_log, 'a');

        if(!$write_log){
            fclose($debug_file);
            $this->app_log('Metasync: Error opening plugin error log for writing');
            return [
                'success' => false,
                'error' => 'Cannot open write log',
                'lines_processed' => 0
            ];
        }

        # Seek to last known position
        if($last_position > 0){
            fseek($debug_file, $last_position);
            $this->app_log('Metasync: Resuming from position: ' . $last_position);
        }

        # Get last processed line time
        $last_line_time = $this->track_last_logline();

        # Initialize counters
        $lines_processed = 0;
        $lines_written = 0;
        $current_position = $last_position;
        $completed = false;

        # Process lines in batches
        while(($line = fgets($debug_file)) !== false){

            # Check execution time every 100 lines to reduce overhead
            if($lines_processed % 100 === 0 && $this->should_stop_processing($start_time)){
                $this->app_log('Metasync: Execution time limit reached, saving progress. Lines processed: ' . $lines_processed);
                break;
            }

            # Check batch size limit
            if($lines_processed >= $this->batch_size){
                $this->app_log('Metasync: Batch size limit reached. Lines processed: ' . $lines_processed);
                break;
            }

            # Parse date from line efficiently
            $time = $this->parse_log_date($line);

            # Only process lines with valid timestamps newer than last processed
            if($time !== false && $time > $last_line_time){
                # Write the line to plugin error log
                $this->write_error_log_line($write_log, $line);

                # Update last line time
                $last_line_time = $time;
                $lines_written++;
            }

            # Update position and counter
            $current_position = ftell($debug_file);
            $lines_processed++;
        }

        # Check if we completed the entire file
        if(feof($debug_file)){
            $completed = true;
            $this->app_log('Metasync: Completed processing debug log. Total lines: ' . $lines_processed);
        }

        # Save progress
        $this->track_file_position($current_position);
        $this->track_last_logline($last_line_time);

        # Close file handles
        fclose($debug_file);
        fclose($write_log);

        # Return status information
        return [
            'success' => true,
            'completed' => $completed,
            'lines_processed' => $lines_processed,
            'lines_written' => $lines_written,
            'file_position' => $current_position,
            'file_size' => $file_size,
            'progress_percentage' => $file_size > 0 ? round(($current_position / $file_size) * 100, 2) : 100
        ];
    }

}