<?php

/**
 * Example usage of Google Index API
 * 
 * This file demonstrates how to use the Google Index API
 * to index WordPress posts directly with Google's Indexing API
 * 
 * PREREQUISITES:
 * 1. Service account must be saved to WordPress options using google_index_save_service_account()
 * 2. Google Search Console must have the service account added as a user
 * 3. Domain must be verified in Google Search Console
 */

// Prevent direct access
if (!defined('ABSPATH')) {
    exit;
}

// Include the initialization file (loads all necessary components)
require_once __DIR__ . '/google-index-init.php';

/**
 * Example 1: Index a single post by ID
 */
function example_index_single_post() 
{
    $google_index = google_index_direct();
    
    // Index post ID 123 (replace with actual post ID)
    $post_id = 123;
    $result = $google_index->index_post($post_id, 'post', 'update');
    
    if ($result['success']) {
        echo "✅ Successfully indexed post ID {$post_id}\n";
        echo "Response: " . json_encode($result['data'], JSON_PRETTY_PRINT) . "\n";
    } else {
        echo "❌ Failed to index post ID {$post_id}\n";
        echo "Error: " . $result['error']['message'] . "\n";
    }
    
    return $result;
}

/**
 * Example 2: Check status of a URL
 */
function example_check_url_status()
{
    $google_index = google_index_direct();
    
    // Check status of your homepage
    $url = get_home_url();
    $result = $google_index->get_url_status($url);
    
    if ($result['success']) {
        echo "✅ Successfully retrieved status for {$url}\n";
        
        // Display indexing information
        $data = $result['data'];
        if (isset($data['urlNotificationMetadata'])) {
            $metadata = $data['urlNotificationMetadata'];
            echo "Last Update Type: " . ($metadata['latestUpdate']['type'] ?? 'Unknown') . "\n";
            echo "Update Time: " . ($metadata['latestUpdate']['notifyTime'] ?? 'Unknown') . "\n";
        } else {
            echo "URL has not been submitted for indexing yet.\n";
        }
    } else {
        echo "❌ Failed to get status for {$url}\n";
        echo "Error: " . $result['error']['message'] . "\n";
    }
    
    return $result;
}

/**
 * Example 3: Index multiple posts
 */
function example_index_multiple_posts()
{
    $google_index = google_index_direct();
    
    // Get the 5 most recent published posts
    $posts = get_posts([
        'numberposts' => 5,
        'post_status' => 'publish',
        'post_type' => 'post'
    ]);
    
    $post_ids = array_map(function($post) {
        return $post->ID;
    }, $posts);
    
    echo "Indexing " . count($post_ids) . " posts...\n";
    
    $results = $google_index->index_multiple_posts($post_ids, 'post', 'update');
    
    foreach ($results as $post_id => $result) {
        if ($result['success']) {
            echo "✅ Post ID {$post_id}: Successfully indexed\n";
        } else {
            echo "❌ Post ID {$post_id}: " . $result['error']['message'] . "\n";
        }
    }
    
    return $results;
}

/**
 * Example 4: Index a custom URL
 */
function example_index_custom_url()
{
    $google_index = google_index_direct();
    
    // Index a specific URL (e.g., a custom page)
    $url = get_home_url() . '/custom-page/';
    $result = $google_index->index_url($url, 'update');
    
    if ($result['success']) {
        echo "✅ Successfully indexed custom URL: {$url}\n";
    } else {
        echo "❌ Failed to index custom URL: {$url}\n";
        echo "Error: " . $result['error']['message'] . "\n";
    }
    
    return $result;
}

/**
 * Example 5: Test the API connection
 */
function example_test_connection()
{
    $google_index = google_index_direct();
    
    echo "🔧 Testing Google Indexing API connection...\n\n";
    
    $test_results = $google_index->test_connection();
    
    // Check credentials
    echo "📋 Service Account Info:\n";
    $creds = $test_results['credentials_test'];
    echo "  Client Email: " . $creds['client_email'] . "\n";
    echo "  Project ID: " . $creds['project_id'] . "\n";
    echo "  Has Private Key: " . ($creds['has_private_key'] ? '✅ Yes' : '❌ No') . "\n\n";
    
    // Check token generation
    echo "🔑 Access Token Test:\n";
    $token = $test_results['token_test'];
    echo "  Token Generated: " . ($token['success'] ? '✅ Yes' : '❌ No') . "\n";
    echo "  Token Cached: " . ($token['cached'] ? '✅ Yes' : '❌ No') . "\n\n";
    
    // Check homepage status
    echo "🏠 Homepage Status Check:\n";
    $homepage = $test_results['homepage_test'];
    if ($homepage['success']) {
        echo "  ✅ Successfully retrieved homepage status\n";
        if (isset($homepage['data']['urlNotificationMetadata'])) {
            echo "  📊 Homepage has been indexed before\n";
        } else {
            echo "  📊 Homepage has not been indexed yet\n";
        }
    } else {
        echo "  ❌ Failed to retrieve homepage status\n";
        echo "  Error: " . $homepage['error']['message'] . "\n";
    }
    
    return $test_results;
}

/**
 * Example 6: Manual indexing integration
 */
function manual_index_post_example($post_id, $post_type = 'post')
{
    $google_index = google_index_direct();
    
    // Index the post manually
    $result = $google_index->index_post($post_id, $post_type, 'update');

    // Log errors only
    // if (!$result['success']) {
    //     error_log("Failed to index {$post_type} ID {$post_id}: " . $result['error']['message']);
    // }

    return $result;
}

/**
 * Example usage in code or cron job
 * Note: Admin interface and automatic hooks have been removed for simplicity
 * Use these functions directly in your code for manual/synchronous indexing as needed
 */

// Example: Index a specific post manually
// manual_index_post_example(123, 'post');
