<?php

/**
 * Copyright 2019 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 * Licensed under the Apache License, Version 2.0 (the "License").
 * You may not use this file except in compliance with the License.
 * A copy of the License is located at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * or in the "license" file accompanying this file. This file is distributed
 * on an "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either
 * express or implied. See the License for the specific language governing
 * permissions and limitations under the License.
 */
namespace Mediavine\Create\Amazon\ProductAdvertisingAPI\v1;

/**
 * Configuration Class Doc Comment
 * PHP version 5
 *
 * @category Class
 * @package  Amazon\ProductAdvertisingAPI\v1
 * @author   Product Advertising API team
 */
class Configuration
{
    private static $defaultConfiguration;
    /**
     * Access Key for AWS V4 authentication
     *
     * @var string
     */
    protected $accessKey = '';
    /**
     * Secret Key for AWS V4 authentication
     *
     * @var string
     */
    protected $secretKey = '';
    /**
     * The host
     *
     * @var string
     */
    protected $host = '';
    /**
     * The region
     *
     * @var string
     */
    protected $region = '';
    /**
     * User agent of the HTTP request
     *
     * @var string
     */
    protected $userAgent = 'paapi5-php-sdk-1.0.0';
    /**
     * Debug switch (default set to false)
     *
     * @var bool
     */
    protected $debug = \false;
    /**
     * Debug file location (log to STDOUT by default)
     *
     * @var string
     */
    protected $debugFile = 'php://output';
    /**
     * Debug file location (log to STDOUT by default)
     *
     * @var string
     */
    protected $tempFolderPath;
    /**
     * Constructor
     */
    public function __construct()
    {
        $this->tempFolderPath = \sys_get_temp_dir();
    }
    /**
     * Sets the access key for AWS V4 authentication
     *
     * @param string $accessKey Access Key for AWS V4 authentication
     *
     * @return $this
     */
    public function setAccessKey($accessKey)
    {
        $this->accessKey = $accessKey;
        return $this;
    }
    /**
     * Gets the access key for AWS V4 authentication
     *
     * @return string Access Key for AWS V4 authentication
     */
    public function getAccessKey()
    {
        return $this->accessKey;
    }
    /**
     * Sets the secret key for AWS V4 authentication
     *
     * @param string $secretKey Secret Key for AWS V4 authentication
     *
     * @return $this
     */
    public function setSecretKey($secretKey)
    {
        $this->secretKey = $secretKey;
        return $this;
    }
    /**
     * Gets the secret key for AWS V4 authentication
     *
     * @return string Secret Key for AWS V4 authentication
     */
    public function getSecretKey()
    {
        return $this->secretKey;
    }
    /**
     * Sets the host
     *
     * @param string $host Host
     *
     * @return $this
     */
    public function setHost($host)
    {
        $this->host = 'https://' . $host;
        return $this;
    }
    /**
     * Gets the host
     *
     * @return string Host
     */
    public function getHost()
    {
        return $this->host;
    }
    /**
     * Sets the region
     *
     * @param string $region Region
     *
     * @return $this
     */
    public function setRegion($region)
    {
        $this->region = $region;
        return $this;
    }
    /**
     * Gets the region
     *
     * @return string Region
     */
    public function getRegion()
    {
        return $this->region;
    }
    /**
     * Gets the user agent of the api client
     *
     * @return string user agent
     */
    public function getUserAgent()
    {
        return $this->userAgent;
    }
    /**
     * Sets debug flag
     *
     * @param bool $debug Debug flag
     *
     * @return $this
     */
    public function setDebug($debug)
    {
        $this->debug = $debug;
        return $this;
    }
    /**
     * Gets the debug flag
     *
     * @return bool
     */
    public function getDebug()
    {
        return $this->debug;
    }
    /**
     * Sets the debug file
     *
     * @param string $debugFile Debug file
     *
     * @return $this
     */
    public function setDebugFile($debugFile)
    {
        $this->debugFile = $debugFile;
        return $this;
    }
    /**
     * Gets the debug file
     *
     * @return string
     */
    public function getDebugFile()
    {
        return $this->debugFile;
    }
    /**
     * Sets the temp folder path
     *
     * @param string $tempFolderPath Temp folder path
     *
     * @return $this
     */
    public function setTempFolderPath($tempFolderPath)
    {
        $this->tempFolderPath = $tempFolderPath;
        return $this;
    }
    /**
     * Gets the temp folder path
     *
     * @return string Temp folder path
     */
    public function getTempFolderPath()
    {
        return $this->tempFolderPath;
    }
    /**
     * Gets the default configuration instance
     *
     * @return Configuration
     */
    public static function getDefaultConfiguration()
    {
        if (self::$defaultConfiguration === null) {
            self::$defaultConfiguration = new Configuration();
        }
        return self::$defaultConfiguration;
    }
    /**
     * Sets the detault configuration instance
     *
     * @param Configuration $config An instance of the Configuration Object
     *
     * @return void
     */
    public static function setDefaultConfiguration(Configuration $config)
    {
        self::$defaultConfiguration = $config;
    }
    /**
     * Gets the essential information for debugging
     *
     * @return string The report for debugging
     */
    public static function toDebugReport()
    {
        $report = 'PHP SDK (Amazon\\ProductAdvertisingAPI\\v1) Debug Report:' . \PHP_EOL;
        $report .= '    OS: ' . \php_uname() . \PHP_EOL;
        $report .= '    PHP Version: ' . \PHP_VERSION . \PHP_EOL;
        $report .= '    OpenAPI Spec Version: 1.0.0' . \PHP_EOL;
        $report .= '    SDK Package Version: 1.0.0' . \PHP_EOL;
        $report .= '    Temp Folder Path: ' . self::getDefaultConfiguration()->getTempFolderPath() . \PHP_EOL;
        return $report;
    }
}
