<?php

namespace Mediavine\Create\Helpers;

use stdClass;
use Countable;
use ArrayAccess;
use Traversable;
use JsonSerializable;
use IteratorAggregate;

/**
 * Class Collection
 */
class Collection implements ArrayAccess, Countable, IteratorAggregate, JsonSerializable {


	/**
	 * The items contained in the collection.
	 *
	 * @var array
	 */
	protected $items = [];

	/**
	 * Create a new collection.
	 *
	 * @param  mixed $items
	 * @return void
	 */
	public function __construct( $items = [] ) {
		$this->items = $this->getArrayableItems( $items );
	}

	/**
	 * Create a new collection instance if the value isn't one already.
	 *
	 * @param  mixed $items
	 * @return static
	 */
	public static function make( $items = [] ) {
		return new static( $items );
	}

	/**
	 * Alias for merge.
	 *
	 * @param array|mixed $item
	 * @return static
	 */
	public function add( $item ) {
		return $this->merge( $item );
	}

	/**
	 * Wrap the given value in a collection if applicable.
	 *
	 * @param  mixed $value
	 * @return static
	 */
	public static function wrap( $value ) {
		return $value instanceof self
			? new static( $value )
			: new static( Arr::wrap( $value ) );
	}

	/**
	 * Get the underlying items from the given collection if applicable.
	 *
	 * @param  array|static $value
	 * @return array
	 */
	public static function unwrap( $value ): array {
		return $value instanceof self ? $value->all() : $value;
	}

	/**
	 * Create a new collection by invoking the callback a given amount of times.
	 *
	 * @param  int      $number
	 * @param  callable $callback
	 * @return static
	 */
	public static function times( $number, callable $callback = null ) {
		if ( $number < 1 ) {
			return new static();
		}

		if ( is_null( $callback ) ) {
			return new static( range( 1, $number ) );
		}

		return ( new static( range( 1, $number ) ) )->map( $callback );
	}

	/**
	 * Get all of the items in the collection.
	 *
	 * @return array
	 */
	public function all(): array {
		return $this->items;
	}

	/**
	 * Get the average value of a given key.
	 *
	 * @param  callable|string|null $callback
	 * @return mixed
	 */
	public function avg( $callback = null ): mixed {
		$callback = $this->valueRetriever( $callback );

		$items = $this->map(
			function ( $value ) use ( $callback ) {
			return $callback( $value );
			}
		)->filter(
			function ( $value ) {
				return ! is_null( $value );
				}
		);

		$count = $items->count();
		if ( $count ) {
			return $items->sum() / $count;
		}
		return null;
	}

	/**
	 * Alias for the "avg" method.
	 *
	 * @param  callable|string|null $callback
	 * @return mixed
	 */
	public function average( $callback = null ): mixed {
		return $this->avg( $callback );
	}

	/**
	 * Get the mode of a given key.
	 *
	 * @param  string|array|null $key
	 * @return array|null
	 */
	public function mode( $key = null ) {
		if ( $this->count() === 0 ) {
			return null;
		}

		$collection = isset( $key ) ? $this->pluck( $key ) : $this;

		$counts = new self();

		$collection->each(
			function ( $value ) use ( $counts ) {
			$counts[ $value ] = isset( $counts[ $value ] ) ? $counts[ $value ] + 1 : 1;
			}
		);

		$sorted = $counts->sort();

		$highestValue = $sorted->last();

		return $sorted->filter(
			function ( $value ) use ( $highestValue ) {
				return $value === $highestValue;
			}
		)->sort()->keys()->all();
	}

	/**
	 * Collapse the collection of items into a single array.
	 *
	 * @return static
	 */
	public function collapse() {
		return new static( Arr::collapse( $this->items ) );
	}

	/**
	 * Alias for the "contains" method.
	 *
	 * @param  mixed $key
	 * @param  mixed $operator
	 * @param  mixed $value
	 * @return bool
	 */
	public function some( $key, $operator = null, $value = null ): bool {
		return $this->contains( ...func_get_args() );
	}

	/**
	 * Determine if an item exists in the collection.
	 *
	 * @param  mixed $key
	 * @param  mixed $operator
	 * @param  mixed $value
	 * @return bool
	 */
	public function contains( $key, $operator = null, $value = null ): bool {
		if ( func_num_args() === 1 ) {
			if ( $this->useAsCallable( $key ) ) {
				$placeholder = new stdClass();

				return $this->first( $key, $placeholder ) !== $placeholder;
			}

			return in_array( $key, $this->items, true );
		}

		return $this->contains( $this->operatorForWhere( ...func_get_args() ) );
	}

	/**
	 * Determine if an item exists in the collection using strict comparison.
	 *
	 * @param  mixed $key
	 * @param  mixed $value
	 * @return bool
	 */
	public function containsStrict( $key, $value = null ): bool {
		if ( func_num_args() === 2 ) {
			return $this->contains(
				function ( $item ) use ( $key, $value ) {
				return mv_data_get( $item, $key ) === $value;
				}
			);
		}

		if ( $this->useAsCallable( $key ) ) {
			return ! is_null( $this->first( $key ) );
		}

		return in_array( $key, $this->items, true );
	}

	/**
	 * Dump the collection.
	 *
	 * @return $this
	 */
	public function dump(): string {
		// phpcs:disable
		return print_r( $this->items, true );
		// phpcs:enable
	}

	/**
	 * Get the items in the collection that are not present in the given items.
	 *
	 * @param  mixed $items
	 * @return static
	 */
	public function diff( $items ) {
		return new static( array_diff( $this->items, $this->getArrayableItems( $items ) ) );
	}

	/**
	 * Get the items in the collection that are not present in the given items.
	 *
	 * @param  mixed    $items
	 * @param  callable $callback
	 * @return static
	 */
	public function diffUsing( $items, callable $callback ) {
		return new static( array_udiff( $this->items, $this->getArrayableItems( $items ), $callback ) );
	}

	/**
	 * Get the items in the collection whose keys and values are not present in the given items.
	 *
	 * @param  mixed $items
	 * @return static
	 */
	public function diffAssoc( $items ) {
		return new static( array_diff_assoc( $this->items, $this->getArrayableItems( $items ) ) );
	}

	/**
	 * Get the items in the collection whose keys and values are not present in the given items.
	 *
	 * @param  mixed    $items
	 * @param  callable $callback
	 * @return static
	 */
	public function diffAssocUsing( $items, callable $callback ) {
		return new static( array_diff_uassoc( $this->items, $this->getArrayableItems( $items ), $callback ) );
	}

	/**
	 * Get the items in the collection whose keys are not present in the given items.
	 *
	 * @param  mixed $items
	 * @return static
	 */
	public function diffKeys( $items ) {
		return new static( array_diff_key( $this->items, $this->getArrayableItems( $items ) ) );
	}

	/**
	 * Get the items in the collection whose keys are not present in the given items.
	 *
	 * @param  mixed    $items
	 * @param  callable $callback
	 * @return static
	 */
	public function diffKeysUsing( $items, callable $callback ) {
		return new static( array_diff_ukey( $this->items, $this->getArrayableItems( $items ), $callback ) );
	}

	/**
	 * Execute a callback over each item.
	 *
	 * @param  callable $callback
	 * @return $this
	 */
	public function each( callable $callback ) {
		foreach ( $this->items as $key => $item ) {
			if ( $callback( $item, $key ) === false ) {
				break;
			}
		}

		return $this;
	}

	/**
	 * Execute a callback over each nested chunk of items.
	 *
	 * @param  callable $callback
	 * @return static
	 */
	public function eachSpread( callable $callback ) {
		return $this->each(
			function ( $chunk, $key ) use ( $callback ) {
			$chunk[] = $key;

			return $callback( ...$chunk );
			}
		);
	}

	/**
	 * Determine if all items in the collection pass the given test.
	 *
	 * @param  string|callable $key
	 * @param  mixed           $operator
	 * @param  mixed           $value
	 * @return bool
	 */
	public function every( $key, $operator = null, $value = null ): bool {
		if ( func_num_args() === 1 ) {
			$callback = $this->valueRetriever( $key );

			foreach ( $this->items as $k => $v ) {
				if ( ! $callback( $v, $k ) ) {
					return false;
				}
			}

			return true;
		}

		return $this->every( $this->operatorForWhere( ...func_get_args() ) );
	}

	/**
	 * Get all items except for those with the specified keys.
	 *
	 * @param  static|mixed $keys
	 * @return static
	 */
	public function except( $keys ) {
		if ( $keys instanceof self ) {
			$keys = $keys->all();
		} elseif ( ! is_array( $keys ) ) {
			$keys = func_get_args();
		}

		return new static( Arr::except( $this->items, $keys ) );
	}

	/**
	 * Run a filter over each of the items.
	 *
	 * @param  callable|null $callback
	 * @return static
	 */
	public function filter( callable $callback = null ) {
		if ( $callback ) {
			return new static( Arr::where( $this->items, $callback ) );
		}

		return new static( array_filter( $this->items ) );
	}

	public function join( $glue, $finalGlue = '' ): string {
		if ( '' === $finalGlue ) {
			return $this->implode( $glue );
		}
		$count = $this->count();
		if ( 0 === $count ) {
			return '';
		}
		if ( 1 === $count ) {
			return $this->last();
		}
		$collection = new static( $this->items );
		$finalItem  = $collection->pop();
		return $collection->implode( $glue ) . $finalGlue . $finalItem;
	}

	/**
	 * Apply the callback if the value is truthy.
	 *
	 * @param  bool     $value
	 * @param  callable $callback
	 * @param  callable $default
	 * @return static
	 */
	public function when( $value, callable $callback, callable $default = null ) {
		if ( $value ) {
			return $callback( $this, $value );
		} elseif ( $default ) {
			return $default( $this, $value );
		}

		return $this;
	}

	/**
	 * Apply the callback if the collection is empty.
	 *
	 * @param  callable $callback
	 * @param  callable $default
	 * @return static
	 */
	public function whenEmpty( callable $callback, callable $default = null ) {
		return $this->when( $this->isEmpty(), $callback, $default );
	}

	/**
	 * Apply the callback if the collection is not empty.
	 *
	 * @param  callable $callback
	 * @param  callable $default
	 * @return static
	 */
	public function whenNotEmpty( callable $callback, callable $default = null ) {
		return $this->when( $this->isNotEmpty(), $callback, $default );
	}

	/**
	 * Apply the callback if the value is falsy.
	 *
	 * @param  bool     $value
	 * @param  callable $callback
	 * @param  callable $default
	 * @return static
	 */
	public function unless( $value, callable $callback, callable $default = null ) {
		return $this->when( ! $value, $callback, $default );
	}

	/**
	 * Apply the callback unless the collection is empty.
	 *
	 * @param  callable $callback
	 * @param  callable $default
	 * @return static
	 */
	public function unlessEmpty( callable $callback, callable $default = null ) {
		return $this->whenNotEmpty( $callback, $default );
	}

	/**
	 * Apply the callback unless the collection is not empty.
	 *
	 * @param  callable $callback
	 * @param  callable $default
	 * @return static
	 */
	public function unlessNotEmpty( callable $callback, callable $default = null ) {
		return $this->whenEmpty( $callback, $default );
	}

	/**
	 * Filter items by the given key value pair.
	 *
	 * @param  string $key
	 * @param  mixed  $operator
	 * @param  mixed  $value
	 * @return static
	 */
	public function where( $key, $operator = null, $value = null ) {
		return $this->filter( $this->operatorForWhere( ...func_get_args() ) );
	}

	/**
	 * Get an operator checker callback.
	 *
	 * @param  string $key
	 * @param  string $operator
	 * @param  mixed  $value
	 * @return \Closure
	 */
	protected function operatorForWhere( $key, $operator = null, $value = null ): \Closure {
		if ( func_num_args() === 1 ) {
			$value = true;

			$operator = '=';
		}

		if ( func_num_args() === 2 ) {
			$value = $operator;

			$operator = '=';
		}

		return function ( $item ) use ( $key, $operator, $value ) {
			$retrieved = mv_data_get( $item, $key );

			$strings = array_filter(
				[ $retrieved, $value ], function ( $value ) {
				return is_string( $value ) || ( is_object( $value ) && method_exists( $value, '__toString' ) );
				}
			);

			if ( count( $strings ) < 2 && 1 === count( array_filter( [ $retrieved, $value ], 'is_object' ) ) ) {
				return in_array( $operator, [ '!=', '<>', '!==' ], true );
			}

			// phpcs:disable
			switch ( $operator ) {
				default:
				case '=':
				case '==':
					return $retrieved == $value;
				case '!=':
				case '<>':
					return $retrieved != $value;
				case '<':
					return $retrieved < $value;
				case '>':
					return $retrieved > $value;
				case '<=':
					return $retrieved <= $value;
				case '>=':
					return $retrieved >= $value;
				case '===':
					return $retrieved === $value;
				case '!==':
					return $retrieved !== $value;
			}
			// phpcs:enable
		};
	}

	/**
	 * Filter items by the given key value pair using strict comparison.
	 *
	 * @param  string $key
	 * @param  mixed  $value
	 * @return static
	 */
	public function whereStrict( $key, $value ) {
		return $this->where( $key, '===', $value );
	}

	/**
	 * Filter items by the given key value pair.
	 *
	 * @param  string $key
	 * @param  mixed  $values
	 * @param  bool   $strict
	 * @return static
	 */
	public function whereIn( $key, $values, $strict = false ) {
		$values = $this->getArrayableItems( $values );

		return $this->filter(
			function ( $item ) use ( $key, $values, $strict ) {
				// phpcs:disable
				return in_array( mv_data_get( $item, $key ), $values, $strict );
				// phpcs:enable
			}
		);
	}

	/**
	 * Filter items by the given key value pair using strict comparison.
	 *
	 * @param  string $key
	 * @param  mixed  $values
	 * @return static
	 */
	public function whereInStrict( $key, $values ) {
		return $this->whereIn( $key, $values, true );
	}

	/**
	 * Filter items such that the value of the given key is between the given values.
	 *
	 * @param  string $key
	 * @param  array  $values
	 * @return static
	 */
	public function whereBetween( $key, $values ) {
		return $this->where( $key, '>=', reset( $values ) )->where( $key, '<=', end( $values ) );
	}

	/**
	 * Filter items such that the value of the given key is not between the given values.
	 *
	 * @param  string $key
	 * @param  array  $values
	 * @return static
	 */
	public function whereNotBetween( $key, $values ) {
		return $this->filter(
			function ( $item ) use ( $key, $values ) {
			return mv_data_get( $item, $key ) < reset( $values ) || mv_data_get( $item, $key ) > end( $values );
			}
		);
	}

	/**
	 * Filter items by the given key value pair.
	 *
	 * @param  string $key
	 * @param  mixed  $values
	 * @param  bool   $strict
	 * @return static
	 */
	public function whereNotIn( $key, $values, $strict = false ) {
		$values = $this->getArrayableItems( $values );

		return $this->reject(
			function ( $item ) use ( $key, $values, $strict ) {
				// phpcs:disable
				return in_array( mv_data_get( $item, $key ), $values, $strict );
				// phpcs:enable
			}
		);
	}

	/**
	 * Filter items by the given key value pair using strict comparison.
	 *
	 * @param  string $key
	 * @param  mixed  $values
	 * @return static
	 */
	public function whereNotInStrict( $key, $values ) {
		return $this->whereNotIn( $key, $values, true );
	}

	/**
	 * Filter the items, removing any items that don't match the given type.
	 *
	 * @param  string $type
	 * @return static
	 */
	public function whereInstanceOf( $type ) {
		return $this->filter(
			function ( $value ) use ( $type ) {
			return $value instanceof $type;
			}
		);
	}

	/**
	 * Get the first item from the collection.
	 *
	 * @param  callable|null $callback
	 * @param  mixed         $default
	 * @return mixed
	 */
	public function first( callable $callback = null, $default = null ): mixed {
		return Arr::first( $this->items, $callback, $default );
	}

	/**
	 * Get the first item by the given key value pair.
	 *
	 * @param  string $key
	 * @param  mixed  $operator
	 * @param  mixed  $value
	 * @return mixed
	 */
	public function firstWhere( $key, $operator, $value = null ): mixed {
		return $this->first( $this->operatorForWhere( $key, $operator, $value ) );
	}

	/**
	 * Get a flattened array of the items in the collection.
	 *
	 * @param  int $depth
	 * @return static
	 */
	public function flatten( $depth = INF ) {
		return new static( Arr::flatten( $this->items, $depth ) );
	}

	/**
	 * Flip the items in the collection.
	 *
	 * @return static
	 */
	public function flip() {
		return new static( array_flip( $this->items ) );
	}

	/**
	 * Remove an item from the collection by key.
	 *
	 * @param  string|array $keys
	 * @return $this
	 */
	public function forget( $keys ) {
		foreach ( (array) $keys as $key ) {
			$this->offsetUnset( $key );
		}

		return $this;
	}

	/**
	 * Get an item from the collection by key.
	 *
	 * @param  mixed $key
	 * @param  mixed $default
	 * @return mixed
	 */
	public function get( $key, $default = null ): mixed {
		if ( $this->offsetExists( $key ) ) {
			return $this->items[ $key ];
		}

		return mv_get_value( $default );
	}

	/**
	 * Group an associative array by a field or using a callback.
	 *
	 * @param  array|callable|string $groupBy
	 * @param  bool                  $preserveKeys
	 * @return static
	 */
	public function groupBy( $groupBy, $preserveKeys = false ) {
		if ( is_array( $groupBy ) ) {
			$nextGroups = $groupBy;

			$groupBy = array_shift( $nextGroups );
		}

		$groupBy = $this->valueRetriever( $groupBy );

		$results = [];

		foreach ( $this->items as $key => $value ) {
			$groupKeys = $groupBy( $value, $key );

			if ( ! is_array( $groupKeys ) ) {
				$groupKeys = [ $groupKeys ];
			}

			foreach ( $groupKeys as $groupKey ) {
				$groupKey = is_bool( $groupKey ) ? (int) $groupKey : $groupKey;

				if ( ! array_key_exists( $groupKey, $results ) ) {
					$results[ $groupKey ] = new static();
				}

				$results[ $groupKey ]->offsetSet( $preserveKeys ? $key : null, $value );
			}
		}

		$result = new static( $results );
		if ( ! empty( $nextGroups ) ) {
			return $result->map(
				function ( $item ) use ( $nextGroups, $preserveKeys ) {
					return $item->groupBy( $nextGroups, $preserveKeys );
				}
			);
		}

		return $result;
	}

	/**
	 * Key an associative array by a field or using a callback.
	 *
	 * @param  callable|string $keyBy
	 * @return static
	 */
	public function keyBy( $keyBy ) {
		$keyBy = $this->valueRetriever( $keyBy );

		$results = [];

		foreach ( $this->items as $key => $item ) {
			$resolvedKey = $keyBy( $item, $key );

			if ( is_object( $resolvedKey ) ) {
				$resolvedKey = (string) $resolvedKey;
			}

			$results[ $resolvedKey ] = $item;
		}

		return new static( $results );
	}

	/**
	 * Determine if an item exists in the collection by key.
	 *
	 * @param  mixed $key
	 * @return bool
	 */
	public function has( $key ): bool {
		$keys = is_array( $key ) ? $key : func_get_args();

		foreach ( $keys as $value ) {
			if ( ! $this->offsetExists( $value ) ) {
				return false;
			}
		}

		return true;
	}

	/**
	 * Concatenate values of a given key as a string.
	 *
	 * @param  string $value
	 * @param  string $glue
	 * @return string
	 */
	public function implode( $value, $glue = null ): string {
		$first = $this->first();

		if ( is_array( $first ) || is_object( $first ) ) {
			return implode( $glue, $this->pluck( $value )->all() );
		}

		return implode( $value, $this->items );
	}

	/**
	 * Intersect the collection with the given items.
	 *
	 * @param  mixed $items
	 * @return static
	 */
	public function intersect( $items ) {
		return new static( array_intersect( $this->items, $this->getArrayableItems( $items ) ) );
	}

	/**
	 * Intersect the collection with the given items by key.
	 *
	 * @param  mixed $items
	 * @return static
	 */
	public function intersectByKeys( $items ) {
		return new static(
			array_intersect_key(
				$this->items, $this->getArrayableItems( $items )
			)
		);
	}

	/**
	 * Determine if the collection is empty or not.
	 *
	 * @return bool
	 */
	public function isEmpty(): bool {
		return empty( $this->items );
	}

	/**
	 * Determine if the collection is not empty.
	 *
	 * @return bool
	 */
	public function isNotEmpty(): bool {
		return ! $this->isEmpty();
	}

	/**
	 * Determine if the given value is callable, but not a string.
	 *
	 * @param  mixed $value
	 * @return bool
	 */
	protected function useAsCallable( $value ): bool {
		return ! is_string( $value ) && is_callable( $value );
	}

	/**
	 * Get the keys of the collection items.
	 *
	 * @return static
	 */
	public function keys() {
		return new static( array_keys( $this->items ) );
	}

	/**
	 * Get the last item from the collection.
	 *
	 * @param  callable|null $callback
	 * @param  mixed         $default
	 * @return mixed
	 */
	public function last( callable $callback = null, $default = null ): mixed {
		return Arr::last( $this->items, $callback, $default );
	}

	/**
	 * Get the values of a given key.
	 *
	 * @param  string|array $value
	 * @param  string|null  $key
	 * @return static
	 */
	public function pluck( $value, $key = null ) {
		return new static( Arr::pluck( $this->items, $value, $key ) );
	}

	/**
	 * Run a map over each of the items.
	 *
	 * @param  callable $callback
	 * @return static
	 */
	public function map( callable $callback ) {
		$keys = array_keys( $this->items );

		$items = array_map( $callback, $this->items, $keys );

		return new static( array_combine( $keys, $items ) );
	}

	/**
	 * Run a dictionary map over the items.
	 *
	 * The callback should return an associative array with a single key/value pair.
	 *
	 * @param  callable $callback
	 * @return static
	 */
	public function mapToDictionary( callable $callback ) {
		$dictionary = [];

		foreach ( $this->items as $key => $item ) {
			$pair = $callback( $item, $key );

			$key = key( $pair );

			$value = reset( $pair );

			if ( ! isset( $dictionary[ $key ] ) ) {
				$dictionary[ $key ] = [];
			}

			$dictionary[ $key ][] = $value;
		}

		return new static( $dictionary );
	}

	/**
	 * Run a grouping map over the items.
	 *
	 * The callback should return an associative array with a single key/value pair.
	 *
	 * @param  callable $callback
	 * @return static
	 */
	public function mapToGroups( callable $callback ) {
		$groups = $this->mapToDictionary( $callback );

		return $groups->map( [ $this, 'make' ] );
	}

	/**
	 * Run an associative map over each of the items.
	 *
	 * The callback should return an associative array with a single key/value pair.
	 *
	 * @param  callable $callback
	 * @return static
	 */
	public function mapWithKeys( callable $callback ) {
		$result = [];

		foreach ( $this->items as $key => $value ) {
			$assoc = $callback( $value, $key );

			foreach ( $assoc as $mapKey => $mapValue ) {
				$result[ $mapKey ] = $mapValue;
			}
		}

		return new static( $result );
	}

	/**
	 * Map a collection and flatten the result by a single level.
	 *
	 * @param  callable $callback
	 * @return static
	 */
	public function flatMap( callable $callback ) {
		return $this->map( $callback )->collapse();
	}

	/**
	 * Map the values into a new class.
	 *
	 * @param  string $class
	 * @return static
	 */
	public function mapInto( $class ) {
		return $this->map(
			function ( $value, $key ) use ( $class ) {
			return new $class( $value, $key );
			}
		);
	}

	/**
	 * Get the max value of a given key.
	 *
	 * @param  callable|string|null $callback
	 * @return mixed
	 */
	public function max( $callback = null ): mixed {
		$callback = $this->valueRetriever( $callback );

		return $this->filter(
			function ( $value ) {
			return ! is_null( $value );
			}
		)->reduce(
			function ( $result, $item ) use ( $callback ) {
			$value = $callback( $item );

			return is_null( $result ) || $value > $result ? $value : $result;
			}
		);
	}

	/**
	 * Merge the collection with the given items.
	 *
	 * @param  mixed $items
	 * @return static
	 */
	public function merge( $items ) {
		return new static( array_merge( $this->items, $this->getArrayableItems( $items ) ) );
	}

	/**
	 * Create a collection by using this collection for keys and another for its values.
	 *
	 * @param  mixed $values
	 * @return static
	 */
	public function combine( $values ) {
		return new static( array_combine( $this->all(), $this->getArrayableItems( $values ) ) );
	}

	/**
	 * Union the collection with the given items.
	 *
	 * @param  mixed $items
	 * @return static
	 */
	public function union( $items ) {
		return new static( $this->items + $this->getArrayableItems( $items ) );
	}

	/**
	 * Get the min value of a given key.
	 *
	 * @param  callable|string|null $callback
	 * @return mixed
	 */
	public function min( $callback = null ): mixed {
		$callback = $this->valueRetriever( $callback );

		return $this->map(
			function ( $value ) use ( $callback ) {
			return $callback( $value );
			}
		)->filter(
			function ( $value ) {
			return ! is_null( $value );
			}
		)->reduce(
			function ( $result, $value ) {
			return is_null( $result ) || $value < $result ? $value : $result;
			}
		);
	}

	/**
	 * Create a new collection consisting of every n-th element.
	 *
	 * @param  int $step
	 * @param  int $offset
	 * @return static
	 */
	public function nth( $step, $offset = 0 ) {
		$new = [];

		$position = 0;

		foreach ( $this->items as $item ) {
			if ( $position % $step === $offset ) {
				$new[] = $item;
			}

			$position++;
		}

		return new static( $new );
	}

	/**
	 * Get the items with the specified keys.
	 *
	 * @param  mixed $keys
	 * @return static
	 */
	public function only( $keys ) {
		if ( is_null( $keys ) ) {
			return new static( $this->items );
		}

		if ( $keys instanceof self ) {
			$keys = $keys->all();
		}

		$keys = is_array( $keys ) ? $keys : func_get_args();

		return new static( Arr::only( $this->items, $keys ) );
	}

	/**
	 * "Paginate" the collection by slicing it into a smaller collection.
	 *
	 * @param  int $page
	 * @param  int $perPage
	 * @return static
	 */
	public function forPage( $page, $perPage ) {
		$offset = max( 0, ( $page - 1 ) * $perPage );

		return $this->slice( $offset, $perPage );
	}

	/**
	 * Partition the collection into two arrays using the given callback or key.
	 *
	 * @param  callable|string $key
	 * @param  mixed           $operator
	 * @param  mixed           $value
	 * @return static
	 */
	public function partition( $key, $operator = null, $value = null ) {
		$partitions = [ new static(), new static() ];

		$callback = func_num_args() === 1
				? $this->valueRetriever( $key )
				: $this->operatorForWhere( $key, $operator, $value );

		foreach ( $this->items as $key => $item ) {
			$partitions[ (int) ! $callback( $item, $key ) ][ $key ] = $item;
		}

		return new static( $partitions );
	}

	/**
	 * Pass the collection to the given callback and return the result.
	 *
	 * @param  callable $callback
	 * @return mixed
	 */
	public function pipe( callable $callback ): mixed {
		return $callback( $this );
	}

	/**
	 * Get and remove the last item from the collection.
	 *
	 * @return mixed
	 */
	public function pop(): mixed {
		return array_pop( $this->items );
	}

	/**
	 * Push an item onto the beginning of the collection.
	 *
	 * @param  mixed $value
	 * @param  mixed $key
	 * @return $this
	 */
	public function prepend( $value, $key = null ) {
		$this->items = Arr::prepend( $this->items, $value, $key );

		return $this;
	}

	/**
	 * Push an item onto the end of the collection.
	 *
	 * @param  mixed $value
	 * @return $this
	 */
	public function push( $value ) {
		$this->offsetSet( null, $value );

		return $this;
	}

	/**
	 * Push all of the given items onto the collection.
	 *
	 * @param  iterable $source
	 * @return static
	 */
	public function concat( $source ) {
		$result = new static( $this );

		foreach ( $source as $item ) {
			$result->push( $item );
		}

		return $result;
	}

	/**
	 * Get and remove an item from the collection.
	 *
	 * @param  mixed $key
	 * @param  mixed $default
	 * @return mixed
	 */
	public function pull( $key, $default = null ): mixed {
		return Arr::pull( $this->items, $key, $default );
	}

	/**
	 * Put an item in the collection by key.
	 *
	 * @param  mixed $key
	 * @param  mixed $value
	 * @return $this
	 */
	public function put( $key, $value ) {
		$this->offsetSet( $key, $value );

		return $this;
	}

	/**
	 * Get one or a specified number of items randomly from the collection.
	 *
	 * @param  int|null $number
	 * @return static
	 *
	 * @throws \InvalidArgumentException
	 */
	public function random( $number = null ) {
		if ( is_null( $number ) ) {
			return Arr::random( $this->items );
		}

		return new static( Arr::random( $this->items, $number ) );
	}

	/**
	 * Reduce the collection to a single value.
	 *
	 * @param  callable $callback
	 * @param  mixed    $initial
	 * @return mixed
	 */
	public function reduce( callable $callback, $initial = null ): mixed {
		return array_reduce( $this->items, $callback, $initial );
	}

	/**
	 * Create a collection of all elements that do not pass a given truth test.
	 *
	 * @param  callable|mixed $callback
	 * @return static
	 */
	public function reject( $callback ) {
		if ( $this->useAsCallable( $callback ) ) {
			return $this->filter(
				function ( $value, $key ) use ( $callback ) {
				return ! $callback( $value, $key );
				}
			);
		}

		return $this->filter(
			function ( $item ) use ( $callback ) {
				return $item !== $callback;
			}
		);
	}

	/**
	 * Reverse items order.
	 *
	 * @return static
	 */
	public function reverse() {
		return new static( array_reverse( $this->items, true ) );
	}

	/**
	 * Search the collection for a given value and return the corresponding key if successful.
	 *
	 * @param  mixed $value
	 * @param  bool  $strict
	 * @return mixed
	 */
	public function search( $value, $strict = false ): mixed {
		if ( ! $this->useAsCallable( $value ) ) {
			// phpcs:disable
			return array_search( $value, $this->items, $strict );
			// phpcs:enable
		}

		foreach ( $this->items as $key => $item ) {
			if ( call_user_func( $value, $item, $key ) ) {
				return $key;
			}
		}

		return false;
	}

	/**
	 * Get and remove the first item from the collection.
	 *
	 * @return mixed
	 */
	public function shift(): mixed {
		return array_shift( $this->items );
	}

	/**
	 * Shuffle the items in the collection.
	 *
	 * @param  int $seed
	 * @return static
	 */
	public function shuffle( $seed = null ) {
		return new static( Arr::shuffle( $this->items, $seed ) );
	}

	/**
	 * Slice the underlying collection array.
	 *
	 * @param  int $offset
	 * @param  int $length
	 * @return static
	 */
	public function slice( $offset, $length = null ) {
		return new static( array_slice( $this->items, $offset, $length, true ) );
	}

	/**
	 * Split a collection into a certain number of groups.
	 *
	 * @param  int $numberOfGroups
	 * @return static
	 */
	public function split( $numberOfGroups ) {
		if ( $this->isEmpty() ) {
			return new static();
		}

		$groups = new static();

		$groupSize = floor( $this->count() / $numberOfGroups );

		$remain = $this->count() % $numberOfGroups;

		$start = 0;

		for ( $i = 0; $i < $numberOfGroups; $i++ ) {
			$size = $groupSize;

			if ( $i < $remain ) {
				$size++;
			}

			if ( $size ) {
				$groups->push( new static( array_slice( $this->items, $start, $size ) ) );

				$start += $size;
			}
		}

		return $groups;
	}

	/**
	 * Chunk the underlying collection array.
	 *
	 * @param  int $size
	 * @return static
	 */
	public function chunk( $size ) {
		if ( $size <= 0 ) {
			return new static();
		}

		$chunks = [];

		foreach ( array_chunk( $this->items, $size, true ) as $chunk ) {
			$chunks[] = new static( $chunk );
		}

		return new static( $chunks );
	}

	/**
	 * Chunk the collection into an array.
	 *
	 * @param int $size
	 * @return array
	 */
	public function chunkToArray( $size ): array {
		if ( $size <= 0 ) {
			return [];
		}
		return array_chunk( $this->items, $size, false );
	}

	/**
	 * Sort through each item with a callback.
	 *
	 * @param  callable|null $callback
	 * @return static
	 */
	public function sort( callable $callback = null ) {
		$items = $this->items;

		$callback
			? uasort( $items, $callback )
			: asort( $items );

		return new static( $items );
	}

	/**
	 * Sort the collection using the given callback.
	 *
	 * @param  callable|string $callback
	 * @param  int             $options
	 * @param  bool            $descending
	 * @return static
	 */
	public function sortBy( $callback, $options = SORT_REGULAR, $descending = false ) {
		$results = [];

		$callback = $this->valueRetriever( $callback );

		// First we will loop through the items and get the comparator from a callback
		// function which we were given. Then, we will sort the returned values and
		// and grab the corresponding values for the sorted keys from this array.
		foreach ( $this->items as $key => $value ) {
			$results[ $key ] = $callback( $value, $key );
		}

		$descending ? arsort( $results, $options )
			: asort( $results, $options );

		// Once we have sorted all of the keys in the array, we will loop through them
		// and grab the corresponding model so we can set the underlying items list
		// to the sorted version. Then we'll just return the collection instance.
		foreach ( array_keys( $results ) as $key ) {
			$results[ $key ] = $this->items[ $key ];
		}

		return new static( $results );
	}

	/**
	 * Sort the collection in descending order using the given callback.
	 *
	 * @param  callable|string $callback
	 * @param  int             $options
	 * @return static
	 */
	public function sortByDesc( $callback, $options = SORT_REGULAR ) {
		return $this->sortBy( $callback, $options, true );
	}

	/**
	 * Sort the collection keys.
	 *
	 * @param  int  $options
	 * @param  bool $descending
	 * @return static
	 */
	public function sortKeys( $options = SORT_REGULAR, $descending = false ) {
		$items = $this->items;

		$descending ? krsort( $items, $options ) : ksort( $items, $options );

		return new static( $items );
	}

	/**
	 * Sort the collection keys in descending order.
	 *
	 * @param  int $options
	 * @return static
	 */
	public function sortKeysDesc( $options = SORT_REGULAR ) {
		return $this->sortKeys( $options, true );
	}

	/**
	 * Splice a portion of the underlying collection array.
	 *
	 * @param  int      $offset
	 * @param  int|null $length
	 * @param  mixed    $replacement
	 * @return static
	 */
	public function splice( $offset, $length = null, $replacement = [] ) {
		if ( func_num_args() === 1 ) {
			return new static( array_splice( $this->items, $offset ) );
		}

		return new static( array_splice( $this->items, $offset, $length, $replacement ) );
	}

	/**
	 * Get the sum of the given values.
	 *
	 * @param  callable|string|null $callback
	 * @return mixed
	 */
	public function sum( $callback = null ): mixed {
		if ( is_null( $callback ) ) {
			return array_sum( $this->items );
		}

		$callback = $this->valueRetriever( $callback );

		return $this->reduce(
			function ( $result, $item ) use ( $callback ) {
			return $result + $callback( $item );
			}, 0
		);
	}

	/**
	 * Take the first or last {$limit} items.
	 *
	 * @param  int $limit
	 * @return static
	 */
	public function take( $limit ) {
		if ( $limit < 0 ) {
			return $this->slice( $limit, abs( $limit ) );
		}

		return $this->slice( 0, $limit );
	}

	/**
	 * Pass the collection to the given callback and then return it.
	 *
	 * @param  callable $callback
	 * @return $this
	 */
	public function tap( callable $callback ) {
		$callback( new static( $this->items ) );

		return $this;
	}

	/**
	 * Pass  the collection into a callback and return a new collection from the result.
	 *
	 * @param callable $callback
	 * @return $this
	 */
	public function tapInto( callable $callback ) {
		return new static( $this->getArrayableItems( $callback( $this->all() ) ) );
	}

	/**
	 * Pass the collection into a callback and return the results with a set of values.
	 *
	 * @param callable          $callback
	 * @param ArrayAccess|array $with
	 * @return static
	 */
	public function newFrom( callable $callback, $with = [] ) {
		return $this->tapInto( $callback )->merge( $this->getArrayableItems( $with ) );
	}

	/**
	 * Pass the collection through a callback.
	 *
	 * Some callbacks return an arrayable response with values that you want,
	 * but with the values you already have.
	 *
	 * @param callable $callback
	 * @return static
	 */
	public function passThrough( callable $callback ) {
		return $this->newFrom( $callback, $this );
	}

	/**
	 * Transform each item in the collection using a callback.
	 *
	 * @param  callable $callback
	 * @return $this
	 */
	public function transform( callable $callback ) {
		$this->items = $this->map( $callback )->all();

		return $this;
	}

	/**
	 * Return only unique items from the collection array.
	 *
	 * @param  string|callable|null $key
	 * @param  bool                 $strict
	 * @return static
	 */
	public function unique( $key = null, $strict = false ) {
		$callback = $this->valueRetriever( $key );

		$exists = [];

		return $this->reject(
			function ( $item, $key ) use ( $callback, $strict, &$exists ) {
				$id = $callback( $item, $key );
				// phpcs:disable
				if ( in_array( $id, $exists, $strict ) ) {
					return true;
				}
				// phpcs:enable

				$exists[] = $id;
			}
		);
	}

	/**
	 * Return only unique items from the collection array using strict comparison.
	 *
	 * @param  string|callable|null $key
	 * @return static
	 */
	public function uniqueStrict( $key = null ) {
		return $this->unique( $key, true );
	}

	/**
	 * Reset the keys on the underlying array.
	 *
	 * @return static
	 */
	public function values() {
		return new static( array_values( $this->items ) );
	}

	/**
	 * Get a value retrieving callback.
	 *
	 * @param  string $value
	 * @return callable
	 */
	protected function valueRetriever( $value ): callable {
		if ( $this->useAsCallable( $value ) ) {
			return $value;
		}

		return function ( $item ) use ( $value ) {
			return mv_data_get( $item, $value );
		};
	}

	/**
	 * Zip the collection together with one or more arrays.
	 *
	 * e.g. new Collection([1, 2, 3])->zip([4, 5, 6]);
	 *      => [[1, 4], [2, 5], [3, 6]]
	 *
	 * @param  mixed ...$items
	 * @return static
	 */
	public function zip( $items ) {
		$arrayableItems = array_map(
			function ( $items ) {
			return $this->getArrayableItems( $items );
			}, func_get_args()
		);

		$params = array_merge(
			[
				function () {
							return new static( func_get_args() );
						},
				$this->items,
			], $arrayableItems
		);

		return new static( call_user_func_array( 'array_map', $params ) );
	}

	/**
	 * Pad collection to the specified length with a value.
	 *
	 * @param  int   $size
	 * @param  mixed $value
	 * @return static
	 */
	public function pad( $size, $value ) {
		return new static( array_pad( $this->items, $size, $value ) );
	}

	/**
	 * Get the collection of items as a plain array.
	 *
	 * @return array
	 */
	public function toArray(): array {
		return array_map(
			function ( $value ) {
			return $value instanceof Collection ? $value->all() : $value;
			}, $this->items
		);
	}

	/**
	 * Convert the object into something JSON serializable.
	 *
	 * @return array
	 */
	public function jsonSerialize(): array {
		return array_map(
			function ( $value ) {
			if ( is_object( $value ) && method_exists( $value, 'jsonSerialize' ) ) {
				return $value->jsonSerialize();
			} elseif ( $value instanceof Collection ) {
				return json_decode( $value->toJson(), true );
			} elseif ( is_object( $value ) && method_exists( $value, 'toArray' ) ) {
				return $value->toArray();
			}

			return $value;
			}, $this->items
		);
	}

	/**
	 * Get the collection of items as JSON.
	 *
	 * @param  int $options
	 * @return string
	 */
	public function toJson( $options = 0 ): string {
		return json_encode( $this->jsonSerialize(), $options );
	}

	/**
	 * Get an iterator for the items.
	 *
	 * @return \ArrayIterator
	 */
	public function getIterator(): Traversable {
		foreach ( $this->items as $item ) {
			yield $item;
		}
	}

	/**
	 * Count the number of items in the collection.
	 *
	 * @return int
	 */
	public function count(): int {
		return count( $this->items );
	}

	/**
	 * Get a base Support collection instance from this collection.
	 *
	 * @return static
	 */
	public function toBase() {
		return new self( $this );
	}

	/**
	 * Determine if an item exists at an offset.
	 *
	 * @param  mixed $key
	 * @return bool
	 */
	public function offsetExists( $key ): bool {
		return array_key_exists( $key, $this->items );
	}

	/**
	 * Get an item at a given offset.
	 *
	 * @param  mixed $key
	 * @return mixed
	 */
	public function offsetGet( $key ): mixed {
		return $this->items[ $key ];
	}

	/**
	 * Set the item at a given offset.
	 *
	 * @param  mixed $key
	 * @param  mixed $value
	 * @return void
	 */
	public function offsetSet( $key, $value ): void {
		if ( is_null( $key ) ) {
			$this->items[] = $value;
		} else {
			$this->items[ $key ] = $value;
		}
	}

	/**
	 * Unset the item at a given offset.
	 *
	 * @param  string $key
	 * @return void
	 */
	public function offsetUnset( $key ): void {
		unset( $this->items[ $key ] );
	}

	/**
	 * Convert the collection to its string representation.
	 *
	 * @return string
	 */
	public function __toString(): string {
		return $this->toJson();
	}

	/**
	 * Results array of items from Collection or Arrayable.
	 *
	 * @param  mixed $items
	 * @return array
	 */
	protected function getArrayableItems( $items ): array {
		if ( is_array( $items ) ) {
			return $items;
		} elseif ( $items instanceof self ) {
			return $items->all();
		} elseif ( is_object( $items ) && method_exists( $items, 'toArray' ) ) {
			return $items->toArray();
		} elseif ( is_object( $items ) && method_exists( $items, 'toJson' ) ) {
			return json_decode( $items->toJson(), true );
		} elseif ( is_object( $items ) && method_exists( $items, 'jsonSerialize' ) ) {
			return $items->jsonSerialize();
		} elseif ( $items instanceof Traversable ) {
			return iterator_to_array( $items );
		}

		return (array) $items;
	}

}
