<?php
namespace Login_With_AJAX\TwoFA\Method;

use Login_With_AJAX\QRCode;
use WP_User;

class TOTP extends Method {
	
	public static $method = 'totp';
	public static $authentication_resend = false;
	
	public static $needs_setup = true;
	public static $svg_icon = '<svg xmlns="http://www.w3.org/2000/svg" xml:space="preserve" viewBox="0 0 512 512"><path d="M196.967 477.863h15.667c9.816 0 17.766-7.951 17.766-17.766V444.43c0-9.816-7.951-17.766-17.766-17.766h-15.667c-9.816 0-17.766 7.951-17.766 17.766v15.667c0 9.816 7.951 17.766 17.766 17.766zm-.699-33.433c0-.39.31-.7.7-.7h15.667c.39 0 .7.31.7.7v15.667c0 .39-.31.7-.7.7h-15.667c-.39 0-.7-.31-.7-.7V444.43z"/><path d="M443.733 119.463c-29.549 0-53.856-7.511-73.367-19.989-6.766-4.327-12.441-8.945-17.054-13.53-1.437-1.429-2.578-2.654-3.446-3.642V39.023C349.867 17.472 332.391 0 310.835 0H98.765C77.209 0 59.733 17.472 59.733 39.023V478.19c0 18.673 15.136 33.809 33.809 33.809h222.515c18.673 0 33.809-15.136 33.809-33.809V364.579c15.299-5.398 29.482-13.494 42.559-23.892 37.98-30.195 59.841-77.585 59.841-127.844v-84.847c.001-4.712-3.82-8.533-8.533-8.533zM332.8 478.188c0 9.248-7.495 16.742-16.742 16.742H93.542c-9.248 0-16.742-7.495-16.742-16.742V418.13h256v60.058zm-256-77.125v-332.8h59.733c4.713 0 8.533-3.82 8.533-8.533s-3.82-8.533-8.533-8.533H76.8V39.023c0-12.124 9.834-21.956 21.965-21.956h212.07c12.131 0 21.965 9.832 21.965 21.956v43.279c-.868.988-2.009 2.214-3.446 3.642-4.613 4.585-10.288 9.203-17.054 13.53-19.511 12.478-43.818 19.989-73.367 19.989-4.713 0-8.533 3.82-8.533 8.533v84.851c0 50.259 21.861 97.649 59.842 127.844.897.713 1.8 1.412 2.708 2.103.172.131.346.258.518.388.705.533 1.413 1.061 2.124 1.581.316.231.633.456.95.684.575.413 1.15.824 1.729 1.229.37.259.742.514 1.114.77.535.367 1.071.731 1.61 1.091.393.262.786.523 1.18.781.532.349 1.066.692 1.602 1.034.394.251.788.503 1.183.75.56.35 1.122.693 1.686 1.034.371.225.74.452 1.112.673.664.395 1.333.781 2.002 1.165.273.157.545.318.819.473.951.538 1.907 1.064 2.868 1.579.15.08.301.156.451.236.819.435 1.64.863 2.466 1.282.283.143.568.281.852.423.699.348 1.399.693 2.103 1.029.319.152.639.3.959.45.681.319 1.365.634 2.051.942.32.144.641.285.963.427.702.309 1.407.611 2.114.908.302.127.604.254.907.378.757.311 1.517.615 2.28.913.253.099.505.199.758.297.885.34 1.775.669 2.667.991.132.048.263.097.395.145.13.046.259.096.389.142v36.475H76.8zm358.4-188.219c0 45.161-19.576 87.598-53.396 114.485-12.465 9.911-25.946 17.417-40.468 22.128-.027-.009-.055-.017-.082-.026-.654-.213-1.304-.436-1.954-.66-.278-.096-.557-.188-.834-.286-.599-.212-1.195-.433-1.791-.654-.326-.121-.653-.24-.979-.364-.559-.213-1.115-.435-1.671-.656-.358-.143-.716-.284-1.073-.43-.528-.216-1.053-.44-1.578-.664-.383-.164-.767-.327-1.148-.494-.496-.218-.991-.441-1.484-.666-.409-.186-.818-.373-1.225-.564-.468-.219-.935-.443-1.401-.668-.429-.208-.858-.417-1.285-.63-.443-.22-.884-.444-1.324-.67-.448-.23-.894-.462-1.339-.698-.42-.222-.839-.445-1.258-.672-.463-.251-.924-.506-1.384-.763-.399-.223-.798-.446-1.195-.674-.476-.272-.949-.55-1.422-.829-.38-.224-.761-.448-1.14-.677-.485-.293-.968-.592-1.451-.891-.364-.226-.729-.451-1.092-.681-.492-.312-.98-.63-1.469-.949-.352-.23-.706-.457-1.057-.691-.485-.323-.967-.654-1.45-.983-.354-.242-.71-.482-1.063-.728-.453-.316-.903-.64-1.353-.962-.381-.272-.763-.541-1.142-.817-.368-.269-.733-.546-1.1-.818-.459-.342-.919-.68-1.375-1.028-.041-.032-.082-.065-.124-.096-.747-.571-1.491-1.15-2.231-1.738-33.82-26.888-53.396-69.325-53.396-114.486V136.34c29.148-1.339 53.79-9.54 74.029-22.484 1.01-.646 1.984-1.292 2.942-1.938.14-.095.276-.189.415-.284.796-.542 1.574-1.082 2.334-1.622.183-.13.365-.26.546-.39.749-.538 1.481-1.075 2.194-1.61.124-.093.251-.186.374-.279 1.689-1.277 3.275-2.541 4.758-3.785.096-.08.187-.159.281-.239.629-.531 1.24-1.058 1.832-1.581.135-.119.268-.237.401-.356.574-.512 1.131-1.02 1.669-1.522.091-.085.186-.171.276-.256.628-.59 1.238-1.174 1.816-1.748 4.848 4.81 11.431 10.232 19.836 15.607 20.239 12.944 44.881 21.145 74.029 22.484v76.507z"/></svg>';
	
	public static function get_name () {
		return esc_html__('Authenticator App', 'login-with-ajax');
	}
	
	/**
	 * @return string
	 */
	public static function get_text_select( $user ){
		return esc_html__('Enter a code fom your 2FA App', 'login-with-ajax-pro');
	}
	
	public static function get_text_request ( $user ) {
		return esc_html__('Enter a verification code generated by the authenticator app on your mobile device to proceed with your login.', 'login-with-ajax');
	}
	
	/**
	 * Overriden to add filter to child function.
	 * @param $user
	 *
	 * @return bool|mixed|null
	 */
	public static function is_ready( $user ) {
		$ready = parent::is_ready( $user );
		return apply_filters('lwa_2FA_method_totp_is_ready', $ready, $user);
	}
	
	/**
	 * Check that user has saved secret in user meta.
	 * @param $user
	 * @return bool|mixed|null
	 */
	public static function is_setup( $user ) {
		$setup = false;
		if( $user ) {
			$setup = static::get_secret( $user ) !== false;
		}
		return apply_filters('lwa_2FA_method_totp_is_setup', $setup, $user);
	}
	
	/**
	 * Assumes that $_REQUEST['2FA_code'] presence, general timeout etc. is checked before firing this filter.
	 * @param array $response
	 * @param WP_User $user
	 * @return array
	 */
	public static function verify( $response, $user ){
		if( !empty($_REQUEST['2FA_code']) && preg_match('/^[0-9 ]{4,9}$/', $_REQUEST['2FA_code']) ) {
			$secret = static::get_secret($user);
			if ( $secret ) {
				include(LOGIN_WITH_AJAX_PATH.'/assets/php/vendor/autoload.php');
				$google2fa = new \PragmaRX\Google2FA\Google2FA();
				$code = str_replace(' ', '', $_REQUEST['2FA_code']);
				if( $google2fa->verifyKey( $secret, $code) ){
					// verified!
					$response['result'] = true;
				}else{
					// invalid code
					$response['result'] = false;
					$response['error'] = esc_html__('Incorrect verification code, please try again.', 'login-with-ajax-pro');
				}
			} else {
				// no valid method
				$response['result'] = false;
				$response['restart'] = true;
				$response['error'] = esc_html__('Verification method unavailable, please try another method or contact us for further assistance.', 'login-with-ajax-pro');
			}
		} else {
			$response['result'] = false;
			$response['error'] = esc_html__('Incorrect verification code, please try again.', 'login-with-ajax-pro');
		}
		return $response;
	}
	
	public static function get_secret( $user ) {
		if( $user instanceof WP_User ) {
			$user_meta = \LoginWithAjax::get_user_meta( $user->ID, '2FA' );
			if ( !empty( $user_meta['methods'][ static::$method ]['secret'] ) ) {
				return $user_meta['methods'][ static::$method ]['secret'];
			}
		}
		return false;
	}
	
	public static function get_form_fields( $user = null ) {
		ob_start();
		?>
		<div class="lwa-2FA-code-input-wrap">
			<div class="lwa-2FA-code-input">
				<input type="text" name="2FA_code" inputmode="numeric" autocomplete="one-time-code" class="lwa-2FA-data-code" placeholder="<?php esc_html_e('Verification Code', 'login-with-ajax-pro'); ?>" style="">
				<button type="submit" class="button-primary u-full-width"><?php esc_html_e('Verify', 'login-with-ajax-pro'); ?></button>
			</div>
		</div>
		<?php
		return ob_get_clean();
	}
	
	/* Setup Function */
	
	public static function generate_secret() {
		include(LOGIN_WITH_AJAX_PATH.'/assets/php/vendor/autoload.php');
		$google2fa = new \PragmaRX\Google2FA\Google2FA();
		return $google2fa->generateSecretKey();
	}
	
	public static function get_qr_code( $user, $secret ) {
		include_once(LOGIN_WITH_AJAX_PATH.'/assets/php/qrcode.php');
		$current_user = urlencode($user->user_email);
		$issuer = urlencode(get_bloginfo('name'));
		return QRCode::svg('otpauth://totp/' . $issuer . ':' . $current_user . '?secret='. $secret .'&issuer=' . $issuer);
	}
	
	public static function get_setup_description () {
		return esc_html__('Generate verification codes on your mobile devices.', 'login-with-ajax');
	}
	
	public static function get_setup_form ( $user ) {
		// get user secret if exists, and generate a new one if they want to change it
		$current_secret = static::get_secret( $user );
		$secret = static::generate_secret();
		// get the QR code
		$qr = static::get_qr_code( $user, $secret );
		?>
		<div class="setup-totp-instructions">
			<div class="setup-totp-secret">
				<div class="totp-qr-code">
					<?php echo static::get_qr_code( $user, $secret ); ?>
				</div>
				<div class="totp-secret">
					<?php echo $secret; ?>
				</div>
			</div>
			<div class="lwa-2FA-method-verification">
				<?php if( $current_secret ) : ?>
					<p><strong><?php esc_html_e('Add this new QR/Secret to your Authenticator App. If you verify correctly, your previous 2FA secret will be replaced.', 'login-with-ajax'); ?></strong></p>
				<?php endif; ?>
				<ol>
					<li><?php esc_html_e('Open or install an Authenticator App on your mobile device.', 'login-with-ajax'); ?></li>
					<li>
						<?php esc_html_e('Choose to create a new code and either scan the QR code or enter the secret key below the QR code mansually.', 'login-with-ajax' ); ?>
					</li>
					<li><?php esc_html_e('Verify the generated code below.', 'login-with-ajax' ); ?></li>
				</ol>
				<div class="setup-verify-form totp-verify-code">
					<input data-name="nonce"  type="hidden" value="<?php echo wp_create_nonce('2FA-setup-verify-' . static::$method . '-' . $user->ID); ?>" class="setup-verify-field">
					<input data-name="secret" type="hidden" value="<?php echo $secret; ?>" class="setup-verify-field">
					<input data-name="code"   type="text"   placeholder="<?php esc_attr_e('Enter Code', 'login-with-ajax'); ?>" class="2FA-totp-verify-code setup-verify-field" aria-label="<?php esc_attr_e('Enter Code', 'login-with-ajax'); ?>"">
					<input data-name="log"    type="hidden" value="<?php echo esc_attr($user->user_login); ?>" class="setup-verify-field">
					<button type="button" class="setup-verify-button setup-verify-button-totp" data-txt="<?php esc_attr_e('Verifying ...', 'login-with-ajax'); ?>"><?php esc_html_e('Verify Code', 'login-with-ajax'); ?></button>
				</div>
				<mark class="error"></mark>
				<?php if( $current_secret ) echo static::get_setup_status_reset_button(); ?>
			</div>
		</div>
		<?php
	}
	
	
	public static function setup_verify( $user ) {
		// check secret against supplied code in $_REQUEST, if authentication is valid, save secret to user meta and confirm verification setup, otherwise return error
		// verify the code
		if( !empty($_REQUEST['code']) && !empty($_REQUEST['secret']) ) {
			if ( preg_match('/^[0-9 ]{4,9}$/', $_REQUEST['code']) ) {
				$secret = $_REQUEST['secret'];
				include_once( LOGIN_WITH_AJAX_PATH . '/assets/php/vendor/autoload.php' );
				$google2fa = new \PragmaRX\Google2FA\Google2FA();
				$code = str_replace( ' ', '', $_REQUEST['code'] );
				if ( $google2fa->verifyKey( $secret, $code ) ) {
					// verified! save secret and return html
					$meta = \LoginWithAjax::get_user_meta( $user->ID, '2FA' );
					if( !is_array($meta) ) $meta = array(); // init if doesn't exist
					$meta['methods'][ static::$method ] = array(
						'enabled' => true,
						'secret' => $secret,
						'verified' => time(),
					);
					if ( \LoginWithAjax::update_user_meta( $user->ID, '2FA', $meta ) ) {
						return array(
							'result'  => true,
							'message' => static::get_setup_status_ready_text($user),
							'status' => static::get_setup_status( $user ),
							'verification_method ' => static::$verification,
						);
					}
				}
			}
		}
		return array(
			'result' => false,
			'error' => esc_html__('Invalid verification code, please try again.', 'login-with-ajax-pro'),
		);
	}
}
add_action('lwa_2FA_loaded', '\Login_With_AJAX\TwoFA\Method\TOTP::init');