<?php
/**
 * Model that houses the Invoice Payment Gateway.
 *
 * @package Invoice_Gateway_For_WooCommerce
 * @subpackage Models/Gateways
 * @since 1.0.0
 * @since 1.1.4 - Applied PHPCS Rules. Compatibility for PHP 8.2+
 */

namespace IGFW\Models\Gateways;

use IGFW\Helpers\Plugin_Constants;

if ( ! defined( 'ABSPATH' ) ) {
    exit; // Exit if accessed directly.
}

/**
 * Model that houses the Invoice Payment Gateway.
 *
 * @since 1.0.0
 */
class IGFW_Invoice_Gateway extends \WC_Payment_Gateway {

    /**
     * Payment gateway instructions
     *
     * @var string
     */
    public $instructions;

    /**
     * Shipping methods this gateway is enabled for
     *
     * @var array
     */
    public $enable_for_methods;

    /**
     * Whether to enable for virtual orders
     *
     * @var bool
     */
    public $enable_for_virtual;

    /**
     * Constructor for the gateway.
     *
     * @since 1.0.0
     */
    public function __construct() {
        $this->id                 = 'igfw_invoice_gateway';
        $this->icon               = '';
        $this->has_fields         = false;
        $this->method_title       = __( 'Invoice Payment', 'invoice-gateway-for-woocommerce' );
        $this->method_description = __( 'Have your customers pay with and invoice generated by your accounting system. You will be responsible for invoicing the customer and entering the invoice ID on their order. Then once you confirm payment is received, you can mark the order as Completed as normal to indicate that it has been paid.', 'invoice-gateway-for-woocommerce' );

        // Load the settings.
        $this->init_form_fields();
        $this->init_settings();

        // Get settings.
        $this->title              = $this->get_option( 'title' );
        $this->description        = $this->get_option( 'description' );
        $this->instructions       = $this->get_option( 'instructions', $this->description );
        $this->enable_for_methods = $this->get_option( 'enable_for_methods', array() );
        $this->enable_for_virtual = $this->get_option( 'enable_for_virtual', 'yes' ) === 'yes';

        // Actions.
        add_action( 'woocommerce_update_options_payment_gateways_' . $this->id, array( $this, 'process_admin_options' ) );
        add_action( 'woocommerce_thankyou_' . $this->id, array( $this, 'thankyou_page' ) );

        // Customer Emails.
        add_action( 'woocommerce_email_before_order_table', array( $this, 'email_instructions' ), 10, 3 );

        // Force gateway to be available via direct hook.
        add_filter( 'woocommerce_available_payment_gateways', array( $this, 'add_to_gateways' ) );

        do_action( 'igfw_invoice_gateway_construct' );
    }

    /**
     * Ensure this gateway is available
     *
     * @param array $gateways Available gateways.
     * @return array Modified gateways.
     */
    public function add_to_gateways( $gateways ) {
        if ( ! isset( $gateways[ $this->id ] ) && $this->is_available() ) {
            $gateways[ $this->id ] = $this;
        }
        return $gateways;
    }

    /**
     * Initialize Gateway Settings Form Fields
     *
     * @since 1.0.0
     */
    public function init_form_fields() {

        $shipping_methods = array();

        if ( is_admin() ) {
            foreach ( WC()->shipping()->load_shipping_methods() as $method ) {
                $shipping_methods[ $method->id ] = $method->get_method_title();
            }
        }

        $this->form_fields = array(
            'enabled'            => array(
                'title'       => __( 'Enable/Disable', 'invoice-gateway-for-woocommerce' ),
                'label'       => __( 'Enable Invoice Payment', 'invoice-gateway-for-woocommerce' ),
                'type'        => 'checkbox',
                'description' => '',
                'default'     => 'no',
            ),
            'title'              => array(
                'title'       => __( 'Title', 'invoice-gateway-for-woocommerce' ),
                'type'        => 'text',
                'description' => __( 'Payment method description that the customer will see on your checkout.', 'invoice-gateway-for-woocommerce' ),
                'default'     => __( 'Invoice Payment', 'invoice-gateway-for-woocommerce' ),
                'desc_tip'    => true,
            ),
            'description'        => array(
                'title'       => __( 'Description', 'invoice-gateway-for-woocommerce' ),
                'type'        => 'textarea',
                'description' => __( 'Payment method description that the customer will see on your website.', 'invoice-gateway-for-woocommerce' ),
                'default'     => __( 'Pay with an invoice processed through our accounting system.', 'invoice-gateway-for-woocommerce' ),
                'desc_tip'    => true,
            ),
            'instructions'       => array(
                'title'       => __( 'Instructions', 'invoice-gateway-for-woocommerce' ),
                'type'        => 'textarea',
                'description' => __( 'Instructions that will be added to the thank you page.', 'invoice-gateway-for-woocommerce' ),
                'default'     => __( 'We\'ll be in touch shortly to deliver your invoice.', 'invoice-gateway-for-woocommerce' ),
                'desc_tip'    => true,
            ),
            'enable_for_methods' => array(
                'title'             => __( 'Enable for shipping methods', 'invoice-gateway-for-woocommerce' ),
                'type'              => 'multiselect',
                'class'             => 'wc-enhanced-select',
                'css'               => 'width: 450px;',
                'default'           => '',
                'description'       => __( 'If invoice payment is only available for certain shipping methods, set it up here. Leave blank to enable for all shipping methods.', 'invoice-gateway-for-woocommerce' ),
                'options'           => $shipping_methods,
                'desc_tip'          => true,
                'custom_attributes' => array(
                    'data-placeholder' => __( 'Select shipping methods', 'invoice-gateway-for-woocommerce' ),
                ),
            ),
            'enable_for_virtual' => array(
                'title'   => __( 'Accept for virtual orders', 'invoice-gateway-for-woocommerce' ),
                'label'   => __( 'Accept invoice payments if the order is virtual', 'invoice-gateway-for-woocommerce' ),
                'type'    => 'checkbox',
                'default' => 'yes',
            ),
        );
    }

    /**
     * Show Purchase Order Number option
     *
     * @since 1.1
     */
    public function payment_fields() {

        if ( get_option( 'igfw_enable_purchase_order_number' ) == 'yes' ) {
            $po_number_title       = apply_filters( 'igfw_purchase_order_number_title', __( 'Purchase Order (optional)', 'invoice-gateway-for-woocommerce' ) );
            $po_number_placeholder = apply_filters( 'igfw_purchase_order_number_placeholder', __( 'PO Number', 'invoice-gateway-for-woocommerce' ) );
            $po_number_desc        = apply_filters( 'igfw_purchase_order_number_desc', __( 'We will generate and send you an invoice for your order, if you have a PO number, please enter it.', 'invoice-gateway-for-woocommerce' ) );
            ?>
            <p><b><?php echo esc_html( $po_number_title ); ?></b></p>
            <p><input type="text" name="igfw_purchase_order_number" placeholder="<?php echo esc_attr( $po_number_placeholder ); ?>"></p>
            <p><?php echo esc_html( $po_number_desc ); ?></p>
            <?php
        } else {
            $description = $this->get_description();
            if ( $description ) {
                echo wpautop(wptexturize(esc_attr($description))); // @codingStandardsIgnoreLine.
            }
        }
    }

    /**
     * Check If The Gateway Is Available For Use.
     *
     * @since 1.0.0
     *
     * @return bool
     */
    public function is_available() {

        $order          = null;
        $needs_shipping = false;

        // Test if shipping is needed first.
        if ( WC()->cart && WC()->cart->needs_shipping() ) {
            $needs_shipping = true;
        } elseif ( is_page( wc_get_page_id( 'checkout' ) ) && 0 < get_query_var( 'order-pay' ) ) {
            $order_id = absint( get_query_var( 'order-pay' ) );
            $order    = wc_get_order( $order_id );

            // Test if order needs shipping.
            if ( 0 < count( $order->get_items() ) ) {
                foreach ( $order->get_items() as $item ) {
                    $_product = $item->get_product();
                    if ( $_product && $_product->needs_shipping() ) {
                        $needs_shipping = true;
                        break;
                    }
                }
            }
        }

        $needs_shipping = apply_filters( 'woocommerce_cart_needs_shipping', $needs_shipping );

        // Virtual order, with virtual disabled.
        if ( ! $this->enable_for_virtual && ! $needs_shipping ) {
            return false;
        }

        // Check methods.
        if ( ! empty( $this->enable_for_methods ) && $needs_shipping ) {
            // Only apply if all packages are being shipped via chosen methods, or order is virtual.
            $chosen_shipping_methods_session = WC()->session->get( 'chosen_shipping_methods' );
            $chosen_shipping_methods         = isset( $chosen_shipping_methods_session ) ? array_unique( $chosen_shipping_methods_session ) : array();
            $check_method                    = false;

            if ( is_object( $order ) ) {
                $shipping_method = $order->get_shipping_method();
                if ( $shipping_method ) {
                    $check_method = $order->shipping_method;
                }
            } elseif ( empty( $chosen_shipping_methods ) || count( $chosen_shipping_methods ) > 1 ) {
                $check_method = false;
            } elseif ( count( $chosen_shipping_methods ) === 1 ) {
                $check_method = $chosen_shipping_methods[0];
            }

            if ( ! $check_method ) {
                return false;
            }

            $found = false;

            foreach ( $this->enable_for_methods as $method_id ) {
                if ( strpos( $check_method, $method_id ) === 0 ) {
                    $found = true;
                    break;
                }
            }

            if ( ! $found ) {
                return false;
            }
        }

        return parent::is_available();
    }

    /**
     * Process the payment and return the result.
     *
     * @since 1.0.0
     * @access public
     *
     * @param int $order_id WC_Order id.
     * @return array
     */
    public function process_payment( $order_id ) {

        $order = wc_get_order( $order_id );

        // Get the default status from settings, fallback to 'on-hold' if not set.
        $default_status = get_option( 'igfw_default_order_status', 'on-hold' );
        
        // Mark as the configured default status.
        $order->update_status(
            apply_filters( 'igfw_invoice_gateway_default_order_status', $default_status ),
            __( 'Awaiting invoice payment. A notification has been sent to the store admin and the customer.', 'invoice-gateway-for-woocommerce' )
        );

        // Save the PO Number if provided and enabled.
        // phpcs:disable WordPress.Security.NonceVerification.Missing
        if ( get_option( 'igfw_enable_purchase_order_number' ) === 'yes' && isset( $_POST['igfw_purchase_order_number'] ) ) {
            $po_number = sanitize_text_field( $_POST['igfw_purchase_order_number'] );
            if ( ! empty( $po_number ) ) {
                $order->update_meta_data( \IGFW\Helpers\Plugin_Constants::PURCHASE_ORDER_NUMBER, $po_number );
                $order->save();
            }
        }
        // phpcs:enable WordPress.Security.NonceVerification.Missing

        // Reduce stock levels.
        wc_reduce_stock_levels( $order_id );

        // Remove cart.
        WC()->cart->empty_cart();

        // Return thankyou redirect.
        return array(
            'result'   => 'success',
            'redirect' => $this->get_return_url( $order ),
        );
    }

    /**
     * Output for the order received page.
     *
     * @since 1.0.0
     * @access public
     */
    public function thankyou_page() {

        if ( $this->instructions ) {
            echo wp_kses_post( wpautop( wptexturize( $this->instructions ) ) );
        }
    }

    /**
     * Add content to the WC emails.
     *
     * @since 1.0.0
     * @access public
     *
     * @param WC_Order $order         WC Order.
     * @param bool     $sent_to_admin Sent to admin flag.
     * @param bool     $plain_text    Plain text flag.
     */
    public function email_instructions( $order, $sent_to_admin, $plain_text = false ) {

        if ( ! $sent_to_admin && 'igfw_invoice_gateway' === $order->get_payment_method() && $order->has_status( 'on-hold' ) ) {
            if ( $this->instructions ) {
                echo wp_kses_post( wpautop( wptexturize( $this->instructions ) ) ) . PHP_EOL;
            }
        }
    }
}
