<?php
/**
 * Helpers utility class for plugin-wide common functionality.
 *
 * @package gutenify
 * @subpackage Utilities
 * @since 1.0.0
 */

namespace gutenify;

/**
 * Prevent direct access to the file.
 *
 * Ensures this file is being loaded within the WordPress environment.
 */
defined( 'ABSPATH' ) || exit;

/**
 * Class Helpers
 *
 * Provides commonly used helper functions across the plugin.
 */
class Helpers {

	/**
	 * Returns an array of active block slugs, filtered for customization.
	 *
	 * @return array List of block slugs.
	 */
	public static function active_blocks() {
		$plugin_constants = self::plugin_constants();
		$function_prefix  = $plugin_constants['plugin_main_function_prefix'];

		$blocks = array(
			'templates-browser',
			'advance-slide-item',
			'advance-slider',
			'buttons',
			'button',
			'container',
			'content-toggle-item',
			'faqs',
			'gallery-carousel',
			'grid',
			'grid-item',
			'icon',
			'info-box',
			'map',
			'post-carousel',
			'post-list',
			'section-title',
			'service',
			'slide-item',
			'slider',
			'star-rating',
			'team-member',
			'team',
			'testimonial',
			'testimonials',
			'wc-product-carousel',
			'wc-product-list',
			'notice-bar',
			'countup-v2',
			'icon-v2',

			// Optional blocks (can be enabled via filters or config).
			/**
			 * Depricated.
			 */
			'countup',
			// 'link-wrapper',
			// 'masonry',
			// 'custom-list',
			// 'social-share',
			// 'slider-v2',
		);

		return apply_filters( "{$function_prefix}_active_blocks", $blocks );
	}

	/**
	 * Returns plugin constants such as slugs, URLs, versions.
	 *
	 * @return array Associative array of plugin constants.
	 */
	public static function plugin_constants() {
		$defaults = array();

		// Filters allow other components or themes to define these constants.
		$filter_name = 'gutenify_plugin_constants';
		return apply_filters( $filter_name, $defaults );
	}

	/**
	 * Reads the asset file generated by wp-scripts and returns dependencies/version.
	 *
	 * @param string $path Path to the asset file.
	 * @return array Array containing 'dependencies' and 'version'.
	 */
	public static function asset_file_values( $path ) {
		$plugin_constants = self::plugin_constants();
		$version          = $plugin_constants['plugin_main_version'];

		if ( file_exists( $path ) ) {
			return include $path;
		}

		// Fallback if asset file doesn't exist.
		return array(
			'dependencies' => array(),
			'version'      => $version,
		);
	}

	/**
	 * Returns the base URL of the plugin's core directory.
	 *
	 * @return string Trailing-slashed URL.
	 */
	public static function core_base_url() {
		$plugin_constants = self::plugin_constants();
		return trailingslashit( $plugin_constants['core_base_url'] );
	}

	/**
	 * Returns the base filesystem path of the plugin's core directory.
	 *
	 * @return string Trailing-slashed directory path.
	 */
	public static function core_base_dir() {
		$plugin_constants = self::plugin_constants();
		return trailingslashit( $plugin_constants['core_base_dir'] );
	}

	/**
	 * Checks whether the PRO version of the plugin is active.
	 *
	 * @return bool True if PRO version is active.
	 */
	public static function is_pro_active() {
		$constants                   = self::plugin_constants();
		$plugin_main_function_prefix = $constants['plugin_main_function_prefix'];
		return apply_filters( $plugin_main_function_prefix . '_pro_activation_status', false );
	}
}
