<?php
/**
 * Transfer Locations tab.
 *
 * @since 4.0
 *
 * @package geo-my-wp
 */

// Exit if accessed directly.
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Transfer Location tab output.
 *
 * @access public
 *
 * @since 4.0
 *
 * @author Eyal Fitoussi
 */
function gmw_output_transfer_locations_tab() {

	?>
	<div class="gmw-settings-panel gmw-export-data-panel">

		<fieldset>

			<legend class="gmw-settings-panel-title"><?php esc_html_e( 'Transfer Location Between Different Sites', 'geo-my-wp' ); ?></legend>

			<div class="gmw-settings-panel-content">

				<div class="gmw-settings-panel-description">

					<div class="gmw-settings-panel-description">

						<p><?php esc_html_e( 'This tool will help you in the process of exporting the post\'s locations created on this site and importing them into a different site.', 'geo-my-wp' ); ?></p>

						<p><?php esc_html_e( 'The export button below will save the posts\' location that were generated by GEO my WP into a specific post custom field named "_gmw_post_locations_export".', 'geo-my-wp' ); ?></p>

						<p><?php esc_html_e( 'You will then be able to export those custom fields using the exporting tool of your choice and import them into another site where you will then use the Import tool to import the location from the custom fields back into GEO my WP.', 'geo-my-wp' ); ?></p>
					</div>

					<div class="gmw-settings-panel-field">
						<?php
						$importer = new GMW_Location_To_Meta_Fields_Importer();
						$importer->output();
						?>
					</div>
				</div>
			</div>
		</fieldset>
	</div>

	<?php
}
add_action( 'gmw_import_export_transfer_locations_tab', 'gmw_output_transfer_locations_tab' );

/**
 * Locations importer class
 *
 * @since 3.0
 *
 * @author Eyal Fitoussi
 */
class GMW_Location_To_Meta_Fields_Importer extends GMW_Locations_Importer {

	/**
	 * Records to import per batch.
	 *
	 * @var integer
	 */
	protected $records_per_batch = 25;

	/**
	 * Import button label.
	 *
	 * @return [type] [description]
	 */
	public function get_import_button_label() {
		return __( 'Export locations to custom fields', 'geo-my-wp' );
	}

	/**
	 * Form fields.
	 */
	public function form_fields() {
		?>
		<div class="gmw-settings-panel-field gmw-transfer-location-settings-wrapper">
			<div class="gmw-settings-panel-header">
				<label class="gmw-settings-label"><?php esc_attr_e( 'Select the location type to export:', 'geo-my-wp' ); ?></label>
			</div>
			<div class="gmw-settings-panel-input-container">
				<select name="gmw_meta_export_object_type">
					<option value="post"><?php esc_html_e( 'Posts', 'geo-my-wp' ); ?></option>
					<option value="user"><?php esc_html_e( 'Users', 'geo-my-wp' ); ?></option>
					<option value="bp_group"><?php esc_html_e( 'BP Groups', 'geo-my-wp' ); ?></option>
				</select>
			</div>
		</div>
		<?php
	}

	/**
	 * Get locations.
	 *
	 * @return [type] [description]
	 */
	public function query_locations() {

		global $wpdb;

		if ( empty( $_POST['formData'] ) ) { // phpcs:ignore WordPress.Security.NonceVerification.Missing, WordPress.Security.NonceVerification.Recommended, CSRF ok.
			return array();
		}

		// Sanitized below.
		parse_str( wp_unslash( $_POST['formData'] ), $form_data ); // phpcs:ignore WordPress.Security.NonceVerification.Missing,WordPress.Security.ValidatedSanitizedInput.MissingUnslash, WordPress.Security.NonceVerification.Recommended, CSRF ok.

		if ( empty( $form_data['gmw_meta_export_object_type'] ) ) {
			return array();
		}

		$this->object_type = sanitize_text_field( $form_data['gmw_meta_export_object_type'] );

		// Count rows only when init the importer.
		$count_rows = absint( $this->total_locations ) === 0 ? 'SQL_CALC_FOUND_ROWS' : '';

		// get locations.
		if ( 'post' === $this->object_type ) {

			$locations = $wpdb->get_results(
				$wpdb->prepare(
					"
					SELECT {$count_rows} gmwloc.*
					FROM {$wpdb->prefix}gmw_locations gmwloc
					INNER JOIN {$wpdb->prefix}posts wpposts
					ON gmwloc.object_id = wpposts.ID
					WHERE gmwloc.object_type = %s
					ORDER BY gmwloc.object_id
					LIMIT %d, %d",
					array(
						$this->object_type,
						$this->records_completed,
						$this->records_per_batch,
					)
				),
				ARRAY_A
			); // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared, WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching.

		} elseif ( 'user' === $this->object_type ) {

			$locations = $wpdb->get_results(
				$wpdb->prepare(
					"
					SELECT {$count_rows} gmwloc.*
					FROM {$wpdb->prefix}gmw_locations gmwloc
					INNER JOIN {$wpdb->prefix}users users
					ON gmwloc.object_id = users.ID
					WHERE gmwloc.object_type = %s
					ORDER BY gmwloc.object_id
					LIMIT %d, %d",
					array(
						$this->object_type,
						$this->records_completed,
						$this->records_per_batch,
					)
				),
				ARRAY_A
			); // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared, WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching.
		}

		// abort if nothing was found.
		if ( empty( $locations ) ) {
			return array();
		}

		// count all rows only when init the importer.
		$this->total_locations = absint( $this->total_locations ) === 0 ? $wpdb->get_var( 'SELECT FOUND_ROWS()' ) : $this->total_locations; // WPCS: db call ok, cache ok, unprepared SQL ok.

		return $locations;
	}

	/**
	 * Loop and export locations to meta fields.
	 *
	 * @return void
	 */
	public function import_locations() {

		global $wpdb;

		// loop locations.
		foreach ( $this->locations as $location ) {

			// Get location meta, we are going to check if this location was already imported.
			$check_meta = get_metadata( $this->object_type, $location['object_id'], '_gmw_locations_export' );

			if ( ! empty( $check_meta ) ) {

				$exists = false;

				foreach ( $check_meta as $this_meta ) {

					// Abort if location exists.
					if ( ! empty( $this_meta['location']['ID'] ) && absint( $this_meta['location']['ID'] ) === absint( $location['ID'] ) ) {

						$exists = true;

						continue;
					}
				}

				// Skip this location if already imported.
				if ( $exists ) {

					$this->location_exist( $location, $location );

					continue;
				}
			}

			// get location meta.
			$location_meta = $wpdb->get_results(
				$wpdb->prepare(
					"
					SELECT {$count_rows} *
					FROM {$wpdb->prefix}gmw_locationmeta
					WHERE location_id = %d
					ORDER BY location_id",
					array(
						$location['ID'],
					)
				),
				ARRAY_A
			); // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared, WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching.

			// Combine location and location meta into a single array.
			$new_location = array(
				'location'      => $location,
				'location_meta' => $location_meta,
			);

			// Import data to custom meta.
			$meta_id = add_metadata( $this->object_type, $location['object_id'], '_gmw_locations_export', $new_location );

			// if location failed importing.
			if ( empty( $meta_id ) ) {

				$this->location_failed( $location, $location );

				continue;
			}

			$this->location_imported( $location['ID'], $location, $location );
		}
	}
}
