<?php

namespace WPDesk\FPF\Free\Validation\Rule;

use WPDesk\FPF\Free\Helper\DateFormatConverter;

/**
 * Supports "Excluded days of week" validation rule for fields.
 */
class WeekDaysExcludedRule implements RuleInterface {

	/**
	 * {@inheritdoc}
	 */
	public function validate_value( array $field_data, array $field_type, $value ): bool {
		if ( ! ( $field_type['has_days_excluded'] ?? false ) ) {
			return true;
		}

		$days_excluded = $field_data['days_excluded'] ?? [];
		if ( ! $days_excluded ) {
			return true;
		}

		$date_format = DateFormatConverter::to_php( $field_data['date_format'] ?? '' );
		$delimiter   = DateFormatConverter::get_dates_delimiter( $field_data );
		$dates       = ( $value ) ? array_map( 'trim', explode( $delimiter, $value ) ) : [];
		foreach ( $dates as $date ) {
			$datetime = \DateTime::createFromFormat( $date_format, $date );
			if ( $datetime === false || in_array( $datetime->format( 'w' ), $days_excluded, false ) ) {
				return false;
			}
		}
		return true;
	}

	/**
	 * {@inheritdoc}
	 */
	public function get_error_message( array $field_data ): string {
		return sprintf(
		/* translators: %s: field label */
			__( '%s has a date with an excluded day of the week.', 'flexible-product-fields' ),
			'<strong>' . $field_data['title'] . '</strong>'
		);
	}
}
