<?php

namespace WPDesk\FPF\Free\Validation\Rule;

use WPDesk\FPF\Free\Helper\DateFormatConverter;

/**
 * Supports "Time of day closing" validation rule for fields.
 */
class TodayMaxHourRule implements RuleInterface {

	/**
	 * {@inheritdoc}
	 */
	public function validate_value( array $field_data, array $field_type, $value ): bool {
		if ( ! ( $field_type['has_today_max_hour'] ?? false ) ) {
			return true;
		}

		$max_hour = $field_data['today_max_hour'] ?? '';
		if ( ! $max_hour ) {
			return true;
		}
		$max_hour_datetime = \DateTime::createFromFormat( 'H:i', $max_hour );
		if ( $max_hour_datetime === false || wp_date( 'H:i' ) <= $max_hour_datetime->format( 'H:i' ) ) {
			return true;
		}

		$date_today = wp_date( 'Ymd' );

		$date_format = DateFormatConverter::to_php( $field_data['date_format'] ?? '' );
		$delimiter   = DateFormatConverter::get_dates_delimiter( $field_data );
		$dates       = ( $value ) ? array_map( 'trim', explode( $delimiter, $value ) ) : [];
		foreach ( $dates as $date ) {
			$datetime = \DateTime::createFromFormat( $date_format, $date );
			if ( $datetime === false || $datetime->format( 'Ymd' ) === $date_today ) {
				return false;
			}
		}
		return true;
	}

	/**
	 * {@inheritdoc}
	 */
	public function get_error_message( array $field_data ): string {
		return sprintf(
		/* translators: %s: field label */
			__( '%s has a current date that is no longer available.', 'flexible-product-fields' ),
			'<strong>' . $field_data['title'] . '</strong>'
		);
	}
}
