<?php
/**
 * Template Functions
 *
 * @package     EPL
 * @subpackage  Functions/Templates
 * @copyright   Copyright (c) 2020, Merv Barrett
 * @license     http://opensource.org/licenses/gpl-2.0.php GNU Public License
 * @since       1.0
 */

// Exit if accessed directly.
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

// phpcs:disable WordPress.NamingConventions.ValidVariableName
// phpcs:disable WordPress.Security.NonceVerification

/**
 * Property Object
 *
 * @param WP_Post $post Post object.
 *
 * @since 2.2
 * @since 3.4.38 Support for third & fourth agents.
 */
function epl_reset_property_object( $post ) {

	global $epl_author, $epl_author_secondary, $epl_author_third, $epl_author_fourth;

	if ( ! is_epl_post() ) {
		return;
	}

	global $property;
	$property = new EPL_Property_Meta( $post );
	$ID       = epl_listing_has_primary_agent(); //phpcs:ignore
	if ( $ID ) {
		$epl_author = new EPL_Author_meta( $ID ); //phpcs:ignore

	} else {
		$epl_author = new EPL_Author_meta( $post->post_author );
	}

	$SEC_ID = epl_listing_has_secondary_author();

	if ( $SEC_ID ) {
		$epl_author_secondary = new EPL_Author_meta( $SEC_ID );
	}

	$third_ID = epl_listing_has_third_agent();

	if ( $third_ID ) {
		$epl_author_third = new EPL_Author_meta( $third_ID );
	}

	$fourth_ID = epl_listing_has_fourth_agent();

	if ( $fourth_ID ) {
		$epl_author_fourth = new EPL_Author_meta( $fourth_ID );
	}
}
add_action( 'the_post', 'epl_reset_property_object' );

/**
 * Make $property global available for hooks before the_post
 *
 * @since 2.2
 * @since 3.5 Support for third & fourth agent.
 */
function epl_create_property_object() {

	global $post,$property,$epl_author,$epl_author_secondary, $epl_author_third, $epl_author_fourth;

	if ( is_author() ) {
		$author_id  = get_query_var( 'author' );
		$epl_author = new EPL_Author_meta( $author_id );
	}
	if ( is_null( $post ) ) {
		return;
	}
	$epl_author = new EPL_Author_meta( $post->post_author );

	if ( is_epl_post() ) {
		$property = new EPL_Property_Meta( $post );
		$ID       = epl_listing_has_secondary_author();
		if ( $ID ) {
			$epl_author_secondary = new EPL_Author_meta( $ID );
		}
		$third_ID = epl_listing_has_third_agent();

		if ( $third_ID ) {
			$epl_author_third = new EPL_Author_meta( $third_ID );
		}

		$fourth_ID = epl_listing_has_fourth_agent();

		if ( $fourth_ID ) {
			$epl_author_fourth = new EPL_Author_meta( $fourth_ID );
		}
	}

	add_filter( 'excerpt_length', 'epl_archive_custom_excerpt_length', 999 );
}

add_action( 'wp', 'epl_create_property_object' );

/**
 * Selecting Card Display Style
 *
 * @since 1.0
 * @since 3.4.4 Removed default template check for single templates as this caused incorrect templates to load in some cases.
 */
function epl_property_single() {
	global $epl_settings;

	$action_check = has_action( 'epl_single_template' );
	if ( ! empty( $action_check ) ) {
		do_action( 'epl_single_template' );
	} else {
		epl_property_single_default();
	}
}
add_action( 'epl_property_single', 'epl_property_single', 10, 1 );

/**
 * Featured Image template now loading through filter
 *
 * @param      string  $image_size   The image size.
 * @param      string  $image_class  The image class.
 * @param      boolean $link         The link.
 *
 * @since      1.2.0
 * @since      3.4.8 Corrected missing parameter count to 3.
 * @since      3.4.38 Added filter epl_property_featured_image_args to control all parameters & epl_no_property_featured_image action.
 * @since      3.4.39 Added missing arguments variable to epl_no_property_featured_image action.
 */
function epl_property_featured_image( $image_size = 'index_thumbnail', $image_class = 'index-thumbnail', $link = true ) {

	$args = apply_filters(
		'epl_property_featured_image_args',
		array(
			'image_size'  => $image_size,
			'image_class' => $image_class,
			'link'        => $link,
		)
	);
	/**
	 * Filter: Allow user or extension to enable or disable link behaviour on featured image.
	 */
	$args['link'] = apply_filters( 'epl_property_featured_image_link', $args['link'] );

	if ( has_post_thumbnail() ) { ?>
		<div class="entry-image">
			<div class="epl-featured-image it-featured-image">
				<?php if ( true === $args['link'] ) { ?>
					<a href="<?php the_permalink(); ?>">
				<?php } ?>
						<?php the_post_thumbnail( $args['image_size'], array( 'class' => $args['image_class'] ) ); ?>
				<?php if ( true === $args['link'] ) { ?>
					</a>
				<?php } ?>
			</div>
		</div>
		<?php
	} else {
		do_action( 'epl_no_property_featured_image', $args );
	}

}
add_action( 'epl_property_featured_image', 'epl_property_featured_image', 10, 3 );
add_action( 'epl_single_featured_image', 'epl_property_featured_image', 10, 3 );

/**
 * Featured Image on archive template now loading through filter
 *
 * @param string $image_size  The image size.
 * @param string $image_class The image class.
 * @param bool   $link        Enable or disable the link with true/false. Default true.
 * @param bool   $stickers    Enable or disable the stickers with true/false. Default true.
 *
 * @since 2.2
 * @since 3.4.27 New: Additional param to disable / enable stickers
 * @since 3.4.30 Fix: Missing parameter filter. Increased from 3 to 4.
 * @since 3.4.38 Added filter epl_property_archive_featured_image to control all parameters & epl_no_archive_featured_image action if no featured image.
 * @since 3.5.0  Replaced stickers html with an action hook epl_stickers_featured_image.
 */
function epl_property_archive_featured_image( $image_size = 'epl-image-medium-crop', $image_class = 'teaser-left-thumb', $link = true, $stickers = true ) {

	if ( empty( $image_size ) ) {
		$image_size = 'epl-image-medium-crop';
	}

	$args = apply_filters(
		'epl_property_archive_featured_image_args',
		array(
			'image_size'  => $image_size,
			'image_class' => $image_class,
			'link'        => $link,
			'stickers'    => $stickers,
		)
	);
	/**
	 * Filter: Allow user or extension to enable or disable link behaviour on archive image.
	 */
	$args['link'] = apply_filters( 'epl_property_archive_featured_image_link', $args['link'] );

	if ( has_post_thumbnail() ) {
		?>
		<div class="epl-archive-entry-image">
			<?php if ( true === $args['link'] ) { ?>
				<a href="<?php the_permalink(); ?>">
			<?php } ?>
				<div class="epl-blog-image">
					<?php do_action( 'epl_stickers_featured_image', $args ); ?>
					<?php the_post_thumbnail( $args['image_size'], array( 'class' => $args['image_class'] ) ); ?>
				</div>
			<?php if ( true === $args['link'] ) { ?>
				</a>
			<?php } ?>
		</div>
		<?php
	} else {
		do_action( 'epl_no_archive_featured_image', $args );
	}

}
add_action( 'epl_property_archive_featured_image', 'epl_property_archive_featured_image', 10, 4 );

/**
 * Featured image sticker callback.
 *
 * @since 3.5.0
 *
 * @param array $args   Arguments.
 */
function epl_stickers_featured_image( $args ) {
	if ( $args['stickers'] ) :
		?>
		<div class="epl-stickers-wrapper">
			<?php echo wp_kses_post( epl_get_price_sticker() ); ?>
		</div>
		<?php
	endif;
}
add_action( 'epl_stickers_featured_image', 'epl_stickers_featured_image' );

/**
 * Featured Image in widgets
 *
 * @since      2.2
 *
 * @param      string  $image_size   The image size.
 * @param      string  $image_class  The image class.
 * @param      boolean $link         The link.
 */
function epl_property_widgets_featured_image( $image_size = 'epl-image-medium-crop', $image_class = 'teaser-left-thumb', $link = true ) {

	if ( has_post_thumbnail() ) {
		?>
		<div class="epl-archive-entry-image">
			<?php if ( $link ) { ?>
				<a href="<?php the_permalink(); ?>">
			<?php } ?>
				<div class="epl-blog-image">
					<?php the_post_thumbnail( $image_size, array( 'class' => $image_class ) ); ?>
				</div>
			<?php if ( $link ) { ?>
				</a>
			<?php } ?>
		</div>
		<?php
	}
}
add_action( 'epl_property_widgets_featured_image', 'epl_property_widgets_featured_image', 10, 3 );

/**
 * Single Listing Templates
 *
 * @since      1.0
 */
function epl_property_single_default() {

	global $epl_settings;
	if ( isset( $epl_settings['epl_feeling_lucky'] ) && 'on' === $epl_settings['epl_feeling_lucky'] ) {
		epl_get_template_part( 'content-listing-single-compatibility.php' );
	} else {
		$single_tpl = 'content-listing-single.php';
		$single_tpl = apply_filters( 'epl_property_single_default', $single_tpl );
		epl_get_template_part( $single_tpl );
	}
}

/**
 * Template Path
 *
 * @since      2.0
 *
 * @return string
 */
function epl_get_content_path() {
	return apply_filters( 'epl_templates_base_path', EPL_PATH_TEMPLATES_CONTENT );
}

/**
 * Template Fallback Path
 *
 * @return mixed|void
 * @since      3.0
 */
function epl_get_fallback_content_path() {
	return apply_filters( 'epl_templates_fallback_base_path', EPL_PATH_TEMPLATES_CONTENT );
}

/**
 * Attempts to load templates in order of priority
 *
 * @param string $template Template name.
 * @param array  $arguments Options to pass to template.
 * @param bool   $default_template Pass a default template.
 *
 * @since 3.0
 * @since 3.4.38 New: Additional parameter default_template to pass the default template which will be used if the template is not found.
 * @since 3.4.41 Fix: Only the default template was loading.
 * @since 3.5.3  Hook to filter find array as well as final template.
 * @since 3.5.11 Pass the original template in the filter.
 */
function epl_get_template_part( $template, $arguments = array(), $default_template = false ) {

	$original_template = $template;
	$base_path         = epl_get_content_path();
	$default           = $default_template ? $default_template : $template;
	$find[]            = epl_template_path() . $template;
	$find              = apply_filters( 'epl_template_part_find', $find, $template, $arguments, $default_template );
	$template_location = locate_template( array_unique( $find ) );

	if ( ! $template_location ) {
		$template          = $base_path . $template;
		$original_template = $template;
		if ( ! file_exists( $template ) ) {
			// Fallback to core.
			$base_path = epl_get_fallback_content_path();
			$template  = $base_path . $default;
		}
	} else {
		$template = $template_location;
	}

	if ( ! isset( $arguments['epl_author'] ) ) {
		global $epl_author;
	}

	foreach ( $arguments as $key => $val ) {
		${$key} = $val;
	}

	include apply_filters( 'epl_get_template_part', $template, $arguments, $default_template, $original_template );
}

/**
 * Modify the Excerpt length on Archive pages
 *
 * @since 1.0
 * @since 3.4.27 Alter length only for EPL posts.
 * @param string $length Excerpt word length.
 * @return int|string
 */
function epl_archive_custom_excerpt_length( $length ) {

	if ( ! is_epl_post() ) {
		return $length;
	}

	return epl_get_option( 'display_excerpt_length', 22 );
}


/**
 * Filter which listing status should not be displayed
 *
 * @return mixed|void
 * @since 3.1.20
 * @since 3.4.45 Added deleted status.
 */
function epl_hide_listing_statuses() {
	return apply_filters( 'epl_hide_listing_statuses', array( 'withdrawn', 'offmarket', 'deleted' ) );
}

/**
 * Selecting Card Display Style
 *
 * Allows the use of one function where we can then select a different template
 * when needed
 *
 * @param string $template The template.
 * @param string $default  Optional template name.
 *
 * @since 1.0.0
 * @since 3.4.4 Removed default template check for loop templates as this caused incorrect templates to load in some cases.
 * @since 3.4.23 Removed compatibility template for loop as we are passing the class using post_class filter.
 * @since 3.4.36 New: Additional action support for listing templates. Actions are: epl_loop_template_{post_type}, epl_loop_template_listing.
 * @since 3.4.38 New: Additional parameters default to pass the default template which will be used if the template is not found.
 * @since 3.5 New: Additional args for custom template action : epl_loop_template_{$post_type}, epl_loop_template.
 */
function epl_property_blog( $template = '', $default = 'default' ) {

	if ( empty( $template ) || 'blog' === $template ) {
		$template = 'default';
	}
	$template = str_replace( '_', '-', $template );

	global $epl_settings, $property;

	if ( is_null( $property ) ) {
		return;
	}

	$property_status = $property->get_property_meta( 'property_status' );
	// Status Removal Do Not Display Withdrawn or OffMarket listings.
	if ( ! in_array( $property_status, epl_hide_listing_statuses(), true ) ) {
		// Do Not Display Withdrawn or OffMarket listings.

		$action_check_type = has_action( 'epl_loop_template_' . $property->post->post_type );
		$action_check      = has_action( 'epl_loop_template' );
		$action_check_core = false;
		$action_exists     = false;

		if ( in_array( $property->post->post_type, epl_get_core_post_types(), true ) ) {
			$action_check_core = has_action( 'epl_loop_template_listing' );
		}

		// Check for action in order of priority : epl_loop_template_{post_type} > epl_loop_template_listing ( only for core )  > epl_loop_template.
		if ( ! empty( $action_check_type ) ) {
				do_action( 'epl_loop_template_' . $property->post->post_type, get_defined_vars() );
				$action_exists = true;
		} elseif ( ! empty( $action_check_core ) ) {
				do_action( 'epl_loop_template_listing', get_defined_vars() );
				$action_exists = true;
		} elseif ( ! empty( $action_check ) ) {
				do_action( 'epl_loop_template', get_defined_vars() );
				$action_exists = true;
		}

		// Fallback to core template.
		if ( ! $action_exists ) {
			$tpl_name = 'loop-listing-blog-' . $template . '.php';
			$tpl_name = apply_filters( 'epl_property_blog_template', $tpl_name, get_defined_vars() );
			epl_get_template_part( $tpl_name, array(), 'loop-listing-blog-' . $default . '.php' );
		}
	} // End Status Removal.
}
add_action( 'epl_property_blog', 'epl_property_blog', 10, 2 );

/**
 * Renders default author box
 *
 * @since 3.2
 * @since 3.4.38 Support for third & fourth agent.
 */
function epl_property_author_default() {
	global $epl_author_secondary, $epl_author_third, $epl_author_fourth;
	epl_get_template_part( 'content-author-box.php' );
	if ( is_epl_post() ) {
		if ( epl_listing_has_secondary_author() ) {
			epl_get_template_part( 'content-author-box.php', array( 'epl_author' => $epl_author_secondary ) );
		}
		if ( epl_listing_has_third_agent() ) {
			epl_get_template_part( 'content-author-box.php', array( 'epl_author' => $epl_author_third ) );
		}
		if ( epl_listing_has_fourth_agent() ) {
			epl_get_template_part( 'content-author-box.php', array( 'epl_author' => $epl_author_fourth ) );
		}
		epl_reset_post_author();
	}
}
/**
 * AUTHOR CARD : Tabbed Style
 *
 * @since      1.0
 */
function epl_property_author_box() {

	if ( has_action( 'epl_author_template' ) ) {
		do_action( 'epl_author_template' );
	} else {
		epl_property_author_default();
	}
}
add_action( 'epl_single_author', 'epl_property_author_box', 10 );

/**
 * Reset post author
 *
 * @since 1.0
 */
function epl_reset_post_author() {
	global $post, $epl_author;
	if ( class_exists( 'EPL_Author_meta' ) ) {

		$ID = epl_listing_has_primary_agent();

		if ( is_epl_post() && $ID ) {
			$epl_author = new EPL_Author_meta( $ID );
		} else {
			$epl_author = new EPL_Author_meta( $post->post_author );
		}
	}
}

/**
 * AUTHOR CARD : Standard
 *
 * @since 1.0
 * @since 3.4.38 Support for third & fourth agent.
 */
function epl_property_author_box_simple_card() {

	global $epl_author_secondary, $epl_author_third, $epl_author_fourth;
	epl_get_template_part( 'content-author-box-simple-card.php' );
	if ( is_epl_post() ) {
		if ( epl_listing_has_secondary_author() ) {
			epl_get_template_part( 'content-author-box-simple-card.php', array( 'epl_author' => $epl_author_secondary ) );
		}
		if ( epl_listing_has_third_agent() ) {
			epl_get_template_part( 'content-author-box-simple-card.php', array( 'epl_author' => $epl_author_third ) );
		}
		if ( epl_listing_has_fourth_agent() ) {
			epl_get_template_part( 'content-author-box-simple-card.php', array( 'epl_author' => $epl_author_fourth ) );
		}
		epl_reset_post_author();
	}
}

/**
 * AUTHOR CARD : Gravatar
 *
 * @since 1.0
 * @since 3.4.38 Support for third & fourth agent.
 */
function epl_property_author_box_simple_grav() {

	global $epl_author_secondary, $epl_author_third, $epl_author_fourth;
	epl_get_template_part( 'content-author-box-simple-grav.php' );
	if ( is_epl_post() ) {
		if ( epl_listing_has_secondary_author() ) {
			epl_get_template_part( 'content-author-box-simple-grav.php', array( 'epl_author' => $epl_author_secondary ) );
		}
		if ( epl_listing_has_third_agent() ) {
			epl_get_template_part( 'content-author-box-simple-grav.php', array( 'epl_author' => $epl_author_third ) );
		}
		if ( epl_listing_has_fourth_agent() ) {
			epl_get_template_part( 'content-author-box-simple-grav.php', array( 'epl_author' => $epl_author_fourth ) );
		}
		epl_reset_post_author();
	}
}

/**
 * WIDGET LISTING : Listing Card
 *
 * @param      string $display            The display.
 * @param      string $image              The image.
 * @param      string $title              The title.
 * @param      string $icons              The icons.
 * @param      string $more_text          The more text with epl_property_widget_read_more filter.
 * @param      string $d_excerpt          The d excerpt.
 * @param      string $d_suburb           The d suburb.
 * @param      string $d_street           The d street.
 * @param      string $d_price            The d price.
 * @param      string $d_more             The d more.
 * @param      string $d_inspection_time  The d inspection time.
 * @param      string $d_ical_link        The d ical link.
 *
 * @since 1.0.0
 * @since 3.4.13 for custom display, file extension not required and file name format enforced to the format widget-content-listing-{$display}.php
 * @since 3.4.38 Support for third & fourth agent.
 * @since 3.4.38 PHP 8.0 fix for parameter following optional parameter. Moved $more_text. Added epl_property_widget_read_more filter.
 */
function epl_property_widget( $display, $image, $title, $icons, $more_text, $d_excerpt, $d_suburb, $d_street, $d_price, $d_more, $d_inspection_time, $d_ical_link ) {
	global $property;

	if ( is_null( $property ) ) {
		return;
	}

	if ( empty( $more_text ) ) {
		$more_text = apply_filters( 'epl_property_widget_read_more', __( 'Read More', 'easy-property-listings' ) );
	}

	$property_status = $property->get_property_meta( 'property_status' );

	switch ( $display ) {
		case 'list':
			$tpl = 'widget-content-listing-list.php';
			break;
		case 'hide-image':
			$tpl = 'widget-content-listing-hide-image.php';
			break;
		case 'image-only':
			$tpl = 'widget-content-listing-image.php';
			break;
		case 'image':
			$tpl = 'widget-content-listing.php';
			break;
		default:
			$tpl = $display;
			if ( ! epl_starts_with( $tpl, 'widget-content-listing' ) ) {
				$tpl = 'widget-content-listing-' . $tpl;
			}
			if ( ! epl_ends_with( $tpl, '.php' ) ) {
				$tpl .= '.php';
			}
			break;
	}

	// Status Removal.
	if ( ! in_array( $property_status, epl_hide_listing_statuses(), true ) ) {
		// Do Not Display Withdrawn or OffMarket listings.
		$arg_list = get_defined_vars();
		if ( has_action( 'epl_listing_widget_template' ) ) {
			do_action( 'epl_listing_widget_template', $tpl, $arg_list );
		} else {
			epl_get_template_part( $tpl, $arg_list );
		}
	} // End Status Removal.
}

/**
 * WIDGET LISTING : Listing List
 *
 * @since      1.0
 */
function epl_property_widget_list_option() {
	$property_status = get_post_meta( get_the_ID(), 'property_status', true );
	// Status Removal.
	if ( ! in_array( $property_status, epl_hide_listing_statuses(), true ) ) {
		epl_get_template_part( 'widget-content-listing-list.php' );
	}
}

/**
 * WIDGET LISTING : Image Only
 *
 * @since      1.0
 *
 * @param      string $image  The image.
 */
function epl_property_widget_image_only_option( $image ) {
	$property_status = get_post_meta( get_the_ID(), 'property_status', true );
	// Status Removal.
	if ( ! in_array( $property_status, epl_hide_listing_statuses(), true ) ) {
		$arg_list = get_defined_vars();
	}
	epl_get_template_part( 'widget-content-listing-image.php', $arg_list );
}


/**
 * WIDGET LISTING : Widget Tall Card
 *
 * @since 1.0
 * @since 3.3 Revised.
 *
 * @param      string $d_image   The d image.
 * @param      string $d_icons   The d icons.
 * @param      string $d_bio     The d bio.
 * @param      string $username  The username.
 */
function epl_property_author_box_simple_card_tall( $d_image, $d_icons, $d_bio, $username ) {

	if ( ! empty( $username ) ) {
		epl_show_author_widget_by_username( $d_image, $d_icons, $d_bio, $username );
		return;
	}

	global $property,$epl_author,$epl_author_secondary,$epl_author_third, $epl_author_fourth;
	if ( is_null( $epl_author ) ) {
		return;
	}

	$arg_list = get_defined_vars();
	epl_get_template_part( 'widget-content-author-tall.php', $arg_list );

	// Second Author.
	if ( is_single() && ! is_null( $property ) ) {

		if ( is_epl_post() ) {
			if ( epl_listing_has_secondary_author() ) {
				$epl_author = $epl_author_secondary;
				epl_get_template_part( 'widget-content-author-tall.php', $arg_list );
			}
			if ( epl_listing_has_third_agent() ) {
				$epl_author = $epl_author_third;
				epl_get_template_part( 'widget-content-author-tall.php', $arg_list );
			}
			if ( epl_listing_has_fourth_agent() ) {
				$epl_author = $epl_author_fourth;
				epl_get_template_part( 'widget-content-author-tall.php', $arg_list );
			}
			epl_reset_post_author();
		}
	}
}

/**
 * Display widget by username
 *
 * @since      3.3
 *
 * @param      string $d_image   The d image.
 * @param      string $d_icons   The d icons.
 * @param      string $d_bio     The d bio.
 * @param      string $username  The username.
 */
function epl_show_author_widget_by_username( $d_image, $d_icons, $d_bio, $username ) {
	$username = explode( ',', $username );
	$username = array_filter( $username );
	foreach ( $username as $uname ) {
		$author = get_user_by( 'login', sanitize_user( $uname ) );
		if ( false !== $author ) {
			$epl_author = new EPL_Author_meta( $author->ID );
			$arg_list   = get_defined_vars();
			epl_get_template_part( 'widget-content-author-tall.php', $arg_list );
		}
	}
}

/**
 * Get the full address
 *
 * @return string  The full address of the listing.
 *
 * @since 1.0.0
 */
function epl_property_get_the_full_address() {
	global $property;

		$address = '';
		$sub_num = $property->get_property_meta( 'property_address_sub_number' );
	if ( ! empty( $sub_num ) ) {
		$address .= $property->get_property_meta( 'property_address_sub_number' ) . '/';
	}
		$address .= $property->get_property_meta( 'property_address_street_number' ) . ' ';
		$address .= $property->get_property_meta( 'property_address_street' ) . ' ';
		$address .= $property->get_property_meta( 'property_address_suburb' ) . ' ';
		$address .= $property->get_property_meta( 'property_address_city' ) . ', ';
		$address .= $property->get_property_meta( 'property_address_state' ) . ' ';
		$address .= $property->get_property_meta( 'property_address_postal_code' ) . ' ';
		$address .= $property->get_property_meta( 'property_address_country' );

	return $address;
}

/**
 * Get the full address
 *
 * @hooked epl_property_title
 * @hooked property_tab_address
 *
 * @param bool   $full Set to false too only display the street address.
 * @param bool   $street_separator Display the street separator.
 * @param string $separator_symbol Symbol to use as the street separator, default is a comma.
 *
 * @since 1.0
 * @since 3.3.3 Revised.
 * @since 3.4.8 Corrected separator location to appear AFTER the street name and options to control display.
 * @since 3.4.9 Added option allowing passing of $full parameter as false to restrict output to street items only.
 * @since 3.5.14 Corrected escaping function to esc_html.
 */
function epl_property_the_address( $full = true, $street_separator = true, $separator_symbol = ',' ) {

	global $property, $epl_settings;

	if ( ! is_bool( $full ) ) {
		$full = true;
	}

	$epl_property_address_separator        = apply_filters( 'epl_property_address_separator', ',' );
	$epl_property_address_separator_suburb = apply_filters( 'epl_property_address_separator_suburb', false );
	$epl_property_address_separator_city   = apply_filters( 'epl_property_address_separator_city', false );

	?>
	<?php if ( 'yes' === $property->get_property_meta( 'property_address_display' ) ) { ?>
		<span class="item-street"><?php echo wp_kses_post( $property->get_formatted_property_address( $street_separator, $separator_symbol ) ); ?></span>
	<?php } ?>

	<?php
	if ( true === $full ) {
		?>
		<span class="entry-title-sub">
			<?php
			if ( 'commercial' === $property->post_type || 'business' === $property->post_type ) {
				if ( 'yes' === $property->get_property_meta( 'property_com_display_suburb' ) || 'yes' === $property->get_property_meta( 'property_address_display' ) ) {
					?>
					<span class="item-suburb"><?php echo esc_html( $property->get_property_meta( 'property_address_suburb' ) ); ?></span>
					<?php
					if ( true === $epl_property_address_separator_suburb && strlen( trim( $property->get_property_meta( 'property_address_suburb' ) ) ) ) {
						echo '<span class="item-separator">' . esc_html( $epl_property_address_separator ) . '</span>';
					}
				}
			} else {
				?>
				<span class="item-suburb"><?php echo esc_html( $property->get_property_meta( 'property_address_suburb' ) ); ?></span>
				<?php
				if ( true === $epl_property_address_separator_suburb && strlen( trim( $property->get_property_meta( 'property_address_suburb' ) ) ) ) {
					echo '<span class="item-separator">' . esc_html( $epl_property_address_separator ) . '</span>';
				}
			}
			?>

			<?php
			if ( 'yes' === $property->get_epl_settings( 'epl_enable_city_field' ) ) {
				?>
				<span class="item-city"><?php echo esc_html( $property->get_property_meta( 'property_address_city' ) ); ?></span>
				<?php
				if ( true === $epl_property_address_separator_city && strlen( trim( $property->get_property_meta( 'property_address_city' ) ) ) ) {
					echo '<span class="item-separator">' . esc_html( $epl_property_address_separator ) . '</span>';
				}
			}
			?>
			<span class="item-state"><?php echo esc_html( $property->get_property_meta( 'property_address_state' ) ); ?></span>
			<span class="item-pcode"><?php echo esc_html( $property->get_property_meta( 'property_address_postal_code' ) ); ?></span>
			<?php
			if ( 'yes' === $property->get_epl_settings( 'epl_enable_country_field' ) ) {
				?>
				<span class="item-country"><?php echo esc_html( $property->get_property_meta( 'property_address_country' ) ); ?></span>
				<?php
			}
			?>
		</span>
		<?php
	}
}
add_action( 'epl_property_title', 'epl_property_the_address', 10, 3 );
add_action( 'epl_property_tab_address', 'epl_property_the_address', 10, 3 );
add_action( 'epl_property_address', 'epl_property_the_address', 10, 3 );

/**
 * Suburb Name Kept for listing templates extensions which use this function
 *
 * @since 1.3
 * @since 3.1.18 Revised.
 * @since 3.5.14 Corrected escaping function to esc_html.
 */
function epl_property_suburb() {
	global $property;
	// Commercial and Business Address.
	if ( 'commercial' === $property->post_type || 'business' === $property->post_type ) {
		?>

		<span class="entry-title-sub">
			<?php if ( 'yes' === $property->get_property_meta( 'property_com_display_suburb' ) ) { ?>
				<span class="item-suburb"><?php echo esc_html( $property->get_property_meta( 'property_address_suburb' ) ); ?></span>
			<?php } else { ?>
				<?php $prop_addr_city = $property->get_property_meta( 'property_address_city' ); ?>
				<?php if ( ! empty( $prop_addr_city ) ) { ?>
					<span class="item-city"><?php echo esc_html( $property->get_property_meta( 'property_address_city' ) ) . ' '; ?></span>
				<?php } ?>

				<span class="item-state"><?php echo esc_html( $property->get_property_meta( 'property_address_state' ) ) . ' '; ?></span>
				<span class="item-pcode"><?php echo esc_html( $property->get_property_meta( 'property_address_postal_code' ) ); ?></span>
			<?php } ?>
		</span>

		<?php
	} else {
		?>
		<span class="entry-title-sub">
			<span class="item-suburb"><?php echo esc_html( $property->get_property_meta( 'property_address_suburb' ) ); ?></span>
		</span>
		<?php
	}
}
add_action( 'epl_property_suburb', 'epl_property_suburb' );

/**
 * Get the price
 *
 * @since      1.0 @hooked property_price
 * @hooked property_price_content
 */
function epl_property_price() {
	echo wp_kses_post( epl_get_property_price() );
}
add_action( 'epl_property_price', 'epl_property_price' );
add_action( 'epl_property_price_content', 'epl_property_price' );

/**
 * Get Property icons
 *
 * @param      array  $args        The arguments.
 * @param      string $returntype  The returntype.
 *
 * @return false|string
 *
 * @since 1.0
 * @since 3.3.3 Added switch.
 */
function epl_get_property_icons( $args = array(), $returntype = 'i' ) {

	global $property;

	$defaults = array( 'bed', 'bath', 'parking', 'ac', 'pool' );

	$icons      = apply_filters( 'epl_get_property_icons', $defaults );
	$returntype = apply_filters( 'epl_icons_return_type', $returntype );

	ob_start();
	//phpcs:disable
	foreach ( $icons as $icon ) {

		if ( ! empty( $args ) && ! in_array( $icon, $args, true ) ) {
			continue;
		}

		switch ( $icon ) {

			case 'bed':
				echo $property->get_property_bed( $returntype );
				break;

			case 'bath':
				echo $property->get_property_bath( $returntype );
				break;

			case 'parking':
				echo $property->get_property_parking( $returntype );
				break;

			case 'ac':
				echo $property->get_property_air_conditioning( $returntype );
				break;

			case 'pool':
				echo $property->get_property_pool( $returntype );
				break;

			default:
				// Action to hook additional icons.
				do_action( 'epl_get_property_icon_' . $icon );
				break;
		}
	}
	//phpcs:enable
	return ob_get_clean();
}

/**
 * Property icons
 *
 * @param string $returntype The returntype.
 *
 * @since 1.0.0
 * @since 3.3.0 Revised.
 */
function epl_property_icons( $returntype = 'i' ) {
	$returntype = empty( $returntype ) ? 'i' : $returntype;
	echo epl_get_property_icons( array(), $returntype ); //phpcs:ignore
}
add_action( 'epl_property_icons', 'epl_property_icons', 10, 1 );

/**
 * Property bed/bath icons.
 *
 * @since 1.0
 *
 * @return string
 */
function epl_get_property_bb_icons() {
	global $property;
	return $property->get_property_bed() . ' ' .
		$property->get_property_bath();
}

/**
 * Property land category
 *
 * @hooked property_land_category
 *
 * @since 1.0
 * @since 3.4.44 Fixed warning for preg_replace when value is empty.
 */
function epl_property_land_category() {
	global $property;
	$land_category = $property->get_property_land_category();

	if ( empty( $land_category ) ) {
		return $land_category;
	}

	echo wp_kses_post( $land_category );
}
add_action( 'epl_property_land_category', 'epl_property_land_category' );

/**
 * Property Commercial category
 *
 * @since      1.0 @hooked property_commercial_category
 */
function epl_property_commercial_category() {
	global $property;
	if ( 'commercial' === $property->post_type ) {
		if ( 1 === (int) $property->get_property_meta( 'property_com_plus_outgoings' ) ) {
			echo '<div class="price-type">' . esc_html( apply_filters( 'epl_property_sub_title_plus_outgoings_label', __( 'Plus Outgoings', 'easy-property-listings' ) ) ) . '</div>';
		}
		echo wp_kses_post( $property->get_property_commercial_category() );
	}
}
add_action( 'epl_property_commercial_category', 'epl_property_commercial_category' );

/**
 * Property Available Dates
 *
 * @since      1.0 @hooked property_available_dates
 */
function epl_property_available_dates() {
	global $property;
	$date_avail = $property->get_property_meta( 'property_date_available' );
	if ( 'rental' === $property->post_type &&
		! empty( $date_avail )
		&& 'leased' !== $property->get_property_meta( 'property_status' ) ) {
		// Rental Specifics.
		echo '<div class="property-meta date-available">' . wp_kses_post( apply_filters( 'epl_property_sub_title_available_from_label', __( 'Available from', 'easy-property-listings' ) ) ) . ' ', wp_kses_post( $property->get_property_available() ) , '</div>';
	}
}
add_action( 'epl_property_available_dates', 'epl_property_available_dates' );

/**
 * Property Inspection Times
 *
 * @since 1.0 @hooked property_inspection_times
 * @since 3.4.44 Improvements for PHP 8.2
 */
function epl_property_inspection_times() {
	global $property;
	$property_inspection_times = $property->get_property_inspection_times();
	$label_home_open           = '';

	if ( ! empty( $property_inspection_times ) ) {
		$property_inspection_times = trim( $property_inspection_times );
	} else {
		return;
	}

	if ( ! empty( $property_inspection_times ) ) {
		$label_home_open = $property->get_epl_settings( 'label_home_open' );

		$label_home_open = apply_filters( 'epl_inspection_times_label', $label_home_open );
		?>
		<div class="epl-inspection-times">
			<span class="epl-inspection-times-label">
				<?php echo wp_kses_post( $label_home_open ); ?>
			</span>
			<?php echo wp_kses_post( $property_inspection_times ); ?>
		</div>
		<?php
	}
}
add_action( 'epl_property_inspection_times', 'epl_property_inspection_times' );

/**
 * Getting heading/title of the listing.
 *
 * @param mixed $listing listing instance.
 *
 * @since 2.3.1
 *
 * @return string listing heading or title
 */
function epl_get_property_heading( $listing = null ) {
	if ( null === $listing ) {
		global $property;
	} elseif ( $listing instanceof EPL_Property_Meta ) {
		$property = $listing;
	} elseif ( $listing instanceof WP_Post ) {
		$property = new EPL_Property_Meta( $listing );
	} else {
		$property = get_post( $listing );
		$property = new EPL_Property_Meta( $property );
	}

	if ( $property ) {
		$property_heading = $property->get_property_meta( 'property_heading' );
		if ( strlen( trim( $property_heading ) ) ) {
			return $property_heading;
		}
		return get_the_title( $property->post->ID );
	}
	return '';
}

/**
 * Property Heading
 *
 * @since 1.0
 * @since      1.0 @hooked the_property_heading
 *
 * @param      string $listing  The listing.
 */
function epl_property_heading( $listing = null ) {
	echo wp_kses_post( epl_get_property_heading( $listing ) );
}
add_action( 'epl_property_heading', 'epl_property_heading' );

/**
 * Property Heading
 *
 * @since 1.0 @hooked property_secondary_heading
 * @since 3.4.41 Fix: Added land category.
 * @since 3.5.14 Corrected escaping function to esc_html.
 */
function epl_property_secondary_heading() {
	global $property;

	if ( in_array( $property->post_type, array( 'rental', 'property' ), true ) ) {
		echo wp_kses_post( $property->get_property_category( 'span', 'epl-property-category' ) );
	}

	if ( 'rural' === $property->post_type ) {
		echo wp_kses_post( $property->get_property_rural_category( 'span', 'epl-rural-category' ) );
	}

	if ( 'land' === $property->post_type ) {
		echo wp_kses_post( $property->get_property_land_category( 'span', 'epl-land-category' ) );
	}

	if ( 'commercial' === $property->post_type || 'commercial_land' === $property->post_type ) {
		echo wp_kses_post( $property->get_property_commercial_category( 'span', 'epl-commercial-category' ) );
	}

	if ( 'sold' === $property->get_property_meta( 'property_status' ) ) {
		echo ' <span class="sold-status">' . esc_html( $property->label_sold ) . '</span>';
	}
	echo ' <span class="suburb"> - ' . wp_kses_post( $property->get_property_meta( 'property_address_suburb' ) ) . ' </span>';
	echo ' <span class="state">' . wp_kses_post( $property->get_property_meta( 'property_address_state' ) ) . '</span>';
}
add_action( 'epl_property_secondary_heading', 'epl_property_secondary_heading' );

/**
 * Property Category
 *
 * @param string $tag The div tag.
 * @param string $class The css class name.
 *
 * @since 1.0.0
 * @since 3.4.9 Removed passed 'value' option, added epl_property_category hook and passing of tag and class.
 */
function epl_property_category( $tag = 'div', $class = 'property-category' ) {
	global $property;

	if ( empty( $tag ) ) {
		$tag = 'div';
	}

	echo wp_kses_post( $property->get_property_category( $tag, $class ) );
}
add_action( 'epl_property_category', 'epl_property_category', 10, 2 );

/**
 * Video type
 *
 * @param string $url    The url.
 * @return string
 *
 * @since 3.3.0
 */
function epl_get_video_host( $url ) {

	$host = 'unknown';

	if ( strpos( $url, 'youtu' ) > 0 ) {
		$host = 'youtube';
	} elseif ( strpos( $url, 'vimeo' ) > 0 ) {
		$host = 'vimeo';
	}

	return $host;
}

/**
 * Property Video HTML
 *
 * @param      string  $property_video_url  The property video url.
 * @param      integer $width               The width.
 *
 * @return     string
 *
 * @since 1.0
 * @since 3.3
 * @since 3.4.38 Support for external/local hosted video formats like mp4, mov etc.
 * @since 3.4.42 Added support for youtube shorts & filter for video URL.
 */
function epl_get_video_html( $property_video_url = '', $width = 600 ) {

		$property_video_url = apply_filters( 'epl_property_video_url', $property_video_url );

	// Remove related videos from YouTube.
	if ( 'youtube' === epl_get_video_host( $property_video_url ) ) {

		$property_video_url = epl_convert_youtube_embed_url( $property_video_url );

		if ( strpos( $property_video_url, '?' ) > 0 ) {
			$property_video_url .= '&rel=0';
		} else {
			$property_video_url .= '?rel=0';
		}
	}
	$width = epl_get_option( 'epl_video_width', $width );
	if ( ! empty( $property_video_url ) ) {
		$video_html = '<div class="epl-video-container videoContainer">';

			$video_embed_html = wp_oembed_get(
				$property_video_url,
				array( 'width' => apply_filters( 'epl_property_video_width', $width ) )
			);

		if ( $video_embed_html ) {
			$video_html .= $video_embed_html;
		} else {
			$video_html .= '<video class="epl-local-video" controls width="' . $width . '">
						<source src="' . $property_video_url . '">
					</video>';
		}

		$video_html .= '</div>';

		return $video_html;
	}
}

/**
 * Convert embed URLs to non embed URLs for WP Oembed compatibility.
 *
 * @param  string $url The url.
 *
 * @return string   converted URL
 * @since  3.4.27
 * @since 3.4.4 Added support for youtube shorts videos.
 */
function epl_convert_youtube_embed_url( $url ) {

	if ( strpos( $url, 'embed' ) > 0 || strpos( $url, 'shorts' ) > 0 ) {

		$video_id = epl_get_youtube_id_from_url( $url );

		if ( ! empty( $video_id ) ) {
			$url = 'https://www.youtube.com/watch?v=' . $video_id;
		}
	}

	return apply_filters( 'epl_youtube_embed_url', $url );
}

/**
 * Video Output Function
 *
 * @hooked property_after_content
 *
 * @param      integer $width  The width.
 * @since 1.0
 * @since 3.3 Revised.
 * @since 3.4.44 Added check for empty videos.
 */
function epl_property_video_callback( $width = 600 ) {

	global $property;
	$video_width        = ! empty( $width ) ? $width : 600;
	$property_video_url = $property->get_property_meta( 'property_video_url' );

	if ( '' !== $property_video_url ) {
		echo epl_get_video_html( $property_video_url, $video_width ); //phpcs:ignore
	}

}
add_action( 'epl_property_video', 'epl_property_video_callback', 10, 1 );

/**
 * Previous Video Hook, maintained for backward compatibility.
 *
 * @since 3.3
 * @hooked property_after_content
 */
add_action( 'epl_property_content_after', 'epl_property_video_callback', 10, 1 );

/**
 * Property Tab section details output
 *
 * @since 1.0.0
 * @since 3.4.14 Fix: Custom features' callback output wrongly placed.
 * @since 3.4.30 Fix: Property Features title set to pass basic html.
 * @since 3.5.0  New: epl_property_tab_section hook is replaced with epl_property_features, older hook is kept for backward compatibility.
 * @since 3.5.2  Tweak: Removed additional property category appearing in additional features list.
 * @since 3.5.14 Corrected escaping function to esc_html.
 *
 * @hooked property_tab_section
 */
function epl_property_tab_section() {
	global $property;
	$post_type                 = $property->post_type;
	$the_property_feature_list = apply_filters( 'epl_the_property_feature_list_before', '' );

	$general_features_array = array(
		'category',
		'rural_category',
		'commercial_category',
		'bed',
		'bath',
		'rooms',
		'year_built',
		'parking',
		'ac',
		'pool',
		'security',
		'land_value',
		'building_value',
		'energy_rating',
		'new_construction',
	);

	$general_features_array = apply_filters( 'epl_property_general_features_list', $general_features_array );

	foreach ( $general_features_array as $general_feature ) {

		switch ( $general_feature ) {

			case 'category':
				if ( 'property' === $post_type || 'rental' === $post_type ) {
					$the_property_feature_list .= $property->get_property_category( 'li' );
				}

				break;

			case 'rural_category':
				if ( 'rural' === $post_type ) {
					$the_property_feature_list .= $property->get_property_rural_category( 'li' );
				}

				break;

			case 'commercial_category':
				if ( 'commercial' === $post_type || 'commercial_land' === $post_type || 'business' === $post_type ) {
					$the_property_feature_list .= $property->get_property_commercial_category( 'li' );
				}

				break;

			case 'bed':
				$the_property_feature_list .= $property->get_property_bed( 'l' ) . ' ';

				break;

			case 'bath':
				$the_property_feature_list .= $property->get_property_bath( 'l' ) . ' ';

				break;

			case 'rooms':
				$the_property_feature_list .= $property->get_property_rooms( 'l' ) . ' ';

				break;

			case 'year_built':
				$the_property_feature_list .= $property->get_property_year_built( 'l' ) . ' ';

				break;

			case 'parking':
				$the_property_feature_list .= $property->get_property_parking( 'l' ) . ' ';

				break;

			case 'ac':
				$the_property_feature_list .= $property->get_property_air_conditioning( 'l' ) . ' ';

				break;

			case 'pool':
				$the_property_feature_list .= $property->get_property_pool( 'l' );

				break;

			case 'security':
				$the_property_feature_list .= $property->get_property_security_system( 'l' ) . ' ';

				break;

			case 'land_value':
				$the_property_feature_list .= $property->get_property_land_value( 'l' );

				break;

			case 'building_value':
				$the_property_feature_list .= $property->get_property_building_area_value( 'l' ) . ' ';

				break;

			case 'energy_rating':
				$the_property_feature_list .= $property->get_property_energy_rating( 'l' );

				break;

			case 'new_construction':
				$the_property_feature_list .= $property->get_property_new_construction( 'l' );

				break;

			default:
				ob_start();
				do_action( 'epl_property_general_feature_' . $general_feature );
				$the_property_feature_list .= ob_get_clean();

				break;

		}
	}

	$the_property_feature_list .= apply_filters( 'epl_the_property_feature_list_before_common_features', '' );

	$common_features = array(
		'property_toilet',
		'property_ensuite',
		'property_pet_friendly',
		'property_garage',
		'property_carport',
		'property_open_spaces',
		'property_com_parking_comments',
		'property_com_car_spaces',
		'property_holiday_rental',
		'property_furnished',
	);
	$common_features = apply_filters( 'epl_property_common_features_list', $common_features );

	foreach ( $common_features as $common_feature ) {
		$the_property_feature_list .= $property->get_additional_features_html( $common_feature );
	}

	$the_property_feature_list .= apply_filters( 'epl_the_property_feature_list_before_additional_features', '' );

	$additional_features = array(
		'property_remote_garage',
		'property_secure_parking',
		'property_study',
		'property_dishwasher',
		'property_built_in_robes',
		'property_gym',
		'property_workshop',
		'property_rumpus_room',
		'property_floor_boards',
		'property_broadband',
		'property_pay_tv',
		'property_vacuum_system',
		'property_intercom',
		'property_spa',
		'property_tennis_court',
		'property_balcony',
		'property_deck',
		'property_courtyard',
		'property_outdoor_entertaining',
		'property_shed',
		'property_open_fire_place',
		'property_ducted_heating',
		'property_ducted_cooling',
		'property_split_system_heating',
		'property_hydronic_heating',
		'property_split_system_aircon',
		'property_gas_heating',
		'property_reverse_cycle_aircon',
		'property_evaporative_cooling',
		'property_land_fully_fenced',
	);
	$additional_features = apply_filters( 'epl_property_additional_features_list', $additional_features );

	if ( 'property' === $property->post_type || 'rental' === $property->post_type || 'rural' === $property->post_type ) {
		foreach ( $additional_features as $additional_feature ) {
			$the_property_feature_list .= $property->get_additional_features_html( $additional_feature );
		}
	}

	$the_property_feature_list .= apply_filters( 'epl_the_property_feature_list_after', '' );

	if ( 'land' !== $property->post_type || 'business' !== $property->post_type ) {
		?>
		<?php $property_features_title = apply_filters( 'epl_property_sub_title_property_features', __( 'Property Features', 'easy-property-listings' ) ); ?>
		<h5 class="epl-tab-title epl-tab-title-property-features tab-title"><?php echo wp_kses_post( $property_features_title ); ?></h5>
			<div class="epl-tab-content tab-content">
				<ul class="epl-property-features listing-info epl-tab-<?php echo esc_attr( $property->get_epl_settings( 'display_feature_columns' ) ); ?>-columns">
					<?php echo wp_kses_post( $the_property_feature_list . ' ' . $property->get_features_from_taxonomy() ); ?>
				</ul>
			</div>
	<?php } ?>

	<div class="epl-tab-content epl-tab-content-additional tab-content">
		<?php
			// Land Category.
		if ( 'land' === $property->post_type || 'commercial_land' === $property->post_type ) {
			echo '<div class="epl-land-category">' . wp_kses_post( $property->get_property_land_category( 'value' ) ) . '</div>';
		}

			// Commercial Options.
		if ( 'commercial' === $property->post_type ) {
			if ( 1 === (int) $property->get_property_meta( 'property_com_plus_outgoings' ) ) {
				echo '<div class="epl-commercial-outgoings price-type">' . wp_kses_post( apply_filters( 'epl_property_sub_title_plus_outgoings', __( 'Plus Outgoings', 'easy-property-listings' ) ) ) . '</div>';
			}
		}
		?>
	</div>
	<?php
}
add_action( 'epl_property_tab_section', 'epl_property_tab_section' );
add_action( 'epl_property_features', 'epl_property_tab_section' );


/**
 * Property Tab section details output for commercial, business and commercial
 * land
 *
 * @hooked property_after_tab_section
 *
 * @since 1.0.0
 * @since 3.4.39 Using correctly spelt get_additional_commercial_features_html function.
 * @since 3.5.7  Filters added; epl_property_commercial_features_list and epl_property_rural_features_list filters.
 */
function epl_property_tab_section_after() {
	global $property;
	$post_type = $property->post_type;
	if ( 'commercial' === $post_type || 'business' === $post_type || 'commercial_land' === $post_type ) {

		$the_property_commercial_feature_list = '';
		$features_lists                       = array(
			'property_com_further_options',
			'property_com_highlight_1',
			'property_com_highlight_2',
			'property_com_highlight_3',
			'property_com_zone',
		);

		$features_lists = apply_filters( 'epl_property_commercial_features_list', $features_lists );

		// Check for values in the commercial features.
		$commercial_value = '';

		$result = array();

		foreach ( $features_lists as $feature ) {

			$commercial_value = $property->get_property_meta( $feature );

			if ( ! empty( $commercial_value ) ) {
				$result[] = $commercial_value;
			}
		}

		// Display results if $result array is not empty.
		if ( ! empty( $result ) ) {

			foreach ( $features_lists as $features_list ) {
				$the_property_commercial_feature_list .= $property->get_additional_commercial_features_html( $features_list );
			}

			?>
			<div class="epl-tab-section epl-tab-section-commercial-features">
				<h5 class="epl-tab-title epl-tab-title-commercial-features tab-title"><?php echo wp_kses_post( apply_filters( 'epl_property_sub_title_commercial_features', __( 'Commercial Features', 'easy-property-listings' ) ) ); ?></h5>
				<div class="epl-tab-content tab-content">
					<div class="epl-commercial-features listing-info">
						<?php echo wp_kses_post( $the_property_commercial_feature_list ); ?>
					</div>
				</div>
			</div>
			<?php
		}
	}

	if ( 'rural' === $property->post_type ) {
		$the_property_rural_feature_list = '';
		$features_lists                  = array(
			'property_rural_fencing',
			'property_rural_annual_rainfall',
			'property_rural_soil_types',
			'property_rural_improvements',
			'property_rural_council_rates',
			'property_rural_irrigation',
			'property_rural_carrying_capacity',
		);

		$features_lists = apply_filters( 'epl_property_rural_features_list', $features_lists );

		// Check for values in the rural features.
		$rural_value = '';

		$result = array();

		foreach ( $features_lists as $feature ) {

			$rural_value = $property->get_property_meta( $feature );

			if ( ! empty( $rural_value ) ) {
				$result[] = $rural_value;
			}
		}

		// Display results if $result array is not empty.
		if ( ! empty( $result ) ) {

			foreach ( $features_lists as $features_list ) {
				$the_property_rural_feature_list .= $property->get_additional_rural_features_html( $features_list );
			}

			?>
			<div class="epl-tab-section epl-tab-section-rural-features">
				<h5 class="epl-tab-title epl-tab-title-rural-features tab-title"><?php echo wp_kses_post( apply_filters( 'epl_property_sub_title_rural_features', __( 'Rural Features', 'easy-property-listings' ) ) ); ?></h5>
				<div class="epl-tab-content tab-content">
					<div class="epl-rural-features listing-info">
						<?php echo wp_kses_post( $the_property_rural_feature_list ); ?>
					</div>
				</div>
			</div>
			<?php
		}
	}
}
add_action( 'epl_property_tab_section_after', 'epl_property_tab_section_after' );

/**
 * Get price sticker
 *
 * @return string
 * @since  1.0
 */
function epl_get_price_sticker() {
	global $property;
	return $property->get_price_sticker();
}

/**
 * Get Property Price
 *
 * @return string
 * @since 1.0
 */
function epl_get_property_price() {
	global $property;
	return $property->get_price();
}

/**
 * Get listing Address for Widget
 *
 * @since      1.0
 *
 * @param      string $d_suburb  The d suburb.
 * @param      string $d_street  The d street.
 */
function epl_widget_listing_address( $d_suburb = '', $d_street = '' ) {
	global $property;
	if ( 'commercial' === $property->post_type || 'business' === $property->post_type ) {
		// Address Display not Commercial or Business type.
		if ( 'yes' === $property->get_property_meta( 'property_address_display' ) ) {
			?>
			<?php
			// Suburb.
			if ( 'on' === $d_suburb && 'yes' === $property->get_property_meta( 'property_com_display_suburb' ) ) {
				?>
				<div class="property-meta suburb-name"><?php echo esc_html( $property->get_property_meta( 'property_address_suburb' ) ); ?></div>
			<?php } ?>

			<?php
			// Street.
			if ( 'on' === $d_street ) {
				?>
				<div class="property-meta street-name"><?php echo wp_kses_post( $property->get_formatted_property_address() ); ?></div>
			<?php } ?>
		<?php } else { ?>
			<?php
			// Suburb.
			if ( 'on' === $d_suburb && 'yes' === $property->get_property_meta( 'property_com_display_suburb' ) ) {
				?>
				<div class="property-meta suburb-name"><?php echo esc_html( $property->get_property_meta( 'property_address_suburb' ) ); ?></div>
			<?php } ?>
			<?php
		}
	} else {
		// Address Display not Commercial or Business type.
		if ( 'yes' === $property->get_property_meta( 'property_address_display' ) ) {
			?>
			<?php
			// Suburb.
			if ( 'on' === $d_suburb ) {
				?>
				<div class="property-meta suburb-name"><?php echo esc_html( $property->get_property_meta( 'property_address_suburb' ) ); ?></div>
			<?php } ?>

			<?php
			// Street.
			if ( 'on' === $d_street ) {
				?>
				<div class="property-meta street-name"><?php echo wp_kses_post( $property->get_formatted_property_address() ); ?></div>
			<?php } ?>
		<?php } else { ?>
			<?php
			// Suburb.
			if ( 'on' === $d_suburb ) {
				?>
				<div class="property-meta suburb-name"><?php echo wp_kses_post( $property->get_property_meta( 'property_address_suburb' ) ); ?></div>
			<?php } ?>
			<?php
		}
	}
}

/**
 * Get Sorting Options
 *
 * @param      boolean $post_type  The post type.
 *
 * @return     mixed|void
 *
 * @since      2.0
 * @since      3.4.44 Option to sort by title.
 */
function epl_sorting_options( $post_type = null ) {
	// phpcs:disable WordPress.Security.NonceVerification
	if ( is_null( $post_type ) ) {
		$post_type = isset( $_GET['post_type'] ) ? sanitize_text_field( wp_unslash( $_GET['post_type'] ) ) : 'property';
	}

	return apply_filters(
		'epl_sorting_options',
		array(
			array(
				'id'      => 'high',
				'label'   => __( 'Price: High to Low', 'easy-property-listings' ),
				'type'    => 'meta',
				'key'     => 'property_price_global',
				'order'   => 'DESC',
				'orderby' => 'meta_value_num',
			),
			array(
				'id'      => 'low',
				'label'   => __( 'Price: Low to High', 'easy-property-listings' ),
				'type'    => 'meta',
				'key'     => 'property_price_global',
				'order'   => 'ASC',
				'orderby' => 'meta_value_num',

			),
			array(
				'id'    => 'new',
				'label' => __( 'Date: Newest First', 'easy-property-listings' ),
				'type'  => 'post',
				'key'   => 'post_date',
				'order' => 'DESC',

			),
			array(
				'id'    => 'old',
				'label' => __( 'Date: Oldest First', 'easy-property-listings' ),
				'type'  => 'post',
				'key'   => 'post_date',
				'order' => 'ASC',
			),
			array(
				'id'      => 'status_asc',
				'label'   => __( 'Status : Current First', 'easy-property-listings' ),
				'type'    => 'meta',
				'key'     => 'property_status',
				'order'   => 'ASC',
				'orderby' => 'meta_value',

			),
			array(
				'id'      => 'status_desc',
				'label'   => __( 'Status : Sold/Leased First', 'easy-property-listings' ),
				'type'    => 'meta',
				'key'     => 'property_status',
				'order'   => 'DESC',
				'orderby' => 'meta_value',

			),
			array(
				'id'      => 'location_asc',
				'label'   => epl_labels( 'label_suburb' ) . __( ' A-Z', 'easy-property-listings' ),
				'type'    => 'meta',
				'key'     => 'property_address_suburb',
				'order'   => 'ASC',
				'orderby' => 'meta_value',

			),
			array(
				'id'      => 'location_desc',
				'label'   => epl_labels( 'label_suburb' ) . __( ' Z-A', 'easy-property-listings' ),
				'type'    => 'meta',
				'key'     => 'property_address_suburb',
				'order'   => 'DESC',
				'orderby' => 'meta_value',

			),
			array(
				'id'    => 'title',
				'label' => __( 'Title', 'easy-property-listings' ),
				'type'  => 'post',
				'key'   => 'post_title',
				'order' => 'ASC',

			),
		)
	);
}

/**
 * Switch Sorting Wrapper
 *
 * @param array $attributes Attributes.
 *
 * @since 3.3
 * @since 3.4.44 Get shortcode attributes as parameter.
 */
function epl_tools_utility_wrapper( $attributes = array() ) {

	// Wrapper Start.
	do_action( 'epl_archive_utility_wrap_start', $attributes );

		do_action( 'epl_add_custom_menus', $attributes );

	// Wrapper End.
	do_action( 'epl_archive_utility_wrap_end', $attributes );

}
add_action( 'epl_property_loop_start', 'epl_tools_utility_wrapper', 10 );

/**
 * Switch Sorting Wrapper
 *
 * @param array $attributes Attributes.
 * @param array $args       Arguments.
 *
 * @since 3.4.44 Get shortcode attributes as parameter.
 * @since 3.5 Args missing second value.
 * @since 2.0
 * @since 3.3 Revised.
 */
function epl_listing_toolbar_items( $attributes = array(), $args = array() ) {
	echo get_epl_listing_toolbar_items( $attributes, $args ); //phpcs:ignore;
}
add_action( 'epl_add_custom_menus', 'epl_listing_toolbar_items', 10, 2 );

/**
 * Retrieves the switch and sorting options normally right aligned
 *
 * @param array $attributes Attributes.
 * @param array $args       Arguments.
 *
 * @return string
 *
 * @since      3.3
 * @since      3.5 gets the shortcode attributes as param
 */
function get_epl_listing_toolbar_items( $attributes = array(), $args = array() ) {

	$defaults = array(
		'switch_views',
		'sorting_tool',
	);

	$tools = apply_filters( 'epl_listing_toolbar_items', $defaults );

	ob_start();

	// Wrapper.
	if ( ! empty( $defaults ) ) {
		?>
		<div class="epl-loop-tools epl-loop-tools-switch-sort epl-switching-sorting-wrap">
			<?php

			foreach ( $tools as $tool ) {

				if ( ! empty( $args ) && ! in_array( $tool, $args, true ) ) {
					continue;
				}

				switch ( $tool ) {

					case 'switch_views':
						do_action( 'epl_switch_views', $attributes );
						break;

					case 'sorting_tool':
						do_action( 'epl_sorting_tool', $attributes );
						break;

					default:
						// Action to hook additional tools.
						do_action( 'epl_listing_toolbar_' . $tool, $attributes );
						break;
				}
			}
			?>
		</div>
		<?php
	}
	return ob_get_clean();
}

/**
 * Switch Views
 *
 * @since      2.0
 */
function epl_switch_views() {
	?>
	<div class="epl-loop-tool epl-tool-switch epl-switch-view">
		<ul>
			<li title="<?php echo esc_attr( apply_filters( 'epl_switch_views_sorting_title_list', __( 'List', 'easy-property-listings' ) ) ); ?>" class="epl-current-view view-list" data-view="list">
			</li>
			<li title="<?php echo esc_attr( apply_filters( 'epl_switch_views_sorting_title_grid', __( 'Grid', 'easy-property-listings' ) ) ); ?>" class="view-grid" data-view="grid">
			</li>
		</ul>
	</div>
	<?php
}
add_action( 'epl_switch_views', 'epl_switch_views' );

/**
 * Displays the Switch Sorting select options
 *
 * @param array $attributes Attributes.
 *
 * @since 2.0
 * @since 3.3 Revised.
 * @since 3.4.44 Unique ID for sort dropdown per instance.
 * @since 3.4.44 Get shortcode attributes as parameter.
 * @since 3.5 Added accessibility labels to select elements.
 * @since 3.5.7 Instance ID default to 1 to enable sorting. The $attributes option is required.
 * @since 3.5.14 Corrected escaping function to esc_html.
 */
function epl_sorting_tool( $attributes = array() ) {

	$instance_id     = ! empty( $attributes['instance_id'] ) ? sanitize_text_field( $attributes['instance_id'] ) : '1';
	$sortby          = '';
	$set_instance_id = '';
	if ( ! empty( $_GET['sortby'] ) ) {
		$sortby = sanitize_text_field( wp_unslash( $_GET['sortby'] ) );
	}
	if ( ! empty( $_GET['instance_id'] ) ) {
		$set_instance_id = sanitize_text_field( wp_unslash( $_GET['instance_id'] ) );
	}
	$sorters = epl_sorting_options();
		$id  = ! empty( $instance_id ) ? $instance_id : '1';
	?>

	<div class="epl-loop-tool epl-tool-sorting epl-properties-sorting epl-clearfix">
		<select aria-label="<?php esc_attr_e( 'Sort Listings', 'easy-property-listings' ); ?>" class="epl-sort-listings" id="epl-sort-listings-<?php echo esc_attr( $instance_id ); ?>" data-instance-id="<?php echo esc_attr( $instance_id ); ?>">
			<option <?php selected( $sortby, '' ); ?> value="">
				<?php echo esc_html( apply_filters( 'epl_switch_views_sorting_title_sort', __( 'Sort', 'easy-property-listings' ) ) ); ?>
			</option>
			<?php
			foreach ( $sorters as $sorter ) {

				if ( ! empty( $instance_id ) ) {
					$sortbyinstance = $set_instance_id . '_' . $sortby;
					$sorter_id      = $instance_id . '_' . $sorter['id'];
				} else {
					$sortbyinstance = $sortby;
					$sorter_id      = $sorter['id'];
				}
				?>
				<option <?php selected( $sortbyinstance, $sorter_id ); ?> value="<?php echo esc_attr( $sorter['id'] ); ?>">
					<?php echo esc_html( $sorter['label'] ); ?>
				</option>
				<?php
			}
			?>
		</select>
	</div>
	<?php
}
add_action( 'epl_sorting_tool', 'epl_sorting_tool' );

/**
 * Displays the Sorting tabs
 *
 * @since 3.3
 * @since 3.5.14 Corrected escaping function to esc_html.
 */
function epl_sorting_tabs() {
	$sortby = '';
	if ( ! empty( $_GET['sortby'] ) ) {
		$sortby = sanitize_text_field( wp_unslash( $_GET['sortby'] ) );
	}
	$sorters = epl_sorting_options();

	global $wp;
	$get_data    = $_GET;
	$get_data    = array_map( 'sanitize_text_field', $get_data );
	$current_url = home_url( add_query_arg( array( $get_data ), $wp->request ) );
	?>

	<div class="epl-loop-tool epl-tool-sorting-tabs epl-properties-sorting epl-clearfix">
		<ul id="epl-sort-tabs-listings">
			<?php
			foreach ( $sorters as $sorter ) {
				$href  = epl_add_or_update_params( $current_url, 'sortby', $sorter['id'] );
				$class = $sortby === $sorter['id'] ? 'epl-sortby-selected' : '';
				?>
					<li class="epl-sortby-list <?php echo esc_attr( $class ); ?>">
						<a href="<?php echo esc_url( $href ); ?>">
							<?php echo esc_html( $sorter['label'] ); ?>
						</a>
					</li>
					<?php
			}
			?>
		</ul>
	</div>
	<?php
}

/**
 * Update parameters
 *
 * @param string $url The url.
 * @param string $key The key.
 * @param string $value The value.
 *
 * @return string
 * @since 3.3
 * @since 3.5 Fix: Warning when url empty.
 */
function epl_add_or_update_params( $url, $key, $value ) {

	if ( empty( $url ) ) {
		return $url;
	}

	$a     = wp_parse_url( $url );
	$query = isset( $a['query'] ) ? $a['query'] : '';
	parse_str( $query, $params );
	$params[ $key ] = $value;
	$query          = http_build_query( $params );
	$result         = '';
	if ( $a['scheme'] ) {
		$result .= $a['scheme'] . ':';
	}
	if ( $a['host'] ) {
		$result .= '//' . $a['host'];
	}
	if ( ! empty( $a['port'] ) ) {
		$result .= ':' . $a['port'];
	}
	if ( $a['path'] ) {
		$result .= $a['path'];
	}
	if ( $query ) {
		$result .= '?' . $query;
	}
	return $result;
}

/**
 * Archive Sorting
 *
 * @param  WP_Query $query  The query.
 *
 * @since  2.0
 */
function epl_archive_sorting( $query ) {
	$post_types_sold   = array( 'property', 'land', 'commercial', 'business', 'commercial_land', 'location_profile', 'rural' );
	$post_types_rental = array( 'rental' );

	if ( ! $query->is_main_query() ) {
		return;
	}

	if ( is_post_type_archive( $post_types_sold ) || is_post_type_archive( $post_types_rental ) || is_tax( 'location' ) || is_tax( 'tax_feature' ) || is_tax( 'tax_business_listing' ) || epl_is_search() ) {

		if ( ! empty( $_GET['sortby'] ) ) {

			$orderby = sanitize_text_field( wp_unslash( $_GET['sortby'] ) );
			$sorters = epl_sorting_options( $query->get( 'post_type' ) );

			foreach ( $sorters as $sorter ) {

				if ( $orderby === $sorter['id'] ) {

					if ( 'meta' === $sorter['type'] ) {
						$query->set( 'orderby', $sorter['orderby'] );
						$query->set( 'meta_key', $sorter['key'] );
					} else {
						$query->set( 'orderby', $sorter['key'] );
					}
					$query->set( 'order', $sorter['order'] );
					break;
				}
			}
		}
	}
}
add_action( 'pre_get_posts', 'epl_archive_sorting' );

/**
 * Author Tabs
 *
 * @param array|null $epl_author EPL author object.
 *
 * @return array
 *
 * @since 1.0.0
 * @since 3.5.2 Passes param $epl_author so it can be passed to tabs filter.
 */
function epl_author_tabs( $epl_author = null ) {

	if ( is_null( $epl_author ) ) {
		global $epl_author;
	}
	$author_tabs = array(
		'author_id'    => __( 'About', 'easy-property-listings' ),
		'description'  => __( 'Bio', 'easy-property-listings' ),
		'video'        => __( 'Video', 'easy-property-listings' ),
		'contact_form' => __( 'Contact', 'easy-property-listings' ),
	);
	$author_tabs = apply_filters( 'epl_author_tabs', $author_tabs, $epl_author );
	return $author_tabs;
}

/**
 * Author Class
 *
 * @since 2.0
 *
 * @param string $classes  The classes.
 */
function epl_author_class( $classes ) {
	$classes = explode( ' ', $classes . ' epl-author-box author-box' );
	$classes = array_filter( array_unique( $classes ) );
	$classes = apply_filters( 'epl_author_class', $classes );
	if ( ! empty( $classes ) ) {
		$classes = implode( ' ', $classes );
		echo esc_attr( $classes );
	}
}

/**
 * Author Tab ID
 *
 * @since      2.0
 *
 * @param      array $epl_author  The epl author.
 *
 * @return false|string
 */
function epl_author_tab_author_id( $epl_author = array() ) {

	if ( empty( $epl_author ) ) {
		global $epl_author;
	}

	$permalink    = apply_filters( 'epl_author_profile_link', get_author_posts_url( $epl_author->author_id ), $epl_author );
	$author_title = apply_filters( 'epl_author_profile_title', get_the_author_meta( 'display_name', $epl_author->author_id ), $epl_author );

	$arg_list = get_defined_vars();

	ob_start();

	epl_get_template_part( 'content-author-box-tab-details.php', $arg_list );

	return ob_get_clean();
}

/**
 * Author Tab Image
 *
 * @since      2.0
 *
 * @param      array $epl_author  The epl author.
 */
function epl_author_tab_image( $epl_author = array() ) {

	if ( empty( $epl_author ) ) {
		global $epl_author;
	}

	if ( function_exists( 'get_avatar' ) ) {
		echo wp_kses_post( apply_filters( 'epl_author_tab_image', get_avatar( $epl_author->email, '150' ), $epl_author ) );
	}
}
add_action( 'epl_author_thumbnail', 'epl_author_tab_image', 10, 2 );

/**
 * Author Tab Description
 *
 * @since      1.0
 *
 * @param      array $epl_author  The epl author.
 */
function epl_author_tab_description( $epl_author = array() ) {
	if ( empty( $epl_author ) ) {
		global $epl_author;
	}
	echo wp_kses_post( $epl_author->get_description_html() );
}

/**
 * Author Tab Video
 *
 * @since      1.0
 *
 * @param      array $epl_author  The epl author.
 */
function epl_author_tab_video( $epl_author = array() ) {
	if ( empty( $epl_author ) ) {
		global $epl_author;
	}
	$video_html = $epl_author->get_video_html();
	if ( ! empty( $video_html ) ) {
		echo '<div class="epl-author-video author-video epl-video-container">' . $video_html . '</div>'; //phpcs:ignore
	}
}

/**
 * Author Tab Contact Form
 *
 * @since      1.0
 *
 * @param      array $epl_author  The epl author.
 */
function epl_author_tab_contact_form( $epl_author = array() ) {
	if ( empty( $epl_author ) ) {
		global $epl_author;
	}
	echo $epl_author->get_author_contact_form(); //phpcs:ignore
}

/**
 * Archive Utility Wrapper Before
 *
 * @since      1.0
 */
function epl_archive_utility_wrap_before() {
	echo '<div class="epl-loop-tools-wrap epl-archive-utility-wrapper epl-clearfix">';
}
add_action( 'epl_archive_utility_wrap_start', 'epl_archive_utility_wrap_before' );

/**
 * Archive Utility Wrapper After
 *
 * @since      1.0
 */
function epl_archive_utility_wrap_after() {
	echo '</div>';
}
add_action( 'epl_archive_utility_wrap_end', 'epl_archive_utility_wrap_after' );

/**
 * Listing Image Gallery
 *
 * @since 1.0
 * @since 3.3 Revised.
 */
function epl_property_gallery() {

	$d_gallery = (int) epl_get_option( 'display_single_gallery' );

	$d_gallery_n = epl_get_option( 'display_gallery_n' );

	if ( 1 !== $d_gallery ) {
		return;
	}

	$attachments = get_children(
		array(
			'post_parent'    => get_the_ID(),
			'post_type'      => 'attachment',
			'post_mime_type' => 'image',
		)
	);

	if ( $attachments ) {
		?>

		<div class="epl-gallery property-gallery">
			<!-- Gallery -->
			<div class="epl-gallery-entry entry-gallery epl-clearfix">
				<?php
					$gallery_shortcode = '[gallery columns="' . $d_gallery_n . '" link="file"]';
					$gallery           = apply_filters( 'epl_property_gallery_shortcode', $gallery_shortcode, $d_gallery_n );
					echo do_shortcode( $gallery ); // phpcs:ignore
				?>
			</div>
		</div>
		<?php
	}
}
add_action( 'epl_property_gallery', 'epl_property_gallery' );

/**
 * Get the template path.
 *
 * @return     string
 * @since      1.0
 */
function epl_template_path() {
	return apply_filters( 'epl_template_path', 'easypropertylistings/' );
}

/**
 * Outputs a wrapper div before the first button
 *
 * @since      1.3
 */
function epl_buttons_wrapper_before() {
	echo '<div class="epl-button-wrapper epl-clearfix">';
}

/**
 * Outputs a wrapper div after the last button
 *
 * @since      1.3
 */
function epl_buttons_wrapper_after() {
	echo '</div>';
}
add_action( 'epl_buttons_single_property', 'epl_buttons_wrapper_before', 1 );
add_action( 'epl_buttons_single_property', 'epl_buttons_wrapper_after', 99 );

/**
 * Used to mark home inspection on Apple devices
 *
 * @param string $start The start.
 * @param string $end The end.
 * @param string $name The name.
 * @param string $description The description.
 * @param string $location The location.
 * @param null   $post_id The post ID.
 *
 * @since 2.0.0
 * @since 3.4.9 Corrected issue where output was trimmed, added better unique ID and URL to output.
 * @since 3.5.7 Updated to allow passing of extra details to ical.
 */
function epl_create_ical_file( $start = '', $end = '', $name = '', $description = '', $location = '', $post_id = null ) {

	if ( is_null( $post_id ) ) {
		$post_id = get_the_ID();
	}

	$description = str_replace( "\n", "\\n", str_replace( ';', '\;', str_replace( ',', '\,', $description ) ) );
	$uid         = $post_id . current_time( 'timestamp' );
	$url         = get_permalink( $post_id );
	$prodid      = '-//' . get_bloginfo( 'name' ) . '/EPL//NONSGML v1.0//EN';
	$args        = get_defined_vars();
	$args        = apply_filters( 'epl_ical_args', $args );
	$data        = "BEGIN:VCALENDAR\nVERSION:2.0\nPRODID:" . $args['prodid'] . "\nMETHOD:PUBLISH\nBEGIN:VEVENT\nDTSTART:" . date( 'Ymd\THis', strtotime( $args['start'] ) ) . "\nDTEND:" . date( 'Ymd\THis', strtotime( $args['end'] ) ) . "\nLOCATION:" . $args['location'] . "\nURL:" . $args['url'] . "\nTRANSP:OPAQUE\nSEQUENCE:0\nUID:" . $args['uid'] . "\nDTSTAMP:" . date( 'Ymd\THis\Z' ) . "\nSUMMARY:" . $args['name'] . "\nDESCRIPTION:" . $args['description'] . "\nPRIORITY:1\nCLASS:PUBLIC\nBEGIN:VALARM\nTRIGGER:-PT10080M\nACTION:DISPLAY\nDESCRIPTION:Reminder\nEND:VALARM\nEND:VEVENT\nEND:VCALENDAR\n";

	header( 'Content-type:text/calendar' );
	header( 'Content-Disposition: attachment; filename="' . $name . '.ics"' );
	Header( 'Content-Length: ' . strlen( $data ) );
	Header( 'Connection: close' );
	echo $data; //phpcs:ignore
	die;
}

/**
 * Output iCal clickable dates. Use the epl_ical_args filter to alter the labels.
 *
 * @since      2.0
 * @since      3.5.7 Different subject for auction.
 */
function epl_process_event_cal_request() {
	global $epl_settings;
	if ( isset( $_GET['propid'] ) && isset( $_GET['epl_cal_dl'] ) && 1 === (int) $_GET['epl_cal_dl'] && intval( $_GET['propid'] ) > 0 ) {
		if ( isset( $_GET['cal'] ) ) {
			$type = sanitize_text_field( wp_unslash( $_GET['cal'] ) );
			switch ( $type ) {
				case 'ical':
					$item = base64_decode( sanitize_text_field( wp_unslash( $_GET['dt'] ) ) ); //phpcs:ignore
					if ( is_numeric( $item[0] ) ) {
						$post_id   = isset( $_GET['propid'] ) ? intval( $_GET['propid'] ) : 0;
						$timearr   = explode( ' ', $item );
						$starttime = current( $timearr );
						if ( isset( $timearr[1] ) ) {
							$starttime .= ' ' . $timearr[1];
						}
						$endtime = current( $timearr ) . ' ' . end( $timearr );
						$post    = get_post( $post_id );
						if ( is_null( $post ) ) {
							return;
						}

						$subject = $epl_settings['label_home_open'] . ' - ' . get_post_meta( $post_id, 'property_heading', true );

						if ( isset( $_GET['event_type'] ) && 'auction' == sanitize_text_field( wp_unslash( $_GET['event_type'] ) ) ) {
							$subject = __( 'Auction', 'easy-property-listings' ) . ' - ' . get_post_meta( $post_id, 'property_heading', true );
						}

						$address      = '';
						$prop_sub_num = get_post_meta( $post_id, 'property_address_sub_number', true );
						if ( ! empty( $prop_sub_num ) ) {
							$address .= get_post_meta( $post_id, 'property_address_sub_number', true ) . '/';
						}
						$address .= get_post_meta( $post_id, 'property_address_street_number', true ) . ' ';
						$address .= get_post_meta( $post_id, 'property_address_street', true ) . ' ';
						$address .= get_post_meta( $post_id, 'property_address_suburb', true ) . ', ';
						$address .= get_post_meta( $post_id, 'property_address_state', true ) . ' ';
						$address .= get_post_meta( $post_id, 'property_address_postal_code', true );

						epl_create_ical_file( $starttime, $endtime, $subject, wp_strip_all_tags( $post->post_content ), $address, $post_id );
					}
					break;
			}
		}
	}
}
add_action( 'init', 'epl_process_event_cal_request' );

/**
 * Add coordinates to meta for faster loading on second view
 *
 * @since 2.1
 * @since 3.5.3 Removed nopriv access & added nonce.
 * @since 3.5.4 Fix: Check user permissions on coordinate generation.
 */
function epl_update_listing_coordinates() {

	check_ajax_referer( 'epl_ajax_nonce', '_epl_nonce' );

	if ( ! current_user_can( 'manage_options' ) ) {
		wp_die( 'fail' );
	}

	if ( ( ! isset( $_POST['listid'] ) || 0 === intval( $_POST['listid'] ) ) || empty( $_POST['coordinates'] ) ) {
		return;
	}
	$coordinates = rtrim( ltrim( sanitize_text_field( wp_unslash( $_POST['coordinates'] ) ), '(' ), ')' ); //phpcs:ignore
	if ( update_post_meta( intval( wp_unslash( $_POST['listid'] ) ), 'property_address_coordinates', $coordinates ) ) { //phpcs:ignore
		wp_die( 'success' );
	} else {
		wp_die( 'fail' );
	}
}
add_action( 'wp_ajax_epl_update_listing_coordinates', 'epl_update_listing_coordinates' );

/**
 * Adapted from wp core to add additional filters
 *
 * @param      string $id        The identifier.
 * @param      string $taxonomy  The taxonomy.
 * @param      string $before    The before.
 * @param      string $sep       The separator.
 * @param      string $after     The after.
 *
 * @return bool|false|string|WP_Error|WP_Term[]
 *
 * @since 2.1
 * @since 3.3 Revised.
 * @since 3.5.1 Dynamic class name for item based on taxonomy.
 */
function epl_get_the_term_list( $id, $taxonomy, $before = '', $sep = '', $after = '' ) {
	$terms = get_the_terms( $id, $taxonomy );

	if ( is_wp_error( $terms ) ) {
		return $terms;
	}

	if ( empty( $terms ) ) {
		return false;
	}

	foreach ( $terms as $term ) {

		$link = get_term_link( $term, $taxonomy );
		if ( is_wp_error( $link ) ) {
			return $link;
		}

		if ( true === apply_filters( 'epl_features_taxonomy_link_filter', true ) ) {
			$term_links[] = '<li class="epl-tax-' . $taxonomy . ' ' . $term->slug . ' ">' .
						'<a href="' . esc_url( $link ) . '" rel="tag">' . $term->name . '</a>'
					. '</li>' . $sep;
		} else {
			$term_links[] = '<li class="epl-tax-' . $taxonomy . ' ' . $term->slug . ' ">' . $term->name . '</li>' . $sep;
		}
	}

	$term_links = apply_filters( "term_links-$taxonomy", $term_links ); //phpcs:ignore

	$html = $before;
	foreach ( $term_links as $term_link ) {
		$html .= $term_link;
	}
	$html .= $after;

	return $html;
}

/**
 * Get Property Meta
 *
 * @param string $key Meta key.
 *
 * @return  string $key  The property meta.
 * @since 2.1
 */
function get_property_meta( $key ) {
	global $property;
	return $property->get_property_meta( $key );
}

/**
 * The Property Meta
 *
 * @since 2.1
 *
 * @param string $key    The key.
 */
function the_property_meta( $key ) {
	global  $property;
	echo wp_kses_post( $property->get_property_meta( $key ) );
}

/**
 * Template Class
 *
 * @since      2.1
 *
 * @param      boolean $class    The class.
 * @param      string  $context  The context.
 *
 * @return mixed|void
 */
function epl_template_class( $class = false, $context = 'single' ) {

	if ( $class ) {
		$class = 'epl-template-' . $class;
	} else {
		$class = 'epl-template-blog';
	}

	return apply_filters( 'epl_template_class', $class, $context );
}

/**
 * Pagination
 *
 * @since      2.1
 *
 * @param      array $query  The query.
 */
function epl_pagination( $query = array() ) {
	global $epl_settings;
	$fancy_on = 1 === (int) epl_get_option( 'use_fancy_navigation', 0 ) ? 1 : 0;
	if ( $fancy_on ) {
		epl_fancy_pagination( $query );
	} else {
		epl_wp_default_pagination( $query );
	}
}
add_action( 'epl_pagination', 'epl_pagination' );

/**
 * Returns active theme name as a lowercase name
 *
 * @since      3.0
 *
 * @return string
 */
function epl_get_active_theme() {
	if ( function_exists( 'wp_get_theme' ) ) { // wp version >= 3.4.
		$active_theme = wp_get_theme();
		$active_theme = $active_theme->get( 'Name' );

	} else {
		// older versions.
		$active_theme = wp_get_theme(); //phpcs:ignore
	}
	$active_theme = str_replace( ' ', '', strtolower( $active_theme ) );
	return apply_filters( 'epl_active_theme', $active_theme );
}

/**
 * Returns active theme name as a css class with prefix for use in default
 * templates
 *
 * @return mixed|void
 * @since      2.1.2
 */
function epl_get_active_theme_name() {
	$epl_class_prefix = apply_filters( 'epl_active_theme_prefix', 'epl-active-theme-' );
	$active_theme     = epl_get_active_theme();
	return apply_filters( 'epl_active_theme_name', $epl_class_prefix . $active_theme );
}

/**
 * Returns core shortcode names
 *
 * @return array
 * @since 3.3
 * @since 3.4.44 Added epl_get_shortcode_list filter.
 */
function epl_get_shortcode_list() {
	return apply_filters(
		'epl_get_shortcode_list',
		array(
			'listing',
			'listing_category',
			'listing_open',
			'listing_feature',
			'listing_location',
			'listing_auction',
			'listing_advanced',
		)
	);
}

/**
 * Wrapper for wp_doing_ajax with fallback for lower WP versions
 *
 * @return     bool  True if it's an ajax request
 * @since      3.4.17
 */
function epl_wp_doing_ajax() {

	if ( function_exists( 'wp_doing_ajax' ) ) {
		return wp_doing_ajax();
	} else {
		return apply_filters( 'wp_doing_ajax', defined( 'DOING_AJAX' ) && DOING_AJAX );
	}
}

/**
 * Pagination fix for home
 *
 * @param WP_Query $query  The query.
 *
 * @since 2.1.2
 * @since 3.3 Revised.
 */
function epl_home_pagination_fix( $query ) {

	global $wp_query;
	$queried_post_type = isset( $query->query_vars['post_type'] ) ? (array) $query->query_vars['post_type'] : array();
	$diff              = array_diff( $queried_post_type, epl_get_core_post_types() );

	if ( isset( $wp_query->query['paged'] ) && 0 === count( $diff ) ) {
		$query->set( 'paged', $wp_query->query['paged'] );
	}

	$shortcodes = epl_get_shortcode_list();

	if ( $query->get( 'is_epl_shortcode' ) &&
		in_array( $query->get( 'epl_shortcode_name' ), $shortcodes, true ) && ! epl_wp_doing_ajax() ) {

		if ( isset( $_GET['pagination_id'] ) && $_GET['pagination_id'] === $query->get( 'instance_id' ) ) {
			$query->set( 'paged', $query->get( 'paged' ) );
		} else {
			$query->set( 'paged', 1 );
		}
	}
}
add_action( 'pre_get_posts', 'epl_home_pagination_fix', 99 );

/**
 * Returns status class
 *
 * @since      2.1.10
 */
function epl_property_widget_status_class() {
	global $property;
	echo 'epl-widget-status-' . esc_attr( $property->get_property_meta( 'property_status' ) );
}
add_action( 'epl_property_widget_status_class', 'epl_property_widget_status_class' );

/**
 * Ability to hide map on single listings
 *
 * @since      2.1.8
 */
function epl_hide_map_from_front() {
	$epl_posts = epl_get_active_post_types();
	$epl_posts = array_keys( $epl_posts );

	global $post,$property;

	if ( is_single() && in_array( $post->post_type, $epl_posts, true ) ) {

		$hide_map = get_post_meta( $post->ID, 'property_address_hide_map', true );
		if ( 'yes' === $hide_map ) {
			remove_all_actions( 'epl_property_map' );
		}
	}
}
add_action( 'wp', 'epl_hide_map_from_front', 10 );

/**
 * Disable paging on listing widget
 *
 * @since 2.1.8
 *
 * @param  WP_Query $query  The query.
 */
function epl_nopaging( $query ) {
	$restrict_paging = $query->get( 'epl_nopaging' );
	if ( true === $restrict_paging ) {
		$query->set( 'paged', 1 );
	}
}
add_action( 'pre_get_posts', 'epl_nopaging' );

/**
 * Ability to hide author box on single listings
 *
 * @since      2.1.11
 */
function epl_hide_author_box_from_front() {
	$epl_posts = epl_get_active_post_types();
	$epl_posts = array_keys( $epl_posts );

	global $post,$property;

	if ( is_single() && in_array( $post->post_type, $epl_posts, true ) ) {

		$hide_author_box = get_post_meta( $post->ID, 'property_agent_hide_author_box', true );
		if ( 'yes' === $hide_author_box ) {
			remove_all_actions( 'epl_single_author' );
		}
	}
}
add_action( 'wp', 'epl_hide_author_box_from_front', 10 );

/**
 * Retain user grid/list view
 *
 * @since      2.1.11
 */
function epl_update_default_view() {

	$view = isset( $_POST['view'] ) ? sanitize_text_field( wp_unslash( $_POST['view'] ) ) : '';

	if ( in_array( $view, array( 'list', 'grid' ), true ) ) {

		setcookie( 'preferredView', $view, 0, '/' );
	}
	wp_die( 'success' );
}
add_action( 'wp_ajax_epl_update_default_view', 'epl_update_default_view' );
add_action( 'wp_ajax_nopriv_epl_update_default_view', 'epl_update_default_view' );

/**
 * Custom the_content filter
 *
 * @since      2.2
 * @since      3.4.35 Added support for WP blocks rendering
 */
function epl_the_content_filters() {

	if ( ! has_filter( 'epl_get_the_content', 'wptexturize' ) ) {

		add_filter( 'epl_get_the_content', 'wptexturize' );
		add_filter( 'epl_get_the_content', 'convert_smilies' );
		add_filter( 'epl_get_the_content', 'convert_chars' );
		add_filter( 'epl_get_the_content', 'wpautop' );
		add_filter( 'epl_get_the_content', 'shortcode_unautop' );
		add_filter( 'epl_get_the_content', 'prepend_attachment' );
		$vidembed = new WP_Embed();
		add_filter( 'epl_get_the_content', array( &$vidembed, 'run_shortcode' ), 8 );
		add_filter( 'epl_get_the_content', array( &$vidembed, 'autoembed' ), 8 );
		add_filter( 'epl_get_the_content', 'do_shortcode', 11 );
		add_filter( 'epl_get_the_content', 'do_blocks', 11 );
	}

	add_filter( 'epl_get_the_excerpt', 'epl_trim_excerpt' );
}
add_action( 'init', 'epl_the_content_filters', 1 );

/**
 * Disable property-box left and right class
 *
 * @since      2.2
 */
function epl_compatibility_archive_class_callback() {
	$class = '-disable';
	echo esc_attr( $class );
}

/**
 * Apply I'm feeling lucky theme options
 *
 * @since      2.2
 */
function epl_apply_feeling_lucky_config() {

	global $epl_settings;

	$epl_posts = epl_get_active_post_types();
	$epl_posts = array_keys( $epl_posts );

	// remove epl featured image on single pages in lucky mode.
	if ( 'on' === epl_get_option( 'epl_lucky_disable_single_thumb' ) ) {

		if ( is_single() && in_array( get_post_type(), $epl_posts, true ) ) {
			remove_all_actions( 'epl_property_featured_image' );
		}
	}

	// remove active theme's featured image on single pages in lucky mode.
	if ( 'on' === epl_get_option( 'epl_lucky_disable_theme_single_thumb' ) ) {

		if ( is_single() && in_array( get_post_type(), $epl_posts, true ) ) {
			add_filter( 'post_thumbnail_html', 'epl_remove_single_thumbnail', 20, 5 );
		}
	}

	// remove featured image on archive pages in lucky mode.
	if ( 'on' === epl_get_option( 'epl_lucky_disable_archive_thumb' ) ) {

		if ( is_post_type_archive( $epl_posts ) ) {
			add_filter( 'post_thumbnail_html', 'epl_remove_archive_thumbnail', 20, 5 );
		}
	}

	// remove epl featured image on archive pages in lucky mode.
	if ( 'on' === epl_get_option( 'epl_lucky_disable_epl_archive_thumb' ) ) {

		if ( is_post_type_archive( $epl_posts ) ) {
			remove_all_actions( 'epl_property_archive_featured_image' );

			// Adds class to disable property-box right and left.
			add_action( 'epl_compatibility_archive_class', 'epl_compatibility_archive_class_callback' );
		}
	}

}
add_action( 'wp', 'epl_apply_feeling_lucky_config', 1 );

/**
 * A workaround to avoid duplicate thumbnails for single listings being
 * displayed on archive pages via theme & epl attempts to null the post
 * thumbnail image called from theme & display thumbnail image called from epl
 *
 * @param string $html               The html.
 * @param string $post_id            The post identifier.
 * @param string $post_thumbnail_id  The post thumbnail identifier.
 * @param string $size               The size.
 * @param string $attr               The attribute.
 * @return null
 *
 * @since 2.2
 */
function epl_remove_archive_thumbnail( $html, $post_id, $post_thumbnail_id, $size, $attr ) {

	if ( is_admin() ) {
		return $html;
	}

	if ( is_epl_post_archive() ) { //phpcs:ignore
		// Allow archive listing images as well as widget images.
		if ( //phpcs:ignore
			doing_action( 'epl_property_archive_featured_image' ) ||
			doing_action( 'epl_property_widgets_featured_image' ) ||
			doing_action( 'epl_author_thumbnail' ) ||
			doing_action( 'epl_author_widget_thumbnail' )
		) {

		} else {
			$html = '';
		}
	}

	return $html;
}

/**
 * A workaround to avoid duplicate thumbnails for single listings
 *
 * @since      2.2
 *
 * @param      string $html               The html.
 * @param      string $post_id            The post identifier.
 * @param      string $post_thumbnail_id  The post thumbnail identifier.
 * @param      string $size               The size.
 * @param      string $attr               The attribute.
 *
 * @return     string  ( description_of_the_return_value )
 */
function epl_remove_single_thumbnail( $html, $post_id, $post_thumbnail_id, $size, $attr ) {

	if ( is_admin() ) {
		return $html;
	}

	if ( is_epl_post() ) {
		// Allow single listing images as well as widget images.
		if ( doing_action( 'epl_property_featured_image' ) || doing_action( 'epl_property_widgets_featured_image' ) ) { //phpcs:ignore

		} else {
			$html = '';
		}
	}
	return $html;
}

/**
 * Custom property the_content
 *
 * @since      2.2
 */
function epl_the_content() {

	global $property;
	$content = apply_filters( 'epl_get_the_content', get_the_content() );
	echo str_replace( ']]>', ']]&gt;', $content ); //phpcs:ignore
}
add_action( 'epl_property_the_content', 'epl_the_content' );

/**
 * Custom property the_content
 *
 * @param string $content  The content.
 *
 * @return false|string
 *
 * @since 2.2
 * @since 3.4.45 Fix: Compatibility mode accidentally styling in admin area.
 */
function epl_feeling_lucky( $content ) {

	global $epl_settings;

	if ( is_admin() ) {
		return $content;
	}

	if ( ! isset( $epl_settings['epl_feeling_lucky'] ) || 'on' !== $epl_settings['epl_feeling_lucky'] ) {
		return $content;
	}

	$epl_posts = epl_get_active_post_types();
	$epl_posts = array_keys( $epl_posts );

	if ( is_single() && in_array( get_post_type(), $epl_posts, true ) ) {
		ob_start();
		do_action( 'epl_property_single' );
		return ob_get_clean();
	} elseif ( is_post_type_archive( $epl_posts ) ) {
		ob_start();
		do_action( 'epl_property_blog' );
		/**
		* Using return VS echo resolves issues with Yoast SEO repeating content in some cases but breaks the template loading correctly in compatibility mode.
		*/
		echo ob_get_clean(); //phpcs:ignore

	} else {
		return $content;
	}
}

add_filter( 'the_content', 'epl_feeling_lucky' );

/**
 * Custom property the_excerpt
 *
 * @since      2.2
 *
 * @param      string $text   The text.
 *
 * @return mixed|void
 */
function epl_trim_excerpt( $text = '' ) {

	$raw_excerpt = $text;
	if ( empty( $text ) ) {
		$text = get_the_content( '' );

		$text = strip_shortcodes( $text );

		$text = apply_filters( 'epl_get_the_content', $text );
		$text = str_replace( ']]>', ']]&gt;', $text );

		$excerpt_length = apply_filters( 'excerpt_length', 55 );
		$excerpt_more   = apply_filters( 'excerpt_more', ' [&hellip;]' );
		$text           = wp_trim_words( $text, $excerpt_length, $excerpt_more );

	}
	return apply_filters( 'epl_trim_excerpt', $text, $raw_excerpt );
}

/**
 * Custom property the_excerpt
 *
 * @since      2.2
 */
function epl_the_excerpt() {
	echo wp_kses_post( apply_filters( 'epl_the_excerpt', epl_get_the_excerpt() ) );
}

/**
 * Custom property the_excerpt
 *
 * @param      string $post The post content.
 *
 * @return mixed|string|void
 *
 * @since 2.2
 * @since 3.4.46 Fix: Support for arguments when in theme compatibility mode and using the_excerpt.
 */
function epl_get_the_excerpt( $post = null ) {

	if ( is_bool( $post ) ) {
		_deprecated_argument( __FUNCTION__, '2.3' );
	}

	if ( $post instanceof WP_Post ) {
		$post = get_post( $post );
	} else {
		$post = get_post();
	}

	if ( empty( $post ) ) {
		return '';
	}

	if ( post_password_required() ) {
		return esc_html__( 'There is no excerpt because this is a protected post.', 'easy-property-listings' );
	}

	return apply_filters( 'epl_get_the_excerpt', $post->post_excerpt );
}

/**
 * Syntax Highlighter
 *
 * @since      2.2
 *
 * @param      string $str    The string.
 * @param      string $class  The class.
 *
 * @return     string
 */
function epl_syntax_highlight( $str = '', $class = '' ) {
	return '<pre><code class="' . $class . '">' . htmlentities( $str ) . '</code></pre>';
}

/**
 * Strip Tags
 *
 * @param      string $value the value.
 * @param      string $allowed_tags allowed tags.
 *
 * @return array|string
 *
 * @since      2.2
 */
function epl_strip_tags( $value, $allowed_tags = '' ) {

	if ( ! is_array( $value ) ) {
		return strip_tags( $value, $allowed_tags );
	}
	return $value;
}

/**
 * Esc Attr
 *
 * @param string $value  The value.
 * @return string|void
 *
 * @since 2.2.0
 * @since 3.4.23 Fix security function check from array to string.
 */
function epl_esc_attr( $value ) {

	if ( is_string( $value ) ) {
		return esc_attr( $value );
	}
	return $value;
}

/**
 * Post Count
 *
 * @param      string $type        The type.
 * @param      string $meta_key    The meta key.
 * @param      string $meta_value  The meta value.
 * @param      string $author_id   The author identifier.
 *
 * @return     null
 *
 * @since      2.2.0
 * @since      3.4.38 PHP 8.0 fix.
 */
function epl_get_post_count( $type, $meta_key, $meta_value, $author_id = '' ) {
	global $wpdb;

	$sql = "
		SELECT count( Distinct p.ID ) AS count
		FROM {$wpdb->prefix}posts AS p
		INNER JOIN $wpdb->postmeta pm  ON (p.ID = pm.post_id)
		INNER JOIN $wpdb->postmeta pm2  ON (p.ID = pm2.post_id)
		WHERE p.post_status = 'publish' ";

	if ( empty( $type ) ) {
		$epl_posts = epl_get_active_post_types();
		$epl_posts = '"' . implode( '","', array_keys( $epl_posts ) ) . '"';

		$sql .= " AND p.post_type IN ( {$epl_posts} )";
	} else {
		$sql .= " AND p.post_type = '{$type}'";
	}

	if ( ! empty( $author_id ) ) {
		$user_info = get_userdata( $author_id );
		$sql      .= " AND (
			p.post_author =  $author_id
			OR (
				pm2.meta_key 	= 'property_second_agent'
				AND
				pm2.meta_value 	= '$user_info->user_login'
			)
		)";
	}
	$sql  .= "
		AND p.ID = pm.post_id
		AND pm.meta_key = '{$meta_key}'
		AND pm.meta_value = '{$meta_value}'
	";
	$count = $wpdb->get_row( $sql ); //phpcs:ignore
	return $count->count;
}

/**
 * Get the inspection date format
 *
 * @since 3.3
 *
 * @return string
 */
function epl_get_inspection_date_format() {

	$date_format = epl_get_option( 'inspection_date_format' ) === 'custom_inspection_date_format' ?
		epl_get_option( 'custom_inspection_date_format' ) : epl_get_option( 'inspection_date_format' );

	if ( empty( $date_format ) ) {
		$date_format = 'd-M-Y';
	}

	return apply_filters( 'epl_inspection_date_format', $date_format );
}

/**
 * Get the inspection time format
 *
 * @since 3.3
 *
 * @return     string
 */
function epl_get_inspection_time_format() {

	$time_format = 'custom_inspection_time_format' === epl_get_option( 'inspection_time_format' ) ?
			epl_get_option( 'custom_inspection_time_format' ) : epl_get_option( 'inspection_time_format' );

	if ( empty( $time_format ) ) {
		$time_format = 'h:i A';
	}

	return apply_filters( 'epl_inspection_time_format', $time_format );

}

/**
 * Inspection Format
 *
 * @since      2.2
 *
 * @param      string $inspection_date  The inspection date.
 *
 * @return     string
 */
function epl_inspection_format( $inspection_date ) {

	$formatted_date  = '';
	$inspection_date = explode( ' ', $inspection_date );

	$date_format = epl_get_inspection_date_format();
	$time_format = epl_get_inspection_time_format();

	$date       = isset( $inspection_date[0] ) ? date( $date_format, strtotime( $inspection_date[0] ) ) : '';
	$time_start = isset( $inspection_date[1] ) ? date( $time_format, strtotime( $inspection_date[1] ) ) : '';
	$time_end   = isset( $inspection_date[3] ) ? date( $time_format, strtotime( $inspection_date[3] ) ) : '';

	return "{$date} {$time_start} to {$time_end}";
}
add_action( 'epl_inspection_format', 'epl_inspection_format' );

/**
 * Counts the total number of contacts.
 *
 * @access     public
 * @since      3.0
 *
 * @return     int   - The total number of contacts.
 */
function epl_count_total_contacts() {
	$counts = wp_count_posts( 'epl_contact' );
	return $counts->publish;
}

/**
 * Hide contacts notes from showing on frontend
 *
 * @param      array  $comments  The comments.
 * @param      string $post_id   The post identifier.
 *
 * @return     array
 * @since      3.0
 */
function epl_filter_listing_comments_array( $comments, $post_id ) {
	foreach ( $comments as $key   => &$comment ) {
		if ( 'epl' === $comment->comment_agent ) {
			unset( $comments[ $key ] );
		}
	}
	return $comments;
}
add_filter( 'comments_array', 'epl_filter_listing_comments_array', 10, 2 );

/**
 * Archive Page Title
 *
 * @return void the archive title
 *
 * @since 3.0.0
 * @since 3.5.2 Added filter epl_archive_title to alter archive title.
 * @since 3.5.14 Corrected escaping function to esc_html.
 */
function epl_archive_title_callback() {
	the_post();

	if ( is_tax() && function_exists( 'epl_is_search' ) && false === epl_is_search() ) { // Tag Archive.
		$term = get_term_by( 'slug', get_query_var( 'term' ), get_query_var( 'taxonomy' ) );
		// Translators: term name.
		$title = sprintf( __( 'Property in %s', 'easy-property-listings' ), $term->name );
	} elseif ( function_exists( 'epl_is_search' ) && epl_is_search() ) { // Search Result.
		$title = apply_filters( 'epl_archive_title_search_result', __( 'Search Result', 'easy-property-listings' ) );
	} elseif ( function_exists( 'is_post_type_archive' ) && is_post_type_archive() && function_exists( 'post_type_archive_title' ) ) { // Post Type Archive.
		$title = post_type_archive_title( '', false );
		$title = apply_filters( 'epl_archive_title_post_type', $title );
	} else { // Default catchall just in case.
		$title = apply_filters( 'epl_archive_title_fallback', __( 'Listing', 'easy-property-listings' ) );
	}

	if ( is_paged() ) {
		// Translators: title, page number.
		printf( '%s &ndash; Page %d', esc_html( $title ), esc_attr( get_query_var( 'paged' ) ) );
	} else {
		echo wp_kses_post( apply_filters( 'epl_archive_title_default', $title ) );
	}

	rewind_posts();
}
add_action( 'epl_the_archive_title', 'epl_archive_title_callback' );

/**
 * Shortcode Sorter
 *
 * @param      array  $args   The arguments.
 * @param      string $type   The type.
 * @param      string $name   The name.
 *
 * @return array $args
 *
 * @since      3.0
 * @since      3.4.44 Option sortby based on instance_id now.
 */
function epl_add_orderby_args( $args, $type = '', $name = '' ) {

	if ( 'shortcode' === $type ) {
		$args['is_epl_shortcode']   = true;
		$args['epl_shortcode_name'] = $name;
	}

	$post_type = isset( $args['post_type'] ) ? current( $args['post_type'] ) : '';
	$post_type = sanitize_text_field( wp_unslash( $post_type ) );
	$orderby   = isset( $_GET['sortby'] ) ? sanitize_text_field( wp_unslash( $_GET['sortby'] ) ) : '';

	$instance_id     = ! empty( $args['instance_id'] ) ? sanitize_text_field( $args['instance_id'] ) : '';
	$set_instance_id = '';

	if ( ! empty( $_GET['instance_id'] ) ) {
		$set_instance_id = sanitize_text_field( wp_unslash( $_GET['instance_id'] ) );
	}

	if ( ! empty( $orderby ) && ( ! empty( $instance_id ) && $instance_id === $set_instance_id ) ) {

		$sorters = epl_sorting_options( $post_type );

		foreach ( $sorters as $sorter ) {

			if ( $orderby === $sorter['id'] ) {

				if ( 'meta' === $sorter['type'] ) {
					$args['orderby']  = $sorter['orderby'];
					$args['meta_key'] = $sorter['key']; //phpcs:ignore
				} else {
					$args['orderby'] = $sorter['key'];
				}
				$args['order'] = $sorter['order'];
				break;
			}
		}
	}
	return $args;
}

/**
 * Shortcode Results Not Found Message
 *
 * @param string $shortcode  The shortcode.
 *
 * @since 3.1.5
 * @since 3.4.48 Added new class for the message removed incorrect class name.
 * @since 3.5 Tweak: Shortcode results message allow basic html.
 */
function epl_shortcode_results_message_callback( $shortcode = 'default' ) {

	$title = apply_filters( 'epl_shortcode_results_message_title', __( 'Nothing found, please check back later.', 'easy-property-listings' ) );

	if ( 'open' === $shortcode ) {
		$title = apply_filters( 'epl_shortcode_results_message_title_open', __( 'Nothing currently scheduled for inspection, please check back later.', 'easy-property-listings' ) );
	}

	echo '<h3 class="epl-alert epl-shortcode-results-message epl-shortcode-results-message-' . esc_attr( $shortcode ) . '">' . wp_kses_post( $title ) . '</h3>';

}
add_action( 'epl_shortcode_results_message', 'epl_shortcode_results_message_callback' );

/**
 * Search Not Found Messages
 *
 * @since 3.1.8
 * @since 3.4.33 Tweak: Search results not found filter epl_property_search_not_found_title allows basic html to be passed.
 * @since 3.4.48 Container added to search not found.
 */
function epl_property_search_not_found_callback() {

	$title   = apply_filters( 'epl_property_search_not_found_title', __( 'Listing not Found', 'easy-property-listings' ) );
	$message = apply_filters( 'epl_property_search_not_found_message', __( 'Listing not found, expand your search criteria and try again.', 'easy-property-listings' ) );

	?>
	<div class="epl-search-not-found">
		<div class="epl-search-not-found-title entry-header clearfix">
			<h3 class="entry-title"><?php echo wp_kses_post( $title ); ?></h3>
		</div>

		<div class="epl-search-not-found-message entry-content clearfix">
			<p><?php echo wp_kses_post( $message ); ?></p>
		</div>
	</div>

	<?php
}
add_action( 'epl_property_search_not_found', 'epl_property_search_not_found_callback' );

/**
 * Add Listing Status and Under Offer to Post Class
 *
 * @param      array $classes  The classes.
 *
 * @return     array
 *
 * @since 3.1.16
 * @since 3.4.23 Added compatibility class.
 */
function epl_property_post_class_listing_status_callback( $classes ) {

	if ( is_epl_post() ) {

		$property_status      = get_property_meta( 'property_status' );
		$property_under_offer = get_property_meta( 'property_under_offer' );
		$commercial_type      = get_property_meta( 'property_com_listing_type' );
		$class_prefix         = 'epl-status-';

		if ( ! empty( $property_status ) ) {
			$classes[] = $class_prefix . strtolower( $property_status );
		}
		if ( 'yes' === $property_under_offer && 'sold' !== $property_status ) {
			$classes[] = $class_prefix . 'under-offer';
		}
		if ( ! empty( $commercial_type ) ) {
			$class_prefix = 'epl-commercial-type-';
			$classes[]    = $class_prefix . strtolower( $commercial_type );
		}
	}

	if ( 'on' === epl_get_option( 'epl_feeling_lucky', 'off' ) && is_epl_post_archive() ) {
		$classes[] = 'epl-property-blog-compatibility';
	}

	return $classes;
}
add_filter( 'post_class', 'epl_property_post_class_listing_status_callback' );

/**
 * Get the author loop
 *
 * @since 3.3
 * @since 3.4.38 Support for third & forth listing agents.
 */
function epl_archive_author_callback() {
	global $epl_author_secondary, $epl_author_third, $epl_author_fourth;
	epl_get_template_part( 'content-author-archive-card.php' );

	if ( is_epl_post() ) {
		if ( epl_listing_has_secondary_author() ) {
			epl_get_template_part( 'content-author-archive-card.php', array( 'epl_author' => $epl_author_secondary ) );
		}
		if ( epl_listing_has_third_agent() ) {
			epl_get_template_part( 'content-author-archive-card', array( 'epl_author' => $epl_author_third ) );
		}
		if ( epl_listing_has_fourth_agent() ) {
			epl_get_template_part( 'content-author-archive-card', array( 'epl_author' => $epl_author_fourth ) );
		}
		epl_reset_post_author();
	}
}
add_action( 'epl_archive_author', 'epl_archive_author_callback' );

/**
 * Contact capture action and messages
 *
 * @since      3.3
 */
function epl_contact_capture_action() {

	$success = array(
		'status' => 'success',
		'msg'    => apply_filters( 'epl_contact_capture_success_msg', __( 'Form submitted successfully', 'easy-property-listings' ) ),
	);

	$fail = array(
		'status' => 'fail',
		'msg'    => apply_filters( 'epl_contact_capture_fail_msg', __( 'Some issues with form submitted', 'easy-property-listings' ) ),
	);

	if (
		! isset( $_POST['epl_contact_widget'] ) ||
		! wp_verify_nonce( sanitize_text_field( wp_unslash( $_POST['epl_contact_widget'] ) ), 'epl_contact_widget' )
	) {
		wp_die( wp_json_encode( $fail ) );
	}

	if ( ! empty( $_POST['epl_contact_anti_spam'] ) ) {
		wp_die( wp_json_encode( $fail ) );
	}

	if ( empty( $_POST['epl_contact_email'] ) ) {
		wp_die(
			wp_json_encode(
				array(
					'status' => 'fail',
					'msg'    => __(
						'Email is required',
						'easy-property-listings'
					),
				)
			)
		);
	}

	$contact = new EPL_contact( sanitize_text_field( wp_unslash( $_POST['epl_contact_email'] ) ) );
	$fname   = isset( $_POST['epl_contact_first_name'] ) ?
	sanitize_text_field( wp_unslash( $_POST['epl_contact_first_name'] ) ) : '';
	$lname   = isset( $_POST['epl_contact_last_name'] ) ?
	sanitize_text_field( wp_unslash( $_POST['epl_contact_last_name'] ) ) : '';
	$phone   = isset( $_POST['epl_contact_phone'] ) ?
	sanitize_text_field( wp_unslash( $_POST['epl_contact_phone'] ) ) : '';
	$title   = isset( $_POST['epl_contact_title'] ) ?
	sanitize_text_field( wp_unslash( $_POST['epl_contact_title'] ) ) : '';
	$title   = trim( $title );
	if ( empty( $title ) && ( ! empty( $fname ) || ! empty( $lname ) ) ) {
		$title = $fname . ' ' . $lname;
	}

	if ( empty( $title ) && ( ! empty( $_POST['epl_contact_email'] ) ) ) {
		$title = sanitize_text_field( wp_unslash( $_POST['epl_contact_email'] ) );
	}

	$contact_listing_id = isset( $_POST['epl_contact_listing_id'] ) ?
	sanitize_text_field( wp_unslash( $_POST['epl_contact_listing_id'] ) ) : false;

	$contact_listing_note = isset( $_POST['epl_contact_note'] ) ?
	sanitize_text_field( wp_unslash( $_POST['epl_contact_note'] ) ) : false;
	if ( empty( $contact->ID ) ) {

		$contact_data = array(
			'name'  => $title,
			'email' => sanitize_email( wp_unslash( $_POST['epl_contact_email'] ) ),
		);
		if ( $contact->create( $contact_data ) ) {
			$contact->update_meta( 'contact_first_name', $fname );
			$contact->update_meta( 'contact_last_name', $lname );
			$contact->update_meta( 'contact_phones', array( 'phone' => $phone ) );
			$contact->update_meta( 'contact_category', 'widget' );
			$contact->attach_listing( $contact_listing_id );
			$contact->add_note( $contact_listing_note, 'note', $contact_listing_id );
			wp_die( wp_json_encode( $success ) );
		} else {
			wp_die( wp_json_encode( $fail ) );
		}
	} else {

		if ( $contact->update( array( 'name' => $title ) ) ) {
			$contact->add_note(
				sanitize_textarea_field( wp_unslash( $_POST['epl_contact_note'] ) ),
				'note',
				$contact_listing_id
			);
			$contact->attach_listing( $contact_listing_id );
			wp_die( wp_json_encode( $success ) );
		} else {
			wp_die( wp_json_encode( $fail ) );
		}
	}
}
add_action( 'wp_ajax_epl_contact_capture_action', 'epl_contact_capture_action' );
add_action( 'wp_ajax_nopriv_epl_contact_capture_action', 'epl_contact_capture_action' );

/**
 * Get Post ID from Unique ID
 *
 * @param  string $unique_id Unique ID.
 * @return mixed false if not found, else Post ID
 * @since 3.5.0
 */
function epl_get_post_id_from_unique_id( $unique_id = '' ) {

	if ( '' === $unique_id ) {
		return false;
	}

	$args = array(
		'meta_key'       => 'property_unique_id',
		'meta_value'     => $unique_id,
		'post_type'      => epl_get_core_post_types(),
		'post_status'    => 'publish',
		'posts_per_page' => -1,
	);

	$posts = get_posts( $args );

	if ( ! empty( $posts ) ) {
		$post = current( $posts );
		return $post->ID;
	}

	return false;
}

/**
 * Renders stickers, based on meta values, an alternative to epl_price_stickers.
 *
 * @param array $options The options.
 * @param array $stickers The stickers.
 *
 * @throws Exception Something.
 * @since 3.4.27
 */
function epl_stickers( $options = array(), $stickers = array() ) {

	global $post;

	if ( ! is_epl_post() ) {
		return;
	}

	$default_options = array(
		'wrap'          => false,
		'wrap_class'    => 'epl-stickers-wrapper',
		'wrapper_tag'   => 'div',
		'sticker_tag'   => 'span',
		'sticker_class' => 'status-sticker',
		'max_stickers'  => 2,
	);

	$sticker_counts = 0;

	$options = array_merge( $default_options, $options );

	if ( empty( $stickers ) ) {
		$stickers = epl_get_stickers_array();
	}

	$options = apply_filters( 'epl_stickers_options', $options );

	if ( $options['wrap'] ) {
		echo '<' . esc_attr( $options['wrapper_tag'] ) . ' class="' . esc_attr( $options['wrap_class'] ) . '">';
	}

	foreach ( $stickers as $key => $sticker ) {

		if ( $sticker_counts === $default_options['max_stickers'] ) {
			break;
		}
		?>

		<<?php echo esc_attr( $options['sticker_tag'] ); ?> class="<?php echo esc_attr( $options['sticker_class'] . ' ' . $sticker['class'] ); ?>">
			<?php
				echo isset( $sticker['before'] ) ?
				wp_kses_post( $sticker['before'] ) : '';
				echo wp_kses_post( $sticker['label'] );
				echo isset( $sticker['after'] ) ?
				wp_kses_post( $sticker['after'] ) : '';
			?>
		</<?php echo esc_attr( $options['sticker_tag'] ); ?>>
			<?php
				$sticker_counts++;
	}

	if ( $options['wrap'] ) {
		echo '</' . esc_attr( $options['wrapper_tag'] ) . '>';
	}

}

/**
 * Returns stickers array based on type, status etc.
 *
 * @param array $sticker_keys Array of keys.
 *
 * @return mixed|void
 *
 * @throws Exception Something.
 * @since 3.4.27
 * @since 3.4.28 Added array and filter.
 */
function epl_get_stickers_array( $sticker_keys = array() ) {

	global $post;

	$stickers = array(

		'under_offer'           => array(
			'conditions' => array(
				'property_status'      => 'current',
				'property_under_offer' => array( 'yes' ),
			),
			'type'       => epl_get_core_post_types(),
			'label'      => epl_get_option( 'label_under_offer' ),
			'before'     => '',
			'after'      => '',
			'class'      => 'under-offer',
		),
		'home_open'             => array(
			'conditions' => array(
				'property_inspection_times' => array( null, '' ),
			),
			'compare'    => '!=',
			'type'       => epl_get_core_post_types(),
			'label'      => epl_get_option( 'label_home_open' ),
			'before'     => '',
			'after'      => '',
			'class'      => 'open',
		),

		'new'                   => array(
			'type'   => epl_get_core_post_types(),
			'label'  => epl_get_option( 'label_new' ),
			'before' => '',
			'after'  => '',
			'class'  => 'new',

		),
		'rental_lease'          => array(
			'conditions' => array(
				'property_status' => 'current',
			),
			'type'       => array( 'rental' ),
			'label'      => __( 'For Lease', 'easy-property-listings' ),
			'before'     => '',
			'after'      => '',
			'class'      => 'for-lease',
		),
		'leased'                => array(
			'conditions' => array(
				'property_status' => 'leased',
			),
			'type'       => array( 'rental', 'commercial_land', 'commercial' ),
			'label'      => epl_get_option( 'label_leased' ),
			'before'     => '',
			'after'      => '',
			'class'      => 'leased',
		),
		'current_sales'         => array(
			'conditions' => array(
				'property_status' => 'current',
			),
			'type'       => array( 'property', 'rural', 'land', 'business' ),
			'label'      => __( 'For Sale', 'easy-property-listings' ),
			'before'     => '',
			'after'      => '',
			'class'      => 'for-sale',
		),
		'sold'                  => array(
			'conditions' => array(
				'property_status' => 'sold',
			),
			'type'       => epl_get_core_post_types(),
			'label'      => epl_get_option( 'label_sold' ),
			'before'     => '',
			'after'      => '',
			'class'      => 'sold',
		),
		'commercial_sale_lease' => array(
			'conditions' => array(
				'property_status'           => 'current',
				'property_com_listing_type' => array( 'both' ),
			),
			'type'       => array( 'commercial', 'commercial_land' ),
			'label'      => __( 'For Sale & Lease', 'easy-property-listings' ),
			'before'     => '',
			'after'      => '',
			'class'      => 'for-sale',
		),
		'commercial_sale'       => array(
			'conditions' => array(
				'property_status'           => 'current',
				'property_com_listing_type' => array( 'sale' ),
			),
			'type'       => array( 'commercial', 'commercial_land' ),
			'label'      => __( 'For Sale', 'easy-property-listings' ),
			'before'     => '',
			'after'      => '',
			'class'      => 'for-sale',
		),
		'commercial_lease'      => array(
			'conditions' => array(
				'property_status'           => 'current',
				'property_com_listing_type' => array( 'lease' ),
			),
			'type'       => array( 'commercial', 'commercial_land' ),
			'label'      => __( 'For Lease', 'easy-property-listings' ),
			'before'     => '',
			'after'      => '',
			'class'      => 'for-lease',
		),

	);

	/**
	 * Hook into this array to add / remove stickers based on conditions.
	 *
	 * @var callable
	 */
	$stickers = apply_filters( 'epl_available_stickers', $stickers );

	$return = array();

	foreach ( $stickers as $key => $sticker ) {

		$add_sticker = false;

		if ( ! empty( $sticker_keys ) && ! in_array( $key, $sticker_keys, true ) ) {
			continue;
		}

		if ( ! empty( $sticker ) && is_array( $sticker ) ) {

			if ( ! in_array( get_post_type(), $sticker['type'], true ) ) {
				continue;
			}

			if ( 'new' === $key ) {

				$date = new DateTime( $post->post_date );
				$now  = new DateTime();

				// php > 5.3.
				if ( method_exists( $now, 'diff' ) ) {

					$diff = $now->diff( $date );
					$diff = $diff->days;
				} else {
					$diff = strtotime( $now->format( 'M d Y ' ) ) - strtotime( $date->format( 'M d Y ' ) );
					$diff = floor( $diff / 3600 / 24 );

				}

				if ( epl_get_option( 'sticker_new_range' ) >= $diff ) {

					$return[ $key ] = array(
						'label'  => $sticker['label'],
						'class'  => $sticker['class'],
						'before' => $sticker['before'],
						'after'  => $sticker['after'],
					);
				}
			} else {

				$conditions = $sticker['conditions'];
				$compare    = isset( $sticker['compare'] ) ? $sticker['compare'] : '=';

				$add_sticker = epl_sticker_is_condition_valid( $conditions, $compare );

			}

			if ( $add_sticker ) {
				$return[ $key ] = array(
					'label'  => $sticker['label'],
					'class'  => $sticker['class'],
					'before' => $sticker['before'],
					'after'  => $sticker['after'],
				);
			}
		}
	}

	/**
	 * List of stickers for current listing, hook here if only need to change labels,
	 * Class etc. for displayed labels.
	 */
	return apply_filters( 'epl_stickers_array', $return );
}

/**
 * Check if sticker condition is valid.
 *
 * @param array  $condition The condition.
 * @param string $compare   Comparison.
 *
 * @return bool
 * @since 3.4.28
 */
function epl_sticker_is_condition_valid( $condition, $compare = '=' ) {

	$condition_met   = 0;
	$total_condition = count( $condition );
	$match           = false;

	foreach ( $condition as $key => $value ) {

		if ( is_array( $value ) ) {

			if ( in_array( get_property_meta( $key ), $value, true ) ) {
				$condition_met++;
			}
		} else {
			if ( get_property_meta( $key ) === $value ) {
				$condition_met++;
			}
		}
	}
	$match = $total_condition === $condition_met ? true : false;

	return '=' === $compare ? $match : ! $match;
}

/**
 * Property Status Labels.
 *
 * @param array $statues The statues.
 * @param array $options The options.
 *
 * @throws Exception Summary.
 * @since 3.4.28
 */
function epl_property_status( $statues = array(), $options = array() ) {

	$statues_defaults = array(
		'commercial_lease',
		'commercial_sale',
		'rental_lease',
		'leased',
		'sold',
		'current_sales',
		'commercial_sale_lease',
	);

	$statues = array_merge( $statues_defaults, (array) $statues );

	$statues = apply_filters( 'epl_property_status_keys', $statues );

	$stickers = epl_get_stickers_array( $statues );

	$default_options = array(
		'wrap'          => true,
		'wrap_class'    => 'epl-listing-status',
		'wrapper_tag'   => 'div',
		'sticker_tag'   => 'span',
		'sticker_class' => 'epl-widget-status-label',
	);

	$options = array_merge( $default_options, (array) $options );
	epl_stickers( $options, $stickers );
}
add_action( 'epl_property_status', 'epl_property_status', 10, 2 );

/**
 * Add Body classes for EPL Single & Archive Pages.
 *
 * @param      array $classes  The classes.
 *
 * @return     array  $classes  Modified classes.
 * @since      3.4.29
 * @since      3.4.38 Added epl-search-results custom body class for search results.
 */
function epl_body_classes( $classes ) {

	if ( is_epl_post_single() ) {
		$classes[] = 'epl-single-listing';
		$classes[] = 'epl-single-' . get_post_type();
	}

	if ( is_epl_post_archive() ) {
		$classes[] = 'epl-post-type-archive';
		$classes[] = 'epl-post-type-archive-' . get_post_type();
	}

	if ( epl_is_search() ) {
		$classes[] = 'epl-post-type-archive';
		$classes[] = 'epl-search-results';
	}

	return $classes;
}
add_filter( 'body_class', 'epl_body_classes', 10 );

/**
 * Get total agent count for listing.
 *
 * @since 3.5.3
 */
function epl_get_total_agent_count() {
	$counter = 1;

	$second_agent = get_property_meta( 'property_second_agent' );
	$third_agent  = get_property_meta( 'property_third_agent' );
	$fourth_agent = get_property_meta( 'property_fourth_agent' );
	$counter      = ! empty( $second_agent ) ? $counter + 1 : $counter;
	$counter      = ! empty( $third_agent ) ? $counter + 1 : $counter;
	$counter      = ! empty( $fourth_agent ) ? $counter + 1 : $counter;

	return $counter;
}

/**
 * Helper function to get user ID by login.
 *
 * @since 3.5.10
 *
 * @param string $login The login name of the user.
 * @return int|false The user ID if found, false otherwise.
 */
function epl_get_agent_id_by_login( $login ) {
	if ( ! empty( $login ) ) {
		$user = get_user_by( 'login', $login );
		return $user ? $user->ID : false;
	}
		return false;
}

/**
 * Get the count of active agents.
 *
 * @since 3.5.3
 * @since 3.5.10 Fix: blank agents.
 *
 * @return int The count of unique active agents.
 */
function epl_get_active_agent_count() {
		global $epl_author;

	$agents = array( $epl_author->author_id );

	$second_agent = get_property_meta( 'property_second_agent' );
	$third_agent  = get_property_meta( 'property_third_agent' );
	$fourth_agent = get_property_meta( 'property_fourth_agent' );

	$second_author_id = epl_get_agent_id_by_login( $second_agent );
	$third_author_id  = epl_get_agent_id_by_login( $third_agent );
	$fourth_author_id = epl_get_agent_id_by_login( $fourth_agent );

	if ( false !== $second_author_id ) {
		$agents[] = $second_author_id;
	}
	if ( false !== $third_author_id ) {
		$agents[] = $third_author_id;
	}
	if ( false !== $fourth_author_id ) {
		$agents[] = $fourth_author_id;
	}

	$agents = array_unique( $agents );
	$count  = count( $agents );

	return apply_filters( 'epl_get_active_agent_count', $count );
}

/**
 * Wrapper Start for Author Box
 *
 * @since 3.5.1
 * @since 3.5.3 Added counter class for number of agents.
 */
function epl_single_author_wrapper_start() {

	$counter = epl_get_total_agent_count();
	$active  = epl_get_active_agent_count();
	?>
	<div class="epl-author-box-wrapper epl-author-box-wrapper--count-<?php echo esc_attr( $active ); ?> epl-author-box-wrapper--total-<?php echo esc_attr( $counter ); ?> ">
	<?php
}
add_action( 'epl_single_author', 'epl_single_author_wrapper_start', 1 );

/**
 * Wrapper end for Author Box
 *
 * @since 3.5.1
 */
function epl_single_author_wrapper_end() {
	?>
	</div>
	<?php
}
add_action( 'epl_single_author', 'epl_single_author_wrapper_end', 99 );

/**
 * Check if the value is truthy / falsy.
 *
 * @param string $value Values.
 *
 * @return bool
 *
 * @since 3.5.13
 */
function epl_value_bool_checker( $value ) {

	$truthy_values = array( 1, 'yes', 'true' );
	$truthy_values = apply_filters( 'epl_truthy_values', $truthy_values );

	if ( in_array( $value, $truthy_values, true ) ) {
		return true;
	} else {
		return false;
	}
}
