import fonts from "../../includes/google-fonts.json";
import Select from 'react-select';
import SettingsContext from "../../contexts/SettingsContext";
import Header from "./Header";
import { showProModal } from "../ProModal";
import { Tooltip } from "react-tooltip";
import LiveEditButton from "./LiveEditButton";

const {
    FormToggle, RangeControl,
    __experimentalRadioGroup: RadioGroup,
    __experimentalRadio: Radio,
} = wp.components;

const { useState, useContext } = React;

export default function Typography() {
    const { isPro } = dracula;

    const { data, setData, isLiveEdit } = useContext(SettingsContext);

    const systemFonts = fonts.filter(font => font.category);
    const googleFonts = fonts.filter(font => !font.category);

    const options = [
        {
            label: wp.i18n.__('System Fonts', 'dracula-dark-mode'),
            options: systemFonts,
        },
        {
            label: wp.i18n.__('Google Fonts', 'dracula-dark-mode'),
            options: googleFonts,
        }
    ];

    const {
        changeFont,
        fontFamily,
        textStroke = 0,
        fontSize = "1",
        customFontSize = 100,
    } = data;

    const [loaded, setLoaded] = useState([]);

    const isVisible = function (ele, container) {
        const { bottom, height, top } = ele.getBoundingClientRect();
        const containerRect = container.getBoundingClientRect();

        return top <= containerRect.top ? containerRect.top - top <= height : bottom - containerRect.bottom <= height;
    };

    return (
        <div className="dracula-settings-content">
            <Header title={wp.i18n.__('Typography Settings', 'dracula-dark-mode')} />

            {/* Change Font */}
            <div className={`settings-field ${!isPro ? 'pro-feature' : ''} field-change-font`}>
                <h4 className={`settings-field-label`}>
                    {wp.i18n.__('Change Font', 'dracula-dark-mode')}
                    {!isPro && <span className="dracula-pro-icon"><i className={`dracula-icon icon-pro`}/></span>}
                </h4>
                <div className="settings-field-content">
                    <FormToggle
                        data-tooltip-id={`tooltip-change-font`}
                        data-tooltip-content={!isPro ? wp.i18n.__('PRO Feature', 'dracula-dark-mode') : ''}
                        checked={changeFont}
                        onChange={() => {
                            if (!isPro) {
                                showProModal();
                                return;
                            }

                            setData({ ...data, changeFont: !changeFont })
                        }}
                    />

                    {!isPro && <Tooltip id={`tooltip-change-font`} place="right" variant="warning" effect="solid" backgroundColor="#ff9966" />}

                    <p className="description">{wp.i18n.__('Use a different font-family for the dark mode scheme.', 'dracula-dark-mode')}</p>
                </div>
            </div>

            {/* Font Family */}
            {(changeFont || !isPro) &&
                <>
                    <div className={`settings-field ${!isPro ? 'pro-feature' : ''}`}>
                        <h4 className={`settings-field-label`}>
                            {wp.i18n.__('Font Family', 'dracula-dark-mode')}
                            {!isPro && <span className="dracula-pro-icon"><i className={`dracula-icon icon-pro`}/></span>}
                        </h4>

                        <div className="settings-field-content">
                            {!isLiveEdit && <LiveEditButton tab={`typography`} />}

                            <div className="settings-wrap"
                                data-tooltip-content={!isPro ? wp.i18n.__('PRO Feature', 'dracula-dark-mode') : ''}
                                data-tooltip-id={`tooltip-fontfamily`} onClick={() => !isPro && showProModal()}>
                                <Select
                                    isDisabled={!isPro}
                                    options={options}
                                    value={fonts.find(item => item.value === fontFamily)}
                                    onChange={selected => {
                                        setData({ ...data, fontFamily: selected.value });
                                    }}
                                    className={`dracula-select select-font-family dracula-ignore`}
                                    classNamePrefix={`dracula-select`}

                                    menuPosition={'fixed'}

                                    onMenuOpen={() => {

                                        setTimeout(() => {
                                            const $ = jQuery;

                                            $('.dracula-select__menu-list').on('scroll', function () {
                                                $('.dracula-select__option').each((index, element) => {

                                                    if (isVisible(element, this)) {
                                                        const fontFamily = $(element).text();

                                                        if (!loaded.includes(fontFamily)) {
                                                            setLoaded(loaded => [...loaded, fontFamily]);
                                                            $(element).css('font-family', fontFamily);
                                                            // Load font family
                                                            const link = document.createElement('link');
                                                            link.href = `https://fonts.googleapis.com/css?family=${fontFamily.replace(/ /g, '+')}`;
                                                            link.rel = 'stylesheet';
                                                            document.head.appendChild(link);
                                                        }
                                                    }

                                                });
                                            });

                                        }, 10);
                                    }}

                                    onMenuClose={() => {
                                        const $ = jQuery;
                                        $('.dracula-select__menu-list').off('scroll');
                                    }}

                                    styles={{
                                        singleValue: (base, { data }) => ({
                                            ...base,
                                            fontFamily: data.label,
                                            fontSize: '1rem',
                                        }),
                                    }}

                                />
                            </div>

                            {!isPro && <Tooltip id={`tooltip-fontfamily`} effect="solid" place="right" variant="warning" backgroundColor="#ff9966" className={'dracula-tooltip'} />}

                            <p className="description">{wp.i18n.__('Select a font family for the dark mode scheme.', 'dracula-dark-mode')}</p>
                        </div>
                    </div>

                    {/*  Font Size */}
                    <div className={`settings-field ${!isPro ? 'pro-feature' : ''}`}>
                        <h4 className={`settings-field-label`}>
                            {wp.i18n.__('Font Size', 'dracula-dark-mode')}
                            {!isPro && <span className="dracula-pro-icon"><i className={`dracula-icon icon-pro`}/></span>}
                        </h4>

                        <div className="settings-field-content">

                            <RadioGroup
                                data-tooltip-id={`tooltip-fontsize`}
                                data-tooltip-content={!isPro ? wp.i18n.__('PRO Feature', 'dracula-dark-mode') : ''}
                                onChange={fontSize => {
                                    if (!isPro) {
                                        showProModal();
                                        return;
                                    }
                                    setData({ ...data, fontSize })
                                }}
                                checked={fontSize}
                                className={'dracula-radio-group'}>
                                <Radio value=".75">
                                    <span className="dracula-btn-icon icon-size-small"></span>
                                    <span>{wp.i18n.__('Small', 'dracula-dark-mode')}</span>
                                </Radio>
                                <Radio value="1">
                                    <span className="dracula-btn-icon icon-size-normal"></span>
                                    <span>{wp.i18n.__('Medium', 'dracula-dark-mode')}</span>
                                </Radio>
                                <Radio value="1.2">
                                    <span className="dracula-btn-icon icon-size-large"></span>
                                    <span>{wp.i18n.__('Large', 'dracula-dark-mode')}</span>
                                </Radio>
                                <Radio value="custom">
                                    <span className="dracula-btn-icon icon-custom"></span>
                                    <span>{wp.i18n.__('Custom', 'dracula-dark-mode')}</span>
                                </Radio>
                            </RadioGroup>

                            {!isPro && <Tooltip id={`tooltip-fontsize`} effect="solid" place="right" backgroundColor="#ff9966" variant="warning" className={'dracula-tooltip'} />}

                            <p className="description">{wp.i18n.__('Set the font-size in the dark mode.', 'dracula-dark-mode')}</p>
                        </div>

                        {/* Custom Font Size */}
                        {fontSize === "custom" &&
                            <div className={`settings-field-sub`}>
                                <div className={`settings-field`}>
                                    <h4 className={`settings-field-label`}>{wp.i18n.__('Custom Font Size', 'dracula-dark-mode')}</h4>
                                    <div className="settings-field-content">
                                        <RangeControl
                                            allowReset={true}
                                            resetFallbackValue={100}
                                            step={10}
                                            isShiftStepEnabled={false}
                                            renderTooltipContent={() => customFontSize + '%'}
                                            value={parseInt(customFontSize)}
                                            onChange={customFontSize => setData({ ...data, customFontSize })}
                                            min={50}
                                            max={200}
                                            className={`dracula-range-control`}
                                        />
                                    </div>
                                </div>
                            </div>
                        }

                    </div>

                    {/*  textStroke */}
                    <div className={`settings-field ${!isPro ? 'pro-feature' : ''}`}>
                        <h4 className={`settings-field-label`}>
                            {wp.i18n.__('Text Stroke', 'dracula-dark-mode')}
                            {!isPro && <span className="dracula-pro-icon"><i className={`dracula-icon icon-pro`}/></span>}
                        </h4>
                        <div className="settings-field-content">

                            <div className="settings-wrap"
                                data-tooltip-id={`text-stroke-tooltip`}
                                data-tooltip-content={!isPro ? wp.i18n.__('PRO Feature', 'dracula-dark-mode') : ''}
                                onClick={() => !isPro && showProModal()}>
                                <RangeControl
                                    value={parseFloat(textStroke)}
                                    onChange={textStroke => {
                                        if (!isPro) {
                                            showProModal();
                                            return;
                                        }
                                        setData({ ...data, textStroke: parseFloat(textStroke) })
                                    }}
                                    min={0}
                                    max={1}
                                    step={0.1}
                                    allowReset={true}
                                    resetFallbackValue={0}
                                    className={'dracula-range-control'}
                                />
                            </div>

                            <p className="description">{wp.i18n.__('Set the text stroke in the dark mode.', 'dracula-dark-mode')}</p>
                        </div>
                    </div>
                </>
            }

        </div>
    )
}