import SettingsContext from "../../contexts/SettingsContext";
import Header from "./Header";

const {FormToggle} = wp.components;

const {useContext} = React;

export default function Tools() {

    const context = useContext(SettingsContext);

    const {data, setData, updateSettings} = context;

    const {autoSave = false} = data;

    const exportSettings = () => {
        //export data to json
        const json = JSON.stringify(data);
        //create blob
        const blob = new Blob([json], {type: 'application/json'});
        //create url
        const url = URL.createObjectURL(blob);
        //create link
        const link = document.createElement('a');
        link.href = url;
        link.download = 'dracula-settings.json';
        //click link
        link.click();
        //remove url
        URL.revokeObjectURL(url);
    }

    const importSettings = () => {

        //create file input
        const input = document.createElement('input');
        input.type = 'file';
        input.accept = 'application/json';
        input.onchange = () => {
            //get file
            const file = input.files[0];
            //read file
            const reader = new FileReader();
            reader.readAsText(file);
            reader.onload = () => {
                //get data
                const json = JSON.parse(reader.result);

                //set data
                setData(json);

                updateSettings(null, json);
            }
        }

        //click file input
        input.click();
    }

    return (
        <div className="dracula-settings-content">

            <Header title={wp.i18n.__('Tools', 'dracula-dark-mode')}/>

            {/* Enable Auto Save */}
            <div className={`settings-field field-show-toggle`}>
                <h4 className={`settings-field-label`}>{wp.i18n.__('Enable Auto-save Settings', 'dracula-dark-mode')}</h4>
                <div
                    className={`settings-field-content`}>
                    <div className="settings-wrap">
                        <FormToggle
                            checked={autoSave}
                            onChange={() => {

                                setData({...data, autoSave: !autoSave});
                            }}
                        />
                    </div>

                    <p className="description">{wp.i18n.__('Enable automatic saving when you change a setting', 'dracula-dark-mode')}</p>
                </div>
            </div>

            {/*Export Settings*/}
            <div className="settings-field">
                <h4 className={`settings-field-label`}>{wp.i18n.__('Export Settings', 'dracula-dark-mode')}</h4>
                <div className="settings-field-content">
                    <button type={`button`} className={`dracula-btn btn-primary`} onClick={exportSettings}>
                        <span className="dracula-btn-icon icon-export"></span>
                        <span>{wp.i18n.__('Export', 'dracula-dark-mode')}</span>
                    </button>

                    <p className="description">{wp.i18n.__('Export current settings to a JSON file.', 'dracula-dark-mode')}</p>
                </div>
            </div>

            {/*Import Settings*/}
            <div className="settings-field">
                <h4 className={`settings-field-label`}>{wp.i18n.__('Import Settings', 'dracula-dark-mode')}</h4>
                <div className="settings-field-content">
                    <button type={`button`} className={`dracula-btn btn-info`} onClick={importSettings}>
                        <span className="dracula-btn-icon icon-import"></span>
                        <span>{wp.i18n.__('Import', 'dracula-dark-mode')}</span>
                    </button>

                    <p className="description">{wp.i18n.__('Import settings from a JSON file.', 'dracula-dark-mode')}</p>
                </div>
            </div>

            {/*  Reset Settings */}
            <div className="settings-field">
                <h4 className={`settings-field-label`}>{wp.i18n.__('Reset Settings', 'dracula-dark-mode')}</h4>
                <div className="settings-field-content">
                    <button type={`button`} className={`dracula-btn btn-danger`} onClick={() => {
                        Swal.fire({
                            title: 'Are you sure?',
                            text: "We recommend you to export your current settings before resetting them.",
                            icon: 'warning',
                            showCancelButton: true,
                            confirmButtonColor: '#3085d6',
                            cancelButtonColor: '#d33',
                            confirmButtonText: 'Reset',
                            reverseButtons: true,
                            showLoaderOnConfirm: true,
                            customClass: {
                                container: 'dracula-swal',
                            },
                            preConfirm: () => {
                                return new Promise((resolve) => {
                                    updateSettings(null, {}).then(() => {
                                        resolve();
                                    });
                                });
                            }
                        }).then((result) => {
                            if (result.value) {
                                setData({});

                                Swal.fire({
                                    title: 'Reset',
                                    text: 'Settings have been reset.',
                                    icon: 'success',
                                    toast: true,
                                    position: 'top-end',
                                    showConfirmButton: false,
                                    timer: 3000,
                                    customClass: {
                                        container: 'dracula-swal',
                                    },
                                });
                            }
                        });
                    }}>
                        <span className="dracula-btn-icon icon-reset"></span>
                        <span>{wp.i18n.__('Reset', 'dracula-dark-mode')}</span>
                    </button>

                    <p className="description">{wp.i18n.__('Reset all settings to default.', 'dracula-dark-mode')}</p>
                </div>
            </div>


        </div>
    )
}