import SettingsContext from "../../contexts/SettingsContext";
import {Tooltip} from "react-tooltip";
import Toggles from "./Toggles";
import Select from "react-select";
import Header from "./Header";
import LiveEditButton from "./LiveEditButton";
import SelectElement from "../LiveEdit/SelectElement";

import {showProModal} from "../ProModal";

const {
    __experimentalRadioGroup: RadioGroup,
    __experimentalRadio: Radio,
    FormToggle,
    RangeControl,
    CheckboxControl,
    TextControl,
    SelectControl,
    Button,
} = wp.components;

const {useContext, useState, useEffect} = React;

const typeOptions = [
    {
        label: wp.i18n.__('Select Action', 'dracula-dark-mode'),
        value: '',
    },
    {
        label: wp.i18n.__('Toggle Dark Mode', 'dracula-dark-mode'),
        value: 'switch-toggle-mode',
    },
    {
        label: wp.i18n.__('Switch to Dark Mode', 'dracula-dark-mode'),
        value: 'switch-dark-mode',
    },
    {
        label: wp.i18n.__('Switch to Light Mode', 'dracula-dark-mode'),
        value: 'switch-light-mode',
    }
];

const {isPro, isBlockTheme} = dracula;

export default function Switch() {

    const {data, setData, isLiveEdit, isEditor} = useContext(SettingsContext);

    const {
        showToggle = true,
        floatingDevices = ["desktop", "tablet", "mobile"],
        toggleStyle = 1,

        showTooltip,
        lightTooltipText = wp.i18n.__('Toggle Dark Mode', 'dracula-dark-mode'),
        darkTooltipText = wp.i18n.__('Toggle Light Mode', 'dracula-dark-mode'),

        showMenuTooltip,
        menuTooltipText = wp.i18n.__('Toggle Dark Mode', 'dracula-dark-mode'),

        attentionEffect = 'none',

        toggleSize = 'normal',
        customToggleSize = 100,
        togglePosition = 'right',
        toggleSide = 'right',
        bottomOffset = 30,
        sideOffset = 30,
        draggableToggle = false,
        absoluteDraggableToggle = false,
        delayToggle = false,
        delayToggleTime = 5,
        displayInMenu,
        toggleMenus = [],
        menuToggleStyle = 14,
        menuToggleSize = 'normal',
        customMenuToggleSize = 100,
        menuTogglePosition = 'end',
        toggleTriggers = [],
    } = data;

    const [innerTab, setInnerTab] = useState('floating');

    const handleFloatingDevicesChange = (checked, device) => {
        setData({
            ...data,
            floatingDevices: checked ? [...floatingDevices, device] : floatingDevices.filter(item => item !== device)
        });
    }

    const updateTrigger = (index, key, value) => {
        setData({
            ...data,
            toggleTriggers: [...toggleTriggers.slice(0, index), {
                ...toggleTriggers[index],
                [key]: value
            }, ...toggleTriggers.slice(index + 1)]
        });
    }

    const removeTrigger = (index) => {
        const newTriggers = [...toggleTriggers.slice(0, index), ...toggleTriggers.slice(index + 1)];

        setData({...data, toggleTriggers: newTriggers});
    }

    // Toggle attention effect
    useEffect(() => {
        clearInterval(window.draculaEffectInterval);

        const element = jQuery('.dracula-toggle-wrap.floating .dracula-toggle, .toggle-wrap.active .dracula-toggle');
        if (!element.length) return;

        function removeEffectClass() {
            const matchedClasses = (element.attr('class').match(/\bdracula-effect-\S+/g) || []);
            element.removeClass(matchedClasses.join(' '));
        }

        if (!attentionEffect || 'none' === attentionEffect) {
            removeEffectClass();
            return;
        }

        element.addClass(`dracula-effect-${attentionEffect}`);

        window.draculaEffectInterval = setInterval(() => {
            // 1. Remove the class
            removeEffectClass();

            // 2. Small delay before re-adding (optional but may help)
            setTimeout(() => {
                element.addClass(`dracula-effect-${attentionEffect}`);
            }, 50);

        }, 5000);

        return () => {
            clearInterval(window.draculaEffectInterval);
        }

    }, [attentionEffect]);

    let docUrl = 'https://softlabbd.com/docs/how-to-display-customize-floating-toggle-switch/';

    if ('menu' === innerTab) {
        docUrl = 'https://softlabbd.com/docs/how-to-use-customize-dark-mode-switch-in-website-menu/';
    } else if ('custom-trigger' === innerTab) {
        docUrl = 'https://softlabbd.com/docs/how-to-trigger-dark-mode-with-any-element';
    }

    return (
        <div className="dracula-settings-content">

            {!isPro &&
                <Tooltip
                    anchorSelect={'.dracula-pro-icon, .pro-disabled'}
                    place="right"
                    variant="warning"
                    content={wp.i18n.__('PRO Feature', 'dracula-dark-mode')}
                    className={`dracula-tooltip`}
                />
            }

            <Header title={wp.i18n.__('Toggle Settings', 'dracula-dark-mode')}/>

            <div className="inner-tabs">

                <Button
                    className={`tab-item dracula-btn ${innerTab === 'floating' ? 'active' : ''}`}
                    onClick={() => setInnerTab('floating')}
                    icon={<i className="dracula-btn-icon icon-floating-toggle"></i>}
                    text={wp.i18n.__('Floating Toggle', 'dracula-dark-mode')}
                    label={wp.i18n.__('Floating Toggle Settings', 'dracula-dark-mode')}
                />

                <Button
                    className={`tab-item dracula-btn menu-toggle ${innerTab === 'menu' ? 'active' : ''}`}
                    onClick={() => setInnerTab('menu')}
                    icon={<i className="dracula-btn-icon icon-menu-toggle"></i>}
                    text={wp.i18n.__('Menu Toggle', 'dracula-dark-mode')}
                    label={wp.i18n.__('Menu Toggle Settings', 'dracula-dark-mode')}
                />

                <Button
                    className={`tab-item dracula-btn custom-trigger ${innerTab === 'custom-trigger' ? 'active' : ''}`}
                    onClick={() => setInnerTab('custom-trigger')}
                    icon={<i className="dracula-btn-icon icon-custom-trigger"></i>}
                    text={wp.i18n.__('Custom Triggers', 'dracula-dark-mode')}
                    label={wp.i18n.__('Custom Dark Mode Triggers Settings', 'dracula-dark-mode')}
                />

                <Button
                    className={`dracula-btn btn-info doc-btn`}
                    href={docUrl}
                    icon={<i className="dashicons dashicons-info"></i>}
                    text={wp.i18n.__('Documentation', 'dracula-dark-mode')}
                    label={wp.i18n.__('View Documentation', 'dracula-dark-mode')}
                />

            </div>

            {/* Show Toggle */}
            {'floating' === innerTab &&
                <div className="settings-field field-show-toggle">
                    <h4 className={`settings-field-label`}>{wp.i18n.__('Show Floating Toggle', 'dracula-dark-mode')}</h4>
                    <div className="settings-field-content">
                        <FormToggle
                            checked={showToggle}
                            onChange={() => setData({...data, showToggle: !showToggle})}
                        />

                        <p className="description">{wp.i18n.__('Show/ hide the toggle switch button in the frontend for the users to toggle between the light and dark mode.', 'dracula-dark-mode')}</p>
                    </div>

                    {showToggle &&
                        <div className="settings-field-sub">
                            <div className="settings-field">
                                <h4 className={`settings-field-label`}>{wp.i18n.__('Display On', 'dracula-dark-mode')}</h4>
                                <div className="settings-field-content">
                                    <div className="display-devices">
                                        <CheckboxControl
                                            label={
                                                <button className={`dracula-btn`}
                                                        onClick={() => handleFloatingDevicesChange(!floatingDevices.includes('mobile'), 'mobile')}>
                                                    <span className="dracula-btn-icon icon-mobile"></span>
                                                    <span>{wp.i18n.__('Mobile', 'dracula-dark-mode')}</span>
                                                </button>}
                                            checked={floatingDevices.includes('mobile')}
                                            onChange={(checked) => handleFloatingDevicesChange(checked, 'mobile')}
                                        />
                                        <CheckboxControl
                                            label={
                                                <button className={`dracula-btn`}
                                                        onClick={() => handleFloatingDevicesChange(!floatingDevices.includes('tablet'), 'tablet')}>
                                                    <span className="dracula-btn-icon icon-tablet"></span>
                                                    <span>{wp.i18n.__('Tablet', 'dracula-dark-mode')}</span>
                                                </button>}
                                            checked={floatingDevices.includes('tablet')}
                                            onChange={(checked) => handleFloatingDevicesChange(checked, 'tablet')}
                                        />
                                        <CheckboxControl
                                            label={
                                                <button className={`dracula-btn`}
                                                        onClick={() => handleFloatingDevicesChange(!floatingDevices.includes('desktop'), 'desktop')}>
                                                    <span className="dracula-btn-icon icon-desktop"></span>
                                                    <span>{wp.i18n.__('Desktop', 'dracula-dark-mode')}</span>
                                                </button>}
                                            checked={floatingDevices.includes('desktop')}
                                            onChange={(checked) => handleFloatingDevicesChange(checked, 'desktop')}
                                        />
                                    </div>

                                    <p className="description">{wp.i18n.__('Select the devices on which the toggle switch will be displayed.', 'dracula-dark-mode')}</p>
                                </div>
                            </div>
                        </div>
                    }

                </div>
            }

            {/* Toggle Style */}
            {'floating' === innerTab && showToggle &&
                <div className={`settings-field toggle-style ${!isPro ? 'pro-feature' : ''}`}>
                    <h4 className={`settings-field-label`}>{wp.i18n.__('Floating Toggle Style', 'dracula-dark-mode')}</h4>

                    <div className="settings-field-content">
                        {!isLiveEdit && <LiveEditButton tab={`toggle`}/>}

                        <p className="description">{wp.i18n.__('Choose the style of the floating toggle switch.', 'dracula-dark-mode')}</p>

                        <Toggles
                            value={toggleStyle}
                            attentionEffect={attentionEffect}
                            onChange={(toggleStyle) => {
                                setData({...data, toggleStyle});
                            }}
                        />

                    </div>
                </div>
            }


            {/* Delay Toggle Switch */}
            {'floating' === innerTab &&
                <div className={`settings-field field-delay-switch ${!isPro ? 'pro-feature' : ''}`}>

                    <h4 className={`settings-field-label`}>
                        {wp.i18n.__('Switch Visibility Delay', 'dracula-dark-mode')}
                        {!isPro && <span className="dracula-pro-icon"><i className={`dracula-icon icon-pro`}/></span>}
                    </h4>

                    <div className="settings-field-content">
                        <FormToggle
                            data-tooltip-id={'switch-visibility-delay-tooltip'}
                            data-tooltip-content={!isPro ? wp.i18n.__('Pro Feature') : ''}
                            checked={isPro && delayToggle}
                            onChange={() => {
                                if (!isPro) {
                                    showProModal(wp.i18n.__(`Upgrade to PRO enable Delay Toggle Switch.`, 'dracula-dark-mode'));
                                    return;
                                }
                                setData({...data, delayToggle: !delayToggle})
                            }}
                        />

                        {!isPro &&
                            <Tooltip id="switch-visibility-delay-tooltip" place="right" effect="solid" variant="warning"
                                     backgroundColor="#ff9966"/>}

                        <p className="description">{wp.i18n.__('Display the toggle switch after a specified delay following page load.', 'dracula-dark-mode')}</p>
                    </div>

                    {/* Delay Toggle Time */}
                    {delayToggle &&
                        <div className={`settings-field-sub`}>
                            <div className={`settings-field`}>
                                <h4 className={`settings-field-label`}>{wp.i18n.__('Delay Toggle Time', 'dracula-dark-mode')}</h4>

                                <div className="settings-field-content">
                                    <TextControl
                                        value={delayToggleTime}
                                        onChange={delayToggleTime => setData({...data, delayToggleTime})}
                                        className={`dracula-text-control`}
                                        type="number"
                                    />

                                    <p className="description">{wp.i18n.__('Set time to show the toggle switch after page load. Default is 5 seconds.', 'dracula-dark-mode')}</p>
                                </div>
                            </div>
                        </div>
                    }

                </div>
            }

            {/* Attention Effect */}
            {'floating' === innerTab && showToggle &&
                <div className={`settings-field field-attention-effect  ${!isPro ? 'pro-feature' : ''}`}>

                    <h4 className={`settings-field-label`}>
                        {wp.i18n.__('Attention Effect', 'dracula-dark-mode')}
                        {!isPro && <span className="dracula-pro-icon"><i className={`dracula-icon icon-pro`}/></span>}
                    </h4>

                    <div className="settings-field-content">
                        <SelectControl
                            data-tooltip-id={'attention-effect-tooltip'}
                            data-tooltip-content={!isPro ? wp.i18n.__('Pro Feature') : ''}
                            value={attentionEffect}
                            options={[
                                {label: wp.i18n.__('None', 'dracula-dark-mode'), value: 'none'},
                                {label: wp.i18n.__('Pulse', 'dracula-dark-mode'), value: 'pulse'},
                                {label: wp.i18n.__('Shake', 'dracula-dark-mode'), value: 'shake'},
                                {label: wp.i18n.__('Bounce', 'dracula-dark-mode'), value: 'bounce'},
                                {label: wp.i18n.__('Glow', 'dracula-dark-mode'), value: 'glow'},
                                {label: wp.i18n.__('Shadow', 'dracula-dark-mode'), value: 'shadow'},
                                {label: wp.i18n.__('Ripple', 'dracula-dark-mode'), value: 'ripple'},
                                {label: wp.i18n.__('Rotate', 'dracula-dark-mode'), value: 'rotate'},
                                {label: wp.i18n.__('Flip', 'dracula-dark-mode'), value: 'flip'},
                                {label: wp.i18n.__('Swing', 'dracula-dark-mode'), value: 'swing'},
                                {label: wp.i18n.__('Tilt', 'dracula-dark-mode'), value: 'tilt'},
                                {label: wp.i18n.__('Wobble', 'dracula-dark-mode'), value: 'wobble'},
                                {label: wp.i18n.__('Jiggle', 'dracula-dark-mode'), value: 'jiggle'},
                                {label: wp.i18n.__('Fade', 'dracula-dark-mode'), value: 'fade'},
                                {label: wp.i18n.__('Slide', 'dracula-dark-mode'), value: 'slide'},
                                {label: wp.i18n.__('Color Transition', 'dracula-dark-mode'), value: 'color'},
                            ]}
                            onChange={attentionEffect => {
                                if (!isPro) {
                                    showProModal(wp.i18n.__(`Tooltip is a PRO feature. Upgrade to PRO to enable tooltip for the toggle switch.`, 'dracula-dark-mode'));
                                    return;
                                }

                                setData({...data, attentionEffect})
                            }}
                            className={`dracula-select`}
                        />

                        {!isPro && <Tooltip id="attention-effect-tooltip" place="right" effect="solid" variant="warning"
                                            backgroundColor="#ff9966"/>}


                        <p className="description">{wp.i18n.__('Choose the attention effect for the toggle switch.', 'dracula-dark-mode')}</p>
                    </div>
                </div>
            }

            {/* Show Tooltip */}
            {'floating' === innerTab &&
                <div className={`settings-field field-show-tooltip ${!isPro ? 'pro-feature' : ''}`}>
                    <h4 className={`settings-field-label`}>
                        {wp.i18n.__('Show Tooltip', 'dracula-dark-mode')}
                        {!isPro && <span className="dracula-pro-icon"><i className={`dracula-icon icon-pro`}/></span>}
                    </h4>

                    <div className="settings-field-content">
                        <FormToggle
                            data-tooltip-id={'show-tooltip-tooltip'}
                            data-tooltip-content={!isPro ? wp.i18n.__('Pro Feature') : ''}
                            checked={showTooltip}
                            onChange={() => {
                                if (!isPro) {
                                    showProModal(wp.i18n.__(`Tooltip is a PRO feature. Upgrade to PRO to enable tooltip for the toggle switch.`, 'dracula-dark-mode'));
                                    return;
                                }

                                setData({...data, showTooltip: !showTooltip});
                            }}
                        />

                        {!isPro && <Tooltip id="show-tooltip-tooltip" variant="warning" place="right" effect="solid"
                                            backgroundColor="#ff9966"/>}

                        <p className="description">{wp.i18n.__('Show/ hide the toggle button tooltip text.', 'dracula-dark-mode')}</p>
                    </div>

                    {showTooltip &&
                        <>
                            <div className="settings-field-sub">
                                {/* Tooltip Text */}
                                <div className="settings-field">
                                    <h4 className={`settings-field-label`}>{wp.i18n.__('Light Tooltip Text', 'dracula-dark-mode')}</h4>
                                    <div className="settings-field-content">
                                        <TextControl
                                            value={lightTooltipText}
                                            onChange={lightTooltipText => setData({...data, lightTooltipText})}
                                            className={`dracula-text-control`}
                                        />

                                        <p className="description">{wp.i18n.__('Enter the toggle switch button tooltip text for light mode.', 'dracula-dark-mode')}</p>
                                    </div>
                                </div>
                            </div>
                            <div className="settings-field-sub">
                                {/* Tooltip Text */}
                                <div className="settings-field">
                                    <h4 className={`settings-field-label`}>{wp.i18n.__('Dark Tooltip Text', 'dracula-dark-mode')}</h4>
                                    <div className="settings-field-content">
                                        <TextControl
                                            value={darkTooltipText}
                                            onChange={darkTooltipText => setData({...data, darkTooltipText})}
                                            className={`dracula-text-control`}
                                        />

                                        <p className="description">{wp.i18n.__('Enter the toggle switch button tooltip text for dark mode.', 'dracula-dark-mode')}</p>
                                    </div>
                                </div>
                            </div>
                        </>
                    }

                </div>
            }

            {/* Draggable Toggle */}
            {'floating' === innerTab && showToggle &&
                <div className={`settings-field field-draggable-toggle ${!isPro ? 'pro-feature' : ''}`}>
                    <h4 className={`settings-field-label`}>
                        {wp.i18n.__('Draggable Floating Toggle', 'dracula-dark-mode')}
                        {!isPro && <span className="dracula-pro-icon"><i className={`dracula-icon icon-pro`}/></span>}
                    </h4>
                    <div className="settings-field-content">
                        <FormToggle
                            data-tooltip-id={'draggable-toggle-tooltip'}
                            data-tooltip-content={!isPro ? wp.i18n.__('Pro Feature') : ''}
                            checked={isPro && draggableToggle}
                            onChange={() => {
                                if (!isPro) {
                                    showProModal(wp.i18n.__(`Upgrade to PRO to enable users to drag and adjust the floating toggle position.`, 'dracula-dark-mode'));
                                    return;
                                }
                                setData({...data, draggableToggle: !draggableToggle})
                            }}
                        />

                        {!isPro && <Tooltip id="draggable-toggle-tooltip" variant="warning" place="right" effect="solid"
                                            backgroundColor="#ff9966"/>}

                        <p className="description">{wp.i18n.__('Allow users to drag the floating toggle button to any position on the screen.', 'dracula-dark-mode')}</p>
                    </div>
                </div>
            }

            {/* Toggle Position */}
            {'floating' === innerTab && showToggle &&
                <div className={`settings-field`}>
                    <h4 className={`settings-field-label`}>{wp.i18n.__('Floating Toggle Position', 'dracula-dark-mode')}</h4>
                    <div className="settings-field-content">

                        <RadioGroup
                            onChange={togglePosition => {
                                if (!isPro && 'custom' === togglePosition) {
                                    showProModal(wp.i18n.__(`Upgrade to PRO to set custom floating toggle position.`, 'dracula-dark-mode'));
                                    return;
                                }

                                setData({...data, togglePosition})
                            }}
                            checked={togglePosition}
                            className={'dracula-radio-group'}
                        >
                            <Radio value="left">
                                <span className="dracula-btn-icon icon-position-left"></span>
                                <span>{wp.i18n.__('Left', 'dracula-dark-mode')}</span>
                            </Radio>
                            <Radio value="right">
                                <span className="dracula-btn-icon icon-position-right"></span>
                                <span>{wp.i18n.__('Right', 'dracula-dark-mode')}</span>
                            </Radio>
                            <Radio
                                data-tooltip-content={!isPro ? wp.i18n.__('Pro Feature') : ''}
                                data-tooltip-id={'custom-position-pro'}
                                value="custom">
                                <span className="dracula-btn-icon icon-custom"></span>
                                <span>{wp.i18n.__('Custom', 'dracula-dark-mode')}</span>
                            </Radio>

                            {!isPro &&
                                <Tooltip id={'custom-position-pro'} variant="warning" effect="solid" place="right"
                                         backgroundColor="#ff9966"/>}

                        </RadioGroup>

                        {!isLiveEdit && <LiveEditButton tab="toggle"/>}

                        <p className="description">{wp.i18n.__('Choose the position of the floating toggle switch.', 'dracula-dark-mode')}</p>
                    </div>

                    {'custom' === togglePosition &&
                        <div className="settings-field-sub custom-toggle-position">
                            <div className="settings-field field-toggle-side">
                                <h4 className={`settings-field-label`}>{wp.i18n.__('Toggle Side', 'dracula-dark-mode')}</h4>

                                <div className="settings-field-content">
                                    <RadioGroup label="Side Selection"
                                                onChange={toggleSide => setData({...data, toggleSide})}
                                                checked={toggleSide}
                                                className={'dracula-radio-group'}>
                                        <Radio value="left">
                                            <span className="dracula-btn-icon icon-position-left"></span>
                                            <span>{wp.i18n.__('Left', 'dracula-dark-mode')}</span>
                                        </Radio>
                                        <Radio value="right">
                                            <span className="dracula-btn-icon icon-position-right"></span>
                                            <span>{wp.i18n.__('Right', 'dracula-dark-mode')}</span>
                                        </Radio>
                                    </RadioGroup>

                                    <p className="description">{wp.i18n.__('Choose the side of the page where the toggle switch will be shown.', 'dracula-dark-mode')}</p>
                                </div>

                                {/* Side Offset */}
                                <h4 className={`settings-field-label`}>{wp.i18n.__('Side Offset', 'dracula-dark-mode')}</h4>
                                <div className="settings-field-content">
                                    <RangeControl
                                        allowReset={true}
                                        resetFallbackValue={30}
                                        step={5}
                                        isShiftStepEnabled
                                        renderTooltipContent={() => sideOffset}
                                        value={parseInt(sideOffset)}
                                        onChange={sideOffset => setData({...data, sideOffset})}
                                        min={0}
                                        max={1000}
                                        className={`dracula-range-control`}
                                    />

                                    <p className="description"> {wp.i18n.__('Set the side offset of the toggle switch from the left or right side of the screen.', 'dracula-dark-mode')}</p>
                                </div>

                                {/* Bottom Offset */}
                                <h4 className={`settings-field-label`}>{wp.i18n.__('Bottom Offset', 'dracula-dark-mode')}</h4>
                                <div className="settings-field-content">
                                    <RangeControl
                                        allowReset={true}
                                        resetFallbackValue={30}
                                        step={5}
                                        isShiftStepEnabled
                                        renderTooltipContent={() => bottomOffset}
                                        value={parseInt(bottomOffset)}
                                        onChange={bottomOffset => setData({...data, bottomOffset})}
                                        min={0}
                                        max={1000}
                                        className={`dracula-range-control`}
                                    />

                                    <p className="description">{wp.i18n.__('Set the bottom offset of the toggle switch from the bottom of the screen.', 'dracula-dark-mode')}</p>
                                </div>

                            </div>

                        </div>
                    }

                </div>
            }

            {/* Absolute Toggle Position */}
            {'floating' === innerTab && showToggle &&
                <div className={`settings-field field-draggable-toggle ${!isPro ? 'pro-feature' : ''}`}>

                    <h4 className={`settings-field-label`}>
                        {wp.i18n.__('Absolute Toggle Position', 'dracula-dark-mode')}
                        {!isPro && <span className="dracula-pro-icon"><i className={`dracula-icon icon-pro`}/></span>}
                    </h4>

                    <div className="settings-field-content">
                        <FormToggle
                            data-tooltip-id={'absolute-toggle-pro'}
                            data-tooltip-content={!isPro ? wp.i18n.__('Pro Feature') : ''}
                            checked={isPro && absoluteDraggableToggle}
                            onChange={() => {
                                if (!isPro) {
                                    showProModal(wp.i18n.__(`Upgrade to PRO enable Absolute Toggle position.`, 'dracula-dark-mode'));
                                    return;
                                }
                                setData({...data, absoluteDraggableToggle: !absoluteDraggableToggle})
                            }}
                        />

                        {!isPro && <Tooltip id={'absolute-toggle-pro'} variant="warning" place="right" effect="solid"
                                            backgroundColor="#ff9966"/>}

                        <p className="description">{wp.i18n.__('Enable to make the floating toggle scroll from it\'s position with page scrolling.', 'dracula-dark-mode')}</p>
                    </div>
                </div>
            }

            {'menu' === innerTab &&
                <>

                    {isBlockTheme ?

                        <>
                            <div className="dracula-placeholder block-theme-notice">
                                <div className="dracula-placeholder__label">
                                    <span className="dashicons dashicons-info-outline"></span>
                                    {wp.i18n.__('Navigation Menus Not Supported', 'dracula-dark-mode')}
                                </div>

                                <div className="dracula-placeholder__instructions">
                                    <p>
                                        {wp.i18n.__(
                                            'Your active theme is a Block Theme. Block themes do not support classic navigation menus. To display the Dark Mode toggle in a menu, please use the “Dracula Dark Mode Switch” block inside your site’s Navigation block.',
                                            'dracula-dark-mode'
                                        )}
                                    </p>
                                </div>

                                <div className="dracula-placeholder__actions">
                                    <a
                                        href="https://softlabbd.com/docs/how-to-use-customize-dark-mode-switch-in-website-menu/"
                                        target="_blank"
                                        rel="noopener noreferrer"
                                        className="button button-primary"
                                    >
                                        {wp.i18n.__('Learn More', 'dracula-dark-mode')}
                                    </a>
                                </div>
                            </div>

                        </>
                        :
                        <>
                            {/* Display In Menu */}
                            <div className={`settings-field field-display-menu-toggle`}>
                                <h4 className={`settings-field-label`}>{wp.i18n.__('Display Toggle In Menu', 'dracula-dark-mode')}</h4>
                                <div className="settings-field-content">
                                    <FormToggle
                                        checked={displayInMenu}
                                        onChange={() => {
                                            setData({...data, displayInMenu: !displayInMenu})
                                        }}
                                    />

                                    <p className="description">{wp.i18n.__('Turn on to display the dark mode toggle button in a menu.', 'dracula-dark-mode')}</p>
                                </div>
                            </div>

                            {(displayInMenu || !isPro) &&
                                <div className={`settings-field field-menu-toggle-placement`}>
                                    <h4 className={`settings-field-label`}>{wp.i18n.__('Menu Toggle Placement', 'dracula-dark-mode')}</h4>

                                    <div className="settings-field-content">
                                        <h3 className="settings-field-label">
                                            {wp.i18n.__(
                                                'Add the Dracula Dark Mode Switch to your WordPress menus',
                                                'dracula-dark-mode'
                                            )}
                                        </h3>

                                        <p>
                                            {wp.i18n.__(
                                                'Drag & drop the Dark Mode Switcher into any menu from Appearance → Menus in your WordPress dashboard.',
                                                'dracula-dark-mode'
                                            )}
                                        </p>

                                        <div className="toggle-placement-actions">
                                            <a
                                                href={`${dracula.adminUrl}nav-menus.php#dracula-dark-mode`}
                                                className="button button-primary dracula-btn"
                                            >
                                                {wp.i18n.__('Open Menu Settings', 'dracula-dark-mode')}
                                                <span className="dashicons dashicons-arrow-right-alt2"/>
                                            </a>

                                            <a
                                                href="https://softlabbd.com/docs/how-to-use-customize-dark-mode-switch-in-website-menu/"
                                                target="_blank"
                                                rel="noopener noreferrer"
                                                className="button button-primary learn-more-btn"
                                            >
                                                <i className={"dashicons dashicons-external"}/>
                                                {wp.i18n.__('Learn More', 'dracula-dark-mode')}
                                            </a>
                                        </div>

                                    </div>
                                </div>
                            }
                        </>
                    }

                </>
            }

            {/*  Toggle Size */}
            {(('floating' === innerTab && showToggle) || ('menu' === innerTab && displayInMenu && !isBlockTheme)) &&
                <div className={`settings-field`}>
                    <h4 className={`settings-field-label`}>{'floating' === innerTab ? wp.i18n.__(`Floating Toggle Size`, 'dracula-dark-mode') : wp.i18n.__(`Menu Toggle Size`, 'dracula-dark-mode')}</h4>
                    <div className="settings-field-content">
                        <RadioGroup
                            onChange={size => {

                                if (!isPro && 'custom' === size) {
                                    showProModal(wp.i18n.__(`Upgrade to PRO to set custom toggle size.`, 'dracula-dark-mode'));
                                    return;
                                }

                                if ('floating' === innerTab) {
                                    setData({...data, toggleSize: size});
                                } else {
                                    setData({...data, menuToggleSize: size});
                                }
                            }}
                            checked={'floating' === innerTab ? toggleSize : menuToggleSize}
                            className={'dracula-radio-group'}>
                            <Radio value="small">
                                <span className="dracula-btn-icon icon-size-small"></span>
                                <span>{wp.i18n.__('Small', 'dracula-dark-mode')}</span>
                            </Radio>
                            <Radio value="normal">
                                <span className="dracula-btn-icon icon-size-normal"></span>
                                <span>{wp.i18n.__('Medium', 'dracula-dark-mode')}</span>
                            </Radio>
                            <Radio value="large">
                                <span className="dracula-btn-icon icon-size-large"></span>
                                <span>{wp.i18n.__('Large', 'dracula-dark-mode')}</span>
                            </Radio>
                            <Radio
                                data-tooltip-content={!isPro ? wp.i18n.__('Pro Feature') : ''}
                                data-tooltip-id={'custom-size-pro'}
                                value="custom">
                                <span className="dracula-btn-icon icon-custom"></span>
                                <span>{wp.i18n.__('Custom', 'dracula-dark-mode')}</span>
                            </Radio>

                            {!isPro && <Tooltip id={'custom-size-pro'} variant="warning" effect="solid" place="right"
                                                backgroundColor="#ff9966"/>}

                        </RadioGroup>

                        {!isLiveEdit && <LiveEditButton tab={`toggle`}/>}

                        <p className="description">{wp.i18n.__('Choose the size of the floating toggle switch.', 'dracula-dark-mode')}</p>
                    </div>

                    {/* Custom Size */}
                    {(('floating' === innerTab && 'custom' === toggleSize) || ('menu' === innerTab && 'custom' === menuToggleSize)) &&
                        <div className="settings-field-sub">
                            <div className="settings-field">
                                <h4 className={`settings-field-label`}>{wp.i18n.__('Custom Toggle Size', 'dracula-dark-mode')}</h4>
                                <div className="settings-field-content">
                                    <RangeControl
                                        allowReset={true}
                                        resetFallbackValue={100}
                                        step={5}
                                        isShiftStepEnabled
                                        value={parseInt('floating' === innerTab ? customToggleSize : customMenuToggleSize)}
                                        onChange={size => {
                                            if ('floating' === innerTab) {
                                                setData({...data, customToggleSize: size});
                                            } else {
                                                setData({...data, customMenuToggleSize: size});
                                            }
                                        }}
                                        min={50}
                                        max={300}
                                        className={`dracula-range-control`}
                                    />

                                    <p className="description">{wp.i18n.__('Set the custom size (px) of the dark mode toggle button.', 'dracula-dark-mode')}</p>
                                </div>
                            </div>
                        </div>
                    }

                </div>
            }

            {/* custom trigger */}
            {'custom-trigger' === innerTab &&
                <div className={`settings-field`}>

                    <h4 className={`settings-field-label`}>
                        {wp.i18n.__('Custom Triggers', 'dracula-dark-mode')}
                        {!isPro && <span className="dracula-pro-icon"><i className={`dracula-icon icon-pro`}/></span>}
                    </h4>

                    <div className="settings-field-content">

                        {toggleTriggers.map((triger, index) => {
                            const {selector, type} = triger;

                            return (
                                <>
                                    <div
                                        data-tooltip-content={!isPro ? wp.i18n.__('PRO Feature', 'dracula-dark-mode') : ''}
                                        data-tooltip-id={`dracula-trigger-group`}
                                        className="dracula-trigger-group"
                                        key={index}
                                    >
                                        <div className="group-item">

                                            {index === 0 &&
                                                <div className={`group-item-header`}>
                                                    <span
                                                        className={`header-title`}>{wp.i18n.__('Element Selector', 'dracula-dark-mode')}</span>
                                                </div>
                                            }

                                            <div className="input-group">
                                                <input
                                                    type="text"
                                                    className={`${!isPro ? 'pro-feature' : ''}`}
                                                    placeholder="Enter CSS Selector (e.g. .my-class, #my-id)"
                                                    value={selector}
                                                    onChange={e => {
                                                        if (!isPro) {
                                                            showProModal(wp.i18n.__('Upgrade to PRO to use the Custom Trigger.', 'dracula-dark-mode'));
                                                            return;
                                                        }

                                                        updateTrigger(index, 'selector', e.target.value);
                                                    }}/>

                                                {!!isLiveEdit &&
                                                    <SelectElement
                                                        index={index}
                                                        data={data}
                                                        setData={setData}
                                                    />
                                                }

                                            </div>

                                        </div>

                                        <div className="group-item">

                                            {index === 0 &&
                                                <div className={`group-item-header`}>
                                                    <span
                                                        className={`header-title`}>{wp.i18n.__('Action', 'dracula-dark-mode')}</span>
                                                </div>
                                            }

                                            <div className="input-group">
                                                <SelectControl
                                                    options={typeOptions}
                                                    value={type}
                                                    onChange={value => {

                                                        if (!isPro) {
                                                            showProModal(wp.i18n.__('Upgrade to PRO to use the Custom Trigger.', 'dracula-dark-mode'));
                                                            return;
                                                        }

                                                        updateTrigger(index, 'type', value);
                                                    }}
                                                />

                                                <button
                                                    type={`button`}
                                                    className={`dracula-btn btn-danger ${!isPro ? 'pro-feature' : ''}`}
                                                    onClick={() => {
                                                        if (!isPro) {
                                                            showProModal(wp.i18n.__('Upgrade to PRO to use the Custom Trigger.', 'dracula-dark-mode'));
                                                            return;
                                                        }

                                                        removeTrigger(index)
                                                    }}
                                                >
                                                    <i className="dashicons dashicons-trash"></i>
                                                </button>
                                            </div>

                                        </div>


                                    </div>

                                    {!isPro && <Tooltip id={'dracula-trigger-group'} effect="solid" place="right"
                                                        backgroundColor="#ff9966"/>}
                                </>
                            )
                        })}

                        <button
                            data-tooltip-content={!isPro ? wp.i18n.__('PRO Feature', 'dracula-dark-mode') : ''}
                            data-tooltip-id="dracual-add-new-trigger-btn"
                            type={`button`}
                            className={`dracula-btn btn-primary ${!isPro ? 'pro-feature' : ''}`}
                            onClick={() => {
                                if (!isPro) {
                                    showProModal(wp.i18n.__('Upgrade to PRO to use the Custom Trigger.', 'dracula-dark-mode'));
                                    return;
                                }

                                setData({
                                    ...data, toggleTriggers: [...toggleTriggers, {
                                        selector: '',
                                    }]
                                })
                            }}>
                            <i className={`dashicons dashicons-plus-alt2`}></i>
                            <span>{wp.i18n.__('Add New Trigger', 'dracula-dark-mode')}</span>
                        </button>

                        {!isPro &&
                            <Tooltip id={'dracual-add-new-trigger-btn'} effect="solid" place="right" variant="warning"
                                     backgroundColor="#ff9966"/>}

                        <p className="description">
                            {wp.i18n.__('You can toggle the dark mode using any custom elements or buttons of your choice.', 'dracula-dark-mode')}
                            <a href="https://softlabbd.com/docs/how-to-use-custom-elements-and-buttons-to-switch-dark-mode/"
                               target="_blank"> {wp.i18n.__('Learn More', 'dracula-dark-mode')}</a>
                        </p>

                    </div>
                </div>
            }
        </div>
    )
}