import {Tooltip} from "react-tooltip";
import Select from "react-select";
import SettingsContext from "../../contexts/SettingsContext";
import {showProModal} from "../ProModal";
import fonts from "../../includes/google-fonts.json";
import ColorPickerWrapper from "../../includes/ColorPickerWrapper";

const {
    __experimentalRadioGroup: RadioGroup,
    __experimentalRadio: Radio,
    FormToggle,
    CheckboxControl,
    GradientPicker,
    SelectControl,
    RangeControl
} = wp.components;

const { useContext, useState } = React;

export default function ReadingModeLayoutSettings(){
    const {isPro} = dracula;

    const context = useContext(SettingsContext);

    const { data, setData } = context;

    const {
        linkNewTab = false,
        toolbar = true,
        tools = ['appearance', 'typography', 'print', 'scroll', 'fullScreen'],
        showSourceURL = true,
        showReadingTime = true,
        showDate = true,
        showAuthor = true,
        contentWidth = 'medium',
        theme = 'system',
        fontFamily = 'System',
        fontSize = "1",
        enableReadingProgress = true,
        progressPosition = 'top',
        progressbarStyle = 'solid',
        progressbarHeight = 7,
        progressbarColor = '#7C7EE5',
        progressbarColorGradient = 'linear-gradient(90deg, #004AFF 80%, rgba(96, 239, 255, 0) 113.89%)',
    } = data;

    // layout
    const toolsOptions = [
        {label: 'Appearance', value: 'appearance', icon: 'dashicons dashicons-admin-appearance'},
        {label: 'Print', value: 'print', icon: 'dashicons dashicons-printer'},
        {label: 'Table of Contents', value: 'toc', icon: 'dashicons dashicons-list-view'},
        {label: 'Links', value: 'links', icon: 'dashicons dashicons-admin-links'},
        {label: 'Auto Scroll', value: 'scroll', icon: 'dashicons dashicons-admin-settings'},
        {label: 'Full-screen', value: 'fullScreen', icon: 'dashicons dashicons-editor-expand'},
    ];

    const systemFonts = fonts.filter(font => font.category);
    const googleFonts = fonts.filter(font => !font.category);

    const fontOptions = [
        {
            label: wp.i18n.__('System Fonts','dracula-dark-mode'),
            options: [{value: 'system', label: 'System'}, ...systemFonts],
        },
        {
            label: wp.i18n.__('Google Fonts','dracula-dark-mode'),
            options: googleFonts,
        }
    ];
    const [loaded, setLoaded] = useState([]);

    return(
        <>
            <h2>{wp.i18n.__('Components', 'dracula-dark-mode')}</h2>
            <div className={`settings-field ${!dracula.isPro ? 'pro-feature' : ''}`}>
                <h4 className="settings-field-label">
                    {wp.i18n.__('Show Toolbar', 'dracula-dark-mode')}
                    {!isPro && <span className="dracula-pro-icon"><i className={`dracula-icon icon-pro`}/></span>}    
                </h4>
                <div className={`settings-field-content`}>
                    <FormToggle
                        data-tooltip-content={wp.i18n.__('PRO Feature', 'dracula-dark-mode')}
                        data-tooltip-id={`show-toolbar`}
                        checked={toolbar}
                        onChange={() => {
                            if (!dracula.isPro) {
                                showProModal(
                                    wp.i18n.__('Upgrade to PRO to use custom button icon.', 'dracula-dark-mode'),
                                    {id: '-eL20lz83Ec', title: wp.i18n.__('Distraction-free Reading Mode', 'dracula-dark-mode')}
                                );
                                return;
                            }

                            setData({...data, toolbar: !toolbar})
                        }}
                    />

                    {!dracula.isPro &&
                        <Tooltip
                            id="show-toolbar" 
                            place="right"
                            variant="warning"
                            backgroundColor="#ff9966"
                            effect="solid"/>
                    }                                            
                    <p className={`description`}>{wp.i18n.__('Show/ hide the toolbar tools for the users in the reading mode.', 'dracula-dark-mode')}</p>
                </div>
            </div>

            {/* Toolbar Tools */}
            { toolbar &&
                <div className={`settings-field`}>
                    <h4 className="settings-field-label">
                        {wp.i18n.__('Toolbar Tools', 'dracula-dark-mode')}
                        {!isPro && <span className="dracula-pro-icon"><i className={`dracula-icon icon-pro`}/></span>}    
                    </h4>
                    <div className={`settings-field-content`}>
                        <div className="checkbox-wrap tool-options">
                            {
                                toolsOptions.map(tool => (
                                    <>
                                        <CheckboxControl
                                            data-tooltip-content={wp.i18n.__('PRO Feature', 'dracula-dark-mode')}
                                            data-tooltip-id={`tool-${tool.value}`}
                                            label={<div
                                                data-tooltip-content={wp.i18n.__('PRO Feature', 'dracula-dark-mode')}
                                                data-tooltip-id={`tool-${tool.value}`}
                                                className={`tool-option ${!dracula.isPro && ['speech', 'translate'].includes(tool.value) ? 'disabled' : ''}`}>
                                                <i className={tool.icon}></i> {tool.label} </div>}
                                            checked={tools.includes(tool.value)}
                                            onChange={() => {

                                                if (!dracula.isPro) {
                                                    showProModal(
                                                        wp.i18n.__('Upgrade to PRO to enable ', 'dracula-dark-mode') + tool.label,
                                                        {id: '-eL20lz83Ec', title: wp.i18n.__('Distraction-free Reading Mode', 'dracula-dark-mode')}
                                                    );
                                                    return;
                                                }

                                                if (tools.includes(tool.value)) {
                                                    setData({
                                                        ...data,
                                                        tools: tools.filter(item => item !== tool.value)
                                                    });
                                                } else {
                                                    setData({...data, tools: [...tools, tool.value]});
                                                }
                                            }}
                                            className={`${!dracula.isPro ? 'disabled' : ''}`}
                                        />

                                        {!dracula.isPro &&
                                            <Tooltip
                                                id={`tool-${tool.value}`}
                                                place="right"
                                                variant="warning"
                                                effect="solid"
                                                backgroundColor="#ff9966"
                                            />
                                        }
                                    </>
                                ))
                            }
                        </div>

                        <p className={`description`}>{wp.i18n.__('Select the tools that will be displayed in the reading mode.', 'dracula-dark-mode')}</p>
                    </div>
                </div>
            }

            {/* Show Source URL */}
            <div className={`settings-field ${!dracula.isPro ? 'pro-feature' : ''}`}>
                <h4 className="settings-field-label">
                    {wp.i18n.__('Source URL', 'dracula-dark-mode')}
                    {!isPro && <span className="dracula-pro-icon"><i className={`dracula-icon icon-pro`}/></span>}    
                </h4>

                <div className={`settings-field-content`}>
                    <FormToggle
                        data-tooltip-content={wp.i18n.__('PRO Feature', 'dracula-dark-mode')}
                        data-tooltip-id={`source-url`}
                        checked={showSourceURL}
                        onChange={() => {
                            if (!dracula.isPro) {
                                showProModal(
                                    wp.i18n.__('Upgrade to PRO to use Source URL.', 'dracula-dark-mode'),
                                    {id: '-eL20lz83Ec', title: wp.i18n.__('Distraction-free Reading Mode', 'dracula-dark-mode')}
                                );
                                return;
                            }

                            setData({...data, showSourceURL: !showSourceURL})
                        }}
                    />

                    {!dracula.isPro &&
                        <Tooltip
                            id="source-url" 
                            place="right" 
                            effect="solid"
                            variant="warning"
                            backgroundColor="#ff9966"
                        />
                    }

                    <p className={`description`}>{wp.i18n.__('Show/ hide post/page/article source URL in the reading mode.', 'dracula-dark-mode')}</p>
                </div>
            </div>

            {/* Show Reading Time */}
            <div className={`settings-field ${!dracula.isPro ? 'pro-feature' : ''}`}>
                <h4 className="settings-field-label">
                    {wp.i18n.__('Reading Time', 'dracula-dark-mode')}
                    {!isPro && <span className="dracula-pro-icon"><i className={`dracula-icon icon-pro`}/></span>}    
                </h4>

                <div className={`settings-field-content`}>
                    <FormToggle
                        data-tooltip-content={wp.i18n.__('PRO Feature', 'dracula-dark-mode')}
                        data-tooltip-id={`reading-time`}
                        checked={showReadingTime}
                        onChange={() => {
                            if (!dracula.isPro) {
                                showProModal(
                                    wp.i18n.__('Upgrade to PRO to use reading time.', 'dracula-dark-mode'),
                                    {id: '-eL20lz83Ec', title: wp.i18n.__('Distraction-free Reading Mode', 'dracula-dark-mode')}
                                );
                                return;
                            }

                            setData({...data, showReadingTime: !showReadingTime})
                        }}
                    />

                    {!dracula.isPro &&
                        <Tooltip
                            id="reading-time" 
                            place="right" 
                            effect="solid"
                            variant="warning"
                            backgroundColor="#ff9966"
                        />
                    }

                    <p className={`description`}>{wp.i18n.__('Show/ hide the estimated reading time in the reading mode.', 'dracula-dark-mode')}</p>
                </div>
            </div>

            {/* Show Date */}
            <div className={`settings-field ${!dracula.isPro ? 'pro-feature' : ''}`}>
                <h4 className="settings-field-label">
                    {wp.i18n.__('Post Date', 'dracula-dark-mode')}
                    {!isPro && <span className="dracula-pro-icon"><i className={`dracula-icon icon-pro`}/></span>}    
                </h4>

                <div className={`settings-field-content`}>
                    <FormToggle
                        data-tooltip-content={wp.i18n.__('PRO Feature', 'dracula-dark-mode')}
                        data-tooltip-id={`show-date`}
                        checked={showDate}
                        onChange={() => {
                            if (!dracula.isPro) {
                                showProModal(
                                    wp.i18n.__('Upgrade to PRO to use show date.', 'dracula-dark-mode'),
                                    {id: '-eL20lz83Ec', title: wp.i18n.__('Distraction-free Reading Mode', 'dracula-dark-mode')}
                                );
                                return;
                            }

                            setData({...data, showDate: !showDate})
                        }}
                    />

                    {!dracula.isPro &&
                        <Tooltip
                            id="show-date" 
                            place="right" 
                            effect="solid"
                            variant="warning"
                            backgroundColor="#ff9966"
                        />
                    }

                    <p className={`description`}>{wp.i18n.__('Show/ hide the post publish date in the reading mode.', 'dracula-dark-mode')}</p>
                </div>
            </div>

            {/* Show Author */}
            <div className={`settings-field ${!dracula.isPro ? 'pro-feature' : ''}`}>
                <h4 className="settings-field-label">
                    {wp.i18n.__('Author Name', 'dracula-dark-mode')}
                    {!isPro && <span className="dracula-pro-icon"><i className={`dracula-icon icon-pro`}/></span>}    
                </h4>

                <div className={`settings-field-content`}>
                    <FormToggle
                        data-tooltip-content={wp.i18n.__('PRO Feature', 'dracula-dark-mode')}
                        data-tooltip-id={`author-name`}
                        checked={showAuthor}
                        onChange={() => {
                            if (!dracula.isPro) {
                                showProModal(
                                    wp.i18n.__('Upgrade to PRO to use Author Name.', 'dracula-dark-mode'),
                                    {id: '-eL20lz83Ec', title: wp.i18n.__('Distraction-free Reading Mode', 'dracula-dark-mode')}
                                );
                                return;
                            }

                            setData({...data, showAuthor: !showAuthor})
                        }}
                    />

                    {!dracula.isPro &&
                        <Tooltip
                            id="author-name"
                            place="right" 
                            variant="warning"
                            effect="solid"
                            backgroundColor="#ff9966"
                        />
                    }

                    <p className={`description`}>{wp.i18n.__('Show/ hide the author name in the reading mode.', 'dracula-dark-mode')}</p>
                </div>
            </div>

            {/* Open Link in New Tab */}
            <div className={`settings-field ${!dracula.isPro ? 'pro-feature' : ''}`}>
                <h4 className="settings-field-label">
                    {wp.i18n.__('Open Link in New Tab', 'dracula-dark-mode')}
                    {!isPro && <span className="dracula-pro-icon"><i className={`dracula-icon icon-pro`}/></span>}    
                </h4>

                <div className={`settings-field-content`}>
                    <FormToggle
                        data-tooltip-content={wp.i18n.__('PRO Feature', 'dracula-dark-mode')}
                        data-tooltip-id={`open-link`}
                        checked={linkNewTab}
                        onChange={() => {
                            if (!dracula.isPro) {
                                showProModal(
                                    wp.i18n.__('Upgrade to PRO to use Open Link in New Tab.', 'dracula-dark-mode'),
                                    {id: '-eL20lz83Ec', title: wp.i18n.__('Distraction-free Reading Mode', 'dracula-dark-mode')}
                                );
                                return;
                            }

                            setData({...data, linkNewTab: !linkNewTab})
                        }}
                    />

                    {!dracula.isPro &&
                        <Tooltip
                            id="open-link" 
                            place="right" 
                            effect="solid"
                            variant="warning"
                            backgroundColor="#ff9966"
                        />
                    }

                    <p className={`description`}>{wp.i18n.__('If enabled, on Reading Mode all the links in the content will be opened in a new tab.', 'dracula-dark-mode')}</p>
                </div>
            </div>

            <h2>{wp.i18n.__('Appearance', 'dracula-dark-mode')}</h2>

            {/*Content Width*/}
            <div className={`settings-field ${!dracula.isPro ? 'pro-feature' : ''}`}>
                <h4 className="settings-field-label">
                    {wp.i18n.__('Content Width', 'dracula-dark-mode')}
                    {!isPro && <span className="dracula-pro-icon"><i className={`dracula-icon icon-pro`}/></span>}    
                </h4>

                <div className={`settings-field-content`}>
                    <RadioGroup label="Content Width"
                        data-tooltip-content={wp.i18n.__('PRO Feature', 'dracula-dark-mode')}
                        data-tooltip-id={`content-width`}
                        onChange={contentWidth => {
                            if (!dracula.isPro) {
                                showProModal(
                                    wp.i18n.__('Upgrade to PRO to use Open Link in New Tab.', 'dracula-dark-mode'),
                                    {id: '-eL20lz83Ec', title: wp.i18n.__('Distraction-free Reading Mode', 'dracula-dark-mode')}
                                );
                                return;
                            }

                            setData({...data, contentWidth})
                        }}
                        checked={contentWidth}
                        className={`dracula-radio-group`} >
                        <Radio value="small">{wp.i18n.__('Small', 'dracula-dark-mode')}</Radio>
                        <Radio value="medium">{wp.i18n.__('Medium', 'dracula-dark-mode')}</Radio>
                        <Radio value="large">{wp.i18n.__('Large', 'dracula-dark-mode')}</Radio>
                    </RadioGroup>

                    {!dracula.isPro &&
                        <Tooltip
                            id="content-width" 
                            place="right" 
                            effect="solid"
                            variant="warning"
                            backgroundColor="#ff9966"
                        />
                    }

                    <p className={`description`}>{wp.i18n.__('Set the reader mode content body width.', 'dracula-dark-mode')}</p>
                </div>
            </div>

            {/* Theme */}
            <div className={`settings-field ${!dracula.isPro ? 'pro-feature' : ''}`}>
                <h4 className="settings-field-label">
                    {wp.i18n.__('Theme', 'darcula-dark-mode')}
                    {!isPro && <span className="dracula-pro-icon"><i className={`dracula-icon icon-pro`}/></span>}    
                </h4>

                <div className={`settings-field-content`}>
                    <RadioGroup
                        data-tooltip-content={wp.i18n.__('PRO Feature', 'dracula-dark-mode')}
                        data-tooltip-id={`theme`}
                        label={wp.i18n.__('Color Theme', 'dracula-dark-mode')}
                        onChange={theme => {
                            if (!dracula.isPro) {
                                showProModal(
                                    wp.i18n.__('Upgrade to PRO to use Theme.', 'dracula-dark-mode'),
                                    {id: '-eL20lz83Ec', title: wp.i18n.__('Distraction-free Reading Mode', 'dracula-dark-mode')}
                                );
                                return;
                            }

                            setData({...data, theme})
                        }}
                        checked={theme}
                        className={`dracula-radio-group`} >
                        <Radio value="light">
                            <span className="theme-option-icon icon-light"></span>
                            <span>{wp.i18n.__('Light', 'dracula-dark-mode')}</span>
                        </Radio>
                        <Radio value="dark">
                            <span className="theme-option-icon icon-dark"></span>
                            <span>{wp.i18n.__('Dark', 'dracula-dark-mode')}</span>
                        </Radio>
                        <Radio value="system">
                            <span className="theme-option-icon icon-system"></span>
                            <span>{wp.i18n.__('System', 'dracula-dark-mode')}</span>
                        </Radio>
                    </RadioGroup>

                    {!dracula.isPro &&
                        <Tooltip
                            id="theme" 
                            place="right" 
                            effect="solid"
                            variant="warning"
                            backgroundColor="#ff9966"
                        />
                    }

                    <p className={`description`}>{wp.i18n.__('Choose the color theme for the reader mode.', 'dracula-dark-mode')}</p>
                </div>
            </div>

            {/*Font Family*/}
            <div className={`settings-field ${!dracula.isPro ? 'pro-feature' : ''}`}>
                <h4 className="settings-field-label">
                    {wp.i18n.__('Font Family', 'dracula-dark-mode')}
                    {!isPro && <span className="dracula-pro-icon"><i className={`dracula-icon icon-pro`}/></span>}
                </h4>
                <div 
                    data-tooltip-content={wp.i18n.__('PRO Feature', 'dracula-dark-mode')}
                    data-tooltip-id={`font-family`}
                    className={`settings-field-content`}>
                    <Select
                        options={fontOptions}
                        value={'System' === fontFamily ? {label: 'System'} : fonts.find(item => item.label === fontFamily)}
                        onChange={selected => {
                            if (!dracula.isPro) {
                                showProModal(
                                    wp.i18n.__('Upgrade to PRO to use Open Link in New Tab.', 'dracula-dark-mode'),
                                    {id: '-eL20lz83Ec', title: wp.i18n.__('Distraction-free Reading Mode', 'dracula-dark-mode')}
                                );
                                return;
                            }

                            setData({...data, fontFamily: selected.label});
                        }}
                        className={`dracula-select select-font-family`}
                        classNamePrefix={`dracula-select`}
                        onMenuOpen={() => {

                            setTimeout(() => {
                                const $ = jQuery;

                                $('.reading-mode-select__menu-list').on('scroll', function () {
                                    $('.reading-mode-select__option').each((index, element) => {

                                        if (isVisible(element, this)) {
                                            const fontFamily = $(element).text();
                                            const id = `${fontFamily.replace(/ /g, '-')}-font`;

                                            $(element).css('font-family', fontFamily);

                                            if (!$(`#${id}`).length) {

                                                if (!loaded.includes(fontFamily)) {
                                                    setLoaded(loaded => [...loaded, fontFamily]);
                                                    // Load font family
                                                    $('head').append(`<link href="https://fonts.googleapis.com/css?family=${fontFamily.replace(/ /g, '+')}" rel="stylesheet" id="${id}" >`);
                                                }
                                            }
                                        }

                                    });
                                });

                            }, 10);
                        }}

                        onMenuClose={() => {
                            const $ = jQuery;
                            $('.reading-mode-select__menu-list').off('scroll');
                            setLoaded([]);
                        }}

                        styles={{
                            singleValue: (base, {data}) => ({
                                ...base,
                                fontFamily: data.label,
                                fontSize: '1.1rem',
                            }),
                        }}

                    />

                    <p className="description">{wp.i18n.__('Choose the font family for the reader mode.', 'dracula-dark-mode')}</p>
                </div>
                {!dracula.isPro &&
                    <Tooltip
                        id="font-family" 
                        place="right" 
                        effect="solid"
                        variant="warning"
                        backgroundColor="#ff9966"
                    />
                }
            </div>

            {/*  Font Size */}
            <div className={`settings-field ${!dracula.isPro ? 'pro-feature' : ''}`}>
                <h4 className="settings-field-label">
                    {wp.i18n.__('Font Size', 'dracula-dark-mode')}
                    {!isPro && <span className="dracula-pro-icon"><i className={`dracula-icon icon-pro`}/></span>}
                </h4>
                <div className={`settings-field-content`}>

                    <RadioGroup
                        data-tooltip-content={wp.i18n.__('PRO Feature', 'dracula-dark-mode')}
                        data-tooltip-id={`font-size`}
                        label="Toggle Position" 
                        onChange={fontSize => {
                            if (!dracula.isPro) {
                                showProModal(
                                    wp.i18n.__('Upgrade to PRO to use Font size.', 'dracula-dark-mode'),
                                    {id: '-eL20lz83Ec', title: wp.i18n.__('Distraction-free Reading Mode', 'dracula-dark-mode')}
                                );
                                return;
                            }

                            setData({...data, fontSize})
                        }}
                        checked={fontSize}
                        className={`dracula-radio-group`} >
                        <Radio value=".75">{wp.i18n.__('Small', 'dracula-dark-mode')}</Radio>
                        <Radio value="1">{wp.i18n.__('Medium', 'dracula-dark-mode')}</Radio>
                        <Radio value="1.2">{wp.i18n.__('Large', 'dracula-dark-mode')}</Radio>
                    </RadioGroup>

                    {!dracula.isPro &&
                        <Tooltip
                            id="font-size" 
                            place="right" 
                            effect="solid"
                            variant="warning"
                            backgroundColor="#ff9966"
                        />
                    }

                    <p className="description">{wp.i18n.__('Set the font-size for the reading mode.', 'dracula-dark-mode')}</p>
                </div>
            </div>

            <h2>{wp.i18n.__('Progress Bar Settings', 'dracula-dark-mode')}</h2>
            
            {/* Reading Progress */}
            <div className={`settings-field ${!dracula.isPro ? 'pro-feature' : ''}`}>
                <h4 className="settings-field-label">
                    {wp.i18n.__('Enable Reading Progress', 'dracula-dark-mode')}
                    {!isPro && <span className="dracula-pro-icon"><i className={`dracula-icon icon-pro`}/></span>}
                </h4>

                <div className="settings-field-content">
                    <FormToggle
                        data-tooltip-content={wp.i18n.__('PRO Feature', 'dracula-dark-mode')}
                        data-tooltip-id={`progress-bar`}
                        checked={enableReadingProgress}
                        onChange={() => {
                            if (!dracula.isPro) {
                                showProModal(
                                    wp.i18n.__('Upgrade to PRO to use Reading Progress Bar.', 'dracula-dark-mode'),
                                    {id: '-eL20lz83Ec', title: wp.i18n.__('Distraction-free Reading Mode', 'dracula-dark-mode')}
                                );
                                return;
                            }

                            setData({...data, enableReadingProgress: !enableReadingProgress})
                        }}
                    />

                    {!dracula.isPro &&
                        <Tooltip
                            id="progress-bar" 
                            place="right" 
                            variant="warning"
                            effect="solid"
                            backgroundColor="#ff9966"
                        />
                    }

                    <p className={`description`}>{wp.i18n.__('Show/ hide the reading progress bar in the general mode posts.', 'dracula-dark-mode')}</p>
                </div>
            </div>
            
            {/* Progress Bar Position */}
            <div className={`settings-field ${!dracula.isPro ? 'pro-feature' : ''}`}>
                <h4 className="settings-field-label">
                    {wp.i18n.__('Progress Bar Position', 'dracula-dark-mode')}
                    {!isPro && <span className="dracula-pro-icon"><i className={`dracula-icon icon-pro`}/></span>}
                </h4>

                <div className="settings-field-content">

                    <SelectControl
                        data-tooltip-content={wp.i18n.__('PRO Feature', 'dracula-dark-mode')}
                        data-tooltip-id={`tooltip-progress-position`}
                        value={progressPosition}
                        options={[
                            {label: 'Top of the page', value: 'top'},
                            {label: 'Bottom of the page', value: 'bottom'},
                        ]}
                        onChange={ (progressPosition) => {
                            if (!dracula.isPro) {
                                showProModal(
                                    wp.i18n.__('Upgrade to PRO to use Progress Bar Position.', 'dracula-dark-mode'),
                                    {id: '-eL20lz83Ec', title: wp.i18n.__('Distraction-free Reading Mode', 'dracula-dark-mode')}
                                );
                                return;
                            }

                            setData({...data, progressPosition})
                        }}
                        className={`dracula-select-control`}
                    />

                    {!dracula.isPro &&
                        <Tooltip
                            id={`tooltip-progress-position`}
                            place="left"
                            variant="warning"
                            effect="solid"
                            backgroundColor="#ff9966"
                        />
                    }

                    <p className={`description`}>{wp.i18n.__('Select display position reading progress bar.', 'dracula-dark-mode')}</p>
                </div>
            </div>

            {/*  Progressbar Color */}
            <div className={`settings-field ${!dracula.isPro ? 'pro-feature' : ''}`}>
                <h4 className="settings-field-label">
                    {wp.i18n.__('Progress Bar Color', 'dracula-dark-mode')}
                    {!isPro && <span className="dracula-pro-icon"><i className={`dracula-icon icon-pro`}/></span>}
                </h4>

                <div
                    data-tooltip-content={wp.i18n.__('PRO Feature', 'dracula-dark-mode')}
                    data-tooltip-id={`tooltip-progress-color`}
                    className="settings-field-content">
                    <RadioGroup
                        label={wp.i18n.__('Color Mode', 'dracula-dark-mode')}
                        onChange={progressbarStyle => {

                            if (!dracula.isPro) {
                                showProModal(
                                    wp.i18n.__('Upgrade to PRO to change the progress bar color.', 'dracula-dark-mode'),
                                    {id: '-eL20lz83Ec', title: wp.i18n.__('Distraction-free Reading Mode', 'dracula-dark-mode')}
                                );
                                return;
                            }

                            setData({...data, progressbarStyle})
                        }}
                        defaultChecked={progressbarStyle}
                        className={`dracula-radio-group progress-bar-color-mode`}
                    >
                        <Radio value="solid">{wp.i18n.__('Solid', 'dracula-dark-mode')}</Radio>
                        <Radio value="gradient">{wp.i18n.__('Gradient', 'dracula-dark-mode')}</Radio>
                    </RadioGroup>

                    {progressbarStyle === 'solid' &&
                        <ColorPickerWrapper
                            value={progressbarColor}
                            onChange={progressbarColor => {

                                if (!dracula.isPro) {
                                    showProModal(
                                        wp.i18n.__('Upgrade to PRO to change the progress bar color.', 'dracula-dark-mode'),
                                        {id: '-eL20lz83Ec', title: wp.i18n.__('Distraction-free Reading Mode', 'dracula-dark-mode')}
                                    );
                                    return;
                                }

                                setData({...data, progressbarColor})
                            }}
                        />
                    }

                    {progressbarStyle === 'gradient' &&
                        <GradientPicker
                            value={progressbarColorGradient}
                            onChange={progressbarColorGradient => {

                                if (!dracula.isPro) {
                                    showProModal(
                                        wp.i18n.__('Upgrade to PRO to customize the progress bar color.', 'dracula-dark-mode'),
                                        {id: '-eL20lz83Ec', title: wp.i18n.__('Distraction-free Reading Mode', 'dracula-dark-mode')}
                                    );
                                    return;
                                }

                                setData({...data, progressbarColorGradient})
                            }}
                            className={`dracula-gradient-picker`}
                            gradients={[
                                {
                                    gradient: 'linear-gradient(90deg, #004AFF 80%, rgba(96, 239, 255, 0) 113.89%)',
                                },
                                {
                                    gradient: 'linear-gradient(90deg, #F9AB8F 80%, rgba(249, 171, 143, 0) 113.89%)',
                                },
                                {
                                    gradient: 'linear-gradient(90deg, #1ED7B5 -18.06%, #F0F9A7 113.89%)',
                                },
                                {
                                    gradient: 'linear-gradient(90deg, #A4E9F9 -18.06%, #C5AEF2 48.77%, #8578EA 113.89%)',
                                },
                                {
                                    gradient: 'linear-gradient(90deg, #F2F3E2 -18.06%, #B2E5F8 49.82%, #F4B3EF 113.89%)',
                                },
                                {
                                    gradient: 'linear-gradient(90deg, #595CFF -18.06%, #B3E8F7 113.89%)',
                                },
                                {
                                    gradient: 'linear-gradient(90deg, #2FEAA8 -18.06%, #028CF3 113.89%)',
                                },
                                {
                                    gradient: 'linear-gradient(90deg, #FFA585 -18.06%, #FFEDA0 113.89%)',
                                },
                                {
                                    gradient: 'linear-gradient(90deg, #EED991 -18.06%, #CCF7F4 113.89%)',
                                },
                                {
                                    gradient: 'linear-gradient(90deg, #1A87FF -2.1%, rgba(18, 199, 145, 0.92) 51.12%, rgba(96, 239, 255, 0.49) 95.86%)',
                                },
                            ]}
                        />
                    }

                    <p className={`description`}>{wp.i18n.__('Set the progressbar color.', 'dracula-dark-mode')}</p>

                    {!dracula.isPro &&
                        <Tooltip
                            id={`tooltip-progress-color`}
                            place="left"
                            variant="warning"
                            effect="solid"
                            backgroundColor="#ff9966"
                        />
                    }

                </div>
            </div>

            {/* Progressbar Height */}
            <div className={`settings-field ${!dracula.isPro ? 'pro-feature' : ''}`}>
                <h4 className="settings-field-label">
                    {wp.i18n.__('Progress Bar Height', 'dracula-dark-mode')}
                    {!isPro && <span className="dracula-pro-icon"><i className={`dracula-icon icon-pro`}/></span>}
                </h4>
                <div
                    data-tooltip-content={wp.i18n.__('PRO Feature', 'dracula-dark-mode')}
                    data-tooltip-id={`tooltip-progress-height`}
                    className="settings-field-content">
                    <RangeControl
                        allowReset={true}
                        resetFallbackValue={7}
                        step={1}
                        isShiftStepEnabled
                        renderTooltipContent={() => progressbarHeight}
                        value={parseInt(progressbarHeight)}
                        onChange={progressbarHeight => {

                            if (!dracula.isPro) {
                                showProModal(
                                    wp.i18n.__('Upgrade to PRO to customize the progress bar height.', 'dracula-dark-mode'),
                                    {id: '-eL20lz83Ec', title: wp.i18n.__('Distraction-free Reading Mode', 'dracula-dark-mode')}
                                );
                                return;
                            }

                            setData({...data, progressbarHeight})
                        }}
                        min={1}
                        max={50}
                        className={`dracula-range-control`}
                    />

                    <p className="description">{wp.i18n.__('Set the progressbar height.', 'dracula-dark-mode')}</p>

                    {!dracula.isPro &&
                        <Tooltip
                            id={`tooltip-progress-height`}
                            place="left"
                            variant="warning"
                            effect="solid"
                            backgroundColor="#ff9966"
                        />
                    }
                </div>
            </div>
        </>
    );
}