import {Tooltip} from "react-tooltip";
import SettingsContext from "../../contexts/SettingsContext";
import ReadingModeButtonPreview from "./Preview/ReadingModeButtonPreview";
import {lightenDarkenColor} from "../../includes/functions";
import ColorPickerWrapper from "../../includes/ColorPickerWrapper";
import {showProModal} from "../ProModal";

const {
    __experimentalRadioGroup: RadioGroup,
    __experimentalRadio: Radio,
    FormToggle,
    TextControl,
    SelectControl,
} = wp.components;

const {useContext, useRef} = React;

export default function ReadingModeButtonSettings(){
    const {isPro} = dracula;

    const context = useContext(SettingsContext);

    const { data, setData } = context;

    const {
        buttonPosition = 'aboveContent',
        buttonSize = 'medium',
        buttonAlignment = 'flex-start',
        readingModeStyle = 'default',
        readingModeLabel = true,
        readingModeText = 'Reading Mode',
        readingModeBGColor = 'linear-gradient(-90deg, rgb(255, 255, 255) 0%, rgb(221, 221, 221) 100%)',
        readingModeTextColor = '#2F80ED',
        readingModeIcon = 1,
        customReadingModeIcon = '',
    } = data;

    const frameRef = useRef();

    const uploadIcon = () => {

        if (frameRef.current) {
            frameRef.current.off('select');

            frameRef.current.on('select', () => {
                const attachment = frameRef.current.state().get('selection').first().toJSON();
                setData({...data, customReadingModeIcon: attachment.url});

            }).open();
            return;
        }

        frameRef.current = wp.media({
            title: 'Select Image',
            button: {text: 'Use this Image'},
            multiple: false,
            library: {type: 'image'},
        });

        frameRef.current.on('select', () => {
            const attachment = frameRef.current.state().get('selection').first().toJSON();
            setData({...data, customReadingModeIcon: attachment.url});
        }).open();
    }

    const styles = {
        default: {
            readingModeBGColor: 'linear-gradient(-90deg, rgb(255, 255, 255) 0%, rgb(221, 221, 221) 100%)',
            readingModeTextColor: '#2F80ED',
            readingModeIcon: 1,
        },
        solid: {
            readingModeBGColor: '#2F80ED',
            readingModeTextColor: '#FFF',
            readingModeIcon: 2,
        },
        outline: {
            readingModeBGColor: 'transparent',
            readingModeTextColor: '#2F80ED',
            readingModeIcon: 3,
        },
        shadow: {
            readingModeTextColor: '#FFF',
            readingModeBGColor: '#2F80ED',
            readingModeIcon: 4,
        },
        'shadow-alt': {
            readingModeTextColor: '#FFF',
            readingModeBGColor: '#2F80ED',
            readingModeIcon: 5,
        },
        icon: {
            readingModeTextColor: '#FFF',
            readingModeBGColor: '#2F80ED',
            readingModeIcon: 6,
        },
        'icon-alt': {
            readingModeBGColor: '#2F80ED',
            readingModeTextColor: '#FFF',
            readingModeIcon: 7,
        }
    }

    return(
        <>
            <div className="settings-fields-wrap">
                <div className="settings-fields">
                    {/* Button Style */}
                    <div className="settings-field">
                        <h4 className="settings-field-label">
                            {wp.i18n.__('Button Style', 'dracula-dark-mode')}
                            {!isPro && <span className="dracula-pro-icon"><i className={`dracula-icon icon-pro`}/></span>}
                        </h4>
                        <div className={`settings-field-content`}>
                            <div className="button-styles" style={{'--reading-mode-bg-darker': lightenDarkenColor('#2F80ED', -30)}}>
                                {
                                    Object.keys(styles)?.map((style, index) => {
                                        const isActive = style == readingModeStyle;

                                        const configData = {
                                            readingModeStyle: style,
                                            ...styles[style]
                                        };

                                        return(
                                            <div
                                                data-tooltip-content={wp.i18n.__('PRO Feature', 'dracula-dark-mode')}
                                                data-tooltip-id={`tool-${style}`}
                                                className={`button-style-wrap style-${style} ${isActive} ${!dracula.isPro ? 'disabled' : ''}`}
                                                onClick={()=>{
                                                    if(!dracula.isPro ){
                                                        showProModal(
                                                            wp.i18n.__('Upgrade to PRO to use this button style.', 'dracula-dark-mode'),
                                                            {id: '-eL20lz83Ec', title: wp.i18n.__('Distraction-free Reading Mode', 'dracula-dark-mode')}
                                                        )
                                                        return; 
                                                    }

                                                    setData(data => ({...data, ...configData}));
                                                }}
                                                >
                                                
                                                <ReadingModeButtonPreview key={index} data={configData} />

                                                {isActive && <i className={`active-badge dashicons dashicons-saved`}></i>}
                                                
                                                {!dracula.isPro &&
                                                    <Tooltip
                                                        id={`tool-${style}`}
                                                        place="bottom"
                                                        effect="solid"
                                                        variant="warning"
                                                        backgroundColor="#ff9966"
                                                    />
                                                }
                                            </div>
                                        )
                                    })
                                }
                            </div>
                            <p className="description">{wp.i18n.__('Choose the button style to enable the Reading Mode.', 'dracula-dark-mode')}</p>
                        </div>
                    </div>

                    {/* Button Position */}
                    <div className={`settings-field field-show-toggle ${!dracula.isPro ? 'pro-feature' : ''}`}>
                        <h4 className={`settings-field-label`}>
                            {wp.i18n.__('Button Position', 'dracula-dark-mode')}
                            {!isPro && <span className="dracula-pro-icon"><i className={`dracula-icon icon-pro`}/></span>}
                        </h4>
                        <div className="settings-field-content">
                            
                            <SelectControl
                                data-tooltip-content={wp.i18n.__('Pro Feature', 'dracula-dark-mode')}
                                data-tooltip-id={`button-position`}
                                value={buttonPosition}
                                options={[
                                    {label: 'Above post content', value: 'aboveContent'},
                                    {label: 'Above post title', value: 'aboveTitle'},
                                    {label: 'Below post title', value: 'belowTitle'},
                                ]}
                                onChange={(buttonPosition) => {
                                    if(!dracula.isPro ){
                                        showProModal(
                                            wp.i18n.__('Upgrade to PRO to use this button position.', 'dracula-dark-mode'),
                                            {id: '-eL20lz83Ec', title: wp.i18n.__('Distraction-free Reading Mode', 'dracula-dark-mode')}
                                        );
                                        return; 
                                    }

                                    setData({...data, buttonPosition: buttonPosition})
                                }}
                                className={`dracula-select-control`}
                            />

                            {!dracula.isPro &&
                                <Tooltip
                                    id="button-position"
                                    place="right"
                                    effect="solid"
                                    variant="warning"
                                    backgroundColor="#ff9966"
                                />
                            }

                            <p className="description">{wp.i18n.__('Show/ hide the reading mode button in the frontend for the users to click reading mode.', 'dracula-dark-mode')}</p>
                        </div>
                    </div>

                    {/*  Label Text*/}
                    <div className={`settings-field ${!dracula.isPro ? 'pro-feature' : ''}`}>
                        <h4 className={`settings-field-label`}>
                            {wp.i18n.__('Button Label Text', 'dracula-dark-mode')}
                            {!isPro && <span className="dracula-pro-icon"><i className={`dracula-icon icon-pro`}/></span>}
                        </h4>

                        <div className={`settings-field-content`}>
                            <div className="settings-wrap">
                                <FormToggle
                                    data-tooltip-content={wp.i18n.__('PRO Feature', 'dracula-dark-mode')}
                                    data-tooltip-id={`enable-reading-mode-label`}
                                    checked={ dracula.isPro && readingModeLabel }
                                    onChange={() => {
                                        if (!dracula.isPro) {
                                            showProModal(
                                                wp.i18n.__('Reading Mode Button Label is a pro feature. Upgrade to PRO to enable the dark mode reading mode.', 'dracula-dark-mode'),
                                                {id: '-eL20lz83Ec', title: wp.i18n.__('Distraction-free Reading Mode', 'dracula-dark-mode')}
                                            );
                                            
                                            return;
                                        };

                                        setData({...data, readingModeLabel: !readingModeLabel});
                                    }}
                                />
                                <p className={`description`}>{wp.i18n.__('Enable Reading Mode button label.', 'dracula-dark-mode')}</p>
                            </div>
                            { readingModeLabel && <>
                                <TextControl
                                    data-tooltip-content={wp.i18n.__('Pro Features', 'dracula-dark-mode')}
                                    data-tooltip-id="label-text"
                                    value={readingModeText}
                                    onChange={ (readingModeText) => {
                                        if(!dracula.isPro ){
                                            showProModal(
                                                wp.i18n.__('Upgrade to PRO to use this Label Text.', 'dracula-dark-mode'),
                                                {id: '-eL20lz83Ec', title: wp.i18n.__('Distraction-free Reading Mode', 'dracula-dark-mode')}
                                            )
                                            return; 
                                        }
                                        setData({...data, readingModeText})
                                    }}
                                    className={`dracula-text-control`}
                                />
                                <p className={`description`}>{wp.i18n.__('Enter the Reading Mode button label text.', 'dracula-dark-mode')}</p>
                            </>}
                            

                            {!dracula.isPro &&
                                <>
                                    <Tooltip
                                        id={`enable-reading-mode-label`} 
                                        place="right" 
                                        effect="solid" 
                                        variant="warning"
                                        backgroundColor="#ff9966"
                                        className={`dracula-tooltip`}
                                    />

                                    <Tooltip
                                        id={`label-text`} 
                                        place="right" 
                                        effect="solid" 
                                        variant="warning"
                                        backgroundColor="#ff9966"
                                        className={`dracula-tooltip`}
                                    />
                                </>
                            }
                        </div>

                    </div>

                    {/* Button Icon */}
                    <div className="settings-field">
                        <h4 className={`settings-field-label`}>
                            {wp.i18n.__('Button Icon', 'dracula-dark-mode')}
                            {!isPro && <span className="dracula-pro-icon"><i className={`dracula-icon icon-pro`}/></span>}
                        </h4>
                        <div className={`settings-field-content`}>
                            <div className="btn-icons">

                                <div className={`btn-icon-wrap ${!readingModeIcon ? 'active' : ''}`}>
                                    <div 
                                        data-tooltip-content={wp.i18n.__('PRO Feature', 'reading-mode')}
                                        data-tooltip-id={`tool-upload-icon`}
                                        className={`btn-icon btn-none ${!dracula.isPro ? 'disabled' : ''}`}
                                        onClick={() => {
                                            if (!dracula.isPro) {
                                                showProModal(
                                                    wp.i18n.__('Upgrade to PRO to use button icon.', 'dracula-dark-mode'),
                                                    {id: '-eL20lz83Ec', title: wp.i18n.__('Distraction-free Reading Mode', 'dracula-dark-mode')}
                                                );
                                                return;
                                            }

                                            setData({
                                                ...data,
                                                readingModeIcon: '',
                                                customReadingModeIcon: ''
                                            })
                                        }}>
                                        <i className="dashicons dashicons-no"></i>
                                        {wp.i18n.__('None', 'dracula-dark-mode')}
                                    </div>
                                </div>

                                {
                                    Array.from(Array(21).keys()).map((item, index) => {
                                        const indexValue = index + 1;
                                        return (
                                            <div
                                                className={`btn-icon-wrap ${!dracula.isPro ? 'disabled' : ''} ${!customReadingModeIcon && indexValue === readingModeIcon ? 'active' : ''}`}>
                                            <span
                                                data-tooltip-content={wp.i18n.__('PRO Feature', 'reading-mode')}
                                                data-tooltip-id={`tool-upload-icon`}
                                                style={{'-webkit-mask': `url("${dracula.pluginUrl}/assets/images/icons/reading-mode/${indexValue}.svg") no-repeat center / contain`}}
                                                className={`btn-icon`}
                                                onClick={() => {
                                                    if (!dracula.isPro) {
                                                        showProModal(
                                                            wp.i18n.__('Upgrade to PRO to use button icon.', 'dracula-dark-mode'),
                                                            {id: '-eL20lz83Ec', title: wp.i18n.__('Distraction-free Reading Mode', 'dracula-dark-mode')}
                                                        );
                                                        return;
                                                    }

                                                    setData({
                                                        ...data,
                                                        readingModeIcon: indexValue,
                                                        customReadingModeIcon: ''
                                                    });
                                                }}
                                            />
                                            </div>
                                        )
                                    })
                                }
                            </div>

                            <p className={`description`}>{wp.i18n.__('Choose the reading mode button icon.', 'dracula-dark-mode')}</p>

                            <div
                                data-tooltip-content={wp.i18n.__('PRO Feature', 'reading-mode')}
                                data-tooltip-id={`tool-upload-icon`}

                                className="upload-icon">
                                <button type={`button`}
                                        className={`dracula-btn btn-primary upload-btn ${!dracula.isPro ? 'disabled' : ''}`}
                                        onClick={(e) => {
                                            if (!dracula.isPro) {
                                                showProModal(
                                                    wp.i18n.__('Upgrade to PRO to use custom button icon.', 'dracula-dark-mode'),
                                                    {id: '-eL20lz83Ec', title: wp.i18n.__('Distraction-free Reading Mode', 'dracula-dark-mode')}
                                                );
                                                return;
                                            }

                                            uploadIcon();
                                        }}>
                                    <i className="dashicons dashicons-upload"></i>
                                    <span>{wp.i18n.__('Upload Icon', 'dracula-dark-mode')}</span>
                                </button>

                                {
                                    !!customReadingModeIcon &&
                                    <button type={`button`} className={`dracula-btn btn-danger`}
                                            onClick={() => setData(data => ({...data, customReadingModeIcon: ''}))}
                                    >
                                        <i className="dashicons dashicons-trash"></i>
                                        <span>{wp.i18n.__('Remove', 'dracula-dark-mode')}</span>
                                    </button>
                                }

                                {!dracula.isPro &&
                                    <Tooltip
                                        id={`tool-upload-icon`}
                                        place="right"
                                        variant="warning"
                                        effect="solid"
                                        backgroundColor="#ff9966"
                                        className={`dracula-tooltip`}
                                    />
                                }

                            </div>

                            {
                                !!customReadingModeIcon &&
                                <div className="upload-icon-preview">
                                    <img src={customReadingModeIcon}/>
                                </div>
                            }

                            <p className="description">{wp.i18n.__('You can also upload and select any custom image as the button icon.', 'dracula-dark-mode')}</p>

                        </div>
                    </div>

                    {/* Button size */}
                    <div className={`settings-field ${!dracula.isPro ? 'pro-feature' : ''}`}>
                        <h4 className={`settings-field-label`}>
                            {wp.i18n.__('Button Size', 'dracula-dark-mode')}
                            {!isPro && <span className="dracula-pro-icon"><i className={`dracula-icon icon-pro`}/></span>}
                        </h4>
                        <div className={`settings-field-content`}>
                            <RadioGroup
                                data-tooltip-content={wp.i18n.__('Pro Feature', 'dracula-dark-mode')}
                                data-tooltip-id="button-size"
                                label="Button Size"
                                checked={buttonSize}
                                onChange={buttonSize => {

                                    if (!dracula.isPro) {
                                        showProModal(
                                            wp.i18n.__('Upgrade to PRO to customize the button size.', 'dracula-dark-mode'),
                                            {id: '-eL20lz83Ec', title: wp.i18n.__('Distraction-free Reading Mode', 'dracula-dark-mode')}
                                        );
                                        return;
                                    }

                                    setData({...data, buttonSize});

                                }}
                                className={`dracula-radio-group`}
                            >
                                <Radio value="small">Small</Radio>
                                <Radio value="medium">Medium</Radio>
                                <Radio value="large">Large</Radio>
                            </RadioGroup>
                            {!dracula.isPro &&
                                <Tooltip
                                    id="button-size"
                                    place="right"
                                    variant="warning"
                                    effect="solid"
                                    backgroundColor="#ff9966"
                                    className={`dracula-tooltip`}
                                />
                            }

                            <p className="description">{wp.i18n.__('Choose the button size for the Reader Mode, Text-to-Speech, Translation & Reading Time button.','dracula-dark-mode')}</p>
                        </div>
                    </div>

                    {/* button alignment */}
                    <div className={`settings-field ${!dracula.isPro ? 'pro-feature' : ''}`}>
                        <h4 className={`settings-field-label`}>
                            {wp.i18n.__('Button Alignment', 'dracula-dark-mode')}
                            {!isPro && <span className="dracula-pro-icon"><i className={`dracula-icon icon-pro`}/></span>}
                        </h4>
                        <div className="settings-field-content">
                            <RadioGroup
                                data-tooltip-content={wp.i18n.__(`Pro Feature`, 'dracula-dark-mode')}
                                data-tooltip-id={`button-alignment`}
                                label="Button Alignment"
                                checked={buttonAlignment}
                                onChange={buttonAlignment => {
                                    if (!dracula.isPro) {
                                        showProModal(
                                            wp.i18n.__('Upgrade to PRO to customize the button alignment.', 'dracula-dark-mode'),
                                            {id: '-eL20lz83Ec', title: wp.i18n.__('Distraction-free Reading Mode', 'dracula-dark-mode')}
                                        );
                                        return;
                                    }
                                    setData({...data, buttonAlignment})
                                }}
                                className={`dracula-radio-group`}
                            >
                                <Radio value="flex-start">{wp.i18n.__('Left', 'dracula-dark-mode')}</Radio>
                                <Radio value="center">{wp.i18n.__('Center', 'dracula-dark-mode')}</Radio>
                                <Radio value="flex-end">{wp.i18n.__('Right', 'dracula-dark-mode')}</Radio>
                            </RadioGroup>

                            {!dracula.isPro &&
                                <Tooltip
                                    id={`button-alignment`}
                                    place="right"
                                    variant="warning"
                                    effect="solid"
                                    backgroundColor="#ff9966"
                                    className={`dracula-tooltip`}
                                />
                            }

                            <p className="description">{wp.i18n.__('Choose the button size.', 'dracula-dark-mode')}</p>
                        </div>
                    </div>

                    {/*  Background Color */}
                    <div className={`settings-field ${!dracula.isPro ? 'pro-feature' : ''}`}>
                        <h4 className={`settings-field-label`}>
                            {wp.i18n.__('Background Color', 'dracula-dark-mode')}
                            {!isPro && <span className="dracula-pro-icon"><i className={`dracula-icon icon-pro`}/></span>}
                        </h4>

                        <div
                            data-tooltip-content={wp.i18n.__('PRO Feature', 'dracual-dark-mode')}
                            data-tooltip-id={`tooltip-bg-color`}

                            className={`settings-field-content`}>

                            <ColorPickerWrapper
                                value={readingModeBGColor}
                                onChange={readingModeBGColor => {

                                    if (!dracula.isPro) {
                                        showProModal(
                                            wp.i18n.__('Upgrade to PRO to change the button colors.', 'dracual-dark-mode'),
                                            {id: '-eL20lz83Ec', title: wp.i18n.__('Distraction-free Reading Mode', 'dracula-dark-mode')}
                                        );
                                        return;
                                    }

                                    setData({...data, readingModeBGColor})
                                }}
                            />

                            <p className={`description`}>{wp.i18n.__('Set the reading mode button background color.', 'dracula-dark-mode')}</p>

                            {!dracula.isPro &&
                                <Tooltip
                                    id={`tooltip-bg-color`}
                                    place="left"
                                    variant="warning"
                                    backgroundColor="#ff9966"
                                    effect="solid"
                                />
                            }
                        </div>
                    </div>

                    {/*  Text Color */}
                    <div className={`settings-field ${!dracula.isPro ? 'pro-feature' : ''}`}>
                    <h4 className={`settings-field-label`}>
                        {wp.i18n.__('Text Color', 'dracula-dark-mode')}
                        {!isPro && <span className="dracula-pro-icon"><i className={`dracula-icon icon-pro`}/></span>}
                    </h4>

                        <div
                            data-tooltip-content={wp.i18n.__('PRO Feature', 'dracula-dark-mode')}
                            data-tooltip-id={`tooltip-text-color`}
                            className="settings-field-content">

                            <ColorPickerWrapper
                                value={readingModeTextColor}
                                onChange={readingModeTextColor => {

                                    if (!dracula.isPro) {
                                        showProModal(
                                            wp.i18n.__('Upgrade to PRO to change the button colors.', 'dracual-dark-mode'),
                                            {id: '-eL20lz83Ec', title: wp.i18n.__('Distraction-free Reading Mode', 'dracula-dark-mode')}
                                        );
                                        return;
                                    }

                                    setData({...data, readingModeTextColor})
                                }}
                            />

                            <p className={`description`}>{wp.i18n.__('Set the reading mode button text color.','dracula-dark-mode')}</p>

                            {!dracula.isPro &&
                                <Tooltip
                                    id={`tooltip-text-color`}
                                    place="left"
                                    variant="warning"
                                    backgroundColor="#ff9966"
                                    effect="solid"
                                />
                            }
                        </div>
                    </div>
                </div>

                {/* Button Preview */}
                <div className="button-preview"
                    style={{'--reading-mode-bg-darker': lightenDarkenColor(readingModeBGColor, -30)}}>
                    <div className={`button-preview-title`}>
                        <div>
                            <span className="browser-header-dot"></span>
                            <span className="browser-header-dot"></span>
                            <span className="browser-header-dot"></span>
                        </div>
                        <span className="dracula-btn-icon icon-eye"></span>
                        <span>{wp.i18n.__('Preview', 'dracula-dark-mode')}</span>
                    </div>

                    <div className={`button-preview-body`}>
                        <img src={`${dracula.pluginUrl}/assets/images/settings/reading-btn-preview-top.svg`} className="mb-6" alt="menu" />
                        { 'aboveTitle' === buttonPosition &&
                            <ReadingModeButtonPreview
                                data={{
                                    readingModeStyle,
                                    readingModeLabel,
                                    readingModeText,
                                    readingModeBGColor,
                                    readingModeTextColor,
                                    readingModeIcon,
                                    customReadingModeIcon,
                                    buttonSize,
                                    buttonAlignment
                                }}
                            />
                        }
                        <img src={`${dracula.pluginUrl}/assets/images/settings/post-title.svg`} alt="post-title" />
                        { ( 'belowTitle' === buttonPosition || 'aboveContent' === buttonPosition ) &&
                            <ReadingModeButtonPreview
                                data={{
                                    readingModeStyle,
                                    readingModeLabel,
                                    readingModeText,
                                    readingModeBGColor,
                                    readingModeTextColor,
                                    readingModeIcon,
                                    customReadingModeIcon,
                                    buttonSize,
                                    buttonAlignment
                                }}
                            />
                        }
                        <img src={`${dracula.pluginUrl}/assets/images/settings/post-content.svg`} alt="post-content" />
                    </div>
                </div>
            </div>
        </>
    );
}