import {Tooltip} from "react-tooltip";
import SettingsContext from "../../contexts/SettingsContext";
import {showProModal} from "../ProModal";

const {FormToggle, CheckboxControl} = wp.components;

const {useContext, useState, useEffect, useRef} = React;

const {isPro} = dracula;

export default function GeneralReadingSettings() {

    const context = useContext(SettingsContext);

    const postTypeList = dracula.postTypes;
    const {data, setData} = context;

    const {
        postTypes = ['post'],
        readingMode = false,
        readingModeDisplay = ['single'],
    } = data;

    const displayOptions = [
        {label: wp.i18n.__('Single', 'dracula-dark-mode'), value: 'single'},
        {label: wp.i18n.__('Blog Page', 'dracula-dark-mode'), value: 'blog'},
        {label: wp.i18n.__('Archive Page', 'dracula-dark-mode'), value: 'archive'},
        {label: wp.i18n.__('Search Page', 'dracula-dark-mode'), value: 'search'},
    ];

    return (
        <>
            {/* Enable Reading Mode */}
            <div className={`settings-field field-show-toggle ${!isPro ? 'pro-feature' : ''}`}>
                <h4 className={`settings-field-label`}>
                    {wp.i18n.__('Enable Reading Mode', 'dracula-dark-mode')}
                    {!isPro && <span className="dracula-pro-icon"><i className={`dracula-icon icon-pro`}/></span>}
                </h4>

                <div
                    className={`settings-field-content`}>
                    <div className="settings-wrap">
                        <FormToggle
                            data-tooltip-content={wp.i18n.__('PRO Feature', 'dracula-dark-mode')}
                            data-tooltip-id={`enable-reading-mode`}
                            checked={isPro && readingMode}
                            onChange={() => {
                                if (!isPro) {
                                    showProModal(
                                        wp.i18n.__('Reading Mode is a pro feature. Upgrade to PRO to enable the dark mode reading mode.', 'dracula-dark-mode'),
                                        {
                                            id: '-eL20lz83Ec',
                                            title: wp.i18n.__('Distraction-free Reading Mode', 'dracula-dark-mode')
                                        }
                                    );

                                    return;
                                }
                                ;

                                setData({...data, readingMode: !readingMode});
                            }}
                        />
                    </div>

                    {!isPro &&
                        <Tooltip
                            id="enable-reading-mode"
                            place="right"
                            effect="solid"
                            variant="warning"
                            backgroundColor="#ff9966"
                        />
                    }

                    <p className="description">{wp.i18n.__('Show/ hide the reading mode button in the frontend for the users to click reading mode.', 'dracula-dark-mode')}</p>
                </div>
            </div>

            {(!isPro || readingMode) && <>
                {/* Display Post type */}
                <div className={`settings-field ${!isPro ? 'pro-feature' : ''}`}>
                    <h4 className={`settings-field-label`}>
                        {wp.i18n.__('Display Post Types', 'dracula-dark-mode')}
                        {!isPro && <span className="dracula-pro-icon"><i className={`dracula-icon icon-pro`}/></span>}
                    </h4>

                    <div className="settings-field-content">
                        <div className="checkbox-wrap posttype-options">
                            {
                                postTypeList.map(option => (
                                    <CheckboxControl
                                        data-tooltip-content={wp.i18n.__('PRO Feature', 'dracula-dark-mode')}
                                        data-tooltip-id={`display-post-type`}
                                        label={option.label}
                                        checked={postTypes.includes(option.value)}
                                        onChange={() => {
                                            if (!isPro) {
                                                showProModal(
                                                    wp.i18n.__('Display Post Type is a pro feature. Upgrade to PRO to enable the dark mode display post type.', 'dracula-dark-mode'),
                                                    {
                                                        id: '-eL20lz83Ec',
                                                        title: wp.i18n.__('Distraction-free Reading Mode', 'dracula-dark-mode')
                                                    }
                                                );
                                                return;
                                            }
                                            ;

                                            if (postTypes.includes(option.value)) {
                                                setData({
                                                    ...data,
                                                    postTypes: postTypes.filter(item => item !== option.value)
                                                });
                                            } else {
                                                setData({...data, postTypes: [...postTypes, option.value]});
                                            }
                                        }}
                                    />
                                ))
                            }

                            {!isPro &&
                                <Tooltip
                                    id="display-post-type"
                                    place="right"
                                    effect="solid"
                                    variant="warning"
                                    backgroundColor="#ff9966"
                                />
                            }
                        </div>
                        <p className={`description`}>{wp.i18n.__('Select the post types, where the Reading Mode button will be displayed.', 'dracula-dark-mode')}</p>
                    </div>
                </div>

                {/* Display on */}
                <div className={`settings-field ${!isPro ? 'pro-feature' : ''}`}>
                    <h4 className="settings-field-label">
                        {wp.i18n.__('Display On', 'dracula-dark-mode')}
                        {!isPro && <span className="dracula-pro-icon"><i className={`dracula-icon icon-pro`}/></span>}
                    </h4>
                    <div className="settings-field-content">
                        <div className="checkbox-wrap display-options">
                            {
                                displayOptions.map(option => (
                                    <>
                                        <CheckboxControl
                                            data-tooltip-content={wp.i18n.__('Pro Feature', 'dracula-dark-mode')}
                                            data-tooltip-id={`tool-${option.value}`}
                                            label={option.label}
                                            checked={readingModeDisplay.includes(option.value)}
                                            onChange={() => {
                                                if (!isPro) {
                                                    showProModal(
                                                        wp.i18n.__('Upgrade to PRO to display the Reading Mode button on ', 'dracula-dark-mode') + option.label,
                                                        {
                                                            id: '-eL20lz83Ec',
                                                            title: wp.i18n.__('Distraction-free Reading Mode', 'dracula-dark-mode')
                                                        }
                                                    );
                                                    return;
                                                }

                                                if (readingModeDisplay.includes(option.value)) {
                                                    setData({
                                                        ...data,
                                                        readingModeDisplay: readingModeDisplay.filter(item => item != option.value)
                                                    })
                                                } else {
                                                    setData({
                                                        ...data,
                                                        readingModeDisplay: [...readingModeDisplay, option.value]
                                                    })
                                                }
                                            }}
                                        />

                                        {!isPro && 'single' !== option.value &&
                                            <Tooltip
                                                id={`tool-${option.value}`}
                                                place="top"
                                                type="dark"
                                                variant="warning"
                                                effect="solid"
                                            />
                                        }
                                    </>
                                ))
                            }
                        </div>
                        <p className="description">{wp.i18n.__('Select the pages where the Reading Mode button will be displayed.', 'dracula-dark-mode')}</p>
                    </div>
                </div>
            </>}
        </>
    );
}