import {Tooltip} from "react-tooltip";
import SettingsContext from "../../contexts/SettingsContext";
import Header from "./Header";
import LiveEditButton from "./LiveEditButton";
import ColorPickerWrapper from "../../includes/ColorPickerWrapper";
import {lightenDarkenColor,} from "../../includes/functions";
import {showProModal} from "../ProModal";
import {CustomColorField} from "./CustomColorField";

const {
    __experimentalRadioGroup: RadioGroup,
    __experimentalRadio: Radio,
    TextControl,
    RangeControl,
    FormToggle,
} = wp.components || {};

const {useContext, useEffect} = React;

export default function Colors() {

    const {isPro, presets} = dracula;

    const {menu, data, setData, isLiveEdit} = useContext(SettingsContext);

    const {
        colorType = 'dynamic',
        darkToLight,
        colorMode = 'dynamic',
        preset = 'default',
        activeCustomPreset,
        customPresets = [],
        scrollbarDarkMode = 'auto',
        scrollbarColor = '#181a1b',

        brightness = 100,
        contrast = 90,
        sepia = 10,
        grayscale,
    } = data;

    const defaultPreset = {
        id: new Date().getTime(),
        label: 'Preset name',
        colors: {
            bg: darkToLight ? '#dcdad7' : '#181a1b',
            text: darkToLight ? '#181a1b' : '#e8e6e3',
            link: '#4B8BFF',
            btn_bg: darkToLight ? '#4B8BFF' : '#44475a',
            btn_text: darkToLight ? '#ffffff' : '#f8f8f2',
            input_bg: darkToLight ? '#f0f0f0' : '#282a36',
            input_text: darkToLight ? '#181a1b' : '#e8e6e3',
            border: darkToLight ? '#ccc' : '#44475a',
        }
    };

    // add custom preset
    const addCustomPreset = () => {
        setData({...data, customPresets: [...customPresets, defaultPreset], activeCustomPreset: defaultPreset.id});
    }

    const activePreset = customPresets.find(item => item.id == activeCustomPreset) || defaultPreset;

    useEffect(() => {
        if ('static' === colorType && 'dynamic' === colorMode) {
            setData({...data, colorMode: 'presets'});
        }
    }, [colorType]);

    return (
        <div className="dracula-settings-content">

            {!isPro &&
                <Tooltip
                    anchorSelect={'.dracula-pro-icon, .pro-disabled'}
                    place="right"
                    variant="warning"
                    content={wp.i18n.__('PRO Feature', 'dracula-dark-mode')}
                    className={`dracula-tooltip`}
                />
            }

            {!isLiveEdit &&
                <Header title={wp.i18n.__('Colors Settings', 'dracula-dark-mode')}/>
            }

            {/* Color Type */}
            {(menu === 'colors' || !isLiveEdit) &&
                <div className={`settings-field align-base`}>
                    <h4 className={`settings-field-label`}>{wp.i18n.__('Color Generator', 'dracula-dark-mode')}</h4>

                    <div className="settings-field-content">

                        <RadioGroup
                            onChange={colorType => {
                                setData({...data, colorType});
                            }}
                            checked={colorType}
                            className={'dracula-radio-group'}
                        >
                            <Radio value="dynamic">
                                <span className="dracula-btn-icon icon-color-dynamic"></span>
                                <span>{wp.i18n.__('Dynamic', 'dracula-dark-mode')}</span>
                            </Radio>

                            <Radio value="static">
                                <span className="dracula-btn-icon icon-color-presets"></span>
                                <span>{wp.i18n.__('Static', 'dracula-dark-mode')}</span>
                            </Radio>

                        </RadioGroup>

                        <p className="description">
                            {wp.i18n.__('Choose how dark mode colors are created.', 'dracula-dark-mode')}
                            <br/>
                            {wp.i18n.__('Dynamic adapts automatically from your website base color, while Static applies a fixed formula for consistent output.', 'dracula-dark-mode')}
                        </p>
                    </div>
                </div>
            }

            {/*--- Dark to Light ---*/}
            {(menu === 'colors' || !isLiveEdit) &&
                <div className={`settings-field field-draggable-toggle ${!isPro ? 'pro-feature' : ''}`}>
                    <h4 className={`settings-field-label`}>
                        {wp.i18n.__('Dark to Light', 'dracula-dark-mode')}
                        {!isPro && <span className="dracula-pro-icon"><i className={`dracula-icon icon-pro`}/></span>}
                    </h4>
                    <div className="settings-field-content">
                        <FormToggle
                            data-tooltip-content={!isPro ? wp.i18n.__('Pro Feature') : ''}
                            data-tooltip-id={'dark-to-light-pro'}
                            checked={isPro && darkToLight}
                            onChange={() => {
                                if (!isPro) {
                                    showProModal(
                                        wp.i18n.__(`Dark to Light is a pro feature.`, 'dracula-dark-mode'),
                                    );
                                    return;
                                }
                                setData({...data, darkToLight: !darkToLight})
                            }}
                        />

                        {!isPro && <Tooltip id={'dark-to-light-pro'} place="right" effect="solid" variant="warning"
                                            backgroundColor="#ff9966"/>}

                        <p className="description">{wp.i18n.__('Enable this option if your website has a dark theme and you\'d like to offer a light mode', 'dracula-dark-mode')}</p>
                    </div>
                </div>
            }


            {/*  Dark Mode Color Scheme */}
            {(menu === 'colors' || !isLiveEdit) &&
                <div className={`settings-field align-base`}>
                    <h4 className={`settings-field-label`}>{darkToLight ? wp.i18n.__('Light Mode Color Scheme', 'dracula-dark-mode') : wp.i18n.__('Dark Mode Color Scheme', 'dracula-dark-mode')}</h4>

                    <div className="settings-field-content">

                        <RadioGroup
                            onChange={colorMode => {

                                if (!isPro && 'custom' === colorMode) {
                                    showProModal(wp.i18n.__('Custom color scheme is a pro feature. Upgrade to PRO to customize the dark mode colors.', 'dracula-dark-mode'));
                                    return;
                                }

                                setData({...data, colorMode});
                            }}
                            checked={colorMode}
                            className={'dracula-radio-group'}
                        >
                            {colorType === 'dynamic' &&
                                <Radio value="dynamic">
                                    <span className="dracula-btn-icon icon-color-dynamic"></span>
                                    <span>{wp.i18n.__('Auto', 'dracula-dark-mode')}</span>
                                </Radio>
                            }

                            <Radio value="presets">
                                <span className="dracula-btn-icon icon-color-presets"></span>
                                <span>{wp.i18n.__('Presets', 'dracula-dark-mode')}</span>
                            </Radio>

                            <Radio
                                data-tooltip-content={!isPro ? wp.i18n.__('Pro Feature') : ''}
                                data-tooltip-id={'custom-color-pro'}
                                value="custom">
                                <span className="dracula-btn-icon icon-color-custom"></span>
                                <span>{wp.i18n.__('Custom', 'dracula-dark-mode')}</span>
                            </Radio>

                            {!isPro &&
                                <Tooltip id={'custom-color-pro'} effect="solid" place="right" variant={'warning'}/>}

                        </RadioGroup>

                        <p className="description">{wp.i18n.__('Choose how the color scheme of your website changes when dark mode is enabled.', 'dracula-dark-mode')}</p>

                        {/* Color Preset */}
                        {colorMode === 'presets' && (menu === 'colors' || !isLiveEdit) &&
                            <div className={`settings-sub-field sub-field-presets`}>
                                <div className={`settings-field field-color-presets ${!isPro ? 'pro-feature' : ''}`}>
                                    <h4 className={`settings-field-label`}>{wp.i18n.__('Color Preset', 'dracula-dark-mode')}</h4>

                                    <div className="settings-field-content">
                                        <p className="description">{wp.i18n.__('Choose a color preset for the dark mode color scheme.', 'dracula-dark-mode')}</p>

                                        <div className="color-presets">
                                            {presets.map(item => {
                                                const {key, label, colors} = item;
                                                const isActive = key === preset;

                                                let {bg, text} = colors;
                                                let bgLight = lightenDarkenColor(bg, 30);
                                                let link = lightenDarkenColor(text, -40);

                                                if (darkToLight) {
                                                    bg = lightenDarkenColor(text, -10);
                                                    text = lightenDarkenColor(bg, -50);
                                                    bgLight = lightenDarkenColor(bg, 20);
                                                    link = lightenDarkenColor(text, -30);
                                                }

                                                const lockPro = !isPro && !['default', 'dracula'].includes(key);

                                                return (
                                                    <div key={key}
                                                         className={`color-preset ${isActive ? 'active' : ''} ${lockPro ? 'disabled' : ''}`}
                                                         onClick={() => {

                                                             if (!isPro && !['default', 'dracula'].includes(key)) {
                                                                 showProModal(wp.i18n.__(`The ${label} color preset is a pro feature.`, 'dracula-dark-mode'));
                                                                 return;
                                                             }

                                                             setData({...data, preset: key})
                                                         }}>

                                                        <svg
                                                            xmlns="http://www.w3.org/2000/svg" width="200" height="135"
                                                            viewBox="0 0 1600 1071" fill="none">

                                                            <rect width="1600" height="1071" rx="50" fill={bg}/>

                                                            <path className=""
                                                                  d="M0 50C0 22.3858 22.3858 0 50 0H1550C1577.61 0 1600 22.3858 1600 50V196H0V50Z"
                                                                  fill={bgLight}/>

                                                            <rect x="89" y="68" width="246" height="59" rx="29.5"
                                                                  fill={link}/>
                                                            <rect x="375" y="68" width="247" height="59" rx="29.5"
                                                                  fill={link}/>
                                                            <rect x="662" y="68" width="246" height="59" rx="29.5"
                                                                  fill={link}/>
                                                            <rect x="1228" y="68" width="246" height="59" rx="29.5"
                                                                  fill={link}/>

                                                            <text className="preset-name" x="89" y="384"
                                                                  fill={link}>{label}</text>

                                                            <rect x="89" y="470" width="659" height="52" rx="26"
                                                                  fill={text}/>
                                                            <rect x="89" y="577" width="546" height="52" rx="26"
                                                                  fill={text}/>

                                                            <rect x="89" y="727" width="550" height="200" rx="83"
                                                                  fill={bgLight}/>

                                                            <text className={`preset-select`}
                                                                  x={isActive ? '165' : '224'} y="870"
                                                                  fill={text}>{wp.i18n.__('Select', 'dracula-dark-mode')}</text>

                                                            <rect x="908" y="333" width="566" height="600" rx="50"
                                                                  fill={bgLight}/>
                                                            <path
                                                                d="M1295 657.321V669.773C1295 671.839 1294.91 673.924 1294.73 675.99C1292.2 704.422 1272.45 723 1243.96 723H1166.04C1151.61 723 1139.44 718.317 1130.69 709.826C1127.26 706.677 1124.38 702.978 1122.12 698.737C1125.1 695.119 1128.44 691.158 1131.68 687.188C1137.18 680.601 1142.5 674.095 1145.84 669.863C1150.8 663.728 1163.88 647.576 1182 655.156C1185.7 656.69 1188.95 658.855 1191.92 660.75C1199.23 665.622 1202.29 667.066 1207.43 664.269C1213.12 661.201 1216.81 655.156 1220.69 648.84C1222.76 645.492 1224.84 642.253 1227.09 639.275C1236.92 626.462 1252.07 623.034 1264.7 630.613C1271.01 634.403 1276.42 639.185 1281.47 644.048C1282.55 645.131 1283.64 646.133 1284.63 647.125C1285.98 648.479 1290.49 652.99 1295 657.321Z"
                                                                fill={text}/>
                                                            <path opacity="0.4"
                                                                  d="M1244.05 543H1166.04C1135.47 543 1115 564.385 1115 596.227V669.774C1115 680.863 1117.53 690.707 1122.12 698.738C1125.1 695.12 1128.44 691.159 1131.68 687.179C1137.18 680.602 1142.51 674.096 1145.84 669.864C1150.8 663.728 1163.88 647.577 1182 655.157C1185.7 656.69 1188.95 658.856 1191.92 660.751C1199.23 665.623 1202.29 667.067 1207.43 664.261C1213.12 661.202 1216.81 655.157 1220.69 648.831C1222.77 645.493 1224.84 642.254 1227.09 639.276C1236.92 626.463 1252.07 623.034 1264.7 630.614C1271.01 634.404 1276.42 639.186 1281.47 644.049C1282.56 645.132 1283.64 646.134 1284.63 647.126C1285.98 648.47 1290.49 652.982 1295 657.322V596.227C1295 564.385 1274.53 543 1244.05 543Z"
                                                                  fill={text}/>
                                                            <path
                                                                d="M1200.09 604.17C1200.09 616.847 1189.53 627.404 1176.86 627.404C1164.2 627.404 1153.64 616.847 1153.64 604.17C1153.64 591.502 1164.2 580.936 1176.86 580.936C1189.53 580.936 1200.09 591.502 1200.09 604.17Z"
                                                                fill={text}/>
                                                        </svg>

                                                        {isActive &&
                                                            <i className="active-badge dashicons dashicons-saved"></i>}

                                                        {lockPro &&
                                                            <span className="pro-label">
                                                                <i className={`dracula-icon icon-pro`}></i>
                                                                {wp.i18n.__('Pro', 'dracula-dark-mode')}
                                                            </span>
                                                        }

                                                    </div>
                                                )
                                            })}
                                        </div>
                                    </div>
                                </div>
                            </div>
                        }

                        {/* Custom preset settings */}
                        {isPro && 'custom' === colorMode && (menu === 'colors' || !isLiveEdit) &&
                            <div className={`settings-sub-field sub-field-custom-presets`}>

                                {/* Custom color presets */}
                                <div className="settings-field field-custom-presets">
                                    <h4 className={`settings-field-label`}>{wp.i18n.__('Custom Color Presets', 'dracula-dark-mode')}</h4>

                                    <div className="settings-field-content">
                                        <div className="settings-wrap">
                                            <div className="custom-presets-wrap">
                                                <div className="custom-presets-item">
                                                    {
                                                        customPresets.map((item, index) => {
                                                            return (
                                                                <>
                                                                    <div
                                                                        className={`color-preset ${activeCustomPreset === item.id ? 'active' : ''}`}
                                                                        key={index}>
                                                                        <svg
                                                                            onClick={() => setData({
                                                                                ...data,
                                                                                activeCustomPreset: item.id
                                                                            })}
                                                                            xmlns="http://www.w3.org/2000/svg"
                                                                            width="200" height="135"
                                                                            viewBox="0 0 1600 1071" fill="none">

                                                                            <rect width="1600" height="1071" rx="50"
                                                                                  fill={item?.colors?.bg}/>

                                                                            <path className=""
                                                                                  d="M0 50C0 22.3858 22.3858 0 50 0H1550C1577.61 0 1600 22.3858 1600 50V196H0V50Z"
                                                                                  fill={lightenDarkenColor(item?.colors?.bg, 30)}/>

                                                                            <rect x="89" y="68" width="246" height="59"
                                                                                  rx="29.5" fill={item?.colors?.link}/>
                                                                            <rect x="375" y="68" width="247" height="59"
                                                                                  rx="29.5" fill={item?.colors?.link}/>
                                                                            <rect x="662" y="68" width="246" height="59"
                                                                                  rx="29.5" fill={item?.color?.link}/>
                                                                            <rect x="1228" y="68" width="246"
                                                                                  height="59" rx="29.5"
                                                                                  fill={item?.colors?.link}/>

                                                                            <text className="preset-name" x="89" y="384"
                                                                                  fill={item?.colors?.link}>{item?.label}</text>

                                                                            <rect x="89" y="470" width="659" height="52"
                                                                                  rx="26" fill={item?.colors?.text}/>
                                                                            <rect x="89" y="577" width="546" height="52"
                                                                                  rx="26" fill={item?.colors?.text}/>

                                                                            <rect x="89" y="727" width="550"
                                                                                  height="200" rx="83"
                                                                                  fill={item?.colors?.button?.bg}/>

                                                                            <text className={`preset-select`}
                                                                                  x={'165'} y="870"
                                                                                  fill={item?.colors?.button?.text}>{wp.i18n.__('Select', 'dracula-dark-mode')}</text>

                                                                            <rect x="908" y="333" width="566"
                                                                                  height="600" rx="50"
                                                                                  fill={lightenDarkenColor(item?.colors?.bg, 30)}/>
                                                                            <path
                                                                                d="M1295 657.321V669.773C1295 671.839 1294.91 673.924 1294.73 675.99C1292.2 704.422 1272.45 723 1243.96 723H1166.04C1151.61 723 1139.44 718.317 1130.69 709.826C1127.26 706.677 1124.38 702.978 1122.12 698.737C1125.1 695.119 1128.44 691.158 1131.68 687.188C1137.18 680.601 1142.5 674.095 1145.84 669.863C1150.8 663.728 1163.88 647.576 1182 655.156C1185.7 656.69 1188.95 658.855 1191.92 660.75C1199.23 665.622 1202.29 667.066 1207.43 664.269C1213.12 661.201 1216.81 655.156 1220.69 648.84C1222.76 645.492 1224.84 642.253 1227.09 639.275C1236.92 626.462 1252.07 623.034 1264.7 630.613C1271.01 634.403 1276.42 639.185 1281.47 644.048C1282.55 645.131 1283.64 646.133 1284.63 647.125C1285.98 648.479 1290.49 652.99 1295 657.321Z"
                                                                                fill={item?.colors?.text}/>
                                                                            <path opacity="0.4"
                                                                                  d="M1244.05 543H1166.04C1135.47 543 1115 564.385 1115 596.227V669.774C1115 680.863 1117.53 690.707 1122.12 698.738C1125.1 695.12 1128.44 691.159 1131.68 687.179C1137.18 680.602 1142.51 674.096 1145.84 669.864C1150.8 663.728 1163.88 647.577 1182 655.157C1185.7 656.69 1188.95 658.856 1191.92 660.751C1199.23 665.623 1202.29 667.067 1207.43 664.261C1213.12 661.202 1216.81 655.157 1220.69 648.831C1222.77 645.493 1224.84 642.254 1227.09 639.276C1236.92 626.463 1252.07 623.034 1264.7 630.614C1271.01 634.404 1276.42 639.186 1281.47 644.049C1282.56 645.132 1283.64 646.134 1284.63 647.126C1285.98 648.47 1290.49 652.982 1295 657.322V596.227C1295 564.385 1274.53 543 1244.05 543Z"
                                                                                  fill={item?.colors?.text}/>
                                                                            <path
                                                                                d="M1200.09 604.17C1200.09 616.847 1189.53 627.404 1176.86 627.404C1164.2 627.404 1153.64 616.847 1153.64 604.17C1153.64 591.502 1164.2 580.936 1176.86 580.936C1189.53 580.936 1200.09 591.502 1200.09 604.17Z"
                                                                                fill={item?.colors?.text}/>
                                                                        </svg>

                                                                        <i className="active-badge dashicons dashicons-saved"></i>

                                                                        <span
                                                                            className="delete-preset dashicons dashicons-trash"
                                                                            onClick={() => {
                                                                                const updatedCustomPresets = customPresets.filter((_, currentIndex) => currentIndex !== index);
                                                                                setData({
                                                                                    ...data,
                                                                                    customPresets: updatedCustomPresets
                                                                                });
                                                                            }}></span>
                                                                    </div>
                                                                </>)
                                                        })
                                                    }

                                                    <div className="custom-presets-btn" onClick={addCustomPreset}>
                                                        <img
                                                            src={`${dracula.pluginUrl}/assets/images/settings/color-plus.svg`}/>
                                                    </div>

                                                </div>
                                            </div>
                                        </div>
                                    </div>
                                </div>

                                {customPresets.length > 0 &&
                                    <>
                                        {/* Preset Name */}
                                        <div
                                            className={`settings-field field-preset-name ${!isPro ? 'pro-feature' : ''}`}>
                                            <h4 className={`settings-field-label`}>{wp.i18n.__('Preset Name', 'dracula-dark-mode')}</h4>

                                            <div className="settings-field-content">
                                                <div className="settings-wrap">
                                                    <TextControl
                                                        value={activePreset?.label}
                                                        onChange={(label) => {
                                                            const updatePresetsData = customPresets.map((item) => {
                                                                if (item?.id == activeCustomPreset) {
                                                                    return {...item, label: label}
                                                                } else {
                                                                    return item;
                                                                }
                                                            });

                                                            setData({...data, customPresets: updatePresetsData});
                                                        }}
                                                        className={`dracula-text-control`}
                                                    />
                                                </div>

                                                <p className="description">{wp.i18n.__('Set the custom preset name.', 'dracula-dark-mode')}</p>
                                            </div>
                                        </div>

                                        {/* Background Color */}
                                        <CustomColorField
                                            value={activePreset?.colors?.bg}
                                            onChange={bg => {
                                                if (!isPro) {
                                                    return;
                                                }

                                                const updatePresetsData = customPresets.map((item) => {
                                                    if (item.id == activeCustomPreset) {
                                                        return {
                                                            ...item,
                                                            colors: {
                                                                ...item.colors,
                                                                bg: bg
                                                            }
                                                        }
                                                    } else {
                                                        return item;
                                                    }
                                                });

                                                setData({...data, customPresets: updatePresetsData})
                                            }}
                                            label={wp.i18n.__('Background Color', 'dracula-dark-mode')}
                                            description={wp.i18n.__('Select background color for dark mode.', 'dracula-dark-mode')}
                                        />


                                        {/* Text Color */}
                                        <CustomColorField
                                            value={activePreset?.colors?.text}
                                            onChange={(text) => {
                                                if (!isPro) {
                                                    return;
                                                }

                                                const updatePresetsText = customPresets.map((item) => {
                                                    if (item.id == activeCustomPreset) {
                                                        return {
                                                            ...item,
                                                            colors: {
                                                                ...item.colors,
                                                                text: text
                                                            }
                                                        }
                                                    } else {
                                                        return item;
                                                    }
                                                });

                                                setData({...data, customPresets: updatePresetsText})
                                            }}
                                            label={wp.i18n.__('Text Color', 'dracula-dark-mode')}
                                            description={wp.i18n.__('Select text color for dark mode.', 'dracula-dark-mode')}
                                        />

                                        {/* Link Color */}
                                        <CustomColorField
                                            value={activePreset?.colors?.link}
                                            onChange={(link) => {
                                                if (!isPro) {
                                                    return;
                                                }

                                                const updatePresetsLinkColor = customPresets.map((item) => {
                                                    if (item.id == activeCustomPreset) {
                                                        return {
                                                            ...item,
                                                            colors: {
                                                                ...item.colors,
                                                                link: link
                                                            }
                                                        }
                                                    } else {
                                                        return item;
                                                    }
                                                });

                                                setData({...data, customPresets: updatePresetsLinkColor})

                                            }}
                                            label={wp.i18n.__('Link Color', 'dracula-dark-mode')}
                                            description={wp.i18n.__('Select link color for dark mode.', 'dracula-dark-mode')}
                                        />


                                        {/* Button Background Color */}
                                        <CustomColorField
                                            value={activePreset?.colors?.btn_bg}
                                            onChange={buttonBgColor => {
                                                if (!isPro) {
                                                    return;
                                                }

                                                const updatePresetsButtonBg = customPresets.map((item) => {
                                                    if (item.id == activeCustomPreset) {
                                                        return {
                                                            ...item,
                                                            colors: {
                                                                ...item.colors,
                                                                btn_bg: buttonBgColor
                                                            }
                                                        }
                                                    } else {
                                                        return item;
                                                    }
                                                });

                                                setData({...data, customPresets: updatePresetsButtonBg})
                                            }}
                                            label={wp.i18n.__('Button Background Color', 'dracula-dark-mode')}
                                            description={wp.i18n.__('Select button background color for dark mode.', 'dracula-dark-mode')}
                                        />

                                        {/* Button Text Color */}
                                        <CustomColorField
                                            value={activePreset?.colors?.btn_text}
                                            label={wp.i18n.__('Button Text Color', 'dracula-dark-mode')}
                                            description={wp.i18n.__('Select button text color for dark mode.', 'dracula-dark-mode')}
                                            onChange={(buttonTextColor) => {
                                                if (!isPro) {
                                                    return;
                                                }

                                                const updatePresetsButtonText = customPresets.map((item) => {
                                                    if (item.id == activeCustomPreset) {
                                                        return {
                                                            ...item,
                                                            colors: {
                                                                ...item?.colors,
                                                                btn_text: buttonTextColor
                                                            }
                                                        }
                                                    } else {
                                                        return item;
                                                    }
                                                });

                                                setData({...data, customPresets: updatePresetsButtonText})

                                            }}
                                        />

                                        {/* Input Background Color */}
                                        <CustomColorField
                                            value={activePreset?.colors?.input_bg}
                                            label={wp.i18n.__('Input Background Color', 'dracula-dark-mode')}
                                            description={wp.i18n.__('Select input background color for dark mode.', 'dracula-dark-mode')}
                                            onChange={(inputBgColor) => {
                                                if (!isPro) {
                                                    return;
                                                }

                                                const updatePresetsInputBg = customPresets.map((item) => {
                                                    if (item.id == activeCustomPreset) {
                                                        return {
                                                            ...item,
                                                            colors: {
                                                                ...item?.colors,
                                                                input_bg: inputBgColor
                                                            }
                                                        }
                                                    } else {
                                                        return item;
                                                    }
                                                });

                                                setData({...data, customPresets: updatePresetsInputBg})
                                            }}
                                        />

                                        {/* Input Text Color */}
                                        <CustomColorField
                                            value={activePreset?.colors?.input_text}
                                            label={wp.i18n.__('Input Text Color', 'dracula-dark-mode')}
                                            description={wp.i18n.__('Select input text color for dark mode.', 'dracula-dark-mode')}
                                            onChange={(inputTextColor) => {
                                                if (!isPro) {
                                                    return;
                                                }

                                                const updatePresetsInputText = customPresets.map((item) => {
                                                    if (item.id == activeCustomPreset) {
                                                        return {
                                                            ...item,
                                                            colors: {
                                                                ...item?.colors,
                                                                input_text: inputTextColor
                                                            }
                                                        }
                                                    } else {
                                                        return item;
                                                    }
                                                });

                                                setData({...data, customPresets: updatePresetsInputText});

                                            }}
                                        />

                                        {/* Border Color */}
                                        <CustomColorField
                                            value={activePreset?.colors?.border}
                                            label={wp.i18n.__('Border Color', 'dracula-dark-mode')}
                                            description={wp.i18n.__('Select border color for dark mode.', 'dracula-dark-mode')}
                                            onChange={(borderColor) => {
                                                if (!isPro) {
                                                    return;
                                                }

                                                const updatePresetsBorder = customPresets.map((item) => {
                                                    if (item.id == activeCustomPreset) {
                                                        return {
                                                            ...item,
                                                            colors: {
                                                                ...item?.colors,
                                                                border: borderColor
                                                            }
                                                        }
                                                    } else {
                                                        return item;
                                                    }
                                                });

                                                setData({...data, customPresets: updatePresetsBorder});

                                            }}
                                        />

                                    </>
                                }
                            </div>
                        }


                    </div>
                </div>
            }

            {/* Scrollbar Dark Mode */}
            {(menu === 'colors' || !isLiveEdit) &&
                <div className="settings-field">
                    <h4 className={`settings-field-label`}>{wp.i18n.__('Scrollbar Dark Mode', 'dracula-dark-mode')}</h4>
                    <div className="settings-field-content">

                        <RadioGroup
                            onChange={scrollbarDarkMode => {

                                if (!isPro && 'custom' === scrollbarDarkMode) {
                                    showProModal(wp.i18n.__('Custom color scheme is a pro feature. Upgrade to PRO to customize the dark mode colors.', 'dracula-dark-mode'));
                                    return;
                                }

                                setData({...data, scrollbarDarkMode});
                            }}
                            checked={scrollbarDarkMode}
                            className={'dracula-radio-group'}
                        >
                            <Radio value="disabled">
                                <span>{wp.i18n.__('Disabled', 'dracula-dark-mode')}</span>
                            </Radio>

                            <Radio value="auto">
                                <span>{wp.i18n.__('Auto', 'dracula-dark-mode')}</span>
                            </Radio>

                            <Radio
                                data-tooltip-content={!isPro ? wp.i18n.__('Pro Feature') : ''}
                                data-tooltip-id={'custom-scrollbar-color-pro'}
                                value="custom">
                                <span>{wp.i18n.__('Custom', 'dracula-dark-mode')}</span>
                            </Radio>

                            {!isPro && <Tooltip id={'custom-scrollbar-color-pro'} effect="solid" place="right"
                                                variant="warning" backgroundColor="#ff9966"/>}

                        </RadioGroup>

                        <p className={`description`}>{wp.i18n.__('Set the scrollbar color for dark mode.', 'dracula-dark-mode')}</p>
                    </div>

                    {/* Scrollbar Color */}
                    {scrollbarDarkMode === 'custom' &&
                        <div className="settings-field-sub">
                            <div className="settings-field">
                                <h4 className={`settings-field-label`}>{wp.i18n.__('Scrollbar Color', 'dracula-dark-mode')}</h4>

                                <div className="settings-field-content">
                                    <div className="settings-wrap"
                                         data-tooltip-content={!isPro ? wp.i18n.__('PRO Feature', 'dracula-dark-mode') : ''}
                                         onClick={() => !isPro && showProModal(wp.i18n.__(`Upgrade to PRO to set custom scrollbar color`, 'dracula-dark-mode'))}>

                                        <ColorPickerWrapper
                                            value={scrollbarColor}
                                            onChange={scrollbarColor => {
                                                if (!isPro) {
                                                    return;
                                                }

                                                setData({...data, scrollbarColor})
                                            }}
                                        />

                                        {!isPro &&
                                            <Tooltip effect="solid" place="left" backgroundColor="#ff9966"
                                                     className={'dracula-tooltip'}/>}
                                    </div>

                                    <p className={`description`}>{wp.i18n.__('Set custom scrollbar color for dark mode.', 'dracula-dark-mode')}</p>
                                </div>

                            </div>
                        </div>
                    }

                </div>
            }

            {/* Adjustments */}
            {(menu === 'adjustments' || (!isLiveEdit && colorType === 'dynamic')) &&
                <div className="settings-field field-adjustments" onClick={(e) => e.stopPropagation()}>

                    {!isLiveEdit &&
                        <h4 className={`settings-field-label`}>{wp.i18n.__('Color Adjustments', 'dracula-dark-mode')}</h4>
                    }

                    {!isLiveEdit && <LiveEditButton tab={'colors'}/>}

                    <div className="settings-field-content">

                        {/*  Brightness */}
                        <div className={`settings-field`}>
                            <h4 className={`settings-field-label`}>
                                <svg width="15" height="15" viewBox="0 0 8 8" fill="none"
                                     xmlns="http://www.w3.org/2000/svg">
                                    <path
                                        d="M1.9913 3.99506C1.9913 5.10174 2.8854 5.99896 3.99208 6.00209C5.10189 6.00522 6.00849 5.09861 6.00536 3.98881C6.00224 2.88213 5.10189 1.98803 3.99521 1.99116C2.8854 1.99428 1.9913 2.88838 1.9913 3.99506Z"
                                        fill="#6D6A7E"/>
                                    <path
                                        d="M3.71707 0.687767C3.71707 0.806563 3.71394 0.925359 3.71707 1.04728C3.72019 1.23485 3.83586 1.36303 3.99217 1.36616C4.14849 1.36928 4.27041 1.24111 4.27353 1.05666C4.27666 0.806563 4.27666 0.559592 4.27353 0.309495C4.27041 0.125049 4.15161 0 3.99217 0C3.83274 0.00312621 3.72332 0.128175 3.71707 0.315748C3.71394 0.43767 3.71707 0.562718 3.71707 0.687767Z"
                                        fill="#6D6A7E"/>
                                    <path
                                        d="M7.32155 4.2764C7.4466 4.2764 7.57165 4.27953 7.6967 4.2764C7.87489 4.27015 7.99682 4.1576 7.99994 4.00442C8.00307 3.84498 7.87802 3.72306 7.69357 3.71993C7.44348 3.71681 7.19651 3.71681 6.94641 3.71993C6.76196 3.72306 6.63379 3.84186 6.63379 3.99817C6.63379 4.15448 6.76509 4.27015 6.94953 4.27327C7.07146 4.27953 7.19651 4.2764 7.32155 4.2764Z"
                                        fill="#6D6A7E"/>
                                    <path
                                        d="M0.684838 4.2764C0.803634 4.2764 0.922431 4.27953 1.04435 4.2764C1.2288 4.27327 1.3601 4.15761 1.36323 4.00129C1.36635 3.84498 1.23505 3.72306 1.05373 3.71993C0.803634 3.71681 0.556663 3.71681 0.306566 3.71993C0.12212 3.72306 -0.00292909 3.84186 0.000197053 4.00129C0.00332332 4.16073 0.128372 4.27015 0.315945 4.2764C0.322197 4.2764 0.325324 4.2764 0.331576 4.2764C0.447246 4.2764 0.566042 4.2764 0.684838 4.2764Z"
                                        fill="#6D6A7E"/>
                                    <path
                                        d="M3.71698 7.32142C3.71698 7.44647 3.71385 7.57152 3.71698 7.69657C3.72323 7.87476 3.83265 7.99668 3.98583 7.99981C4.14527 8.00606 4.26719 7.87789 4.27032 7.69657C4.27344 7.44334 4.27657 7.18699 4.27032 6.93377C4.26719 6.7587 4.14527 6.63678 3.99208 6.63678C3.84202 6.63678 3.72323 6.76183 3.71385 6.9369C3.71385 6.94628 3.71385 6.95878 3.71385 6.96816C3.71698 7.0807 3.71698 7.1995 3.71698 7.32142Z"
                                        fill="#6D6A7E"/>
                                    <path
                                        d="M1.06952 6.66484C1.12267 6.72737 1.16956 6.82741 1.24459 6.86492C1.31649 6.89931 1.44779 6.89618 1.50094 6.84929C1.73228 6.64296 1.95424 6.421 2.1637 6.19279C2.25436 6.09275 2.22935 5.9677 2.13556 5.86766C2.04177 5.77075 1.9136 5.73323 1.81356 5.82389C1.57597 6.0396 1.35401 6.27407 1.12892 6.50541C1.10078 6.53354 1.09766 6.58982 1.06952 6.66484Z"
                                        fill="#6D6A7E"/>
                                    <path
                                        d="M6.66207 1.0564C6.56828 1.11267 6.49325 1.1408 6.44011 1.19082C6.25253 1.36902 6.06809 1.55346 5.88989 1.74104C5.76797 1.86609 5.76484 2.01927 5.87114 2.12869C5.97743 2.24123 6.13686 2.24436 6.25879 2.12556C6.45261 1.93799 6.65269 1.75354 6.82776 1.55034C6.88715 1.48156 6.90904 1.34401 6.8809 1.25647C6.85277 1.17832 6.74022 1.12517 6.66207 1.0564Z"
                                        fill="#6D6A7E"/>
                                    <path
                                        d="M1.04731 1.34406C1.10359 1.44098 1.13172 1.51601 1.18174 1.56915C1.35681 1.7536 1.53813 1.93179 1.71945 2.10999C1.85388 2.24129 2.01019 2.24754 2.11961 2.135C2.23215 2.02245 2.22902 1.86614 2.10085 1.73484C1.9164 1.54414 1.72883 1.35969 1.53813 1.17212C1.45372 1.08771 1.34431 1.0502 1.24114 1.11898C1.16611 1.17525 1.11609 1.26278 1.04731 1.34406Z"
                                        fill="#6D6A7E"/>
                                    <path
                                        d="M5.75844 6.03338C5.81158 6.12404 5.83347 6.19594 5.88036 6.24596C6.06168 6.43666 6.24925 6.62424 6.43995 6.80868C6.565 6.93061 6.71506 6.93373 6.8276 6.82431C6.93389 6.72115 6.93702 6.55233 6.82135 6.43354C6.6369 6.24284 6.44933 6.05527 6.25863 5.87082C6.1711 5.78641 6.05855 5.73327 5.94914 5.80829C5.87411 5.86457 5.82409 5.95523 5.75844 6.03338Z"
                                        fill="#6D6A7E"/>
                                </svg>
                                {wp.i18n.__('Brightness', 'dracula-dark-mode')}
                            </h4>
                            <div className="settings-field-content">
                                <div className="settings-wrap">
                                    <RangeControl
                                        value={parseInt(brightness)}
                                        step={5}
                                        onChange={brightness => {
                                            setData({...data, brightness: parseInt(brightness)})
                                        }}
                                        min={0}
                                        max={100}
                                        allowReset={true}
                                        resetFallbackValue={100}
                                        className={'dracula-range-control'}
                                    />
                                </div>

                                <p className="description">{wp.i18n.__('Adjust brightness of dark mode color scheme.', 'dracula-dark-mode')}</p>
                            </div>
                        </div>

                        {/*  Contrast */}
                        <div className={`settings-field`}>
                            <h4 className={`settings-field-label`}>
                                <svg width="15" height="15" viewBox="0 0 8 8" fill="none"
                                     xmlns="http://www.w3.org/2000/svg">
                                    <g clip-path="url(#clip0_4014_11909)">
                                        <path
                                            d="M3.99984 7.33335C3.53873 7.33335 3.10539 7.2458 2.69984 7.07069C2.29428 6.89558 1.9415 6.65813 1.6415 6.35835C1.3415 6.05835 1.10406 5.70558 0.929171 5.30002C0.754282 4.89446 0.666726 4.46113 0.666504 4.00002C0.666504 3.53891 0.754059 3.10558 0.929171 2.70002C1.10428 2.29446 1.34173 1.94169 1.6415 1.64169C1.9415 1.34169 2.29428 1.10424 2.69984 0.929354C3.10539 0.754465 3.53873 0.666909 3.99984 0.666687C4.46095 0.666687 4.89428 0.754243 5.29984 0.929354C5.70539 1.10446 6.05817 1.34191 6.35817 1.64169C6.65817 1.94169 6.89573 2.29446 7.07084 2.70002C7.24595 3.10558 7.33339 3.53891 7.33317 4.00002C7.33317 4.46113 7.24561 4.89446 7.0705 5.30002C6.89539 5.70558 6.65795 6.05835 6.35817 6.35835C6.05817 6.65835 5.70539 6.89591 5.29984 7.07102C4.89428 7.24613 4.46095 7.33358 3.99984 7.33335ZM4.33317 6.64169C4.99428 6.55835 5.5485 6.26802 5.99584 5.77069C6.44317 5.27335 6.66673 4.68313 6.6665 4.00002C6.6665 3.31669 6.44284 2.72646 5.9955 2.22935C5.54817 1.73224 4.99406 1.44191 4.33317 1.35835V6.64169Z"
                                            fill="#6D6A7E"/>
                                    </g>
                                    <defs>
                                        <clipPath id="clip0_4014_11909">
                                            <rect width="8" height="8" fill="white"/>
                                        </clipPath>
                                    </defs>
                                </svg>
                                {wp.i18n.__('Contrast', 'dracula-dark-mode')}
                            </h4>
                            <div className="settings-field-content">

                                <div className="settings-wrap">
                                    <RangeControl
                                        value={parseInt(contrast)}
                                        onChange={contrast => {
                                            setData({...data, contrast: parseInt(contrast)})
                                        }}
                                        min={0}
                                        max={100}
                                        step={5}
                                        allowReset={true}
                                        resetFallbackValue={90}
                                        className={'dracula-range-control'}
                                    />
                                </div>

                                <p className="description">{wp.i18n.__('Set the contrast of the dark mode color scheme.', 'dracula-dark-mode')}</p>
                            </div>
                        </div>

                        {/*  Sepia */}
                        <div className={`settings-field`}>
                            <h4 className={`settings-field-label`}>
                                <svg width="15" height="15" viewBox="0 0 8 8" fill="none"
                                     xmlns="http://www.w3.org/2000/svg">
                                    <path
                                        d="M7.81966 3.96708C7.81966 4.46905 7.72079 4.9661 7.5287 5.42986C7.3366 5.89362 7.05504 6.315 6.7001 6.66995C6.34515 7.02489 5.92377 7.30645 5.46001 7.49855C4.99625 7.69064 4.4992 7.78951 3.99723 7.78951C3.49526 7.78951 2.99821 7.69064 2.53445 7.49855C2.07069 7.30645 1.64931 7.02489 1.29437 6.66995C0.939422 6.315 0.657864 5.89362 0.465769 5.42986C0.273674 4.9661 0.174805 4.46905 0.174805 3.96708C0.174805 3.46511 0.273674 2.96806 0.465769 2.5043C0.657864 2.04054 0.939422 1.61916 1.29437 1.26422C1.64931 0.909271 2.07069 0.627713 2.53445 0.435618C2.99821 0.243523 3.49526 0.144653 3.99723 0.144653C4.4992 0.144653 4.99625 0.243523 5.46001 0.435618C5.92377 0.627713 6.34515 0.909271 6.7001 1.26422C7.05504 1.61916 7.3366 2.04054 7.5287 2.5043C7.72079 2.96806 7.81966 3.46511 7.81966 3.96708Z"
                                        fill="#6E6D7E"/>
                                    <path
                                        d="M6.93002 5.47358C6.66378 6.01354 6.25554 6.47082 5.74909 6.79634C5.24265 7.12186 4.65713 7.30333 4.05536 7.32128C3.45359 7.33923 2.85829 7.19298 2.33335 6.89822C1.80841 6.60347 1.37363 6.17133 1.07568 5.64819L3.95652 4.00745L6.93002 5.47358Z"
                                        fill="#E7E7E9"/>
                                    <path opacity="0.3"
                                          d="M3.69043 0.702836C4.27649 0.655644 4.86457 0.76516 5.39435 1.02015C5.92414 1.27515 6.37659 1.66645 6.7053 2.15394C7.03402 2.64143 7.22718 3.20756 7.26498 3.79431C7.30278 4.38105 7.18386 4.9673 6.92041 5.49293L3.95653 4.00745L3.69043 0.702836Z"
                                          fill="#6D6A7E"/>
                                    <path
                                        d="M1.10553 5.69969C0.806962 5.19669 0.646785 4.62366 0.641261 4.03875C0.635737 3.45384 0.785062 2.87789 1.07408 2.36934C1.3631 1.8608 1.78152 1.43777 2.28686 1.14321C2.79221 0.84864 3.3665 0.693018 3.95144 0.692139L3.95642 4.00744L1.10553 5.69969Z"
                                        fill="#6D6A7E"/>
                                </svg>
                                {wp.i18n.__('Sepia', 'dracula-dark-mode')}
                            </h4>
                            <div className="settings-field-content">

                                <div className="settings-wrap">
                                    <RangeControl
                                        value={parseInt(sepia)}
                                        onChange={sepia => {
                                            setData({...data, sepia: parseInt(sepia)})
                                        }}
                                        min={0}
                                        max={100}
                                        step={5}
                                        allowReset={true}
                                        resetFallbackValue={10}
                                        className={'dracula-range-control'}
                                    />
                                </div>

                                <p className="description">{wp.i18n.__('Adjust the sepia of the dark mode color scheme.', 'dracula-dark-mode')}</p>
                            </div>
                        </div>

                        {/*  Grayscale */}
                        <div className={`settings-field`}>
                            <h4 className={`settings-field-label`}>
                                <svg width="15" height="15" viewBox="0 0 8 8" fill="none"
                                     xmlns="http://www.w3.org/2000/svg">
                                    <ellipse cx="5.24153" cy="2.87715" rx="2.75862" ry="2.80702" fill="#E7E7E9"/>
                                    <ellipse cx="2.75862" cy="2.80702" rx="2.75862" ry="2.80702" fill="#6D6A7E"
                                             fillOpacity="0.8"/>
                                    <ellipse cx="4.06917" cy="5.193" rx="2.75862" ry="2.80702" fill="#6E6D7E"
                                             fillOpacity="0.8"/>
                                </svg>
                                {wp.i18n.__('Grayscale', 'dracula-dark-mode')}
                            </h4>
                            <div className="settings-field-content">

                                <div className="settings-wrap">
                                    <RangeControl
                                        value={parseInt(grayscale)}
                                        onChange={grayscale => {
                                            setData({...data, grayscale: parseInt(grayscale)})
                                        }}
                                        min={0}
                                        max={100}
                                        step={5}
                                        allowReset={true}
                                        resetFallbackValue={0}
                                        className={'dracula-range-control'}
                                    />
                                </div>


                                <p className="description">{wp.i18n.__('Adjust the grayscale of the dark mode color scheme.', 'dracula-dark-mode')}</p>
                            </div>
                        </div>

                    </div>
                </div>
            }

        </div>
    )
}