import {Tooltip} from "react-tooltip";
import SettingsContext from "../../contexts/SettingsContext";
import Header from "./Header";
import {showProModal} from "../ProModal";

const {FormToggle} = wp.components;

const {useContext} = React;

const {isPro} = dracula;

export default function Advanced() {
    const context = useContext(SettingsContext);

    const {data, setData} = context;

    const {
        keyboardShortcuts = false,
        performanceMode = false,
        timeBasedMode = false,
        timeBasedModeStart = '19:00',
        timeBasedModeEnd = '07:00',
        urlParameter = false,
    } = data;

    return (
        <div className="dracula-settings-content">

            {!isPro &&
                <Tooltip
                    anchorSelect={'.dracula-pro-icon, .pro-disabled'}
                    place="right"
                    variant="warning"
                    content={wp.i18n.__('PRO Feature', 'dracula-dark-mode')}
                    className={`dracula-tooltip`}
                />
            }

            <Header title={wp.i18n.__('Advanced Settings', 'dracula-dark-mode')}/>

            {/* Time Based Dark Mode */}
            <div className="settings-field field-time-based">
                <h4 className={`settings-field-label`}>
                    {wp.i18n.__('Time Based Dark Mode', 'dracula-dark-mode')}
                </h4>

                <div className="settings-field-content">
                    <FormToggle
                        checked={timeBasedMode}
                        onChange={() => setData({...data, timeBasedMode: !timeBasedMode})}
                    />

                    {timeBasedMode &&
                        <div className="time-based-mode-settings">
                            <span>{wp.i18n.__('Start Time', 'dracula-dark-mode')}</span>
                            <input type="time" value={timeBasedModeStart}
                                   onChange={(e) => setData({...data, timeBasedModeStart: e.target.value})}/>
                            <span>{wp.i18n.__('End Time', 'dracula-dark-mode')}</span>
                            <input type="time" value={timeBasedModeEnd}
                                   onChange={(e) => setData({...data, timeBasedModeEnd: e.target.value})}/>
                        </div>
                    }

                    <p className="description">
                        {wp.i18n.__('Enable the time-based dark mode to switch your website to dark mode during specific hours, based on the user\'s local time.', 'dracula-dark-mode')}
                    </p>
                </div>
            </div>

            {/* Performance Mode */}
            <div className={`settings-field ${!isPro ? 'pro-feature' : ''}`}>
                <h4 className={`settings-field-label`}>
                    {wp.i18n.__('Performance Mode', 'dracula-dark-mode')}

                    {!isPro &&
                        <span className="dracula-pro-icon">
                            <i className={`dracula-icon icon-pro`}/>
                        </span>
                    }

                </h4>
                <div className="settings-field-content">

                    <div className="settings-wrap"
                         data-tooltip-content={!isPro ? wp.i18n.__('PRO Feature', 'dracula-dark-mode') : ''}
                         data-tooltip-id="performance-mode">
                        <FormToggle
                            checked={performanceMode}
                            onChange={() => {
                                if (!isPro) {
                                    showProModal(wp.i18n.__('Upgrade to PRO to enable performance mode.', 'dracula-dark-mode'));
                                    return;
                                }

                                setData({...data, performanceMode: !performanceMode})
                            }}
                        />
                    </div>

                    {!isPro &&
                        <Tooltip id="performance-mode" place="right" effect="solid" backgroundColor="#ff9966"/>}

                    <p className="description">By default the dark mode scripts files loaded before the page is
                        loaded to prevent the dark mode flashing issue.
                        <br/>
                        If the performance mode is enabled, the dark mode scripts will be loaded parallel to the page
                        load and executed after the page is loaded.
                    </p>
                </div>
            </div>

            {/* URL Parameter */}
            <div className={`settings-field ${!isPro ? 'pro-feature' : ''}`}>
                <h4 className={`settings-field-label`}>
                    {wp.i18n.__('URL Parameter', 'dracula-dark-mode')}

                    {!isPro &&
                        <span className="dracula-pro-icon">
                            <i className={`dracula-icon icon-pro`}/>
                        </span>
                    }

                </h4>
                <div className="settings-field-content">

                    <div className="settings-wrap"
                         data-tooltip-content={!isPro ? wp.i18n.__('PRO Feature', 'dracula-dark-mode') : ''}
                         data-tooltip-id="performance-mode">
                        <FormToggle
                            checked={isPro && urlParameter}
                            onChange={() => {
                                if (!isPro) {
                                    showProModal(wp.i18n.__('Upgrade to PRO to enable/disable dark mode by URL parameter.', 'dracula-dark-mode'));
                                    return;
                                }

                                setData({...data, urlParameter: !urlParameter})
                            }}
                        />
                    </div>

                    {!isPro &&
                        <Tooltip id="performance-mode" place="right" effect="solid" variant="warning"
                                 backgroundColor="#ff9966"/>}

                    <p className="description">
                        {wp.i18n.__('Enable/disable dark mode by adding the following parameter to the URL.', 'dracula-dark-mode')}
                        <br/>
                        <code>?darkmode=1</code> : {wp.i18n.__('Enable dark mode', 'dracula-dark-mode')}
                        <br/>
                        <code>?darkmode=0</code> : {wp.i18n.__('Disable dark mode', 'dracula-dark-mode')}
                    </p>
                </div>
            </div>

            {/*Keyboard Shortcut*/}
            <div className="settings-field">
                <h4 className={`settings-field-label`}>{wp.i18n.__('Keyboard Shortcut', 'dracula-dark-mode')}</h4>
                <div className="settings-field-content">
                    <FormToggle
                        checked={keyboardShortcuts}
                        onChange={() => setData({...data, keyboardShortcuts: !keyboardShortcuts})}
                    />

                    <p className="description">{wp.i18n.__('Enable keyboard shortcut to toggle dark mode.', 'dracula-dark-mode')}
                        <code>Ctrl + Alt + D</code></p>
                </div>
            </div>

        </div>
    )
}