import {formatDate} from "../../includes/functions";

const {useState, useEffect, useRef} = React;

export default function Feedbacks({data, setData, startDate, endDate}) {

    const {feedbacks} = data;

    const [page, setPage] = useState(feedbacks.length >= 20 ? 2 : 0); //initial data loaded in Analytics.js
    const [isLoading, setIsLoading] = useState(false);

    const getFeedbacks = () => {
        setIsLoading(true);

        wp.ajax.send('dracula_get_feedbacks', {
            data: {
                page: page,
                start_date: startDate,
                end_date: endDate,
            },
            success: (response) => {
                setPage(prePage => response.complete ? 0 : prePage + 1);

                setData(data => ({...data, feedbacks: [...data.feedbacks, ...response.feedbacks]}));
            },
            error: (error) => {
                console.log(error);
                setPage(0);
            },
            complete: () => {
                setIsLoading(false);
            }
        })
    }

    const wrapperRef = useRef();

    useEffect(() => {
        if (page < 1 || isLoading) return;

        const handleScroll = () => {
            const wrapper = wrapperRef.current;

            if (wrapper.scrollTop + wrapper.clientHeight >= wrapper.scrollHeight - 1) {
                getFeedbacks();
            }
        }

        const wrapper = wrapperRef.current;
        if (wrapper) {
            wrapper.addEventListener('scroll', handleScroll);
        }

        return () => {
            if (wrapper) {
                wrapper.removeEventListener('scroll', handleScroll);
            }
        }

    }, [page, isLoading]);

    return (
        <div className="analytics-box feedbacks">

            <span className="analytics-box-title">
                {wp.i18n.__('User Feedbacks', 'dracula-dark-mode')}
            </span>

            <div
                ref={wrapperRef}
                className={`table-wrapper`}>
                {feedbacks.length > 0 ?
                    <>
                        <table className="widefat striped">
                            <thead>
                            <tr>
                                <th></th>
                                <th>
                                    <div>{wp.i18n.__('Message', 'dracula-dark-mode')}</div>
                                </th>
                                <th>
                                    <div>{wp.i18n.__('Date', 'dracula-dark-mode')}</div>
                                </th>
                            </tr>
                            </thead>

                            <tbody>
                            {
                                feedbacks.map((feedback, index) => {
                                    const {message, date} = feedback;

                                    return (
                                        <tr key={index}>
                                            <td className={`col-sl`}>
                                                <div className="sl"><span className="sl-no">{index + 1}.</span></div>
                                            </td>
                                            <td className={`col-description`}>{message}</td>
                                            <td className="col-date">{formatDate(date, true)}</td>
                                        </tr>
                                    )
                                })
                            }

                            </tbody>
                        </table>

                        {isLoading && <div className="dracula-spinner spinner-large"></div>}

                    </>
                    : <div className="no-data">{wp.i18n.__('No data found!', 'dracula-dark-mode')}</div>
                }
            </div>


        </div>
    )
}