<?php

namespace Dracula;

defined( 'ABSPATH' ) or die( 'No script kiddies please!' );

class Enqueue {

	private static $instance = null;

	public function __construct() {
		add_action( 'wp_enqueue_scripts', array( $this, 'frontend_scripts' ), 999 );
		add_action( 'login_enqueue_scripts', array( $this, 'frontend_scripts' ) );
		add_action( 'enqueue_block_assets', array( $this, 'frontend_scripts' ) );
		add_action( 'admin_enqueue_scripts', array( $this, 'admin_scripts' ) );
	}

	/**
	 * Enqueue frontend styles and scripts.
	 *
	 * This function enqueues the frontend styles and scripts required for the dark mode feature.
	 *
	 * It registers the frontend styles, frontend script, and localizes the frontend script with the translations.
	 *
	 * It also checks if the frontend dark mode is enabled and if the current page is excluded from the dark mode feature.
	 *
	 * @since 1.0.0
	 */
	public function frontend_scripts() {
		if ( is_admin() ) {
			return;
		}

		wp_register_style( 'dracula-dark-mode', DRACULA_ASSETS . '/css/dark-mode.css', array(), DRACULA_VERSION );
		wp_register_style( 'dracula_sweetalert2', DRACULA_ASSETS . '/vendor/sweetalert2/sweetalert2.min.css', [], DRACULA_VERSION );

		wp_register_style( 'dracula-frontend', DRACULA_ASSETS . '/css/frontend.css', array(), DRACULA_VERSION );
		wp_style_add_data( 'dracula-frontend', 'rtl', 'replace' );

		// JS Scripts
		$deps = [
			'wp-i18n',
			'wp-util',
		];

		$color_type = dracula_get_settings( 'colorType', 'dynamic' );

		wp_register_script( 'dracula-sweetalert2', DRACULA_ASSETS . '/vendor/sweetalert2/sweetalert2.min.js', [], DRACULA_VERSION, true );

		wp_register_script( 'dracula-dark-mode', DRACULA_ASSETS . '/js/' . ( $color_type === 'dynamic' ? 'dark-mode' : 'dark-mode-static' ) . '.js', [], DRACULA_VERSION );
		$deps[] = 'dracula-dark-mode';

		$deps = apply_filters( 'dracula_frontend_scripts_deps', $deps );

		wp_register_script( 'dracula-frontend', DRACULA_ASSETS . '/js/frontend.js', $deps, DRACULA_VERSION, true );
		wp_localize_script( 'dracula-frontend', 'dracula', $this->get_localize_data() );

		wp_localize_script( 'dracula-dark-mode', 'dracula', $this->get_localize_data() );

		$is_active = dracula_get_settings( 'frontendDarkMode', true ) && ! dracula_page_excluded();

		// Live Edit && Setup Scripts
		if ( dracula_is_embed_request() ) {
			$this->enqueue_live_edit_scripts();
		}


		$is_reading_mode = dracula_get_settings( 'readingMode' );

		// Custom Dark Mode Style
		if ( $color_type === 'static' || 'custom' === dracula_get_settings( 'colorMode' ) ) {
			wp_enqueue_style( 'dracula-dark-mode' );
		}

		// Frontend Scripts
		if ( $is_active || $is_reading_mode ) {

			wp_enqueue_style( 'dracula-frontend' );
			wp_enqueue_script( 'dracula-frontend' );

			// Link the script with its translations.
			wp_set_script_translations( 'dracula-frontend', 'dracula-dark-mode', plugin_dir_path( DRACULA_FILE ) . 'languages' );
		}

		// frontend script hook
		do_action( 'dracula_frontend_scripts', $is_active );

	}

	/**
	 * Enqueue admin styles and scripts.
	 *
	 * This function enqueues the admin styles and scripts required for the dark mode feature.
	 *
	 * It registers the admin styles, admin script, and localizes the admin script with the translations.
	 *
	 * It also checks if the user can access dracula pages and if the current page is excluded from the dark mode feature.
	 *
	 * @param string $hook The current admin page.
	 *
	 * @since 1.0.0
	 */
	public function admin_scripts( $hook ) {

		// Check if user can access dracula pages
		if ( ! dracula_is_user_dark_mode() && ! dracula_is_block_editor_page() ) {
			return;
		}

		if ( ! class_exists( 'Admin' ) ) {
			require_once DRACULA_INCLUDES . '/class-admin.php';
		}

		$admin_pages = Admin::instance()->get_admin_pages();

		// By default, style id startWith dracula- ignored by dark mode.
		// that why we need to add dracula_ prefix to the selector where we don't want to ignore dark mode
		wp_register_style( 'dracula_sweetalert2', DRACULA_ASSETS . '/vendor/sweetalert2/sweetalert2.min.css', [], DRACULA_VERSION );

		// Ignore toggle styles from dark mode
		wp_register_style( 'dracula-toggle', DRACULA_ASSETS . '/css/toggle.css', array(), DRACULA_VERSION );

		wp_enqueue_style( 'dracula_admin', DRACULA_ASSETS . '/css/admin.css', array(
			'wp-components',
			'dracula-toggle',
			'dracula_sweetalert2'
		), DRACULA_VERSION );

		wp_style_add_data( 'dracula_admin', 'rtl', 'replace' );

		// Javascript Dependencies
		$deps = [
			'react',
			'react-dom',
			'wp-components',
			'wp-i18n',
			'wp-util',
		];

		if ( ! Admin::instance()->should_exclude_darkmode() ) {
			wp_register_script( 'dracula-dark-mode', DRACULA_ASSETS . '/js/dark-mode.js', [], DRACULA_VERSION );
			$deps[] = 'dracula-dark-mode';
		}

		// If block editor page and !active return
		$block_editor_dark_mode = dracula_get_settings( 'blockEditorDarkMode', true );
		if ( ! $block_editor_dark_mode && dracula_is_block_editor_page() ) {
			$deps = array_diff( $deps, [ 'dracula-dark-mode' ] );
		}

		// Analytics page scripts
		if ( ! empty( $admin_pages['analytics'] ) && $admin_pages['analytics'] === $hook ) {
			wp_register_script( 'dracula-chart', DRACULA_ASSETS . '/vendor/chart.umd.min.js', array( 'jquery-ui-datepicker' ), '4.5.0', true );
			$deps[] = 'dracula-chart';
		}

		wp_register_script( 'dracula-sweetalert2', DRACULA_ASSETS . '/vendor/sweetalert2/sweetalert2.min.js', [], DRACULA_VERSION, true );
		$deps[] = 'dracula-sweetalert2';

		// Enqueue media scripts for settings and toggle builder page
		if ( in_array( $hook, [ $admin_pages['settings'], $admin_pages['toggle_builder'] ] ) ) {
			wp_enqueue_media();
		}

		// CSS Editor Scripts
		if ( $admin_pages['dracula'] === $hook || dracula_is_block_editor_page() || dracula_is_classic_editor_page() ) {
			wp_enqueue_script( 'wp-theme-plugin-editor' );
			wp_enqueue_style( 'wp-codemirror' );

			wp_enqueue_code_editor( array(
				'type'  => 'text/css',
				'theme' => 'dracula',
			) );
		}

		wp_enqueue_script( 'dracula-admin', DRACULA_ASSETS . '/js/admin.js', $deps, DRACULA_VERSION, true );
		wp_localize_script( 'dracula-admin', 'dracula', $this->get_localize_data( $hook ) );
		wp_localize_script( 'dracula-dark-mode', 'dracula', $this->get_localize_data( $hook ) );

		// Link the script with its translations.
		wp_set_script_translations( 'dracula-admin', 'dracula-dark-mode', plugin_dir_path( DRACULA_FILE ) . 'languages' );
	}

	public static function enqueue_live_edit_scripts() {

		wp_enqueue_style( 'dracula_sweetalert2' );

		wp_enqueue_style( 'dracula-live-edit', DRACULA_ASSETS . '/css/live-edit.css', [
			'dashicons',
			'wp-components',
			'dracula-frontend',
			'dracula_sweetalert2',
			'wp-codemirror'
		], DRACULA_VERSION );

		wp_enqueue_media();

		wp_enqueue_script( 'dracula-sweetalert2' );
		wp_enqueue_script( 'wp-theme-plugin-editor' );

		wp_enqueue_script( 'dracula-live-edit', DRACULA_ASSETS . '/js/live-edit.js', [
			'wp-components',
			'dracula-frontend'
		], DRACULA_VERSION, true );
	}

	/**
	 * Get localized data for scripts.
	 *
	 * This function returns an array of localized data to be used in the frontend and admin scripts.
	 *
	 * @param string|bool $hook The current admin page hook. Default is false.
	 *
	 * @return array The localized data.
	 * @since 1.0.0
	 */
	public function get_localize_data( $hook = false ) {

		$data = array(
			'homeUrl'        => home_url(),
			'adminUrl'       => admin_url(),
			'ajaxUrl'        => admin_url( 'admin-ajax.php' ),
			'pluginUrl'      => DRACULA_URL,
			'settings'       => dracula_get_settings(),
			'isPro'          => false,
			'nonce'          => wp_create_nonce( 'dracula' ),
			'switches'       => dracula_get_switches_markups(),
			'customSwitches' => dracula_custom_toggle_switches(),
			'presets'        => dracula_get_preset(),
		);

		if ( is_admin() ) {
			$data['isAdmin'] = true;

			$admin_pages = Admin::instance()->get_admin_pages();

			if ( $admin_pages['dracula'] === $hook ) {
				$data['version']            = DRACULA_VERSION;
				$data['isBlockTheme']       = function_exists( 'wp_is_block_theme' ) && wp_is_block_theme();
				$data['userRoles']          = dracula_get_user_roles();
				$data['excludeList']        = dracula_get_exclude_list();
				$data['excludeReadingList'] = dracula_get_exclude_reading_list();
				$data['excludeTaxList']     = dracula_get_exclude_taxonomy_list();
				$data['showReviewPopup']    = current_user_can( 'manage_options' ) && 'off' != get_option( 'dracula_rating_notice' ) && 'off' != get_transient( 'dracula_rating_notice_interval' );
				$data['postTypes']          = dracula_get_post_type_list();
			}
		}

		$data = apply_filters( 'dracula_localize_script', $data );

		return $data;
	}

	public static function instance() {
		if ( null === self::$instance ) {
			self::$instance = new self;
		}

		return self::$instance;
	}
}


Enqueue::instance();
