/**
 * ConvertKit Broadcasts Divi Module.
 */
class ConvertKitDiviBroadcasts extends React.Component {
	/**
	 * The Divi module name. Must match the slug defined in
	 * the PHP class ConvertKit_Divi_Module_Broadcasts.
	 *
	 * @since 	2.5.7
	 */
	static slug = 'convertkit_broadcasts';

	/**
	 * Renders the frontend output for this module.
	 *
	 * @since 	2.5.7
	 */
	render() {
		const block = window.ConvertkitDiviBuilderData.broadcasts;

		// If no Access Token has been defined in the Plugin, show a message in the module to tell the user what to do.
		if (!block.has_access_token) {
			return ConvertKitDiviNoAccessMessage(block);
		}

		// If no resources exist in ConvertKit, show a message in the module to tell the user what to do.
		if (!block.has_resources) {
			return ConvertKitDiviNoResourcesMessage(block);
		}

		// Return module with text.
		return ConvertKitDiviMessage(
			'Configure this broadcasts module by clicking the settings cog.'
		);
	}
}

/**
 * ConvertKit Form Divi Module.
 */
class ConvertKitDiviForm extends React.Component {
	/**
	 * The Divi module name. Must match the slug defined in
	 * the PHP class ConvertKit_Divi_Module_Form.
	 *
	 * @since 	2.5.6
	 */
	static slug = 'convertkit_form';

	/**
	 * Renders the frontend output for this module.
	 *
	 * @since 	2.5.6
	 */
	render() {
		const form =
			typeof this.props.form !== 'undefined' ? this.props.form : '';

		return ConvertKitDiviRenderResourceModule(
			window.ConvertkitDiviBuilderData.form,
			form,
			form
				? window.ConvertkitDiviBuilderData.form.fields.form.data.forms[
						form
					].name
				: '',
			window.ConvertkitDiviBuilderData.form.fields.form.label
		);
	}
}

/**
 * ConvertKit Form Trigger Divi Module.
 */
class ConvertKitDiviFormTrigger extends React.Component {
	/**
	 * The Divi module name. Must match the slug defined in
	 * the PHP class ConvertKit_Divi_Module_Form.
	 *
	 * @since 	2.5.7
	 */
	static slug = 'convertkit_formtrigger';

	/**
	 * Renders the frontend output for this module.
	 *
	 * @since 	2.5.7
	 */
	render() {
		const form = Number(
			typeof this.props.form !== 'undefined' ? this.props.form : 0
		);

		return ConvertKitDiviRenderResourceModule(
			window.ConvertkitDiviBuilderData.formtrigger,
			form,
			form > 0
				? window.ConvertkitDiviBuilderData.formtrigger.fields.form
						.values[form]
				: '',
			window.ConvertkitDiviBuilderData.formtrigger.fields.form.label
		);
	}
}

/**
 * ConvertKit Product Divi Module.
 */
class ConvertKitDiviProduct extends React.Component {
	/**
	 * The Divi module name. Must match the slug defined in
	 * the PHP class ConvertKit_Divi_Module_Product.
	 *
	 * @since 	2.5.7
	 */
	static slug = 'convertkit_product';

	/**
	 * Renders the frontend output for this module.
	 *
	 * @since 	2.5.7
	 */
	render() {
		const product =
			typeof this.props.product !== 'undefined' ? this.props.product : '';

		return ConvertKitDiviRenderResourceModule(
			window.ConvertkitDiviBuilderData.product,
			product,
			product
				? window.ConvertkitDiviBuilderData.product.fields.product
						.values[product]
				: '',
			window.ConvertkitDiviBuilderData.product.fields.product.label
		);
	}
}

/**
 * Returns the block's notice and instruction text when no access token exists.
 *
 * @since 	2.5.7
 *
 * @param {Object} block Block.
 */
function ConvertKitDiviNoAccessMessage(block) {
	return React.createElement(
		'div',
		{
			className: 'convertkit-divi-module convertkit-no-content',
		},
		[
			React.createElement('p', {}, block.no_access_token.notice),
			React.createElement(
				'p',
				{},
				block.no_access_token.instruction_text
			),
		]
	);
}

/**
 * Returns the block's notice and instruction text when no resources
 * (forms, products etc) exist.
 *
 * @since 	2.5.7
 *
 * @param {Object} block Block.
 */
function ConvertKitDiviNoResourcesMessage(block) {
	return React.createElement(
		'div',
		{
			className: 'convertkit-divi-module convertkit-no-content',
		},
		[
			React.createElement('p', {}, block.no_resources.notice),
			React.createElement('p', {}, block.no_resources.instruction_text),
		]
	);
}

/**
 * Returns the given text.
 *
 * @since 	2.5.7
 *
 * @param {string} message Message to display.
 */
function ConvertKitDiviMessage(message) {
	return React.createElement(
		'div',
		{
			className: 'convertkit-divi-module',
		},
		message
	);
}

/**
 * Renders the Divi module for the given ConvertKit block (form, product, broadcast etc).
 * and message, based on the Plugin and Divi module's configuration.
 *
 * @since 	2.5.7
 *
 * @param {Object} block        Block.
 * @param {string} value        Selected value.
 * @param {string} valueLabel   Selected value's label.
 * @param {string} resourceName Resource name (form,product).
 */
function ConvertKitDiviRenderResourceModule(
	block,
	value,
	valueLabel,
	resourceName
) {
	// If no Access Token has been defined in the Plugin, show a message in the module to tell the user what to do.
	if (!block.has_access_token) {
		return ConvertKitDiviNoAccessMessage(block);
	}

	// If no resources exist in ConvertKit, show a message in the module to tell the user what to do.
	if (!block.has_resources) {
		return ConvertKitDiviNoResourcesMessage(block);
	}

	// Show instructions if no resource selected.
	if (value.length === 0) {
		return ConvertKitDiviMessage(
			'Configure this module by clicking the settings cog.'
		);
	}

	// Return module with text.
	return ConvertKitDiviMessage(
		'Kit ' +
			resourceName +
			' "' +
			valueLabel +
			'" selected. View on the frontend site to see the ' +
			resourceName +
			'.'
	);
}

/**
 * Register Divi modules when the Divi Builder API is ready.
 *
 * @since 	2.5.6
 *
 * @param {Object} event Event.
 * @param {Object} API   Divi Buidler API.
 */
jQuery(window).on('et_builder_api_ready', function (event, API) {
	// eslint-disable-line no-unused-vars
	// Register Divi modules.
	API.registerModules([
		ConvertKitDiviBroadcasts,
		ConvertKitDiviForm,
		ConvertKitDiviFormTrigger,
		ConvertKitDiviProduct,
	]);
});
