<?php

namespace ContentEgg\application\components;

use ContentEgg\application\admin\GeneralConfig;
use ContentEgg\application\components\ai\PrefillPrompt;
use ContentEgg\application\models\PrefillQueueModel;;

defined('\ABSPATH') || exit;

/**
 * PrefillKeywordResolver class file
 *
 * @author keywordrush.com <support@keywordrush.com>
 * @link https://www.keywordrush.com
 * @copyright Copyright &copy; 2025 keywordrush.com
 */

class PrefillKeywordResolver
{
    protected PrefillLogger $logger;
    protected $prompt;

    public function __construct(PrefillLogger $logger, $prompt)
    {
        $this->logger = $logger;
        $this->prompt = $prompt;
    }

    /**
     * Resolve the best keyword for given post and config.
     *
     * @param \WP_Post $post
     * @param array $config
     * @return string
     */
    public function resolve(\WP_Post $post, array $config, array $modules): string
    {
        $source = $config['keyword_source'] ?? '';

        switch ($source)
        {
            case 'post_title':
                return $this->fromPostTitle($post);

            case 'meta_field':
                $meta_key = $config['meta_field_name'] ?? '';
                return $this->fromMetaField($post, $meta_key);

            case 'gtin_woocommerce':
                $meta_key = '_global_unique_id';
                return $this->fromMetaField($post, $meta_key);

            case 'gtin_module':
                $module_id = $config['source_module_gtin'] ?? '';
                return $this->fromModuleData($post, $module_id, 'ean');

            case 'product_title_module':
                $module_id = $config['source_module_title'] ?? '';
                return $this->fromModuleData($post, $module_id, 'title');

            case 'ai':
                return $this->fromAI($post, $modules);

            default:
                $this->logNotice(sprintf(__('Unknown keyword source "%s".', 'content-egg'), $source));
                return '';
        }
    }

    /**
     * Keyword = Post Title.
     */
    protected function fromPostTitle(\WP_Post $post): string
    {
        return trim($post->post_title);
    }

    /**
     * Keyword = Custom Meta Field Value.
     */
    protected function fromMetaField(\WP_Post $post, string $meta_key): string
    {
        if (empty($meta_key))
        {
            $this->logNotice(__('Meta field key is empty.', 'content-egg'));
            return '';
        }

        $value = get_post_meta($post->ID, $meta_key, true);

        if ($value === '' || $value === null)
        {
            $this->logNotice(sprintf(__('Meta field "%s" not found for post ID %d.', 'content-egg'), $meta_key, $post->ID));
            return '';
        }

        if (is_array($value))
        {
            $value = implode(' ', array_filter($value));
        }

        return is_string($value) ? trim($value) : '';
    }

    /**
     * Keyword = Value from existing module product data
     */
    protected function fromModuleData(\WP_Post $post, string $module_id, string $field): string
    {
        if (empty($module_id))
        {
            $this->logNotice(__('Module ID is empty for module keyword source.', 'content-egg'));
            return '';
        }

        $products = \ContentEgg\application\components\ContentManager::getData($post->ID, $module_id);

        if (empty($products) || !is_array($products))
        {
            $this->logNotice(sprintf(__('Module data is empty for module "%s" on post ID %d.', 'content-egg'), $module_id, $post->ID));

            return '';
        }

        foreach ($products as $item)
        {
            if (!empty($item[$field]) && is_scalar($item[$field]))
            {
                return trim((string)$item[$field]);
            }
        }

        $this->logNotice(sprintf(__('Field "%s" not found in products for module "%s" on post ID %d.', 'content-egg'), $field, $module_id, $post->ID));
        return '';
    }

    /**
     * Keyword = Generated by AI
     */
    protected function fromAI(\WP_Post $post, array $modules): string
    {
        $max_keywords = apply_filters('cegg_prefill_ai_keywords_count', 2);

        $keywords = $this->prompt->suggestProductKeywordsForPost(
            $post->post_title,
            $post->post_content,
            $max_keywords,
            $modules,
        );

        PrefillQueueModel::model()->updateAiStat($post->ID, $this->prompt->getLastUsageStat());

        if (!$keywords || !is_array($keywords))
        {
            return '';
        }

        return reset($keywords);
    }

    protected function logNotice(string $message): void
    {
        if ($this->logger)
        {
            $this->logger->notice($message);
        }
    }
}
