<?php
/*
 * This file is part of Contact Form 7 Extension For Mailchimp.
 *
 * Copyright (C) 2010-2025, Renzo Johnson (email: renzo.johnson at gmail.com)
 *
 * For the full copyright and license information, please view the LICENSE
 * file that was distributed with this source code, or visit:
 * https://www.gnu.org/licenses/gpl-2.0.html
 */

defined( 'ABSPATH' ) || exit;


function cmatic_register_rest_routes() {
	register_rest_route(
		'chimpmatic-lite/v1',
		'/lists',
		array(
			'methods'             => 'POST',
			'callback'            => 'cmatic_rest_get_lists',
			'permission_callback' => 'cmatic_rest_permission_check',
			'args'                => array(
				'form_id' => array(
					'required'          => true,
					'type'              => 'integer',
					'sanitize_callback' => 'absint',
					'validate_callback' => function ( $param ) {
						return is_numeric( $param ) && $param > 0;
					},
				),
				'api_key' => array(
					'required'          => true,
					'type'              => 'string',
					'sanitize_callback' => 'sanitize_text_field',
					'validate_callback' => function ( $param ) {
						return preg_match( '/^[a-f0-9]{32}-[a-z]{2,3}\d+$/', $param );
					},
				),
			),
		)
	);

	register_rest_route(
		'chimpmatic-lite/v1',
		'/settings/autoupdate',
		array(
			'methods'             => 'POST',
			'callback'            => 'cmatic_rest_set_autoupdate',
			'permission_callback' => 'cmatic_rest_permission_check',
			'args'                => array(
				'enabled' => array(
					'required'          => true,
					'type'              => 'boolean',
					'sanitize_callback' => 'rest_sanitize_boolean',
				),
			),
		)
	);

	register_rest_route(
		'chimpmatic-lite/v1',
		'/settings/logger',
		array(
			'methods'             => 'POST',
			'callback'            => 'cmatic_rest_set_logger',
			'permission_callback' => 'cmatic_rest_permission_check',
			'args'                => array(
				'enabled' => array(
					'required'          => true,
					'type'              => 'boolean',
					'sanitize_callback' => 'rest_sanitize_boolean',
				),
			),
		)
	);

	register_rest_route(
		'chimpmatic-lite/v1',
		'/logs',
		array(
			'methods'             => 'GET',
			'callback'            => 'cmatic_rest_get_debug_log',
			'permission_callback' => 'cmatic_rest_permission_check',
		)
	);

	register_rest_route(
		'chimpmatic-lite/v1',
		'/logs/clear',
		array(
			'methods'             => 'POST',
			'callback'            => 'cmatic_rest_clear_debug_log',
			'permission_callback' => 'cmatic_rest_permission_check',
		)
	);

	register_rest_route(
		'chimpmatic-lite/v1',
		'/notices/dismiss',
		array(
			'methods'             => 'POST',
			'callback'            => 'cmatic_rest_dismiss_notice',
			'permission_callback' => 'cmatic_rest_permission_check',
		)
	);

	register_rest_route(
		'chimpmatic-lite/v1',
		'/logs/browser',
		array(
			'methods'             => 'POST',
			'callback'            => 'cmatic_rest_log_browser_console',
			'permission_callback' => 'cmatic_rest_permission_check',
			'args'                => array(
				'level'   => array(
					'required'          => true,
					'type'              => 'string',
					'sanitize_callback' => 'sanitize_text_field',
					'validate_callback' => function ( $param ) {
						return in_array( $param, array( 'log', 'info', 'warn', 'error', 'debug' ), true );
					},
				),
				'message' => array(
					'required'          => true,
					'type'              => 'string',
					'sanitize_callback' => 'sanitize_text_field',
				),
				'data'    => array(
					'required'          => false,
					'type'              => 'string',
					'sanitize_callback' => 'sanitize_textarea_field',
				),
			),
		)
	);

	register_rest_route(
		'chimpmatic-lite/v1',
		'/settings/reset',
		array(
			'methods'             => 'POST',
			'callback'            => 'cmatic_rest_reset_settings',
			'permission_callback' => 'cmatic_rest_permission_check',
			'args'                => array(
				'form_id' => array(
					'required'          => true,
					'type'              => 'integer',
					'sanitize_callback' => 'absint',
					'validate_callback' => function ( $param ) {
						return is_numeric( $param ) && $param > 0;
					},
				),
			),
		)
	);

	register_rest_route(
		'chimpmatic-lite/v1',
		'/merge-fields',
		array(
			'methods'             => 'POST',
			'callback'            => 'cmatic_rest_get_merge_fields',
			'permission_callback' => 'cmatic_rest_permission_check',
			'args'                => array(
				'form_id' => array(
					'required'          => true,
					'type'              => 'integer',
					'sanitize_callback' => 'absint',
				),
				'list_id' => array(
					'required'          => true,
					'type'              => 'string',
					'sanitize_callback' => 'sanitize_text_field',
				),
			),
		)
	);

	// Nuclear License Reset endpoint (support teams only)
	register_rest_route(
		'chimpmatic-lite/v1',
		'/license/nuclear-reset',
		array(
			'methods'             => 'POST',
			'callback'            => 'cmatic_rest_nuclear_license_reset',
			'permission_callback' => 'cmatic_rest_permission_check',
		)
	);
}
add_action( 'rest_api_init', 'cmatic_register_rest_routes' );

/**
 * Permission callback for REST API endpoints
 * Checks if user has manage_options capability and verifies nonce
 */
function cmatic_rest_permission_check( $request ) {
	if ( ! current_user_can( 'manage_options' ) ) {
		return new WP_Error(
			'rest_forbidden',
			esc_html__( 'You do not have permission to access this endpoint.', 'chimpmatic-lite' ),
			array( 'status' => 403 )
		);
	}

	// Verify nonce from header
	$nonce = $request->get_header( 'X-WP-Nonce' );
	if ( ! wp_verify_nonce( $nonce, 'wp_rest' ) ) {
		return new WP_Error(
			'rest_cookie_invalid_nonce',
			esc_html__( 'Cookie nonce is invalid.', 'chimpmatic-lite' ),
			array( 'status' => 403 )
		);
	}

	return true;
}


function cmatic_rest_get_lists( $request ) {
	$form_id = $request->get_param( 'form_id' );
	$api_key = $request->get_param( 'api_key' );

	// Track API setup funnel: Sync attempted
	if ( ! mce_get_cmatic( 'api.sync_attempted' ) ) {
		// First time user clicks "Sync Audiences"
		mce_update_cmatic( 'api.sync_attempted', time() );
	}
	// Increment sync attempts counter
	$current_count = (int) mce_get_cmatic( 'api.sync_attempts_count', 0 );
	mce_update_cmatic( 'api.sync_attempts_count', $current_count + 1 );

	$option_name = 'cf7_mch_' . $form_id;
	$cf7_mch     = get_option( $option_name, array() );

	// Use global debug logger setting (not per-form)
	$logfileEnabled = (bool) get_option( CMATIC_LOG_OPTION, false );

	try {
		// Validate API key
		$validation_result = wpcf7_mce_validate_api_key( $api_key, $logfileEnabled, $option_name );
		$api_valid         = $validation_result['api-validation'] ?? 0;

		// Get lists and merge fields if API is valid
		$lists_result      = wpcf7_mce_listasasociadas( $api_key, $logfileEnabled, $option_name, $api_valid );
		$lists_data        = $lists_result['lisdata'] ?? array();
		$merge_fields_data = $lists_result['merge_fields'] ?? array();

		// Format response - send only the data we need for display
		$lists = array();
		if ( $api_valid === 1 && isset( $lists_data['lists'] ) && is_array( $lists_data['lists'] ) ) {
			foreach ( $lists_data['lists'] as $list ) {
				if ( is_array( $list ) && isset( $list['id'], $list['name'] ) ) {
					// Get member count from stats
					$member_count = isset( $list['stats']['member_count'] ) ? intval( $list['stats']['member_count'] ) : 0;

					// Get total field count (merge_fields includes EMAIL by default)
					// The stats object contains merge_field_count which is the total including EMAIL
					$field_count = isset( $list['stats']['merge_field_count'] ) ? intval( $list['stats']['merge_field_count'] ) : 0;

					// Send clean data for dropdown display
					$lists[] = array(
						'id'           => $list['id'],
						'name'         => $list['name'],
						'member_count' => $member_count,
						'field_count'  => $field_count,
					);
				}
			}
		}

		$excluded_types = array( 'address', 'birthday', 'imageurl', 'zip' );
		$merge_fields   = array();

		if ( $logfileEnabled ) {
			error_log( '[ChimpMatic Lite] [REST API] Starting merge field filtering. Raw data count: ' . count( $merge_fields_data['merge_fields'] ?? array() ) );
		}

		// Always add EMAIL as the first field (required)
		$merge_fields[] = array(
			'tag'  => 'EMAIL',
			'name' => 'Subscriber Email',
			'type' => 'email',
		);
		if ( $logfileEnabled ) {
			error_log( '[ChimpMatic Lite] [REST API] Added EMAIL field (required)' );
		}

		if ( isset( $merge_fields_data['merge_fields'] ) && is_array( $merge_fields_data['merge_fields'] ) ) {
			// Sort by display_order to match Mailchimp UI order
			$fields_to_process = $merge_fields_data['merge_fields'];
			usort(
				$fields_to_process,
				function ( $a, $b ) {
					return ( $a['display_order'] ?? 0 ) - ( $b['display_order'] ?? 0 );
				}
			);

			if ( $logfileEnabled ) {
				error_log( '[ChimpMatic Lite] [REST API] Fields sorted by display_order' );
			}

			$count = 1; // Start at 1 since EMAIL is already added
			foreach ( $fields_to_process as $field ) {
				$field_type    = strtolower( $field['type'] ?? '' );
				$field_tag     = $field['tag'] ?? '';
				$display_order = $field['display_order'] ?? 0;

				if ( $logfileEnabled ) {
					error_log( '[ChimpMatic Lite] [REST API] Processing field: ' . $field_tag . ' | Type: ' . $field_type . ' | Display Order: ' . $display_order );
				}

				// Skip EMAIL if it's in the API response (we already added it)
				if ( $field_tag === 'EMAIL' ) {
					if ( $logfileEnabled ) {
						error_log( '[ChimpMatic Lite] [REST API] Skipping EMAIL (already added as first field)' );
					}
					continue;
				}

				// Skip excluded types (address, birthday, imageurl, zip)
				if ( in_array( $field_type, $excluded_types, true ) ) {
					if ( $logfileEnabled ) {
						error_log( '[ChimpMatic Lite] [REST API] Filtering out ' . $field_tag . ' (type: ' . $field_type . ')' );
					}
					continue;
				}

				// Cap at CMATIC_LITE_MAX_FIELDS total fields (including EMAIL)
				if ( $count >= CMATIC_LITE_MAX_FIELDS ) {
					if ( $logfileEnabled ) {
						error_log( '[ChimpMatic Lite] [REST API] Reached ' . CMATIC_LITE_MAX_FIELDS . ' field limit, stopping' );
					}
					break;
				}

				if ( $logfileEnabled ) {
					error_log( '[ChimpMatic Lite] [REST API] Adding field ' . ( $count + 1 ) . ': ' . $field_tag );
				}
				$merge_fields[] = array(
					'tag'  => $field_tag,
					'name' => $field['name'] ?? '',
					'type' => $field_type,
				);
				++$count;
			}
		}

		if ( $logfileEnabled ) {
			error_log( '[ChimpMatic Lite] [REST API] Final filtered merge fields: ' . print_r( $merge_fields, true ) );
		}

		// Save settings with formatted merge fields
		$settings_to_save = array_merge(
			$cf7_mch,
			$validation_result,
			$lists_result,
			array(
				'api'          => $api_key,
				'merge_fields' => $merge_fields,  // Save formatted merge fields
			)
		);
		update_option( $option_name, $settings_to_save );

		return rest_ensure_response(
			array(
				'success'      => true,
				'api_valid'    => $api_valid === 1,
				'lists'        => $lists,
				'total'        => count( $lists ),
				'merge_fields' => $merge_fields,
			)
		);

	} catch ( Exception $e ) {
		$logger = new Cmatic_Logger( true, 'REST-API-Error', 0 );
		$logger->log( 'ERROR', 'REST API list loading failed.', $e->getMessage() );

		return new WP_Error(
			'api_request_failed',
			esc_html__( 'Failed to load Mailchimp lists. Check debug log for details.', 'chimpmatic-lite' ),
			array( 'status' => 500 )
		);
	}
}

/**
 * REST API: Set Auto-Update Setting
 */
function cmatic_rest_set_autoupdate( $request ) {
	$enabled = $request->get_param( 'enabled' );
	$value   = $enabled ? 1 : 0;

	update_option( 'chimpmatic-update', $value );

	return rest_ensure_response(
		array(
			'success' => true,
			'enabled' => $enabled,
			'message' => esc_html__( 'Auto-update setting saved.', 'chimpmatic-lite' ),
		)
	);
}

/**
 * REST API: Set Logger Setting (Global)
 */
function cmatic_rest_set_logger( $request ) {
	$enabled = $request->get_param( 'enabled' );

	update_option( CMATIC_LOG_OPTION, $enabled );

	return rest_ensure_response(
		array(
			'success' => true,
			'enabled' => $enabled,
			'message' => esc_html__( 'Debug logger setting saved.', 'chimpmatic-lite' ),
		)
	);
}

/**
 * REST API: Get Debug Log
 * Returns filtered debug log entries for ChimpMatic Lite
 */
function cmatic_rest_get_debug_log( $request ) {
	$log_file_path = WP_CONTENT_DIR . '/debug.log';

	if ( ! file_exists( $log_file_path ) || 0 === filesize( $log_file_path ) ) {
		return rest_ensure_response(
			array(
				'success' => true,
				'content' => esc_html__( 'The debug.log file was not found or is empty.', 'chimpmatic-lite' ),
				'entries' => array(),
			)
		);
	}

	// Read last 500 lines
	$lines        = file( $log_file_path );
	$recent_lines = array_slice( $lines, -500 );
	$line_count   = count( $recent_lines );
	$filtered_log = '';

	for ( $i = 0; $i < $line_count; $i++ ) {
		// Find lines belonging to our plugin
		if ( strpos( $recent_lines[ $i ], '[ChimpMatic Lite]' ) !== false ) {
			$current_entry = '';
			// Capture this line and subsequent continuation lines
			for ( $j = $i; $j < $line_count; $j++ ) {
				$line_to_check = $recent_lines[ $j ];
				// New entry starts with timestamp
				if ( $j > $i && preg_match( '/^\[\d{2}-\w{3}-\d{4} \d{2}:\d{2}:\d{2} UTC\]/', $line_to_check ) ) {
					break;
				}
				$current_entry .= $line_to_check;
			}
			$filtered_log .= $current_entry;
			$i             = $j - 1;
		}
	}

	if ( empty( $filtered_log ) ) {
		$filtered_log = esc_html__( 'No ChimpMatic Lite entries found in the recent log data. Note: This viewer only shows the last 500 lines of the log file.', 'chimpmatic-lite' );
	}

	return rest_ensure_response(
		array(
			'success' => true,
			'content' => trim( $filtered_log ),
		)
	);
}

/**
 * REST API: Log Browser Console Messages
 * Captures browser console logs and writes them to WordPress debug.log
 */
function cmatic_rest_log_browser_console( $request ) {
	$level   = $request->get_param( 'level' );
	$message = $request->get_param( 'message' );
	$data    = $request->get_param( 'data' );

	// Map console level to WordPress error level
	$level_map = array(
		'log'   => 'INFO',
		'info'  => 'INFO',
		'warn'  => 'WARNING',
		'error' => 'ERROR',
		'debug' => 'DEBUG',
	);

	$wp_level = $level_map[ $level ] ?? 'INFO';

	// Format the log entry
	$log_message = sprintf(
		'[%s] [ChimpMatic Lite] [Browser Console - %s] %s',
		gmdate( 'd-M-Y H:i:s' ) . ' UTC',
		strtoupper( $level ),
		$message
	);

	if ( ! empty( $data ) ) {
		$log_message .= ' | Data: ' . $data;
	}

	// Write to debug.log if WP_DEBUG_LOG is enabled
	if ( defined( 'WP_DEBUG_LOG' ) && WP_DEBUG_LOG ) {
		error_log( $log_message );
	}

	// Also use our custom logger if available
	$cf7_mch = get_option( 'cf7_mch_' . ( $_POST['form_id'] ?? 0 ), array() );
	// Use global debug logger setting (not per-form)
	$logfileEnabled = (bool) get_option( CMATIC_LOG_OPTION, false );

	if ( class_exists( 'Cmatic_Logger' ) ) {
		$logger = new Cmatic_Logger( $logfileEnabled, 'Browser-Console', 0 );
		$logger->log( $wp_level, 'Browser: ' . $message, $data ? json_decode( $data, true ) : null );
	}

	return rest_ensure_response(
		array(
			'success' => true,
			'logged'  => true,
		)
	);
}

/**
 * REST API: Clear Debug Log
 * Completely clears the WordPress debug.log file
 */
function cmatic_rest_clear_debug_log( $request ) {
	$log_file_path = WP_CONTENT_DIR . '/debug.log';

	// Check if file exists
	if ( ! file_exists( $log_file_path ) ) {
		return rest_ensure_response(
			array(
				'success' => true,
				'message' => esc_html__( 'Debug log file does not exist.', 'chimpmatic-lite' ),
				'cleared' => false,
			)
		);
	}

	// Check if file is writable
	if ( ! is_writable( $log_file_path ) ) {
		return new WP_Error(
			'file_not_writable',
			esc_html__( 'Debug log file is not writable.', 'chimpmatic-lite' ),
			array( 'status' => 500 )
		);
	}

	// Clear the file by opening it in write mode and immediately closing
	$file_handle = fopen( $log_file_path, 'w' );

	if ( false === $file_handle ) {
		return new WP_Error(
			'file_clear_failed',
			esc_html__( 'Failed to clear debug log file.', 'chimpmatic-lite' ),
			array( 'status' => 500 )
		);
	}

	fclose( $file_handle );

	// Log this action
	if ( defined( 'WP_DEBUG_LOG' ) && WP_DEBUG_LOG ) {
		error_log(
			sprintf(
				'[%s] [ChimpMatic Lite] Debug log cleared by user: %s',
				gmdate( 'd-M-Y H:i:s' ) . ' UTC',
				wp_get_current_user()->user_login
			)
		);
	}

	return rest_ensure_response(
		array(
			'success' => true,
			'message' => esc_html__( 'Debug log cleared successfully.', 'chimpmatic-lite' ),
			'cleared' => true,
		)
	);
}

/**
 * REST API: Dismiss Admin Notice
 * Dismisses the #mce-notice update notice
 */
function cmatic_rest_dismiss_notice( $request ) {
	// Update the option to hide the notice
	$result = update_site_option( 'mce_show_notice', 0 );

	if ( false === $result ) {
		return new WP_Error(
			'notice_dismiss_failed',
			esc_html__( 'Failed to dismiss notice.', 'chimpmatic-lite' ),
			array( 'status' => 500 )
		);
	}

	return rest_ensure_response(
		array(
			'success'   => true,
			'message'   => esc_html__( 'Notice dismissed successfully.', 'chimpmatic-lite' ),
			'dismissed' => true,
		)
	);
}

/**
 * REST API: Get Merge Fields for a Mailchimp List
 * Returns merge fields (FNAME, LNAME, etc.) for the selected audience
 */
function cmatic_rest_get_merge_fields( $request ) {
	$form_id = $request->get_param( 'form_id' );
	$list_id = $request->get_param( 'list_id' );

	$option_name = 'cf7_mch_' . $form_id;
	$cf7_mch     = get_option( $option_name, array() );

	$api_key = $cf7_mch['api'] ?? '';
	// Use global debug logger setting (not per-form)
	$logfileEnabled = (bool) get_option( CMATIC_LOG_OPTION, false );

	if ( empty( $api_key ) ) {
		return new WP_Error(
			'missing_api_key',
			esc_html__( 'API key not found. Please connect to Mailchimp first.', 'chimpmatic-lite' ),
			array( 'status' => 400 )
		);
	}

	try {
		$merge_fields_result = wpcf7_mce_get_merge_fields( $api_key, $list_id, $logfileEnabled );
		$merge_fields_data   = $merge_fields_result['merge_fields'] ?? array();

		// Format merge fields - filter out certain types and cap at 6 total
		// Free version: filter out address, birthday, imageurl, zip types
		$excluded_types = array( 'address', 'birthday', 'imageurl', 'zip' );
		$merge_fields   = array();

		if ( $logfileEnabled ) {
			error_log( '[ChimpMatic Lite] [REST API] Fetching merge fields for list: ' . $list_id );
		}

		// Always add EMAIL as the first field (required)
		$merge_fields[] = array(
			'tag'  => 'EMAIL',
			'name' => 'Subscriber Email',
			'type' => 'email',
		);

		if ( isset( $merge_fields_data['merge_fields'] ) && is_array( $merge_fields_data['merge_fields'] ) ) {
			// Sort by display_order to match Mailchimp UI order
			$fields_to_process = $merge_fields_data['merge_fields'];
			usort(
				$fields_to_process,
				function ( $a, $b ) {
					return ( $a['display_order'] ?? 0 ) - ( $b['display_order'] ?? 0 );
				}
			);

			$count = 1; // Start at 1 since EMAIL is already added
			foreach ( $fields_to_process as $field ) {
				$field_type = strtolower( $field['type'] ?? '' );
				$field_tag  = $field['tag'] ?? '';

				// Skip EMAIL if it's in the API response (we already added it)
				if ( $field_tag === 'EMAIL' ) {
					continue;
				}

				// Skip excluded types (address, birthday, imageurl, zip)
				if ( in_array( $field_type, $excluded_types, true ) ) {
					if ( $logfileEnabled ) {
						error_log( '[ChimpMatic Lite] [REST API] Filtering out ' . $field_tag . ' (type: ' . $field_type . ')' );
					}
					continue;
				}

				// Cap at CMATIC_LITE_MAX_FIELDS total fields (including EMAIL)
				if ( $count >= CMATIC_LITE_MAX_FIELDS ) {
					if ( $logfileEnabled ) {
						error_log( '[ChimpMatic Lite] [REST API] Reached ' . CMATIC_LITE_MAX_FIELDS . ' field limit, stopping' );
					}
					break;
				}

				$merge_fields[] = array(
					'tag'  => $field_tag,
					'name' => $field['name'] ?? '',
					'type' => $field_type,
				);
				++$count;
			}
		}

		if ( $logfileEnabled ) {
			error_log( '[ChimpMatic Lite] [REST API] Returning ' . count( $merge_fields ) . ' merge fields' );
		}

		// Save merge fields AND selected list to form settings
		$cf7_mch['merge_fields'] = $merge_fields;
		$cf7_mch['list']         = $list_id; // AUTO-SAVE: Save selected audience
		update_option( $option_name, $cf7_mch );

		// Track API setup funnel: Audience selected
		if ( ! mce_get_cmatic( 'api.audience_selected' ) ) {
			mce_update_cmatic( 'api.audience_selected', time() );
		}

		return rest_ensure_response(
			array(
				'success'      => true,
				'merge_fields' => $merge_fields,
			)
		);

	} catch ( Exception $e ) {
		return new WP_Error(
			'api_request_failed',
			esc_html__( 'Failed to load merge fields. Check debug log for details.', 'chimpmatic-lite' ),
			array( 'status' => 500 )
		);
	}
}

/**
 * ✅ ZOMBIE FIX: Nuclear License Reset Handler
 *
 * Completely wipes ALL license-related data from database and cache.
 * This fixes the zombie activation bug by removing:
 * 1. All license options (activation, state, last_check, etc.)
 * 2. ALL transients (regular + site + timeout records)
 * 3. Object cache (Redis/Memcached)
 * 4. Plugin update transients (forces fresh check)
 *
 * @param WP_REST_Request $request REST request object
 * @return WP_REST_Response|WP_Error Response with results or error
 */

/**
 * REST API: Reset Form Settings
 *
 * Clears all ChimpMatic settings for a form when API key is removed.
 * This ensures clean state on page reload.
 *
 * @param WP_REST_Request $request REST request object
 * @return WP_REST_Response Response with success status
 */
function cmatic_rest_reset_settings( $request ) {
	$form_id     = $request->get_param( 'form_id' );
	$option_name = 'cf7_mch_' . $form_id;

	// Clear all form settings from database
	delete_option( $option_name );

	return rest_ensure_response(
		array(
			'success' => true,
			'message' => 'Form settings cleared successfully',
		)
	);
}

function cmatic_rest_nuclear_license_reset( $request ) {
	global $wpdb;

	$current_user   = wp_get_current_user();
	$username       = $current_user->user_login ?? 'unknown';
	$deleted_counts = array();

	error_log( '[ChimpMatic Lite] NUCLEAR RESET initiated by user: ' . $username );

	// Step 1: Delete all license activation options
	$license_options = array(
		'chimpmatic_license_activation',
		'chimpmatic_license_status',
		'chimpmatic_license_state',
		'chimpmatic_license_last_check',
		'chimpmatic_license_error_state',
		'cmatic_license_activated',
		'cmatic_license_api_key',
		'cmatic_product_id',
		'wc_am_client_chimpmatic',
		'wc_am_product_id_chimpmatic',
		'wc_am_client_chimpmatic_activated',
		'wc_am_client_chimpmatic_instance',
		'wc_am_client_chimpmatic_deactivate_checkbox',
		'chimpmatic_product_id',
	);

	$options_deleted = 0;
	foreach ( $license_options as $option ) {
		if ( delete_option( $option ) ) {
			++$options_deleted;
		}
	}
	$deleted_counts['options'] = $options_deleted;
	error_log( '[ChimpMatic Lite] NUCLEAR RESET: Deleted ' . $options_deleted . ' license options' );

	// Step 2: Delete ALL transients (regular, site, and timeout records)
	$transients_deleted           = $wpdb->query(
		"DELETE FROM {$wpdb->options}
        WHERE option_name LIKE '_transient_%chimpmatic%'
           OR option_name LIKE '_transient_timeout_%chimpmatic%'
           OR option_name LIKE '_site_transient_%chimpmatic%'
           OR option_name LIKE '_site_transient_timeout_%chimpmatic%'
           OR option_name LIKE 'site_transient_%chimpmatic%'
           OR option_name LIKE '_transient_%cmatic%'
           OR option_name LIKE '_transient_timeout_%cmatic%'
           OR option_name LIKE '_site_transient_%cmatic%'
           OR option_name LIKE '_site_transient_timeout_%cmatic%'
           OR option_name LIKE 'site_transient_%cmatic%'"
	);
	$deleted_counts['transients'] = (int) $transients_deleted;
	error_log( '[ChimpMatic Lite] NUCLEAR RESET: Deleted ' . $transients_deleted . ' transients' );

	// Step 3: Flush object cache (Redis/Memcached)
	$cache_flushed = false;
	if ( function_exists( 'wp_cache_flush' ) ) {
		$cache_flushed = wp_cache_flush();
		error_log( '[ChimpMatic Lite] NUCLEAR RESET: Object cache flushed - ' . ( $cache_flushed ? 'success' : 'failed' ) );
	}
	$deleted_counts['cache_flushed'] = $cache_flushed;

	// Step 4: Delete update_plugins transient to force fresh plugin update check
	delete_site_transient( 'update_plugins' );
	error_log( '[ChimpMatic Lite] NUCLEAR RESET: Cleared update_plugins transient' );

	// Step 5: Set status explicitly to deactivated
	update_option( 'chimpmatic_license_status', 'deactivated' );
	error_log( '[ChimpMatic Lite] NUCLEAR RESET: Set license status to deactivated' );

	// Final log entry
	error_log(
		sprintf(
			'[ChimpMatic Lite] NUCLEAR RESET completed by %s - Options: %d, Transients: %d, Cache: %s',
			$username,
			$options_deleted,
			$transients_deleted,
			$cache_flushed ? 'flushed' : 'not flushed'
		)
	);

	return rest_ensure_response(
		array(
			'success'        => true,
			'message'        => 'License data completely wiped (nuclear reset)',
			'deleted_counts' => $deleted_counts,
			'performed_by'   => $username,
			'timestamp'      => current_time( 'mysql' ),
			'actions_taken'  => array(
				'Deleted ' . $options_deleted . ' license options',
				'Deleted ' . $transients_deleted . ' cached transients',
				'Flushed object cache: ' . ( $cache_flushed ? 'yes' : 'no' ),
				'Cleared plugin update cache',
				'Set license status to deactivated',
			),
		)
	);
}
