<?php
/*
 * This file is part of Contact Form 7 Extension For Mailchimp.
 *
 * Copyright (C) 2010-2025, Renzo Johnson (email: renzo.johnson at gmail.com)
 *
 * For the full copyright and license information, please view the LICENSE
 * file that was distributed with this source code, or visit:
 * https://www.gnu.org/licenses/gpl-2.0.html
 */

defined( 'ABSPATH' ) || exit;

add_filter( 'wpcf7_editor_panels', 'cmatic_add_editor_panel' );
add_action( 'wpcf7_after_save', 'cmatic_save_form_settings' );


if ( ! defined( 'CMATIC_VERSION' ) ) {
	add_action( 'wpcf7_before_send_mail', 'cmatic_subscribe_submission' );
}

// Add version info to CF7 Status metabox
add_action( 'wpcf7_admin_misc_pub_section', 'cmatic_add_version_to_submitdiv' );

// Render promotional banner in page footer (visible on all tabs)
// Now uses plain HTML form instead of JavaScript injection
add_action( 'wpcf7_admin_footer', 'cmatic_render_footer_banner', 10, 1 );


function cmatic_add_editor_panel( $panels ) {
	// If Chimpmatic Pro is active, let Pro handle the panel
	if ( defined( 'CMATIC_VERSION' ) ) {
		return $panels;
	}

	$panels['Chimpmatic'] = array(
		'title'    => __( 'Chimpmatic', 'chimpmatic-lite' ),
		'callback' => 'cmatic_editor_panel_content',
	);
	return $panels;
}


function cmatic_editor_panel_content( $contact_form ) {
	$form_id  = $contact_form->id();
	$defaults = array();
	$cf7_mch  = get_option( 'cf7_mch_' . $form_id, $defaults );

	// Initialize variables for the view
	$mce_txcomodin = $form_id;

	// Use CF7's native scanning method to get all form tags with types
	$form_tags_with_types = mce_get_form_tags_with_types( $contact_form );

	// Get checkbox and acceptance fields for Required Acceptance dropdown
	$checkbox_fields = mce_get_checkbox_fields( $contact_form );

	$logfileEnabled = isset( $cf7_mch['logfileEnabled'] ) && '1' == $cf7_mch['logfileEnabled'];
	$mceapi         = $cf7_mch['api'] ?? null;
	$apivalid       = $cf7_mch['api-validation'] ?? null;
	$listdata       = isset( $cf7_mch['lisdata'] ) && is_array( $cf7_mch['lisdata'] ) ? $cf7_mch['lisdata'] : null;

	$chm_valid   = '<span class="chmm valid">API Connected</span>';
	$chm_invalid = '<span class="chmm invalid">API Inactive</span>';

	// The view file contains the HTML for the panel.
	include SPARTAN_MCE_PLUGIN_DIR . '/lib/view.php';
}


function cmatic_save_form_settings( $contact_form ) {
	if ( ! isset( $_POST['wpcf7-mailchimp'] ) ) {
		return;
	}

	check_admin_referer( 'wpcf7-save-contact-form_' . $contact_form->id() );

	$form_id            = $contact_form->id();
	$option_name        = 'cf7_mch_' . $form_id;
	$old_settings       = get_option( $option_name, array() );
	$posted_data        = $_POST['wpcf7-mailchimp'];
	$sanitized_settings = array();

	// Define text fields that need sanitization
	$text_fields = array( 'api', 'list', 'accept' );

	// Dynamically add field mappings based on CMATIC_LITE_MAX_FIELDS
	$max_field_index = CMATIC_LITE_MAX_FIELDS + 2; // +2 because fields start at field3
	for ( $i = 3; $i <= $max_field_index; $i++ ) {
		$text_fields[] = 'field' . $i;
	}

	// Add custom fields for advanced settings
	for ( $i = 1; $i <= 10; $i++ ) {
		$text_fields[] = 'CustomValue' . $i;
		$text_fields[] = 'CustomKey' . $i;
	}

	foreach ( $text_fields as $field ) {
		if ( isset( $posted_data[ $field ] ) ) {
			$sanitized_settings[ $field ] = sanitize_text_field( $posted_data[ $field ] );
		}
	}

	// IMPORTANT: Preserve real API key if submitted value is masked (contains bullet character)
	// The UI masks the key for display, but we must never save the masked version
	if ( isset( $sanitized_settings['api'] ) && strpos( $sanitized_settings['api'], '•' ) !== false ) {
		// Submitted key is masked - preserve the old real key
		if ( ! empty( $old_settings['api'] ) && strpos( $old_settings['api'], '•' ) === false ) {
			$sanitized_settings['api'] = $old_settings['api'];
		}
	}

	// Define checkbox fields
	$checkbox_fields = array(
		'cfactive',
		'confsubs',
		'addunsubscr',
		'logfileEnabled',
		'cf-supp',
	);

	foreach ( $checkbox_fields as $field ) {
		// If the checkbox is checked, its value will be in the POST data.
		// If it's not checked, it won't be set.
		$sanitized_settings[ $field ] = isset( $posted_data[ $field ] ) ? '1' : '0';
	}

	// Check if API key is empty - if so, delete all settings instead of saving
	if ( empty( $sanitized_settings['api'] ) ) {
		delete_option( $option_name );
		return; // Exit early - nothing to save
	}

	// Merge sanitized new settings over the old ones
	$updated_settings = array_merge( $old_settings, $sanitized_settings );

	// FREE VERSION SECURITY: Strip any field mappings beyond CMATIC_LITE_MAX_FIELDS
	// This prevents users from bypassing the limit by modifying POST data
	$max_field_index = CMATIC_LITE_MAX_FIELDS + 2; // +2 because fields start at field3
	for ( $i = $max_field_index + 1; $i <= 20; $i++ ) {
		$field_key = 'field' . $i;
		if ( isset( $updated_settings[ $field_key ] ) ) {
			unset( $updated_settings[ $field_key ] );
			error_log( '[ChimpMatic Lite] [SECURITY] Stripped unauthorized field: ' . $field_key );
		}
	}

	update_option( $option_name, $updated_settings );
}


/**
 * Processes the form submission and subscribes the user to Mailchimp.
 */
function cmatic_subscribe_submission( $contact_form ) {
	// Always log that the hook was triggered
	error_log( '[ChimpMatic Lite] [HOOK TRIGGERED] cmatic_subscribe_submission called for form ID: ' . $contact_form->id() );

	$submission = WPCF7_Submission::get_instance();
	if ( ! $submission ) {
		error_log( '[ChimpMatic Lite] [EXIT] No submission instance found.' );
		return;
	}

	$form_id = $contact_form->id();
	$cf7_mch = get_option( 'cf7_mch_' . $form_id );
	error_log( '[ChimpMatic Lite] [DEBUG] Full cf7_mch settings: ' . print_r( $cf7_mch, true ) );

	// Exit if integration is not configured or API is invalid.
	if ( empty( $cf7_mch ) || empty( $cf7_mch['api-validation'] ) || '1' != $cf7_mch['api-validation'] || empty( $cf7_mch['api'] ) ) {
		error_log(
			'[ChimpMatic Lite] [EXIT] Integration not configured or API invalid. Settings: ' . print_r(
				array(
					'has_settings'   => ! empty( $cf7_mch ),
					'api_validation' => $cf7_mch['api-validation'] ?? 'not set',
					'has_api'        => ! empty( $cf7_mch['api'] ),
				),
				true
			)
		);
		return;
	}

	// FIX: Properly check if the logfileEnabled setting is exactly '1'.
	$logfileEnabled = isset( $cf7_mch['logfileEnabled'] ) && '1' == $cf7_mch['logfileEnabled'];
	$logger         = new Cmatic_File_Logger( 'api-events', $logfileEnabled );

	$posted_data = $submission->get_posted_data();
	error_log( '[ChimpMatic Lite] [DEBUG] Posted data: ' . print_r( $posted_data, true ) );

	$regex = '/\[\s*([a-zA-Z_][0-9a-zA-Z:._-]*)\s*\]/';

	// Find EMAIL field from merge_fields - it should be in field3 (first dynamic field)
	$email = '';
	if ( ! empty( $cf7_mch['merge_fields'] ) && is_array( $cf7_mch['merge_fields'] ) ) {
		foreach ( $cf7_mch['merge_fields'] as $idx => $merge_field ) {
			if ( ( $merge_field['tag'] ?? '' ) === 'EMAIL' ) {
				$field_key = 'field' . ( $idx + 3 ); // field3, field4, etc.
				if ( ! empty( $cf7_mch[ $field_key ] ) ) {
					$email = cmatic_tag_replace( $regex, $cf7_mch[ $field_key ], $posted_data );
					error_log( '[ChimpMatic Lite] [DEBUG] Found EMAIL at ' . $field_key . ': ' . $cf7_mch[ $field_key ] );
				}
				break;
			}
		}
	}
	error_log( '[ChimpMatic Lite] [DEBUG] Email after tag replace: ' . $email );

	// Do not proceed if the email is invalid or empty.
	if ( ! is_email( $email ) ) {
		$logger->log( 'WARNING', 'Subscription attempt with invalid email address.', $email );
		error_log( '[ChimpMatic Lite] [EXIT] Invalid email: ' . $email );
		return;
	}

	$list_id = cmatic_tag_replace( $regex, $cf7_mch['list'] ?? '', $posted_data );
	error_log( '[ChimpMatic Lite] [DEBUG] List ID: ' . $list_id );

	// Determine subscription status
	$status = 'subscribed';
	error_log( '[ChimpMatic Lite] [DEBUG] Checking subscription status. confsubs: ' . ( $cf7_mch['confsubs'] ?? 'not set' ) . ' | accept: ' . ( $cf7_mch['accept'] ?? 'not set' ) );

	if ( ! empty( $cf7_mch['confsubs'] ) ) {
		$status = 'pending'; // Double opt-in
		error_log( '[ChimpMatic Lite] [DEBUG] Status set to pending (double opt-in)' );
	} elseif ( ! empty( $cf7_mch['accept'] ) ) {
		$acceptance_value = cmatic_tag_replace( $regex, $cf7_mch['accept'], $posted_data );
		error_log( '[ChimpMatic Lite] [DEBUG] Acceptance field: ' . $cf7_mch['accept'] . ' | Value: ' . print_r( $acceptance_value, true ) );

		if ( empty( $acceptance_value ) ) {
			if ( ! empty( $cf7_mch['addunsubscr'] ) ) {
				$status = 'unsubscribed';
				error_log( '[ChimpMatic Lite] [DEBUG] Status set to unsubscribed' );
			} else {
				$logger->log( 'INFO', 'Subscription skipped: acceptance checkbox was not checked.' );
				error_log( '[ChimpMatic Lite] [EXIT] Acceptance checkbox not checked, subscription skipped' );
				return;
			}
		}
	}
	error_log( '[ChimpMatic Lite] [DEBUG] Final status: ' . $status );

	// Initialize merge_vars
	$merge_vars = array();

	// Process merge fields from the field mapping system (field3, field4, field5, field6, etc.)
	if ( ! empty( $cf7_mch['merge_fields'] ) && is_array( $cf7_mch['merge_fields'] ) ) {
		error_log( '[ChimpMatic Lite] [DEBUG] Processing ' . count( $cf7_mch['merge_fields'] ) . ' merge fields' );

		$field_index = 3; // Start from field3 (field1 is email)
		foreach ( $cf7_mch['merge_fields'] as $merge_field ) {
			$field_key = 'field' . $field_index;
			$merge_tag = $merge_field['tag'] ?? '';

			if ( ! empty( $cf7_mch[ $field_key ] ) && ! empty( $merge_tag ) ) {
				$cf7_field_config = $cf7_mch[ $field_key ];
				$merge_field_type = $merge_field['type'] ?? '';
				error_log( '[ChimpMatic Lite] [DEBUG] Field ' . $field_index . ' - Merge tag: ' . $merge_tag . ' | Type: ' . $merge_field_type . ' | CF7 field config: ' . $cf7_field_config );

				$value = cmatic_tag_replace( $regex, $cf7_field_config, $posted_data );
				error_log( '[ChimpMatic Lite] [DEBUG] Field ' . $field_index . ' - After tag replace: ' . print_r( $value, true ) );

				if ( ! empty( $value ) ) {
					$merge_vars[ $merge_tag ] = $value;
				}
			}

			++$field_index;

			// FREE VERSION SECURITY: Stop at CMATIC_LITE_MAX_FIELDS (EMAIL + additional merge fields)
			$max_field_index = CMATIC_LITE_MAX_FIELDS + 2; // +2 because fields start at field3
			if ( $field_index > $max_field_index ) {
				error_log( '[ChimpMatic Lite] [SECURITY] Reached field limit (' . CMATIC_LITE_MAX_FIELDS . '), stopping processing' );
				break;
			}
		}
	}

	// Remove empty values (empty strings, null, false, 0, etc.)
	$merge_vars = array_filter(
		$merge_vars,
		function ( $value ) {
			return ! empty( $value ) || $value === 0 || $value === '0';
		}
	);

	error_log( '[ChimpMatic Lite] [DEBUG] Final merge_vars after filtering: ' . print_r( $merge_vars, true ) );

	try {
		error_log( '[ChimpMatic Lite] [DEBUG] Entering try block for API call' );
		error_log( '[ChimpMatic Lite] [DEBUG] About to call logger->log()' );
		$logger->log(
			'INFO',
			'Starting subscription process.',
			array(
				'email'   => $email,
				'list_id' => $list_id,
			)
		);
		error_log( '[ChimpMatic Lite] [DEBUG] Passed logger->log() call' );

		// Build payload - only include merge_fields if we have data
		$payload = array(
			'email_address' => $email,
			'status'        => $status,
		);

		if ( ! empty( $merge_vars ) ) {
			$payload['merge_fields'] = (object) $merge_vars;
		}

		error_log( '[ChimpMatic Lite] [DEBUG] Payload prepared: ' . print_r( $payload, true ) );

		$api_key        = $cf7_mch['api'];
		list($key, $dc) = explode( '-', $api_key );
		$url            = "https://{$dc}.api.mailchimp.com/3.0/lists/{$list_id}/members/" . md5( strtolower( $email ) );
		$info           = wp_json_encode( $payload );

		error_log( '[ChimpMatic Lite] [DEBUG] API URL: ' . $url );
		error_log( '[ChimpMatic Lite] [DEBUG] Sending data to Mailchimp...' );
		error_log( '[ChimpMatic Lite] [DEBUG] About to call logger->log() for sending' );
		$logger->log(
			'INFO',
			'Sending data to Mailchimp.',
			array(
				'url'     => $url,
				'payload' => $payload,
			)
		);
		error_log( '[ChimpMatic Lite] [DEBUG] Passed logger->log() call' );

		error_log( '[ChimpMatic Lite] [DEBUG] About to call callApiPut()' );
		$response_data = callApiPut( $api_key, $url, $info ); // Use PUT for idempotency
		error_log( '[ChimpMatic Lite] [DEBUG] Returned from callApiPut()' );

		error_log( '[ChimpMatic Lite] [DEBUG] Mailchimp API Response: ' . print_r( $response_data, true ) );
		$logger->log( 'INFO', 'Mailchimp API Response.', $response_data );

		if ( ! empty( $response_data[0]['errors'] ) ) {
			$php_logger = new Cmatic_File_Logger( 'php-errors', $logfileEnabled );
			foreach ( $response_data[0]['errors'] as $error ) {
				$php_logger->log( 'ERROR', 'Mailchimp API Error received.', $error );
			}
		} else {
			mce_save_contador();
		}
	} catch ( Exception $e ) {
		$logger->log( 'CRITICAL', 'Subscription process failed with exception.', $e->getMessage() );
	}
}

/**
 * Replaces mail-tags in a string with their corresponding values from submitted data.
 */
function cmatic_tag_replace( $pattern, $subject, $posted_data ) {
	if ( preg_match( $pattern, $subject, $matches ) > 0 ) {
		if ( isset( $posted_data[ $matches[1] ] ) ) {
			$submitted = $posted_data[ $matches[1] ];
			return is_array( $submitted ) ? implode( ', ', $submitted ) : $submitted;
		}
		return $matches[0]; // Return the tag itself if no value found
	}
	return $subject; // Return original string if no tag found
}

/**
 * Increments the total number of successful submissions.
 */
function mce_save_contador() {
	$count = (int) mce_get_cmatic( 'stats.sent', 0 );
	mce_update_cmatic( 'stats.sent', $count + 1 );
}



/**
 * Generate HTML for list selection panel
 */
function mce_html_panel_listmail( $apivalid, $listdata, $cf7_mch ) {
	$vlist = isset( $cf7_mch['list'] ) ? sanitize_text_field( $cf7_mch['list'] ) : '';
	$count = isset( $listdata['lists'] ) && is_array( $listdata['lists'] ) ? count( $listdata['lists'] ) : 0;
	?>
	<?php // API validation status now stored in #cmatic_data data-api-valid attribute (see Cmatic_Data_Container). ?>
	<?php if ( $apivalid == 1 && $count > 0 ) : ?>
		<label for="wpcf7-mailchimp-list">
			<?php printf( esc_html__( 'Total Mailchimp Audiences: %d', 'chimpmatic-lite' ), $count ); ?>
		</label><br/>
		<select id="wpcf7-mailchimp-list" name="wpcf7-mailchimp[list]">
			<?php foreach ( $listdata['lists'] as $i => $list ) : ?>
				<?php
				if ( ! is_array( $list ) || ! isset( $list['id'], $list['name'] ) ) {
					continue;}
				?>
				<?php
					// Get member count and field count from stats
					$member_count = isset( $list['stats']['member_count'] ) ? intval( $list['stats']['member_count'] ) : 0;
					$field_count  = isset( $list['stats']['merge_field_count'] ) ? intval( $list['stats']['merge_field_count'] ) : 0;

					// Format: 1:117 Lalito  12 fields #2b68cca472
					$option_text = ( $i + 1 ) . ':' . $member_count . ' ' . $list['name'] . '  ' . $field_count . ' fields #' . $list['id'];
				?>
				<option value="<?php echo esc_attr( $list['id'] ); ?>" <?php selected( $vlist, $list['id'] ); ?>>
					<?php echo esc_html( $option_text ); ?>
				</option>
			<?php endforeach; ?>
		</select>

		<button type="button" id="mce_fetch_fields" class="button" <?php echo empty( $vlist ) ? 'disabled' : ''; ?>>
			<?php esc_html_e( 'Sync Fields', 'chimpmatic-lite' ); ?>
		</button>

	<?php elseif ( $apivalid == 1 ) : ?>
		<p><?php esc_html_e( 'No lists found in your Mailchimp account.', 'chimpmatic-lite' ); ?></p>
		<?php
	endif;
}

/**
 * Render field mapping dropdown using CF7's native mail-tags
 * Uses CF7's scan_form_tags() method - no guesswork!
 *
 * @param string $field_name Field name (email, name, field3, etc.)
 * @param array  $form_tags Array of form tags with name and basetype
 * @param array  $cf7_mch Saved settings
 * @param string $filter Optional filter (email, text, etc.)
 */
function mce_render_field_dropdown( $field_name, $form_tags, $cf7_mch, $filter = '' ) {
	$field_name  = sanitize_key( $field_name );
	$saved_value = isset( $cf7_mch[ $field_name ] ) ? sanitize_text_field( $cf7_mch[ $field_name ] ) : ' ';

	// Set default for email field
	if ( $field_name === 'email' && $saved_value === ' ' ) {
		// Try to find an email field in form_tags
		foreach ( $form_tags as $tag ) {
			if ( $tag['basetype'] === 'email' || strpos( strtolower( $tag['name'] ), 'email' ) !== false ) {
				$saved_value = '[' . $tag['name'] . ']';
				break;
			}
		}
	}

	?>
	<select class="chm-select" id="wpcf7-mailchimp-<?php echo esc_attr( $field_name ); ?>" name="wpcf7-mailchimp[<?php echo esc_attr( $field_name ); ?>]" style="width:95%">
		<?php if ( $field_name !== 'email' ) : ?>
			<option value=" " <?php selected( $saved_value, ' ' ); ?>>
				<?php esc_html_e( 'Choose..', 'chimpmatic-lite' ); ?>
			</option>
		<?php endif; ?>

		<?php foreach ( $form_tags as $tag ) : ?>
			<?php
				// Skip opt-in fields
			if ( $tag['name'] === 'opt-in' ) {
				continue;
			}

				// Apply filter for email field - only show email type fields
			if ( $field_name === 'email' && $tag['basetype'] !== 'email' ) {
				continue;
			}

				$tag_value = '[' . $tag['name'] . ']';
			?>
			<option value="<?php echo esc_attr( $tag_value ); ?>" <?php selected( $saved_value, $tag_value ); ?>>
				<?php echo esc_html( $tag_value ); ?> - type: <?php echo esc_html( $tag['basetype'] ); ?>
			</option>
		<?php endforeach; ?>
	</select>
	<?php
}


function mce_get_form_tags_with_types( $contact_form ) {
	if ( ! $contact_form ) {
		return array();
	}

	// Get mail tags (excludes non-mail fields like submit buttons)
	$mail_tags = $contact_form->collect_mail_tags();

	// Scan all form tags to get type information
	$all_tags = $contact_form->scan_form_tags();

	// Build a map of tag name => basetype
	$tags_with_types = array();
	foreach ( $all_tags as $tag ) {
		if ( ! empty( $tag->name ) && in_array( $tag->name, $mail_tags ) ) {
			$tags_with_types[] = array(
				'name'     => $tag->name,
				'basetype' => $tag->basetype,
			);
		}
	}

	return $tags_with_types;
}

/**
 * Get only checkbox and acceptance fields from CF7 form
 * Used for Required Acceptance dropdown
 *
 * @param WPCF7_ContactForm $contact_form The contact form object
 * @return array Array of form tags with name, type, and basetype
 */
function mce_get_checkbox_fields( $contact_form ) {
	if ( ! $contact_form ) {
		return array();
	}

	// Scan form tags with type filter for checkbox and acceptance
	$tags = $contact_form->scan_form_tags( array( 'type' => array( 'checkbox', 'acceptance' ) ) );

	$checkbox_fields = array();
	foreach ( $tags as $tag ) {
		if ( ! empty( $tag->name ) ) {
			$checkbox_fields[] = array(
				'name'     => $tag->name,
				'type'     => $tag->type,
				'basetype' => $tag->basetype,
			);
		}
	}

	return $checkbox_fields;
}

/**
 * Validate Mailchimp API key
 */
function wpcf7_mce_validate_api_key( $input, $logfileEnabled, $idform = '' ) {
	$logger = new Cmatic_File_Logger( 'API-Validation', $logfileEnabled );
	try {
		if ( empty( $input ) || ! preg_match( '/^[a-f0-9]{32}-[a-z]{2,3}\d+$/', $input ) ) {
			$logger->log( 'ERROR', 'Invalid API Key format provided.', $input );

			// Track API setup funnel: Failure
			if ( ! mce_get_cmatic( 'api.setup_first_failure' ) ) {
				mce_update_cmatic( 'api.setup_first_failure', time() );
			}
			mce_update_cmatic( 'api.setup_last_failure', time() );
			$failure_count = (int) mce_get_cmatic( 'api.setup_failure_count', 0 );
			mce_update_cmatic( 'api.setup_failure_count', $failure_count + 1 );

			return array( 'api-validation' => 0 );
		}

		list($key, $dc) = explode( '-', $input );
		if ( empty( $key ) || empty( $dc ) ) {
			// Track API setup funnel: Failure
			if ( ! mce_get_cmatic( 'api.setup_first_failure' ) ) {
				mce_update_cmatic( 'api.setup_first_failure', time() );
			}
			mce_update_cmatic( 'api.setup_last_failure', time() );
			$failure_count = (int) mce_get_cmatic( 'api.setup_failure_count', 0 );
			mce_update_cmatic( 'api.setup_failure_count', $failure_count + 1 );

			return array( 'api-validation' => 0 );
		}

		$urltoken = "https://{$dc}.api.mailchimp.com/3.0/ping";
		$response = callApiGet( $key, $urltoken );

		if ( is_wp_error( $response[2] ) || 200 !== wp_remote_retrieve_response_code( $response[2] ) ) {
			$error = is_wp_error( $response[2] ) ? $response[2]->get_error_message() : 'HTTP ' . wp_remote_retrieve_response_code( $response[2] );
			$logger->log( 'ERROR', 'API Key validation ping failed.', $error );

			// Track API setup funnel: Failure
			if ( ! mce_get_cmatic( 'api.setup_first_failure' ) ) {
				mce_update_cmatic( 'api.setup_first_failure', time() );
			}
			mce_update_cmatic( 'api.setup_last_failure', time() );
			$failure_count = (int) mce_get_cmatic( 'api.setup_failure_count', 0 );
			mce_update_cmatic( 'api.setup_failure_count', $failure_count + 1 );

			return array( 'api-validation' => 0 );
		}

		$logger->log( 'INFO', 'API Key validated successfully.' );

		// Track API setup funnel: First success
		if ( ! mce_get_cmatic( 'api.setup_first_success' ) ) {
			mce_update_cmatic( 'api.setup_first_success', time() );
		}

		return array( 'api-validation' => 1 );
	} catch ( Exception $e ) {
		$logger->log( 'CRITICAL', 'API validation threw an exception.', $e->getMessage() );

		// Track API setup funnel: Failure
		if ( ! mce_get_cmatic( 'api.setup_first_failure' ) ) {
			mce_update_cmatic( 'api.setup_first_failure', time() );
		}
		mce_update_cmatic( 'api.setup_last_failure', time() );
		$failure_count = (int) mce_get_cmatic( 'api.setup_failure_count', 0 );
		mce_update_cmatic( 'api.setup_failure_count', $failure_count + 1 );

		return array( 'api-validation' => 0 );
	}
}

/**
 * Get associated Mailchimp lists
 */
function wpcf7_mce_listasasociadas( $apikey, $logfileEnabled, $idform, $apivalid ) {
	$logger = new Cmatic_File_Logger( 'List-Retrieval', $logfileEnabled );
	if ( $apivalid != 1 ) {
		return array( 'lisdata' => array() );
	}

	try {
		list($key, $dc) = explode( '-', $apikey );
		if ( empty( $key ) || empty( $dc ) ) {
			return array( 'lisdata' => array() );
		}

		$urltoken = "https://{$dc}.api.mailchimp.com/3.0/lists?count=999";
		$response = callApiGet( $key, $urltoken );

		if ( is_wp_error( $response[2] ) || 200 !== wp_remote_retrieve_response_code( $response[2] ) ) {
			$error = is_wp_error( $response[2] ) ? $response[2]->get_error_message() : 'HTTP ' . wp_remote_retrieve_response_code( $response[2] );
			$logger->log( 'ERROR', 'Failed to retrieve lists from Mailchimp.', $error );
			return array( 'lisdata' => array() );
		}

		$lists_data = $response[0];

		// Two-step flow: Only fetch lists here, fields are fetched separately
		// User will select a list, then click "Fetch Your Fields and Groups" button

		$logger->log( 'INFO', 'Successfully retrieved lists from Mailchimp.', $lists_data );
		return array(
			'lisdata'      => $lists_data,
			'merge_fields' => array(), // Empty - fields fetched separately now
		);
	} catch ( Exception $e ) {
		$logger->log( 'CRITICAL', 'List retrieval threw an exception.', $e->getMessage() );
		return array( 'lisdata' => array() );
	}
}

/**
 * Get merge fields for a specific Mailchimp list
 */
function wpcf7_mce_get_merge_fields( $apikey, $list_id, $logfileEnabled ) {
	$logger = new Cmatic_File_Logger( 'Merge-Fields-Retrieval', $logfileEnabled );

	if ( empty( $apikey ) || empty( $list_id ) ) {
		return array( 'merge_fields' => array() );
	}

	try {
		list($key, $dc) = explode( '-', $apikey );
		if ( empty( $key ) || empty( $dc ) ) {
			return array( 'merge_fields' => array() );
		}

		$urltoken = "https://{$dc}.api.mailchimp.com/3.0/lists/{$list_id}/merge-fields?count=50";
		$response = callApiGet( $key, $urltoken );

		if ( is_wp_error( $response[2] ) || 200 !== wp_remote_retrieve_response_code( $response[2] ) ) {
			$error = is_wp_error( $response[2] ) ? $response[2]->get_error_message() : 'HTTP ' . wp_remote_retrieve_response_code( $response[2] );
			$logger->log( 'ERROR', 'Failed to retrieve merge fields from Mailchimp.', $error );
			return array( 'merge_fields' => array() );
		}

		$merge_fields_data = $response[0];
		$logger->log( 'INFO', 'Successfully retrieved merge fields from Mailchimp.', $merge_fields_data );
		return array( 'merge_fields' => $merge_fields_data );
	} catch ( Exception $e ) {
		$logger->log( 'CRITICAL', 'Merge fields retrieval threw an exception.', $e->getMessage() );
		return array( 'merge_fields' => array() );
	}
}

/**
 * Display help for new users
 */
function chmp_new_usr() {
	$help_url = 'https://chimpmatic.com/how-to-find-your-mailchimp-api-key' . vc_utm() . 'NewUserMC-api';
	?>
	<h2>
		<a href="<?php echo esc_url( $help_url ); ?>" class="helping-field" target="_blank" rel="noopener noreferrer">
			<?php esc_html_e( 'How to get your Mailchimp API key.', 'chimpmatic-lite' ); ?>
		</a>
	</h2>
	<?php
}

/**
 * Add ChimpMatic Lite version info to CF7 Status metabox
 *
 * Displays plugin version, API connection status, and sent count
 * in the Contact Form 7 admin #submitdiv metabox.
 *
 * @param int $post_id Contact form post ID.
 * @since 0.9.27
 */
function cmatic_add_version_to_submitdiv( $post_id ) {
	// Get current form's MCE data
	$cf7_mch   = get_option( 'cf7_mch_' . $post_id, array() );
	$apivalid  = $cf7_mch['api-validation'] ?? null;
	$mce_sents = mce_get_cmatic( 'stats.sent', 0 );

	// Determine API status components
	if ( isset( $apivalid ) && '1' == $apivalid ) {
		$status_icon = '<span class="chmm valid">✓</span>';
		$status_text = '<span class="chmm valid">API Connected</span>';
	} else {
		$status_icon = '<span class="chmm invalid">✗</span>';
		$status_text = '<span class="chmm invalid">API Inactive</span>';
	}

	?>
	<div class="misc-pub-section chimpmatic-info" id="chimpmatic-version-info">
		<div style="margin-bottom: 3px;">
			<strong><?php echo esc_html__( 'ChimpMatic Lite', 'chimpmatic-lite' ) . ' ' . esc_html( SPARTAN_MCE_VERSION ); ?></strong>
		</div>
		<div style="margin-top: 5px;">
			<div class="mc-stats" style="color: #646970; font-size: 12px; margin-bottom: 3px;">
				<?php
				echo esc_html( $mce_sents ) . ' synced contacts in ' .
					esc_html( mce_diferdays_dateact_date() ) . ' days';
				?>
			</div>
			<div style="margin-bottom: 3px;">
				<?php echo $status_text; ?>
			</div>
		</div>
	</div>
	<?php
}


/**
 * Render promotional banner in CF7 form edit page footer
 *
 * Displays the ChimpMatic Lite upgrade banner at the bottom of all CF7 form
 * edit pages using the wpcf7_admin_footer hook. This makes the banner visible
 * on all tabs (Form, Mail, Messages, ChimpMatic) instead of just the ChimpMatic tab.
 *
 * Now uses plain HTML form instead of JavaScript injection since the banner is
 * outside CF7's admin form (so CF7 doesn't strip the form element).
 *
 * @param object $post WPCF7_ContactForm object.
 * @since 0.9.27
 * @since 0.9.28 Updated to use plain HTML form instead of JavaScript injection
 */
function cmatic_render_footer_banner( $post ) {
	// Get current user data for auto-population
	$current_user = wp_get_current_user();
	$user_name    = '';
	$user_email   = '';

	if ( $current_user->ID ) {
		$user_name  = $current_user->display_name;
		$user_email = $current_user->user_email;
	}

	// Fetch dynamic pricing from api.chimpmatic.com (JSON endpoint - no visit limits)
	$pricing_fetcher = new CMatic_Remote_Fetcher(
		array(
			'url'            => 'https://api.chimpmatic.com/promo',
			'cache_key'      => 'cmatic_pricing_data',
			'cache_duration' => DAY_IN_SECONDS,
			'fallback_data'  => array(
				'regular_price'    => 39,
				'sale_price'       => 29.25,
				'discount_percent' => 40,
				'coupon_code'      => 'NOW40',
				'formatted'        => '$39 → $29.25 • Save 40%',
			),
		)
	);

	$pricing_data     = $pricing_fetcher->get_data();
	$pricing_text     = isset( $pricing_data['formatted'] ) ? $pricing_data['formatted'] : '$39 → $29.25 • Save 40%';
	$discount_percent = isset( $pricing_data['discount_percent'] ) ? (int) $pricing_data['discount_percent'] : 40;

	// Get anonymous installation ID from modern cmatic structure.
	$install_id = mce_get_cmatic( 'install.id' );

	// Generate on-demand if missing (shouldn't happen after migration, but safety fallback).
	if ( ! $install_id ) {
		mce_migrate_to_cmatic_option(); // Trigger migration
		$install_id = mce_get_cmatic( 'install.id', 'unknown' );
	}

	// Build destination URL with promo tracking + GA4 campaign parameters.
	$pursuit_addy = add_query_arg(
		array(
			'source'       => $install_id,               // Anonymous install ID for coupon generation
			'promo'        => 'pro' . $discount_percent, // Dynamic promo type (pro40, pro25, etc.)
			'utm_source'   => 'wordpress-admin',
			'utm_medium'   => 'in-plugin-promo',
			'utm_campaign' => 'lite-to-pro-2025',
			'utm_content'  => 'discount-' . $discount_percent,
			'utm_term'     => 'admin-banner',
		),
		'https://chimpmatic.com/almost-there'
	);

	?>
	<div id="informationdiv_aux" class="postbox mce-move mc-lateral" style="display:none">
		<div class="inside bg-f2">
			<h3>Upgrade to PRO</h3>
			<p>Get the best Contact Form 7 and Mailchimp integration tool available. Now with these new features:</p>
			<ul>
				<li>Tag Existing Subscribers</li>
				<li>Group Existing Subscribers</li>
				<li>Email Verification</li>
				<li>AWESOME Support And more!</li>
			</ul>
		</div>
		<div class="promo-2022">
			<h1><?php echo (int) $discount_percent; ?><span>%</span> Off!</h1>
			<p class="interesting">Unlock advanced tagging, subscriber groups, email verification, and priority support for your Mailchimp campaigns.</p>
			<div class="cm-form">

				<form action="<?php echo esc_url( $pursuit_addy ); ?>" target="_blank" method="post" class="cm-promo-form">
					<div class="cmatic-out">
						<label>
							<span class="your-name">
								<input type="text" name="your-name" size="40" class="cm-form-control cm-text" placeholder="Your Name" required value="<?php echo esc_attr( $user_name ); ?>" />
							</span>
						</label>

						<label>
							<span class="your-email">
								<input type="email" name="your-email" size="40" class="cm-form-control cm-email" placeholder="Your Email" required value="<?php echo esc_attr( $user_email ); ?>" />
							</span>
						</label>
						</div>
						<input type="submit" value="Get PRO Now" class="button cm-submit" />
						<br>
						<span class="cm-pricing"><?php echo esc_html( $pricing_text ); ?></span>

  					<input type="hidden" name="_wpcf7" value="10181" />
  					<input type="hidden" name="_wpcf7_version" value="6.1.2" />
  					<input type="hidden" name="_wpcf7_locale" value="en_US" />
  					<input type="hidden" name="_wpcf7_unit_tag" value="wpcf7-f10181-p10182-o1" />
  					<input type="hidden" name="_wpcf7_container_post" value="10182" />
				</form>

			</div>
		</div>
	</div>
	<?php
}

