<?php
/*
 * This file is part of Contact Form 7 Extension For Mailchimp.
 *
 * Copyright (C) 2010-2025, Renzo Johnson (email: renzo.johnson at gmail.com)
 *
 * For the full copyright and license information, please view the LICENSE
 * file that was distributed with this source code, or visit:
 * https://www.gnu.org/licenses/gpl-2.0.html
 */

defined( 'ABSPATH' ) || exit;


class Cmatic_Buster {


	private $plugin_version;


	private $is_debug;


	public function __construct( $plugin_version = SPARTAN_MCE_VERSION, $is_debug = null ) {
		$this->plugin_version = $plugin_version;
		$this->is_debug       = $is_debug ?? ( defined( 'WP_DEBUG' ) && WP_DEBUG );
	}


	public function get_version( $file_path ) {
		$version_parts = array( $this->plugin_version );

		// Add file modification time (updates when file changes)
		if ( file_exists( $file_path ) ) {
			$version_parts[] = filemtime( $file_path );

			// Add short hash of file contents (8 chars for brevity)
			$version_parts[] = substr( md5_file( $file_path ), 0, 8 );
		}

		// In development: add timestamp for aggressive cache busting
		if ( $this->is_debug ) {
			$version_parts[] = time();
		}

		return implode( '-', $version_parts );
	}

	/**
	 * Get singleton instance
	 *
	 * @return Cmatic_Buster
	 */
	public static function instance() {
		static $instance = null;

		if ( null === $instance ) {
			$instance = new self();
		}

		return $instance;
	}
}


function cmatic_enqueue_admin_assets( $hook_suffix ) {

	// Check if we're on any Contact Form 7 admin page
	if ( false === strpos( $hook_suffix, 'wpcf7' ) ) {
		return;
	}

	$css_file_path = SPARTAN_MCE_PLUGIN_DIR . 'assets/css/chimpmatic-lite.css';
	wp_enqueue_style(
		'chimpmatic-lite-css',
		SPARTAN_MCE_PLUGIN_URL . 'assets/css/chimpmatic-lite.css',
		array(),
		Cmatic_Buster::instance()->get_version( $css_file_path )
	);

	cmatic_enqueue_lite_js( $hook_suffix );

	$is_pro_installed = defined( 'CMATIC_VERSION' );
	$is_pro_blessed   = function_exists( 'cmatic_is_blessed' ) && cmatic_is_blessed();

	if ( $is_pro_installed ) {
		cmatic_enqueue_pro_js( $hook_suffix, $is_pro_blessed );
	}
}

add_action( 'admin_enqueue_scripts', 'cmatic_enqueue_admin_assets' );

/**
 * Add body class to CF7 admin pages.
 *
 * @param string $classes Space-separated body classes.
 * @return string Modified body classes.
 */
function cmatic_admin_body_class( $classes ) {
	$screen = get_current_screen();

	if ( $screen && strpos( $screen->id, 'wpcf7' ) !== false ) {
		$classes .= ' chimpmatic-lite';

		// Add Pro class if Chimpmatic Pro is active and licensed.
		if ( function_exists( 'cmatic_is_blessed' ) && cmatic_is_blessed() ) {
			$classes .= ' chimpmatic';
		}
	}

	return $classes;
}
add_filter( 'admin_body_class', 'cmatic_admin_body_class' );


function cmatic_enqueue_lite_js( $hook_suffix ) {

	$js_file_path = SPARTAN_MCE_PLUGIN_DIR . 'assets/js/chimpmatic-lite.js';
	wp_enqueue_script(
		'chimpmatic-lite-js',
		SPARTAN_MCE_PLUGIN_URL . 'assets/js/chimpmatic-lite.js',
		array(),
		Cmatic_Buster::instance()->get_version( $js_file_path ),
		true
	);

	// Get saved merge fields for this form (if on a form edit page)
	$form_id         = isset( $_GET['post'] ) ? absint( $_GET['post'] ) : 0;
	$merge_fields    = array();
	$logging_enabled = false;

	if ( $form_id > 0 ) {
		$option_name = 'cf7_mch_' . $form_id;
		$cf7_mch     = get_option( $option_name, array() );

		// Extract merge fields from saved data
		if ( isset( $cf7_mch['merge_fields'] ) && is_array( $cf7_mch['merge_fields'] ) ) {
			$merge_fields = $cf7_mch['merge_fields'];
		}

		// Check if logging is enabled for this form
		$logging_enabled = ! empty( $cf7_mch['logfileEnabled'] );
	}

	wp_localize_script(
		'chimpmatic-lite-js',
		'chimpmaticLite',
		array(
			'restUrl'         => esc_url_raw( rest_url( 'chimpmatic-lite/v1/' ) ),
			'restNonce'       => wp_create_nonce( 'wp_rest' ),
			'licenseResetUrl' => esc_url_raw( rest_url( 'chimpmatic-lite/v1/license/nuclear-reset' ) ),
			'nonce'           => wp_create_nonce( 'wp_rest' ),
			'pluginUrl'       => SPARTAN_MCE_PLUGIN_URL,
			'mergeFields'     => $merge_fields,
			'loggingEnabled'  => $logging_enabled,
			'i18n'            => array(
				'loading'       => __( 'Loading...', 'chimpmatic-lite' ),
				'error'         => __( 'An error occurred. Check the browser console for details.', 'chimpmatic-lite' ),
				'apiKeyValid'   => __( 'API Connected', 'chimpmatic-lite' ),
				'apiKeyInvalid' => __( 'API Inactive', 'chimpmatic-lite' ),
			),
		)
	);
}


function cmatic_enqueue_pro_js( $hook_suffix, $is_pro_blessed ) {

	$pro_js_path = SPARTAN_MCE_PLUGIN_DIR . 'assets/js/chimpmatic.js';

	wp_enqueue_script(
		'chimpmatic-pro',
		SPARTAN_MCE_PLUGIN_URL . 'assets/js/chimpmatic.js',
		array(),
		Cmatic_Buster::instance()->get_version( $pro_js_path ),
		true
	);

	// PRO configuration (includes blessing status for feature gating)
	wp_localize_script(
		'chimpmatic-pro',
		'chmAjax',
		array(
			'restUrl'   => rest_url( 'chimpmatic/v1/' ),
			'nonce'     => wp_create_nonce( 'wp_rest' ),
			'isBlessed' => $is_pro_blessed, // Pass blessing status to JS
		)
	);

	wp_localize_script(
		'chimpmatic-pro',
		'wpApiSettings',
		array(
			'root'  => esc_url_raw( rest_url() ),
			'nonce' => wp_create_nonce( 'wp_rest' ),
		)
	);
}


function cmatic_enqueue_notices_script( $hook_suffix ) {

	// Only load on CF7 pages (same condition as admin CSS)
	if ( false === strpos( $hook_suffix, 'wpcf7' ) ) {
		return;
	}

	$notices_js_path = SPARTAN_MCE_PLUGIN_DIR . 'assets/js/chimpmatic-lite-notices.js';
	wp_enqueue_script(
		'chimpmatic-lite-notices',
		SPARTAN_MCE_PLUGIN_URL . 'assets/js/chimpmatic-lite-notices.js',
		array(),
		Cmatic_Buster::instance()->get_version( $notices_js_path ),
		true
	);

	wp_localize_script(
		'chimpmatic-lite-notices',
		'chimpmaticNotices',
		array(
			'restUrl'   => esc_url_raw( rest_url( 'chimpmatic-lite/v1' ) ),
			'restNonce' => wp_create_nonce( 'wp_rest' ),
		)
	);

	$promo_js_path = SPARTAN_MCE_PLUGIN_DIR . 'assets/js/chimpmatic-promo-form.js';
	wp_enqueue_script(
		'chimpmatic-promo-form',
		SPARTAN_MCE_PLUGIN_URL . 'assets/js/chimpmatic-promo-form.js',
		array(),
		Cmatic_Buster::instance()->get_version( $promo_js_path ),
		true
	);

	$current_user = wp_get_current_user();
	wp_localize_script(
		'chimpmatic-promo-form',
		'chimpmaticUserData',
		array(
			'userName'   => $current_user->display_name,
			'userEmail'  => $current_user->user_email,
			'isLoggedIn' => is_user_logged_in(),
		)
	);
}
add_action( 'admin_enqueue_scripts', 'cmatic_enqueue_notices_script' );
