<?php
/**
 * ClickSyncr Metrics - Tracker
 *
 * @package ClickSyncr\Metrics\Core
 * @since 1.0.0
 */

/*
 * Copyright (C) 2010-2025, Renzo Johnson
 * License: GPL-2.0+
 */

namespace ClickSyncr\Metrics\Core;

defined( 'ABSPATH' ) || exit;

/**
 * Tracker class
 */
class Tracker {

	/**
	 * @return void
	 */
	public static function init() {
		add_action( 'csyncr_metrics_on_activation', array( __CLASS__, 'on_activation' ) );
		add_action( 'csyncr_metrics_on_deactivation', array( __CLASS__, 'on_deactivation' ) );
	}

	/**
	 * @return void
	 */
	public static function on_activation() {
		Storage::init();
		Storage::record_activation();
		Storage::set_schedule( 'super_frequent' );

		if ( ! wp_next_scheduled( 'csyncr_metrics_heartbeat' ) ) {
			wp_schedule_event( time() + ( 2 * MINUTE_IN_SECONDS ), 'csyncr_2min', 'csyncr_metrics_heartbeat' );
		}

		self::send_event_heartbeat( 'activation' );
	}

	/**
	 * @return void
	 */
	public static function on_deactivation() {
		Storage::record_deactivation();

		$timestamp = wp_next_scheduled( 'csyncr_metrics_heartbeat' );
		if ( $timestamp ) {
			wp_unschedule_event( $timestamp, 'csyncr_metrics_heartbeat' );
		}

		self::send_event_heartbeat( 'deactivation', true );
	}

	/**
	 * @param string $event Event type.
	 * @param bool   $force Force send.
	 * @return void
	 */
	private static function send_event_heartbeat( $event, $force = false ) {
		if ( ! $force && ! Storage::is_enabled() ) {
			return;
		}

		$payload = Collector::collect( $event );
		Sync::send_async( $payload );
	}

	/**
	 * @return void
	 */
	public static function on_opt_out() {
		Storage::increment_disabled_count();

		$payload = Collector::collect( 'opt_out' );
		Sync::send_async( $payload );

		$timestamp = wp_next_scheduled( 'csyncr_metrics_heartbeat' );
		if ( $timestamp ) {
			wp_unschedule_event( $timestamp, 'csyncr_metrics_heartbeat' );
		}
	}

	/**
	 * @return void
	 */
	public static function on_re_enable() {
		if ( ! mce_get_cmatic( 'telemetry.opt_in_date' ) ) {
			mce_update_cmatic( 'telemetry.opt_in_date', time() );
		}

		Storage::set_schedule( 'super_frequent' );

		if ( ! wp_next_scheduled( 'csyncr_metrics_heartbeat' ) ) {
			wp_schedule_event( time() + ( 2 * MINUTE_IN_SECONDS ), 'csyncr_2min', 'csyncr_metrics_heartbeat' );
		}

		$payload = Collector::collect( 'reactivation' );
		Sync::send_async( $payload );
	}
}
