<?php
/**
 * ClickSyncr Metrics - Sync
 *
 * @package ClickSyncr\Metrics\Core
 * @since 1.0.0
 */

/*
 * Copyright (C) 2010-2025, Renzo Johnson
 * License: GPL-2.0+
 */

namespace ClickSyncr\Metrics\Core;

use ClickSyncr\Metrics\Security\Signature;

defined( 'ABSPATH' ) || exit;

/**
 * Sync class
 */
class Sync {

	/** @var string */
	private static $endpoint_url = '';

	/**
	 * @param string $url Endpoint URL.
	 * @return void
	 */
	public static function set_endpoint( $url ) {
		self::$endpoint_url = $url;
	}

	/**
	 * @param array $payload Telemetry payload.
	 * @return bool
	 */
	public static function send( $payload ) {
		try {
			if ( empty( self::$endpoint_url ) ) {
				return false;
			}

			$request = self::prepare_request( $payload );

			$response = wp_remote_post(
				self::$endpoint_url,
				array(
					'body'    => wp_json_encode( $request ),
					'headers' => array( 'Content-Type' => 'application/json' ),
					'timeout' => 5,
				)
			);

			return self::handle_response( $response, $payload );
		} catch ( \Exception $e ) {
			return false;
		}
	}

	/**
	 * @param array $payload Telemetry payload.
	 * @return void
	 */
	public static function send_async( $payload ) {
		try {
			if ( empty( self::$endpoint_url ) ) {
				return;
			}

			$request = self::prepare_request( $payload );

			wp_remote_post(
				self::$endpoint_url,
				array(
					'body'     => wp_json_encode( $request ),
					'headers'  => array( 'Content-Type' => 'application/json' ),
					'timeout'  => 5,
					'blocking' => false,
				)
			);

			Storage::update_last_heartbeat();
		} catch ( \Exception $e ) {
			return;
		}
	}

	/**
	 * @param array $payload Telemetry payload.
	 * @return array
	 */
	private static function prepare_request( $payload ) {
		$install_id   = Storage::get_install_id();
		$timestamp    = time();
		$payload_json = wp_json_encode( $payload );
		$signature    = Signature::generate( $install_id, $timestamp, $payload_json );

		return array(
			'install_id'   => $install_id,
			'timestamp'    => $timestamp,
			'signature'    => $signature,
			'public_key'   => Signature::get_public_key(),
			'payload_json' => $payload_json,
		);
	}

	/**
	 * @param array|WP_Error $response WordPress HTTP response.
	 * @param array          $payload Original payload.
	 * @return bool
	 */
	private static function handle_response( $response, $payload ) {
		if ( is_wp_error( $response ) ) {
			self::handle_failure( $response->get_error_message(), $payload );
			return false;
		}

		$code = wp_remote_retrieve_response_code( $response );

		if ( 200 !== $code && 201 !== $code ) {
			$body = wp_remote_retrieve_body( $response );
			self::handle_failure( "HTTP {$code}: {$body}", $payload );
			return false;
		}

		self::handle_success( $payload );
		return true;
	}

	/**
	 * @param array $payload Original payload.
	 * @return void
	 */
	private static function handle_success( $payload ) {
		Storage::update_last_heartbeat();

		$payload_hash = md5( wp_json_encode( $payload ) );
		Storage::save_payload_hash( $payload_hash );
		Storage::save_error( '' );

		if ( 'sparse' === Storage::get_schedule() ) {
			Scheduler::schedule_next_sparse();
		}
	}

	/**
	 * @param string $error Error message.
	 * @param array  $payload Original payload.
	 * @return void
	 */
	private static function handle_failure( $error, $payload ) {
		Storage::update_last_heartbeat();
		Storage::increment_failed_count();
		Storage::save_error( $error );

		if ( defined( 'WP_DEBUG' ) && WP_DEBUG ) {
			error_log( "[ChimpMatic Metrics] Heartbeat failed: {$error}" );
		}

		if ( 'sparse' === Storage::get_schedule() ) {
			Scheduler::schedule_next_sparse();
		}
	}
}
