<?php
/**
 * ClickSyncr Metrics - Storage
 *
 * @package ClickSyncr\Metrics\Core
 * @since 1.0.0
 */

/*
 * Copyright (C) 2010-2025, Renzo Johnson
 * License: GPL-2.0+
 */

namespace ClickSyncr\Metrics\Core;

defined( 'ABSPATH' ) || exit;

/**
 * Storage class
 */
class Storage {

	/**
	 * @return void
	 */
	public static function init() {
		if ( mce_get_cmatic( 'telemetry.enabled' ) !== null ) {
			return;
		}

		$defaults = array(
			'enabled'             => true,
			'opt_in_date'         => time(),
			'disabled_count'      => 0,
			'heartbeat_interval'  => 48,
			'schedule'            => 'frequent',
			'frequent_started_at' => time(),
			'last_heartbeat'      => 0,
			'heartbeat_count'     => 0,
			'failed_count'        => 0,
			'last_payload_hash'   => '',
			'last_error'          => '',
		);

		foreach ( $defaults as $key => $value ) {
			mce_update_cmatic( "telemetry.{$key}", $value );
		}

		if ( mce_get_cmatic( 'lifecycle.activations' ) === null ) {
			mce_update_cmatic( 'lifecycle.activations', array( time() ) );
			mce_update_cmatic( 'lifecycle.deactivations', array() );
			mce_update_cmatic( 'lifecycle.is_reactivation', false );
		}
	}

	/**
	 * @return bool
	 */
	public static function is_enabled() {
		return (bool) mce_get_cmatic( 'telemetry.enabled', true );
	}

	/**
	 * @return string
	 */
	public static function get_schedule() {
		return mce_get_cmatic( 'telemetry.schedule', 'frequent' );
	}

	/**
	 * @param string $schedule Schedule type.
	 * @return void
	 */
	public static function set_schedule( $schedule ) {
		mce_update_cmatic( 'telemetry.schedule', $schedule );

		if ( 'frequent' === $schedule ) {
			mce_update_cmatic( 'telemetry.frequent_started_at', time() );
		}
	}

	/**
	 * @return int
	 */
	public static function get_heartbeat_interval() {
		return (int) mce_get_cmatic( 'telemetry.heartbeat_interval', 48 );
	}

	/**
	 * @return int
	 */
	public static function get_last_heartbeat() {
		return (int) mce_get_cmatic( 'telemetry.last_heartbeat', 0 );
	}

	/**
	 * @param int $timestamp Unix timestamp.
	 * @return void
	 */
	public static function update_last_heartbeat( $timestamp = null ) {
		if ( null === $timestamp ) {
			$timestamp = time();
		}
		mce_update_cmatic( 'telemetry.last_heartbeat', $timestamp );
	}

	/**
	 * @return void
	 */
	public static function increment_heartbeat_count() {
		$count = (int) mce_get_cmatic( 'telemetry.heartbeat_count', 0 );
		mce_update_cmatic( 'telemetry.heartbeat_count', $count + 1 );
	}

	/**
	 * @return void
	 */
	public static function increment_failed_count() {
		$count = (int) mce_get_cmatic( 'telemetry.failed_count', 0 );
		mce_update_cmatic( 'telemetry.failed_count', $count + 1 );
	}

	/**
	 * @return void
	 */
	public static function increment_disabled_count() {
		$count = (int) mce_get_cmatic( 'telemetry.disabled_count', 0 );
		mce_update_cmatic( 'telemetry.disabled_count', $count + 1 );
	}

	/**
	 * @return int
	 */
	public static function get_frequent_started_at() {
		return (int) mce_get_cmatic( 'telemetry.frequent_started_at', 0 );
	}

	/**
	 * @return bool
	 */
	public static function is_frequent_elapsed() {
		$started_at = self::get_frequent_started_at();
		if ( 0 === $started_at ) {
			return false;
		}

		$elapsed = time() - $started_at;
		return $elapsed >= ( 1 * HOUR_IN_SECONDS );
	}

	/**
	 * @return void
	 */
	public static function record_activation() {
		$activations = mce_get_cmatic( 'lifecycle.activations', array() );

		$is_reactivation = count( $activations ) > 0;
		mce_update_cmatic( 'lifecycle.is_reactivation', $is_reactivation );

		$activations[] = time();
		mce_update_cmatic( 'lifecycle.activations', $activations );
	}

	/**
	 * @return void
	 */
	public static function record_deactivation() {
		$deactivations   = mce_get_cmatic( 'lifecycle.deactivations', array() );
		$deactivations[] = time();
		mce_update_cmatic( 'lifecycle.deactivations', $deactivations );
	}

	/**
	 * @return bool
	 */
	public static function is_reactivation() {
		return (bool) mce_get_cmatic( 'lifecycle.is_reactivation', false );
	}

	/**
	 * @return int
	 */
	public static function get_activation_count() {
		$activations = mce_get_cmatic( 'lifecycle.activations', array() );
		return count( $activations );
	}

	/**
	 * @return int
	 */
	public static function get_deactivation_count() {
		$deactivations = mce_get_cmatic( 'lifecycle.deactivations', array() );
		return count( $deactivations );
	}

	/**
	 * @return string
	 */
	public static function get_install_id() {
		return mce_get_cmatic( 'install.id', '' );
	}

	/**
	 * @param string $error Error message.
	 * @return void
	 */
	public static function save_error( $error ) {
		mce_update_cmatic( 'telemetry.last_error', $error );
	}

	/**
	 * @param string $hash MD5 hash.
	 * @return void
	 */
	public static function save_payload_hash( $hash ) {
		mce_update_cmatic( 'telemetry.last_payload_hash', $hash );
	}
}
