<?php
/**
 * ClickSyncr Metrics - Scheduler
 *
 * @package ClickSyncr\Metrics\Core
 * @since 1.0.0
 */

/*
 * Copyright (C) 2010-2025, Renzo Johnson
 * License: GPL-2.0+
 */

namespace ClickSyncr\Metrics\Core;

defined( 'ABSPATH' ) || exit;

/**
 * Scheduler class
 */
class Scheduler {

	/**
	 * @return void
	 */
	public static function init() {
		add_filter( 'cron_schedules', array( __CLASS__, 'add_cron_intervals' ) );
		add_action( 'csyncr_metrics_heartbeat', array( __CLASS__, 'execute_heartbeat' ) );
		add_action( 'admin_init', array( __CLASS__, 'ensure_schedule' ) );
	}

	/**
	 * @param array $schedules Existing schedules.
	 * @return array
	 */
	public static function add_cron_intervals( $schedules ) {
		$schedules['csyncr_2min'] = array(
			'interval' => 2 * MINUTE_IN_SECONDS,
			'display'  => 'Every 2 Minutes',
		);

		$schedules['csyncr_10min'] = array(
			'interval' => 10 * MINUTE_IN_SECONDS,
			'display'  => 'Every 10 Minutes',
		);

		$interval_hours = Storage::get_heartbeat_interval();
		$schedules['csyncr_sparse'] = array(
			'interval' => $interval_hours * HOUR_IN_SECONDS,
			'display'  => 'Every ' . $interval_hours . ' Hours',
		);

		return $schedules;
	}

	/**
	 * @return void
	 */
	public static function execute_heartbeat() {
		if ( ! Storage::is_enabled() ) {
			return;
		}

		$schedule   = Storage::get_schedule();
		$started_at = Storage::get_frequent_started_at();
		$elapsed    = $started_at > 0 ? time() - $started_at : 0;

		if ( 'super_frequent' === $schedule && $elapsed >= ( 15 * MINUTE_IN_SECONDS ) ) {
			self::transition_to_frequent();
		} elseif ( 'frequent' === $schedule && $elapsed >= ( 1 * HOUR_IN_SECONDS ) ) {
			self::transition_to_sparse();
		}

		$payload = Collector::collect( 'heartbeat' );
		Sync::send( $payload );
	}

	/**
	 * @return void
	 */
	private static function transition_to_frequent() {
		Storage::set_schedule( 'frequent' );

		$timestamp = wp_next_scheduled( 'csyncr_metrics_heartbeat' );
		if ( $timestamp ) {
			wp_unschedule_event( $timestamp, 'csyncr_metrics_heartbeat' );
		}

		wp_schedule_event( time(), 'csyncr_10min', 'csyncr_metrics_heartbeat' );
	}

	/**
	 * @return void
	 */
	private static function transition_to_sparse() {
		Storage::set_schedule( 'sparse' );

		$timestamp = wp_next_scheduled( 'csyncr_metrics_heartbeat' );
		if ( $timestamp ) {
			wp_unschedule_event( $timestamp, 'csyncr_metrics_heartbeat' );
		}

		$interval_hours = Storage::get_heartbeat_interval();
		wp_schedule_single_event(
			time() + ( $interval_hours * HOUR_IN_SECONDS ),
			'csyncr_metrics_heartbeat'
		);
	}

	/**
	 * @return void
	 */
	public static function ensure_schedule() {
		if ( ! Storage::is_enabled() ) {
			return;
		}

		if ( wp_next_scheduled( 'csyncr_metrics_heartbeat' ) ) {
			return;
		}

		$schedule   = Storage::get_schedule();
		$started_at = Storage::get_frequent_started_at();
		$elapsed    = $started_at > 0 ? time() - $started_at : 0;

		if ( 'super_frequent' === $schedule ) {
			if ( $elapsed >= ( 15 * MINUTE_IN_SECONDS ) ) {
				self::transition_to_frequent();
			} else {
				wp_schedule_event( time(), 'csyncr_2min', 'csyncr_metrics_heartbeat' );
			}
		} elseif ( 'frequent' === $schedule ) {
			if ( Storage::is_frequent_elapsed() ) {
				self::transition_to_sparse();
			} else {
				wp_schedule_event( time(), 'csyncr_10min', 'csyncr_metrics_heartbeat' );
			}
		} else {
			$interval_hours = Storage::get_heartbeat_interval();
			$last_heartbeat = Storage::get_last_heartbeat();
			$next_heartbeat = $last_heartbeat + ( $interval_hours * HOUR_IN_SECONDS );

			if ( $next_heartbeat < time() ) {
				$next_heartbeat = time();
			}

			wp_schedule_single_event( $next_heartbeat, 'csyncr_metrics_heartbeat' );
		}
	}

	/**
	 * @return void
	 */
	public static function schedule_next_sparse() {
		if ( 'sparse' !== Storage::get_schedule() ) {
			return;
		}

		$timestamp = wp_next_scheduled( 'csyncr_metrics_heartbeat' );
		if ( $timestamp ) {
			wp_unschedule_event( $timestamp, 'csyncr_metrics_heartbeat' );
		}

		$interval_hours = Storage::get_heartbeat_interval();
		wp_schedule_single_event(
			time() + ( $interval_hours * HOUR_IN_SECONDS ),
			'csyncr_metrics_heartbeat'
		);
	}

	/**
	 * @return void
	 */
	public static function clear_schedule() {
		$timestamp = wp_next_scheduled( 'csyncr_metrics_heartbeat' );
		if ( $timestamp ) {
			wp_unschedule_event( $timestamp, 'csyncr_metrics_heartbeat' );
		}
	}
}
