<?php
/**
 * ClickSyncr Metrics - Collector
 *
 * @package ClickSyncr\Metrics\Core
 * @since 1.0.0
 */

/*
 * Copyright (C) 2010-2025, Renzo Johnson
 * License: GPL-2.0+
 */

namespace ClickSyncr\Metrics\Core;

defined( 'ABSPATH' ) || exit;

/**
 * Collector class
 */
class Collector {

	/**
	 * @param string $event Event type.
	 * @return array
	 */
	public static function collect( $event = 'heartbeat' ) {
		return array(
			'install_id'  => Storage::get_install_id(),
			'timestamp'   => time(),
			'event'       => $event,
			'metadata'    => self::collect_metadata(),
			'lifecycle'   => self::collect_lifecycle(),
			'environment' => self::collect_environment(),
			'api'         => self::collect_api(),
			'submissions' => self::collect_submissions(),
			'features'    => self::collect_features(),
			'forms'       => self::collect_forms(),
			'performance' => self::collect_performance(),
			'plugins'     => self::collect_plugins(),
			'server'      => self::collect_server(),
			'wordpress'   => self::collect_wordpress(),
		);
	}

	/**
	 * @return array
	 */
	private static function collect_metadata() {
		$first_installed = mce_get_cmatic( 'install.quest', 0 );
		$total_uptime    = $first_installed > 0 ? time() - $first_installed : 0;
		$failed_count    = (int) mce_get_cmatic( 'telemetry.failed_count', 0 );

		return array(
			'schedule'              => mce_get_cmatic( 'telemetry.schedule', 'frequent' ),
			'frequent_started_at'   => (int) mce_get_cmatic( 'telemetry.frequent_started_at', 0 ),
			'is_reactivation'       => Storage::is_reactivation(),
			'disabled_count'        => (int) mce_get_cmatic( 'telemetry.disabled_count', 0 ),
			'opt_in_date'           => (int) mce_get_cmatic( 'telemetry.opt_in_date', 0 ),
			'last_heartbeat'        => (int) mce_get_cmatic( 'telemetry.last_heartbeat', 0 ),
			'failed_heartbeats'     => $failed_count,
			'total_uptime_seconds'  => $total_uptime,
			'telemetry_version'     => SPARTAN_MCE_VERSION,
		);
	}

	/**
	 * @return array
	 */
	private static function collect_lifecycle() {
		$activations   = mce_get_cmatic( 'lifecycle.activations', array() );
		$deactivations = mce_get_cmatic( 'lifecycle.deactivations', array() );
		$upgrades      = mce_get_cmatic( 'lifecycle.upgrades', array() );

		$first_activated  = mce_get_cmatic( 'install.quest', 0 );
		$last_activated   = ! empty( $activations ) ? max( $activations ) : 0;
		$last_deactivated = ! empty( $deactivations ) ? max( $deactivations ) : 0;
		$last_upgrade     = ! empty( $upgrades ) ? max( $upgrades ) : 0;

		$days_since_first = 0;
		if ( $first_activated > 0 ) {
			$days_since_first = floor( ( time() - $first_activated ) / DAY_IN_SECONDS );
		}

		$avg_session_length = 0;
		if ( count( $activations ) > 0 && count( $deactivations ) > 0 ) {
			$total_session_time = 0;
			$session_count      = 0;
			foreach ( $activations as $index => $activation_time ) {
				if ( isset( $deactivations[ $index ] ) ) {
					$total_session_time += $deactivations[ $index ] - $activation_time;
					++$session_count;
				}
			}
			if ( $session_count > 0 ) {
				$avg_session_length = floor( $total_session_time / $session_count );
			}
		}

		$version_history         = mce_get_cmatic( 'lifecycle.version_history', array() );
		$previous_version        = mce_get_cmatic( 'lifecycle.previous_version', '' );
		$days_since_last_upgrade = $last_upgrade > 0 ? floor( ( time() - $last_upgrade ) / DAY_IN_SECONDS ) : 0;

		return array(
			'activation_count'            => count( $activations ),
			'deactivation_count'          => count( $deactivations ),
			'upgrade_count'               => count( $upgrades ),
			'first_activated'             => $first_activated,
			'last_activated'              => $last_activated,
			'last_deactivated'            => $last_deactivated,
			'last_upgrade'                => $last_upgrade,
			'days_since_first_activation' => (int) $days_since_first,
			'days_since_last_upgrade'     => (int) $days_since_last_upgrade,
			'avg_session_length_seconds'  => $avg_session_length,
			'total_sessions'              => count( $activations ),
			'active_session'              => empty( $deactivations ) || $last_activated > $last_deactivated,
			'previous_version'            => $previous_version,
			'version_history_count'       => count( $version_history ),
			'install_method'              => mce_get_cmatic( 'lifecycle.install_method', 'unknown' ),
			'days_on_current_version'     => $last_upgrade > 0 ? floor( ( time() - $last_upgrade ) / DAY_IN_SECONDS ) : $days_since_first,
		);
	}

	/**
	 * @return array
	 */
	private static function collect_environment() {
		global $wp_version, $wpdb;

		$php_extensions      = get_loaded_extensions();
		$critical_extensions = array( 'curl', 'json', 'mbstring', 'openssl', 'zip', 'gd', 'xml', 'dom', 'SimpleXML' );
		$loaded_critical     = array_intersect( $critical_extensions, $php_extensions );

		$theme        = wp_get_theme();
		$parent_theme = $theme->parent() ? $theme->parent()->get( 'Name' ) : '';

		return array(
			'php_version'                    => phpversion(),
			'php_sapi'                       => php_sapi_name(),
			'php_os'                         => PHP_OS,
			'php_architecture'               => PHP_INT_SIZE === 8 ? '64-bit' : '32-bit',
			'php_memory_limit'               => ini_get( 'memory_limit' ),
			'php_max_execution_time'         => (int) ini_get( 'max_execution_time' ),
			'php_max_input_time'             => (int) ini_get( 'max_input_time' ),
			'php_max_input_vars'             => (int) ini_get( 'max_input_vars' ),
			'php_post_max_size'              => ini_get( 'post_max_size' ),
			'php_upload_max_filesize'        => ini_get( 'upload_max_filesize' ),
			'php_default_timezone'           => ini_get( 'date.timezone' ),
			'php_display_errors'             => ini_get( 'display_errors' ),
			'php_error_reporting'            => ini_get( 'error_reporting' ),
			'php_log_errors'                 => ini_get( 'log_errors' ),
			'php_extensions_count'           => count( $php_extensions ),
			'php_critical_extensions'        => implode( ',', $loaded_critical ),
			'php_curl_version'               => function_exists( 'curl_version' ) ? curl_version()['version'] : 'none',
			'php_openssl_version'            => OPENSSL_VERSION_TEXT,
			'wp_version'                     => $wp_version,
			'wp_db_version'                  => get_option( 'db_version' ),
			'wp_memory_limit'                => WP_MEMORY_LIMIT,
			'wp_max_memory_limit'            => WP_MAX_MEMORY_LIMIT,
			'wp_debug'                       => defined( 'WP_DEBUG' ) && WP_DEBUG,
			'wp_debug_log'                   => defined( 'WP_DEBUG_LOG' ) && WP_DEBUG_LOG,
			'wp_debug_display'               => defined( 'WP_DEBUG_DISPLAY' ) && WP_DEBUG_DISPLAY,
			'script_debug'                   => defined( 'SCRIPT_DEBUG' ) && SCRIPT_DEBUG,
			'wp_cache'                       => defined( 'WP_CACHE' ) && WP_CACHE,
			'wp_cron_disabled'               => defined( 'DISABLE_WP_CRON' ) && DISABLE_WP_CRON,
			'wp_auto_update_core'            => get_option( 'auto_update_core', 'enabled' ),
			'mysql_version'                  => $wpdb->db_version(),
			'mysql_client_version'           => $wpdb->get_var( 'SELECT VERSION()' ),
			'db_charset'                     => $wpdb->charset,
			'db_collate'                     => $wpdb->collate,
			'db_prefix'                      => strlen( $wpdb->prefix ),
			'server_software'                => isset( $_SERVER['SERVER_SOFTWARE'] ) ? sanitize_text_field( wp_unslash( $_SERVER['SERVER_SOFTWARE'] ) ) : 'unknown',
			'server_protocol'                => isset( $_SERVER['SERVER_PROTOCOL'] ) ? sanitize_text_field( wp_unslash( $_SERVER['SERVER_PROTOCOL'] ) ) : 'unknown',
			'server_port'                    => isset( $_SERVER['SERVER_PORT'] ) ? (int) $_SERVER['SERVER_PORT'] : 0,
			'https'                          => is_ssl(),
			'http_host'                      => hash( 'sha256', isset( $_SERVER['HTTP_HOST'] ) ? sanitize_text_field( wp_unslash( $_SERVER['HTTP_HOST'] ) ) : '' ),
			'locale'                         => get_locale(),
			'timezone'                       => wp_timezone_string(),
			'site_language'                  => get_bloginfo( 'language' ),
			'site_charset'                   => get_bloginfo( 'charset' ),
			'permalink_structure'            => get_option( 'permalink_structure' ),
			'home_url'                       => hash( 'sha256', home_url() ),
			'site_url'                       => hash( 'sha256', site_url() ),
			'admin_email'                    => hash( 'sha256', get_option( 'admin_email' ) ),
			'theme'                          => $theme->get( 'Name' ),
			'theme_version'                  => $theme->get( 'Version' ),
			'theme_author'                   => $theme->get( 'Author' ),
			'parent_theme'                   => $parent_theme,
			'is_child_theme'                 => ! empty( $parent_theme ),
			'theme_supports_html5'           => current_theme_supports( 'html5' ),
			'theme_supports_post_thumbnails' => current_theme_supports( 'post-thumbnails' ),
			'active_plugins_count'           => count( get_option( 'active_plugins', array() ) ),
			'total_plugins_count'            => count( get_plugins() ),
			'must_use_plugins_count'         => count( wp_get_mu_plugins() ),
			'is_multisite'                   => is_multisite(),
			'is_subdomain_install'           => is_multisite() ? ( defined( 'SUBDOMAIN_INSTALL' ) && SUBDOMAIN_INSTALL ) : false,
			'network_count'                  => is_multisite() ? get_blog_count() : 1,
			'is_main_site'                   => is_multisite() ? is_main_site() : true,
			'cf7_version'                    => defined( 'WPCF7_VERSION' ) ? WPCF7_VERSION : 'unknown',
			'cf7_installed'                  => class_exists( 'WPCF7_ContactForm' ),
			'plugin_version'                 => defined( 'SPARTAN_MCE_VERSION' ) ? SPARTAN_MCE_VERSION : 'unknown',
			'user_agent'                     => isset( $_SERVER['HTTP_USER_AGENT'] ) ? sanitize_text_field( wp_unslash( $_SERVER['HTTP_USER_AGENT'] ) ) : 'unknown',
		);
	}

	/**
	 * @return array
	 */
	private static function collect_api() {
		$has_api_key     = false;
		$api_data_center = '';
		$api_key_length  = 0;
		$forms_with_api  = 0;
		$cf7_forms       = self::get_cf7_forms();

		foreach ( $cf7_forms as $form ) {
			$form_id = $form->id();
			$cf7_mch = get_option( 'cf7_mch_' . $form_id, array() );

			if ( ! empty( $cf7_mch['api'] ) ) {
				$has_api_key = true;
				++$forms_with_api;

				if ( empty( $api_data_center ) && preg_match( '/-([a-z0-9]+)$/i', $cf7_mch['api'], $matches ) ) {
					$api_data_center = $matches[1];
				}

				if ( 0 === $api_key_length ) {
					$api_key_length = strlen( $cf7_mch['api'] );
				}
			}
		}

		$total_sent          = (int) mce_get_cmatic( 'stats.sent', 0 );
		$total_attempts      = (int) mce_get_cmatic( 'api.total_attempts', $total_sent );
		$total_successes     = (int) mce_get_cmatic( 'api.total_successes', $total_sent );
		$total_failures      = (int) mce_get_cmatic( 'api.total_failures', 0 );
		$success_rate        = $total_attempts > 0 ? round( ( $total_successes / $total_attempts ) * 100, 2 ) : 0;
		$first_connected     = (int) mce_get_cmatic( 'api.first_connected', 0 );
		$last_success        = (int) mce_get_cmatic( 'api.last_success', 0 );
		$last_failure        = (int) mce_get_cmatic( 'api.last_failure', 0 );
		$avg_response_time   = (int) mce_get_cmatic( 'api.avg_response_time', 0 );

		$error_codes   = mce_get_cmatic( 'api.error_codes', array() );
		$error_summary = array();
		foreach ( $error_codes as $code => $count ) {
			if ( $count > 0 ) {
				$error_summary[ $code ] = (int) $count;
			}
		}

		$rate_limit_hits      = (int) mce_get_cmatic( 'api.rate_limit_hits', 0 );
		$last_rate_limit      = (int) mce_get_cmatic( 'api.last_rate_limit', 0 );
		$consecutive_failures = (int) mce_get_cmatic( 'api.consecutive_failures', 0 );
		$uptime_percentage    = $total_attempts > 0 ? round( ( $total_successes / $total_attempts ) * 100, 2 ) : 100;

		$days_since_first_connection = $first_connected > 0 ? floor( ( time() - $first_connected ) / DAY_IN_SECONDS ) : 0;
		$days_since_last_success     = $last_success > 0 ? floor( ( time() - $last_success ) / DAY_IN_SECONDS ) : 0;
		$days_since_last_failure     = $last_failure > 0 ? floor( ( time() - $last_failure ) / DAY_IN_SECONDS ) : 0;

		return array(
			'is_connected'                => $has_api_key,
			'forms_with_api'              => $forms_with_api,
			'api_data_center'             => $api_data_center,
			'api_key_length'              => $api_key_length,
			'first_connected'             => $first_connected,
			'days_since_first_connection' => $days_since_first_connection,
			'total_attempts'              => $total_attempts,
			'total_successes'             => $total_successes,
			'total_failures'              => $total_failures,
			'success_rate'                => $success_rate,
			'uptime_percentage'           => $uptime_percentage,
			'last_success'                => $last_success,
			'last_failure'                => $last_failure,
			'days_since_last_success'     => $days_since_last_success,
			'days_since_last_failure'     => $days_since_last_failure,
			'avg_response_time_ms'        => $avg_response_time,
			'error_codes'                 => $error_summary,
			'unique_error_count'          => count( $error_summary ),
			'consecutive_failures'        => $consecutive_failures,
			'total_error_codes'           => array_sum( $error_summary ),
			'rate_limit_hits'             => $rate_limit_hits,
			'last_rate_limit'             => $last_rate_limit,
			'days_since_rate_limit'       => $last_rate_limit > 0 ? floor( ( time() - $last_rate_limit ) / DAY_IN_SECONDS ) : 0,
			'api_health_score'            => min( 100, max( 0, $uptime_percentage - ( $consecutive_failures * 5 ) ) ),
			'setup_sync_attempted'        => mce_get_cmatic( 'api.sync_attempted', false ),
			'setup_sync_attempts_count'   => (int) mce_get_cmatic( 'api.sync_attempts_count', 0 ),
			'setup_first_success'         => mce_get_cmatic( 'api.setup_first_success', false ),
			'setup_first_failure'         => mce_get_cmatic( 'api.setup_first_failure', false ),
			'setup_last_failure'          => mce_get_cmatic( 'api.setup_last_failure', false ),
			'setup_failure_count'         => (int) mce_get_cmatic( 'api.setup_failure_count', 0 ),
			'setup_audience_selected'     => mce_get_cmatic( 'api.audience_selected', false ),
		);
	}

	/**
	 * @return array
	 */
	private static function collect_submissions() {
		$total_sent       = (int) mce_get_cmatic( 'stats.sent', 0 );
		$total_failed     = (int) mce_get_cmatic( 'submissions.failed', 0 );
		$first_submission = (int) mce_get_cmatic( 'submissions.first', 0 );
		$last_submission  = (int) mce_get_cmatic( 'submissions.last', 0 );
		$last_success     = (int) mce_get_cmatic( 'submissions.last_success', 0 );
		$last_failure     = (int) mce_get_cmatic( 'submissions.last_failure', 0 );

		$first_activated = mce_get_cmatic( 'install.quest', 0 );

		$days_active = 1;
		if ( $first_activated > 0 ) {
			$days_active = max( 1, floor( ( time() - $first_activated ) / DAY_IN_SECONDS ) );
		}

		$total_submissions = $total_sent + $total_failed;
		$avg_per_day       = $days_active > 0 ? round( $total_submissions / $days_active, 2 ) : 0;
		$success_rate      = $total_submissions > 0 ? round( ( $total_sent / $total_submissions ) * 100, 2 ) : 100;

		$days_since_first = $first_submission > 0 ? floor( ( time() - $first_submission ) / DAY_IN_SECONDS ) : 0;
		$days_since_last  = $last_submission > 0 ? floor( ( time() - $last_submission ) / DAY_IN_SECONDS ) : 0;
		$hours_since_last = $last_submission > 0 ? floor( ( time() - $last_submission ) / HOUR_IN_SECONDS ) : 0;

		$hourly_distribution = mce_get_cmatic( 'submissions.hourly', array() );
		$busiest_hour        = 0;
		$max_submissions     = 0;
		foreach ( $hourly_distribution as $hour => $count ) {
			if ( $count > $max_submissions ) {
				$max_submissions = $count;
				$busiest_hour    = (int) $hour;
			}
		}

		$daily_distribution  = mce_get_cmatic( 'submissions.daily', array() );
		$busiest_day         = 0;
		$max_day_submissions = 0;
		foreach ( $daily_distribution as $day => $count ) {
			if ( $count > $max_day_submissions ) {
				$max_day_submissions = $count;
				$busiest_day         = (int) $day;
			}
		}

		$this_month = (int) mce_get_cmatic( 'submissions.this_month', 0 );
		$last_month = (int) mce_get_cmatic( 'submissions.last_month', 0 );
		$peak_month = (int) mce_get_cmatic( 'submissions.peak_month', 0 );

		$consecutive_successes = (int) mce_get_cmatic( 'submissions.consecutive_successes', 0 );
		$consecutive_failures  = (int) mce_get_cmatic( 'submissions.consecutive_failures', 0 );

		return array(
			'total_sent'                   => $total_sent,
			'total_failed'                 => $total_failed,
			'total_submissions'            => $total_submissions,
			'successful_submissions_count' => $total_sent,
			'failed_count'                 => $total_failed,
			'success_rate'                 => $success_rate,
			'first_submission'             => $first_submission,
			'last_submission'              => $last_submission,
			'last_success'                 => $last_success,
			'last_failure'                 => $last_failure,
			'days_since_first'             => $days_since_first,
			'days_since_last'              => $days_since_last,
			'hours_since_last'             => $hours_since_last,
			'avg_per_day'                  => $avg_per_day,
			'avg_per_week'                 => round( $avg_per_day * 7, 2 ),
			'avg_per_month'                => round( $avg_per_day * 30, 2 ),
			'busiest_hour'                 => $busiest_hour,
			'busiest_day'                  => $busiest_day,
			'submissions_busiest_hour'     => $max_submissions,
			'submissions_busiest_day'      => $max_day_submissions,
			'this_month'                   => $this_month,
			'last_month'                   => $last_month,
			'peak_month'                   => $peak_month,
			'month_over_month_change'      => $last_month > 0 ? round( ( ( $this_month - $last_month ) / $last_month ) * 100, 2 ) : 0,
			'consecutive_successes'        => $consecutive_successes,
			'consecutive_failures'         => $consecutive_failures,
			'longest_success_streak'       => (int) mce_get_cmatic( 'submissions.longest_success_streak', 0 ),
			'active_forms_count'           => (int) mce_get_cmatic( 'submissions.active_forms', 0 ),
			'forms_with_submissions'       => count( mce_get_cmatic( 'submissions.forms_used', array() ) ),
		);
	}

	/**
	 * @return array
	 */
	private static function collect_features() {
		$features = array(
			'double_optin'        => 0,
			'required_consent'    => 0,
			'debug_logger'        => 0,
			'custom_merge_fields' => 0,
			'interest_groups'     => 0,
			'tags_enabled'        => 0,
			'conditional_logic'   => 0,
			'auto_update'         => get_option( 'chimpmatic-update', '0' ) === '1',
			'telemetry_enabled'   => true,
		);

		$cf7_forms = self::get_cf7_forms();

		foreach ( $cf7_forms as $form ) {
			$form_id = $form->id();
			$cf7_mch = get_option( 'cf7_mch_' . $form_id, array() );

			if ( isset( $cf7_mch['confsubs'] ) && '1' === $cf7_mch['confsubs'] ) {
				++$features['double_optin'];
			}

			if ( ! empty( $cf7_mch['accept'] ) && ' ' !== $cf7_mch['accept'] ) {
				++$features['required_consent'];
			}

			if ( isset( $cf7_mch['logfileEnabled'] ) && '1' === $cf7_mch['logfileEnabled'] ) {
				++$features['debug_logger'];
			}

			if ( ! empty( $cf7_mch['merge_fields'] ) && is_array( $cf7_mch['merge_fields'] ) ) {
				++$features['custom_merge_fields'];
			}

			if ( ! empty( $cf7_mch['interest_groups'] ) && is_array( $cf7_mch['interest_groups'] ) ) {
				++$features['interest_groups'];
			}

			if ( ! empty( $cf7_mch['tags'] ) ) {
				++$features['tags_enabled'];
			}

			if ( ! empty( $cf7_mch['conditional_logic'] ) ) {
				++$features['conditional_logic'];
			}
		}

		$feature_first_use = mce_get_cmatic( 'features.first_use', array() );

		return array(
			'double_optin_count'        => $features['double_optin'],
			'required_consent_count'    => $features['required_consent'],
			'debug_logger_count'        => $features['debug_logger'],
			'custom_merge_fields_count' => $features['custom_merge_fields'],
			'interest_groups_count'     => $features['interest_groups'],
			'tags_enabled_count'        => $features['tags_enabled'],
			'conditional_logic_count'   => $features['conditional_logic'],
			'double_optin'              => $features['double_optin'] > 0,
			'required_consent'          => $features['required_consent'] > 0,
			'debug_logger'              => $features['debug_logger'] > 0,
			'custom_merge_fields'       => $features['custom_merge_fields'] > 0,
			'interest_groups'           => $features['interest_groups'] > 0,
			'tags_enabled'              => $features['tags_enabled'] > 0,
			'conditional_logic'         => $features['conditional_logic'] > 0,
			'auto_update'               => $features['auto_update'],
			'telemetry_enabled'         => $features['telemetry_enabled'],
			'total_features_enabled'    => count( array_filter( $features ) ),
			'features_usage_percentage' => round( ( count( array_filter( $features ) ) / count( $features ) ) * 100, 2 ),
			'double_optin_first_use'    => isset( $feature_first_use['double_optin'] ) ? (int) $feature_first_use['double_optin'] : 0,
			'consent_first_use'         => isset( $feature_first_use['consent'] ) ? (int) $feature_first_use['consent'] : 0,
			'debug_logger_first_use'    => isset( $feature_first_use['debug_logger'] ) ? (int) $feature_first_use['debug_logger'] : 0,
			'webhook_enabled'           => (bool) mce_get_cmatic( 'features.webhook_enabled', false ),
			'custom_api_endpoint'       => (bool) mce_get_cmatic( 'features.custom_api_endpoint', false ),
			'email_notifications'       => (bool) mce_get_cmatic( 'features.email_notifications', false ),
		);
	}

	/**
	 * @return array
	 */
	private static function collect_forms() {
		$cf7_forms        = self::get_cf7_forms();
		$total_forms      = count( $cf7_forms );
		$active_forms     = 0;
		$forms_with_api   = 0;
		$forms_with_lists = 0;
		$total_lists      = 0;
		$total_fields     = 0;
		$unique_audiences = array();
		$forms_data       = array();

		foreach ( $cf7_forms as $form ) {
			$form_id = $form->id();
			$cf7_mch = get_option( 'cf7_mch_' . $form_id, array() );

			if ( ! empty( $cf7_mch['api'] ) ) {
				++$forms_with_api;
				++$active_forms;
			}

			$audience_count = 0;
			$has_list       = false;
			$selected_list  = $cf7_mch['list'] ?? '';

			if ( isset( $cf7_mch['lisdata']['lists'] ) && is_array( $cf7_mch['lisdata']['lists'] ) ) {
				$audience_count = count( $cf7_mch['lisdata']['lists'] );
				$has_list       = $audience_count > 0;
				$total_lists   += $audience_count;
				if ( $has_list ) {
					++$forms_with_lists;
				}

				foreach ( $cf7_mch['lisdata']['lists'] as $list ) {
					if ( isset( $list['stats']['member_count'] ) ) {
						$list_id      = $list['id'] ?? '';
						$member_count = (int) $list['stats']['member_count'];
						$is_paired    = ( $list_id === $selected_list );

						if ( ! isset( $unique_audiences[ $list_id ] ) ) {
							$unique_audiences[ $list_id ] = array(
								'audience_id'  => hash( 'sha256', $list_id ),
								'member_count' => $member_count,
								'is_paired'    => $is_paired,
							);
						} elseif ( $is_paired ) {
							$unique_audiences[ $list_id ]['is_paired'] = true;
						}
					}
				}
			}

			$form_fields   = $form->scan_form_tags();
			$total_fields += count( $form_fields );

			$forms_data[] = array(
				'form_id'         => hash( 'sha256', (string) $form_id ),
				'has_api'         => ! empty( $cf7_mch['api'] ),
				'has_list'        => $has_list,
				'audience_count'  => $audience_count,
				'field_count'     => count( $form_fields ),
				'has_double_opt'  => isset( $cf7_mch['confsubs'] ) && '1' === $cf7_mch['confsubs'],
				'has_consent'     => ! empty( $cf7_mch['accept'] ) && ' ' !== $cf7_mch['accept'],
				'submissions'     => (int) get_option( 'cf7_mch_submissions_' . $form_id, 0 ),
				'last_submission' => (int) get_option( 'cf7_mch_last_submission_' . $form_id, 0 ),
			);
		}

		$avg_fields_per_form = $total_forms > 0 ? round( $total_fields / $total_forms, 2 ) : 0;
		$avg_lists_per_form  = $forms_with_lists > 0 ? round( $total_lists / $forms_with_lists, 2 ) : 0;

		$oldest_form = 0;
		$newest_form = 0;
		foreach ( $cf7_forms as $form ) {
			$created   = get_post_field( 'post_date', $form->id(), 'raw' );
			$timestamp = strtotime( $created );

			if ( 0 === $oldest_form || $timestamp < $oldest_form ) {
				$oldest_form = $timestamp;
			}
			if ( 0 === $newest_form || $timestamp > $newest_form ) {
				$newest_form = $timestamp;
			}
		}

		$audience_data          = array_values( $unique_audiences );
		$total_unique_audiences = count( $audience_data );
		$total_contacts         = array_sum( array_column( $audience_data, 'member_count' ) );

		return array(
			'total_forms'                 => $total_forms,
			'active_forms'                => $active_forms,
			'forms_with_api'              => $forms_with_api,
			'forms_with_lists'            => $forms_with_lists,
			'inactive_forms'              => $total_forms - $active_forms,
			'total_audiences'             => $total_unique_audiences,
			'audiences'                   => $audience_data,
			'total_contacts'              => $total_contacts,
			'avg_lists_per_form'          => $avg_lists_per_form,
			'max_lists_per_form'          => $total_lists > 0 ? max( array_column( $forms_data, 'audience_count' ) ) : 0,
			'total_fields_all_forms'      => $total_fields,
			'avg_fields_per_form'         => $avg_fields_per_form,
			'min_fields_per_form'         => $total_forms > 0 ? min( array_column( $forms_data, 'field_count' ) ) : 0,
			'max_fields_per_form'         => $total_forms > 0 ? max( array_column( $forms_data, 'field_count' ) ) : 0,
			'oldest_form_created'         => $oldest_form,
			'newest_form_created'         => $newest_form,
			'days_since_oldest_form'      => $oldest_form > 0 ? floor( ( time() - $oldest_form ) / DAY_IN_SECONDS ) : 0,
			'days_since_newest_form'      => $newest_form > 0 ? floor( ( time() - $newest_form ) / DAY_IN_SECONDS ) : 0,
			'forms_with_submissions'      => count( array_filter( $forms_data, function ( $f ) {
				return $f['submissions'] > 0;
			} ) ),
			'forms_never_submitted'       => count( array_filter( $forms_data, function ( $f ) {
				return 0 === $f['submissions'];
			} ) ),
			'forms_with_double_opt'       => count( array_filter( $forms_data, function ( $f ) {
				return $f['has_double_opt'];
			} ) ),
			'forms_with_consent'          => count( array_filter( $forms_data, function ( $f ) {
				return $f['has_consent'];
			} ) ),
			'total_submissions_all_forms' => array_sum( array_column( $forms_data, 'submissions' ) ),
			'form_utilization_rate'       => $total_forms > 0 ? round( ( $active_forms / $total_forms ) * 100, 2 ) : 0,
		);
	}

	/**
	 * @return array
	 */
	private static function collect_performance() {
		global $wpdb;

		$memory_current       = memory_get_usage( true );
		$memory_peak          = memory_get_peak_usage( true );
		$memory_limit         = ini_get( 'memory_limit' );
		$memory_limit_bytes   = self::convert_to_bytes( $memory_limit );
		$memory_usage_percent = $memory_limit_bytes > 0 ? round( ( $memory_peak / $memory_limit_bytes ) * 100, 2 ) : 0;

		$db_queries = get_num_queries();
		$db_time    = timer_stop( 0, 3 );

		$page_load_time = isset( $_SERVER['REQUEST_TIME_FLOAT'] ) ? ( microtime( true ) - $_SERVER['REQUEST_TIME_FLOAT'] ) * 1000 : 0;

		$object_cache_hits   = 0;
		$object_cache_misses = 0;
		if ( function_exists( 'wp_cache_get_stats' ) ) {
			$cache_stats         = wp_cache_get_stats();
			$object_cache_hits   = isset( $cache_stats['hits'] ) ? $cache_stats['hits'] : 0;
			$object_cache_misses = isset( $cache_stats['misses'] ) ? $cache_stats['misses'] : 0;
		}

		$plugin_load_time = (float) mce_get_cmatic( 'performance.plugin_load_time', 0 );
		$api_avg_response = (float) mce_get_cmatic( 'performance.api_avg_response', 0 );

		return array(
			'memory_current'          => $memory_current,
			'memory_peak'             => $memory_peak,
			'memory_limit'            => $memory_limit,
			'memory_limit_bytes'      => $memory_limit_bytes,
			'memory_usage_percent'    => $memory_usage_percent,
			'memory_available'        => max( 0, $memory_limit_bytes - $memory_peak ),
			'php_max_execution_time'  => (int) ini_get( 'max_execution_time' ),
			'page_load_time_ms'       => round( $page_load_time, 2 ),
			'plugin_load_time_ms'     => round( $plugin_load_time, 2 ),
			'db_queries_count'        => $db_queries,
			'db_query_time_seconds'   => (float) $db_time,
			'db_size_mb'              => self::get_database_size(),
			'api_avg_response_ms'     => round( $api_avg_response, 2 ),
			'api_slowest_response_ms' => (int) mce_get_cmatic( 'performance.api_slowest', 0 ),
			'api_fastest_response_ms' => (int) mce_get_cmatic( 'performance.api_fastest', 0 ),
			'object_cache_enabled'    => wp_using_ext_object_cache(),
			'object_cache_hits'       => $object_cache_hits,
			'object_cache_misses'     => $object_cache_misses,
			'object_cache_hit_rate'   => ( $object_cache_hits + $object_cache_misses ) > 0 ? round( ( $object_cache_hits / ( $object_cache_hits + $object_cache_misses ) ) * 100, 2 ) : 0,
			'opcache_enabled'         => self::is_opcache_enabled(),
			'opcache_hit_rate'        => self::get_opcache_hit_rate(),
		);
	}

	/**
	 * @return bool
	 */
	private static function is_opcache_enabled() {
		if ( ! function_exists( 'opcache_get_status' ) ) {
			return false;
		}
		$status = @opcache_get_status();
		return false !== $status && is_array( $status );
	}

	/**
	 * @param string $value Memory value.
	 * @return int
	 */
	private static function convert_to_bytes( $value ) {
		$value = trim( $value );
		if ( empty( $value ) ) {
			return 0;
		}
		$last  = strtolower( $value[ strlen( $value ) - 1 ] );
		$value = (int) $value;

		switch ( $last ) {
			case 'g':
				$value *= 1024;
			case 'm':
				$value *= 1024;
			case 'k':
				$value *= 1024;
		}

		return $value;
	}

	/**
	 * @return float
	 */
	private static function get_database_size() {
		global $wpdb;

		$size = $wpdb->get_var(
			$wpdb->prepare(
				'SELECT SUM(data_length + index_length) / 1024 / 1024
				FROM information_schema.TABLES
				WHERE table_schema = %s',
				DB_NAME
			)
		);

		return round( (float) $size, 2 );
	}

	/**
	 * @return float
	 */
	private static function get_opcache_hit_rate() {
		if ( ! function_exists( 'opcache_get_status' ) ) {
			return 0;
		}

		$status = @opcache_get_status();
		if ( false === $status || ! is_array( $status ) || ! isset( $status['opcache_statistics'] ) ) {
			return 0;
		}

		$stats  = $status['opcache_statistics'];
		$hits   = isset( $stats['hits'] ) ? (int) $stats['hits'] : 0;
		$misses = isset( $stats['misses'] ) ? (int) $stats['misses'] : 0;

		if ( ( $hits + $misses ) === 0 ) {
			return 0;
		}

		return round( ( $hits / ( $hits + $misses ) ) * 100, 2 );
	}

	/**
	 * @return array
	 */
	private static function collect_plugins() {
		if ( ! function_exists( 'get_plugins' ) ) {
			require_once ABSPATH . 'wp-admin/includes/plugin.php';
		}

		$all_plugins    = get_plugins();
		$active_plugins = get_option( 'active_plugins', array() );
		$mu_plugins     = get_mu_plugins();

		$plugin_list    = array();
		$network_active = is_multisite() ? get_site_option( 'active_sitewide_plugins', array() ) : array();

		foreach ( $all_plugins as $plugin_path => $plugin_data ) {
			$is_active  = in_array( $plugin_path, $active_plugins, true );
			$is_network = isset( $network_active[ $plugin_path ] );

			$status = 'inactive';
			if ( $is_network ) {
				$status = 'network-active';
			} elseif ( $is_active ) {
				$status = 'active';
			}

			$plugin_list[] = array(
				'name'    => $plugin_data['Name'],
				'version' => $plugin_data['Version'],
				'author'  => strip_tags( $plugin_data['Author'] ),
				'status'  => $status,
			);
		}

		foreach ( $mu_plugins as $mu_plugin_path => $mu_plugin_data ) {
			$plugin_list[] = array(
				'name'    => $mu_plugin_data['Name'],
				'version' => $mu_plugin_data['Version'],
				'author'  => strip_tags( $mu_plugin_data['Author'] ),
				'status'  => 'mu-plugin',
			);
		}

		$premium_plugins   = 0;
		$cf7_addons        = 0;
		$mailchimp_plugins = 0;
		$security_plugins  = 0;
		$cache_plugins     = 0;
		$seo_plugins       = 0;

		foreach ( $all_plugins as $plugin_path => $plugin_data ) {
			$name = strtolower( $plugin_data['Name'] );

			if ( strpos( $name, 'pro' ) !== false || strpos( $name, 'premium' ) !== false ) {
				++$premium_plugins;
			}
			if ( strpos( $name, 'contact form 7' ) !== false ) {
				++$cf7_addons;
			}
			if ( strpos( $name, 'mailchimp' ) !== false ) {
				++$mailchimp_plugins;
			}
			if ( strpos( $name, 'security' ) !== false || strpos( $name, 'wordfence' ) !== false || strpos( $name, 'sucuri' ) !== false ) {
				++$security_plugins;
			}
			if ( strpos( $name, 'cache' ) !== false || strpos( $name, 'wp rocket' ) !== false || strpos( $name, 'w3 total cache' ) !== false ) {
				++$cache_plugins;
			}
			if ( strpos( $name, 'seo' ) !== false || strpos( $name, 'yoast' ) !== false ) {
				++$seo_plugins;
			}
		}

		$known_plugins = array(
			'woocommerce' => is_plugin_active( 'woocommerce/woocommerce.php' ),
			'elementor'   => is_plugin_active( 'elementor/elementor.php' ),
			'jetpack'     => is_plugin_active( 'jetpack/jetpack.php' ),
			'wordfence'   => is_plugin_active( 'wordfence/wordfence.php' ),
			'yoast_seo'   => is_plugin_active( 'wordpress-seo/wp-seo.php' ),
			'akismet'     => is_plugin_active( 'akismet/akismet.php' ),
		);

		return array(
			'total_plugins'     => count( $all_plugins ),
			'active_plugins'    => count( $active_plugins ),
			'inactive_plugins'  => count( $all_plugins ) - count( $active_plugins ),
			'mu_plugins'        => count( $mu_plugins ),
			'premium_plugins'   => $premium_plugins,
			'cf7_addons'        => $cf7_addons,
			'mailchimp_plugins' => $mailchimp_plugins,
			'security_plugins'  => $security_plugins,
			'cache_plugins'     => $cache_plugins,
			'seo_plugins'       => $seo_plugins,
			'has_woocommerce'   => $known_plugins['woocommerce'],
			'has_elementor'     => $known_plugins['elementor'],
			'has_jetpack'       => $known_plugins['jetpack'],
			'has_wordfence'     => $known_plugins['wordfence'],
			'has_yoast_seo'     => $known_plugins['yoast_seo'],
			'has_akismet'       => $known_plugins['akismet'],
			'plugin_list'       => $plugin_list,
		);
	}

	/**
	 * @return array
	 */
	private static function collect_server() {
		$server_load = array( 0, 0, 0 );
		if ( function_exists( 'sys_getloadavg' ) ) {
			$load = @sys_getloadavg();
			if ( false !== $load && is_array( $load ) ) {
				$server_load = $load;
			}
		}

		$disk_free  = 0;
		$disk_total = 0;
		if ( function_exists( 'disk_free_space' ) ) {
			$free = @disk_free_space( ABSPATH );
			if ( false !== $free ) {
				$disk_free = $free;
			}
		}
		if ( function_exists( 'disk_total_space' ) ) {
			$total = @disk_total_space( ABSPATH );
			if ( false !== $total ) {
				$disk_total = $total;
			}
		}
		$disk_used          = $disk_total - $disk_free;
		$disk_usage_percent = $disk_total > 0 ? round( ( $disk_used / $disk_total ) * 100, 2 ) : 0;

		$hostname = '';
		if ( function_exists( 'gethostname' ) ) {
			$name = @gethostname();
			if ( false !== $name ) {
				$hostname = $name;
			}
		}

		$architecture = '';
		if ( function_exists( 'php_uname' ) ) {
			$arch = @php_uname( 'm' );
			if ( false !== $arch ) {
				$architecture = $arch;
			}
		}

		return array(
			'load_average_1min'   => isset( $server_load[0] ) ? round( (float) $server_load[0], 2 ) : 0,
			'load_average_5min'   => isset( $server_load[1] ) ? round( (float) $server_load[1], 2 ) : 0,
			'load_average_15min'  => isset( $server_load[2] ) ? round( (float) $server_load[2], 2 ) : 0,
			'disk_free_bytes'     => $disk_free,
			'disk_total_bytes'    => $disk_total,
			'disk_used_bytes'     => $disk_used,
			'disk_usage_percent'  => $disk_usage_percent,
			'disk_free_gb'        => $disk_free ? round( $disk_free / 1024 / 1024 / 1024, 2 ) : 0,
			'disk_total_gb'       => $disk_total ? round( $disk_total / 1024 / 1024 / 1024, 2 ) : 0,
			'server_ip'           => hash( 'sha256', isset( $_SERVER['SERVER_ADDR'] ) ? sanitize_text_field( wp_unslash( $_SERVER['SERVER_ADDR'] ) ) : '' ),
			'server_hostname'     => hash( 'sha256', $hostname ),
			'server_os'           => PHP_OS,
			'server_architecture' => $architecture,
		);
	}

	/**
	 * @return array
	 */
	private static function collect_wordpress() {
		global $wpdb;

		$post_counts    = wp_count_posts( 'post' );
		$page_counts    = wp_count_posts( 'page' );
		$comment_counts = wp_count_comments();
		$user_count     = count_users();
		$media_counts   = wp_count_posts( 'attachment' );
		$category_count = wp_count_terms( 'category' );
		$tag_count      = wp_count_terms( 'post_tag' );
		$revision_count = $wpdb->get_var( "SELECT COUNT(*) FROM {$wpdb->posts} WHERE post_type = 'revision'" );

		return array(
			'posts_published'      => isset( $post_counts->publish ) ? (int) $post_counts->publish : 0,
			'posts_draft'          => isset( $post_counts->draft ) ? (int) $post_counts->draft : 0,
			'pages_published'      => isset( $page_counts->publish ) ? (int) $page_counts->publish : 0,
			'pages_draft'          => isset( $page_counts->draft ) ? (int) $page_counts->draft : 0,
			'media_items'          => isset( $media_counts->inherit ) ? (int) $media_counts->inherit : 0,
			'comments_total'       => isset( $comment_counts->total_comments ) ? (int) $comment_counts->total_comments : 0,
			'comments_approved'    => isset( $comment_counts->approved ) ? (int) $comment_counts->approved : 0,
			'comments_pending'     => isset( $comment_counts->moderated ) ? (int) $comment_counts->moderated : 0,
			'comments_spam'        => isset( $comment_counts->spam ) ? (int) $comment_counts->spam : 0,
			'users_total'          => isset( $user_count['total_users'] ) ? (int) $user_count['total_users'] : 0,
			'users_administrators' => isset( $user_count['avail_roles']['administrator'] ) ? (int) $user_count['avail_roles']['administrator'] : 0,
			'users_editors'        => isset( $user_count['avail_roles']['editor'] ) ? (int) $user_count['avail_roles']['editor'] : 0,
			'users_authors'        => isset( $user_count['avail_roles']['author'] ) ? (int) $user_count['avail_roles']['author'] : 0,
			'users_subscribers'    => isset( $user_count['avail_roles']['subscriber'] ) ? (int) $user_count['avail_roles']['subscriber'] : 0,
			'categories_count'     => is_wp_error( $category_count ) ? 0 : (int) $category_count,
			'tags_count'           => is_wp_error( $tag_count ) ? 0 : (int) $tag_count,
			'revisions_count'      => (int) $revision_count,
			'auto_updates_enabled' => (bool) get_option( 'auto_update_plugins', false ),
		);
	}

	/**
	 * @return array
	 */
	private static function get_cf7_forms() {
		if ( ! class_exists( 'WPCF7_ContactForm' ) ) {
			return array();
		}

		$args = array(
			'post_type'      => 'wpcf7_contact_form',
			'posts_per_page' => -1,
			'post_status'    => 'publish',
		);

		$posts = get_posts( $args );
		$forms = array();

		foreach ( $posts as $post ) {
			$form = \WPCF7_ContactForm::get_instance( $post->ID );
			if ( $form ) {
				$forms[] = $form;
			}
		}

		return $forms;
	}
}
