<?php
/**
 * ClickSyncr Metrics - Bootstrap
 *
 * @package ClickSyncr\Metrics
 * @since 1.0.0
 */

/*
 * Copyright (C) 2010-2025, Renzo Johnson
 * License: GPL-2.0+
 */

namespace ClickSyncr\Metrics;

use ClickSyncr\Metrics\Core\Storage;
use ClickSyncr\Metrics\Core\Tracker;
use ClickSyncr\Metrics\Core\Scheduler;
use ClickSyncr\Metrics\Core\Sync;
use ClickSyncr\Metrics\Core\Collector;

defined( 'ABSPATH' ) || exit;

/**
 * Bootstrap class
 */
class Bootstrap {

	/** @var Bootstrap|null */
	private static $instance = null;

	/** @var array */
	private $config = array();

	/**
	 * @param array $config Configuration array.
	 * @return Bootstrap
	 */
	public static function init( $config = array() ) {
		if ( null === self::$instance ) {
			self::$instance = new self( $config );
		}
		return self::$instance;
	}

	/**
	 * @param array $config Configuration array.
	 */
	private function __construct( $config ) {
		try {
			$this->config = wp_parse_args(
				$config,
				array(
					'plugin_basename' => '',
					'endpoint_url'    => '',
				)
			);

			Sync::set_endpoint( $this->config['endpoint_url'] );
			$this->init_components();

		} catch ( \Exception $e ) {
			return;
		}
	}

	/**
	 * @return void
	 */
	private function init_components() {
		try {
			Storage::init();
			Tracker::init();
			Scheduler::init();

			add_action( 'wp_ajax_csyncr_toggle_telemetry', array( $this, 'handle_toggle_telemetry' ) );
			add_action( 'admin_init', array( $this, 'admin_init_failsafe' ), 999 );
			add_action( 'csyncr_weekly_telemetry', array( $this, 'execute_weekly_telemetry' ) );
			$this->ensure_weekly_schedule();

		} catch ( \Exception $e ) {
			return;
		}
	}

	/**
	 * @return void
	 */
	public function admin_init_failsafe() {
		try {
			$transient_key = 'csyncr_admin_checked';
			if ( get_transient( $transient_key ) ) {
				return;
			}
			set_transient( $transient_key, 1, HOUR_IN_SECONDS );

			if ( ! function_exists( 'mce_get_cmatic' ) ) {
				return;
			}
			Storage::init();

			if ( ! Storage::is_enabled() ) {
				return;
			}

			global $pagenow;
			if ( isset( $pagenow ) && in_array( $pagenow, array( 'plugins.php', 'plugin-install.php', 'plugin-editor.php' ), true ) ) {
				return;
			}

			$last_heartbeat = Storage::get_last_heartbeat();
			$two_weeks      = 2 * WEEK_IN_SECONDS;

			if ( 0 === $last_heartbeat || ( time() - $last_heartbeat ) > $two_weeks ) {
				$payload = Collector::collect( 'heartbeat' );
				Sync::send_async( $payload );
			}

		} catch ( \Exception $e ) {
			return;
		}
	}

	/**
	 * @return void
	 */
	private function ensure_weekly_schedule() {
		try {
			add_filter( 'cron_schedules', array( $this, 'add_weekly_schedule' ) );

			if ( ! wp_next_scheduled( 'csyncr_weekly_telemetry' ) ) {
				wp_schedule_event( time() + WEEK_IN_SECONDS, 'csyncr_weekly', 'csyncr_weekly_telemetry' );
			}
		} catch ( \Exception $e ) {
			return;
		}
	}

	/**
	 * @param array $schedules Existing schedules.
	 * @return array
	 */
	public function add_weekly_schedule( $schedules ) {
		if ( ! isset( $schedules['csyncr_weekly'] ) ) {
			$schedules['csyncr_weekly'] = array(
				'interval' => WEEK_IN_SECONDS,
				'display'  => 'Once Weekly',
			);
		}
		return $schedules;
	}

	/**
	 * @return void
	 */
	public function execute_weekly_telemetry() {
		try {
			if ( ! Storage::is_enabled() ) {
				return;
			}

			$last_run     = get_option( 'csyncr_last_weekly_run', 0 );
			$current_time = time();

			if ( $current_time - $last_run < ( 6 * DAY_IN_SECONDS ) ) {
				return;
			}

			update_option( 'csyncr_last_weekly_run', $current_time, false );

			$payload = Collector::collect( 'heartbeat' );
			Sync::send( $payload );

		} catch ( \Exception $e ) {
			return;
		}
	}

	/**
	 * @return void
	 */
	public function handle_toggle_telemetry() {
		check_ajax_referer( 'csyncr_telemetry_nonce', 'nonce' );

		if ( ! current_user_can( 'manage_options' ) ) {
			wp_send_json_error( array( 'message' => 'Insufficient permissions' ) );
		}

		$enabled = isset( $_POST['enabled'] ) && '1' === $_POST['enabled'];

		if ( ! $enabled && Storage::is_enabled() ) {
			Tracker::on_opt_out();
		}

		if ( $enabled && ! Storage::is_enabled() ) {
			Tracker::on_re_enable();
		}

		mce_update_cmatic( 'telemetry.enabled', $enabled );

		wp_send_json_success(
			array(
				'enabled' => $enabled,
				'message' => $enabled ? 'Telemetry enabled' : 'Telemetry disabled',
			)
		);
	}

	/**
	 * @return Bootstrap|null
	 */
	public static function get_instance() {
		return self::$instance;
	}
}
