<?php
/**
 * Chimpmatic Plugin Links Component
 *
 * Shared plugin action links (Settings, Docs) for Lite and Pro.
 * Designed for Lite but extensible for Pro inheritance.
 *
 * @package    ChimpmaticLite
 * @subpackage Admin
 * @since      0.9.37
 * @author     Renzo Johnson <renzo.johnson@gmail.com>
 * @copyright  2013-2025 Chimpmatic LLC
 * @license    GPL-3.0-or-later
 */

// Exit if accessed directly.
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Class Cmatic_Plugin_Links
 *
 * Generates plugin action links (Settings, Documentation) for WordPress plugins list.
 * Can be used by both Lite and Pro versions.
 *
 * @since 0.9.37
 */
class Cmatic_Plugin_Links {

	/**
	 * Panel key for CF7 tab.
	 *
	 * @var string
	 */
	const PANEL_KEY = 'Chimpmatic';

	/**
	 * Documentation URL.
	 *
	 * @var string
	 */
	const DOCS_URL = 'https://chimpmatic.com/docs';

	/**
	 * Get the settings URL for the plugin.
	 *
	 * @param int|null $form_id Optional. Specific form ID. Default uses latest form.
	 * @return string Settings URL or empty string if no forms exist.
	 */
	public static function get_settings_url( $form_id = null ) {
		// Determine form ID.
		if ( null === $form_id && function_exists( 'cmatic_newest' ) ) {
			$form_id = cmatic_newest();
		}

		// Bail if no form ID.
		if ( empty( $form_id ) ) {
			return '';
		}

		// Build URL using add_query_arg (cleaner than concatenation).
		return add_query_arg(
			array(
				'page'       => 'wpcf7',
				'post'       => $form_id,
				'action'     => 'edit',
				'active-tab' => self::PANEL_KEY,
			),
			admin_url( 'admin.php' )
		);
	}

	/**
	 * Get the settings link HTML.
	 *
	 * @param int|null $form_id Optional. Specific form ID. Default uses latest form.
	 * @return string Settings link HTML or empty string.
	 */
	public static function get_settings_link( $form_id = null ) {
		$url = self::get_settings_url( $form_id );

		if ( empty( $url ) ) {
			return '';
		}

		return sprintf(
			'<a href="%s">%s</a>',
			esc_url( $url ),
			esc_html__( 'Settings', 'chimpmatic-lite' )
		);
	}

	/**
	 * Get the documentation link HTML.
	 *
	 * @return string Documentation link HTML.
	 */
	public static function get_docs_link() {
		return sprintf(
			'<a href="%s" target="_blank" title="%s">%s</a>',
			esc_url( self::DOCS_URL ),
			esc_attr__( 'Chimpmatic Documentation', 'chimpmatic-lite' ),
			esc_html__( 'Docs', 'chimpmatic-lite' )
		);
	}

	/**
	 * Filter plugin action links to add Settings link.
	 *
	 * Usage: add_filter( 'plugin_action_links_' . $basename, array( 'Cmatic_Plugin_Links', 'filter_action_links' ) );
	 *
	 * @param array<int, string> $links Existing plugin action links.
	 * @return array<int, string> Modified links with Settings prepended.
	 */
	public static function filter_action_links( array $links ) {
		$settings_link = self::get_settings_link();

		if ( ! empty( $settings_link ) ) {
			array_unshift( $links, $settings_link );
		}

		return $links;
	}

	/**
	 * Filter plugin row meta to add Documentation link.
	 *
	 * Usage: add_filter( 'plugin_row_meta', array( 'Cmatic_Plugin_Links', 'filter_row_meta' ), 10, 2 );
	 *
	 * @param array<int, string> $links Existing plugin meta links.
	 * @param string             $file  Plugin file basename.
	 * @param string             $match Plugin basename to match.
	 * @return array<int, string> Modified links with Docs appended.
	 */
	public static function filter_row_meta( array $links, string $file, string $match ) {
		if ( $file === $match ) {
			$links[] = self::get_docs_link();
		}

		return $links;
	}
}
