<?php
/**
 * CMatic Remote Content Fetcher
 *
 * Ultra-portable WordPress class to fetch and cache remote content (pricing data)
 * with zero plugin dependencies. Implements retry logic via WP-Cron.
 *
 * @package ChimpMatic
 * @since 0.9.28
 */

// Exit if accessed directly.
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Class CMatic_Remote_Fetcher
 *
 * Fetches and caches remote content from specified URL with automatic retry logic.
 *
 * @since 0.9.28
 */
class CMatic_Remote_Fetcher {

	/**
	 * Configuration array
	 *
	 * @var array
	 */
	private $config;

	/**
	 * Default configuration values
	 *
	 * @var array
	 */
	private $defaults = array(
		'url'             => '',
		'cache_key'       => 'cmatic_remote_data',
		'cache_duration'  => DAY_IN_SECONDS,
		'retry_interval'  => 600, // 10 minutes in seconds
		'max_retries'     => 3,
		'retry_count_key' => 'cmatic_retry_count',
		'cron_hook'       => 'cmatic_fetch_retry',
		'timeout'         => 15,
		'fallback_data'   => array(),
		'parser_callback' => null,
	);

	/**
	 * Constructor
	 *
	 * @param array $config Configuration array.
	 */
	public function __construct( $config = array() ) {
		$this->config = wp_parse_args( $config, $this->defaults );

		// Register WP-Cron hook for retry logic.
		add_action( $this->config['cron_hook'], array( $this, 'cron_retry_fetch' ) );
	}

	/**
	 * Main public method - Get data (from cache or fresh fetch)
	 *
	 * @return array Data array or fallback data on failure.
	 */
	public function get_data() {
		// Try to get cached data first.
		$cached_data = $this->get_cache();

		if ( false !== $cached_data ) {
			return $cached_data;
		}

		// Cache expired or empty - fetch fresh data.
		$fresh_data = $this->fetch_fresh_data();

		if ( false !== $fresh_data ) {
			// Success - cache it and clear any pending retries.
			$this->set_cache( $fresh_data );
			$this->clear_retry_schedule();
			return $fresh_data;
		}

		// Fetch failed - schedule retry via WP-Cron and return fallback.
		$this->schedule_retry();
		return $this->get_fallback_data();
	}

	/**
	 * Get cached data from transient
	 *
	 * @return array|false Cached data or false if not found/expired.
	 */
	public function get_cache() {
		return get_transient( $this->config['cache_key'] );
	}

	/**
	 * Store data in transient cache
	 *
	 * @param array $data Data to cache.
	 * @return bool True on success, false on failure.
	 */
	public function set_cache( $data ) {
		return set_transient(
			$this->config['cache_key'],
			$data,
			$this->config['cache_duration']
		);
	}

	/**
	 * Clear cached data
	 *
	 * @return bool True on success, false on failure.
	 */
	public function clear_cache() {
		return delete_transient( $this->config['cache_key'] );
	}

	/**
	 * Fetch fresh data from remote URL
	 *
	 * @return array|false Parsed data or false on failure.
	 */
	private function fetch_fresh_data() {
		if ( empty( $this->config['url'] ) ) {
			return false;
		}

		// Fetch remote content via WordPress HTTP API.
		$response = wp_remote_get(
			$this->config['url'],
			array(
				'timeout'    => $this->config['timeout'],
				'user-agent' => 'WordPress/' . get_bloginfo( 'version' ) . '; ' . home_url(),
			)
		);

		// Check for HTTP errors.
		if ( is_wp_error( $response ) ) {
			return false;
		}

		// Check response code.
		$response_code = wp_remote_retrieve_response_code( $response );
		if ( 200 !== $response_code ) {
			return false;
		}

		// Get response body.
		$body = wp_remote_retrieve_body( $response );
		if ( empty( $body ) ) {
			return false;
		}

		// Parse the content.
		$parsed_data = $this->parse_content( $body );

		return $parsed_data;
	}

	/**
	 * Parse fetched content
	 *
	 * @param string $content Raw HTML or JSON content.
	 * @return array|false Parsed data or false on failure.
	 */
	private function parse_content( $content ) {
		// If custom parser callback provided, use it.
		if ( is_callable( $this->config['parser_callback'] ) ) {
			return call_user_func( $this->config['parser_callback'], $content );
		}

		// Try JSON parsing first (api.chimpmatic.com/welcome format).
		$json_data = $this->parse_pricing_json( $content );
		if ( false !== $json_data ) {
			return $json_data;
		}

		// Fall back to HTML parsing (chimpmatic.com homepage format).
		return $this->parse_pricing_html( $content );
	}

	/**
	 * Parse JSON pricing data (api.chimpmatic.com/welcome format)
	 *
	 * @param string $content Raw content (might be JSON).
	 * @return array|false Pricing data or false if not valid JSON.
	 */
	private function parse_pricing_json( $content ) {
		// Try to decode JSON.
		$json = json_decode( $content, true );

		// Check if valid JSON and has required fields.
		if ( null === $json || ! is_array( $json ) ) {
			return false;
		}

		if ( ! isset( $json['regular_price'] ) || ! isset( $json['discount_percent'] ) ) {
			return false;
		}

		// Build pricing data from JSON.
		$pricing_data = array(
			'regular_price'    => (int) $json['regular_price'],
			'sale_price'       => isset( $json['sale_price'] ) ? (float) $json['sale_price'] : null,
			'discount_percent' => (int) $json['discount_percent'],
			'coupon_code'      => isset( $json['coupon_code'] ) ? sanitize_text_field( $json['coupon_code'] ) : null,
			'last_updated'     => current_time( 'mysql' ),
		);

		// Calculate sale price if not provided.
		if ( null === $pricing_data['sale_price'] ) {
			$discount_amount            = $pricing_data['regular_price'] * ( $pricing_data['discount_percent'] / 100 );
			$pricing_data['sale_price'] = $pricing_data['regular_price'] - $discount_amount;
		}

		// Auto-generate coupon code if not provided: NOW{discount_percent}.
		if ( null === $pricing_data['coupon_code'] ) {
			$pricing_data['coupon_code'] = 'NOW' . $pricing_data['discount_percent'];
		}

		// Generate formatted string.
		$pricing_data['formatted'] = sprintf(
			'$%d → $%s • Save %d%%',
			$pricing_data['regular_price'],
			number_format( $pricing_data['sale_price'], 2 ),
			$pricing_data['discount_percent']
		);

		return $pricing_data;
	}

	/**
	 * Default parser for chimpmatic.com pricing
	 *
	 * @param string $html Raw HTML content.
	 * @return array|false Pricing data or false on failure.
	 */
	private function parse_pricing_html( $html ) {
		$pricing_data = array(
			'regular_price'    => null,
			'sale_price'       => null,
			'discount_percent' => null,
			'coupon_code'      => null,
			'formatted'        => null,
			'last_updated'     => current_time( 'mysql' ),
		);

		// Pattern 1: Extract "Single Site" pricing like "$39/year".
		if ( preg_match( '/Single\s+Site[^$]*\$(\d+)\/year/i', $html, $matches ) ) {
			$pricing_data['regular_price'] = (int) $matches[1];
		}

		// Pattern 2: Extract discount percentage like "25% Off".
		if ( preg_match( '/(\d+)%\s+Off/i', $html, $matches ) ) {
			$pricing_data['discount_percent'] = (int) $matches[1];
		}

		// Pattern 3: Extract coupon code like "NOW25".
		if ( preg_match( '/coupon\s+code\s+["\']([A-Z0-9]+)["\']/i', $html, $matches ) ) {
			$pricing_data['coupon_code'] = sanitize_text_field( $matches[1] );
		}

		// Calculate sale price if we have both regular price and discount.
		if ( $pricing_data['regular_price'] && $pricing_data['discount_percent'] ) {
			$discount_amount            = $pricing_data['regular_price'] * ( $pricing_data['discount_percent'] / 100 );
			$pricing_data['sale_price'] = $pricing_data['regular_price'] - $discount_amount;
		}

		// Generate formatted string.
		if ( $pricing_data['regular_price'] && $pricing_data['sale_price'] && $pricing_data['discount_percent'] ) {
			$pricing_data['formatted'] = sprintf(
				'$%d → $%s • Save %d%%',
				$pricing_data['regular_price'],
				number_format( $pricing_data['sale_price'], 2 ),
				$pricing_data['discount_percent']
			);
		}

		// Validate we got at least the regular price.
		if ( null === $pricing_data['regular_price'] ) {
			return false;
		}

		return $pricing_data;
	}

	/**
	 * Get fallback data when fetch fails
	 *
	 * @return array Fallback data from config or default values.
	 */
	private function get_fallback_data() {
		if ( ! empty( $this->config['fallback_data'] ) ) {
			return $this->config['fallback_data'];
		}

		// Default fallback pricing.
		return array(
			'regular_price'    => 39,
			'sale_price'       => 29.25,
			'discount_percent' => 25,
			'coupon_code'      => 'NOW25',
			'formatted'        => '$39 → $29.25 • Save 25%',
			'last_updated'     => null,
		);
	}

	/**
	 * Schedule WP-Cron retry on fetch failure
	 *
	 * @return void
	 */
	private function schedule_retry() {
		// Get current retry count.
		$retry_count = (int) get_option( $this->config['retry_count_key'], 0 );

		// Check if we've exceeded max retries.
		if ( $retry_count >= $this->config['max_retries'] ) {
			return;
		}

		// Increment retry count.
		update_option( $this->config['retry_count_key'], $retry_count + 1 );

		// Schedule next retry if not already scheduled.
		if ( ! wp_next_scheduled( $this->config['cron_hook'] ) ) {
			wp_schedule_single_event(
				time() + $this->config['retry_interval'],
				$this->config['cron_hook']
			);
		}
	}

	/**
	 * WP-Cron callback - Retry fetch
	 *
	 * @return void
	 */
	public function cron_retry_fetch() {
		// Attempt fresh fetch.
		$fresh_data = $this->fetch_fresh_data();

		if ( false !== $fresh_data ) {
			// Success - cache it and clear retry schedule.
			$this->set_cache( $fresh_data );
			$this->clear_retry_schedule();
		} else {
			// Failed again - schedule another retry (if within limit).
			$this->schedule_retry();
		}
	}

	/**
	 * Clear retry schedule and reset retry count
	 *
	 * @return void
	 */
	private function clear_retry_schedule() {
		// Clear scheduled event.
		$timestamp = wp_next_scheduled( $this->config['cron_hook'] );
		if ( $timestamp ) {
			wp_unschedule_event( $timestamp, $this->config['cron_hook'] );
		}

		// Reset retry count.
		delete_option( $this->config['retry_count_key'] );
	}

	/**
	 * Manual clear - Clear cache and retry schedule (useful for testing)
	 *
	 * @return void
	 */
	public function clear_all() {
		$this->clear_cache();
		$this->clear_retry_schedule();
	}
}
