<?php
/**
 * Chimpmatic Header Component
 *
 * Reusable header for all Chimpmatic admin panels.
 * Designed for Lite but extensible for Pro inheritance.
 *
 * @package    ChimpmaticLite
 * @subpackage Admin
 * @since      0.9.36
 * @author     Renzo Johnson <renzo.johnson@gmail.com>
 * @copyright  2013-2025 Chimpmatic LLC
 * @license    GPL-3.0-or-later
 */

// Exit if accessed directly.
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Class Cmatic_Header
 *
 * Renders the Chimpmatic admin header with dynamic badge, version, and API status.
 *
 * @since 0.9.36
 */
class Cmatic_Header {

	/**
	 * Plugin version.
	 *
	 * @var string
	 */
	private $version;

	/**
	 * Whether Pro is licensed and active.
	 *
	 * @var bool
	 */
	private $is_pro;

	/**
	 * API connection status.
	 *
	 * @var string|null 'connected', 'disconnected', or null for unknown.
	 */
	private $api_status;

	/**
	 * Review link URL.
	 *
	 * @var string
	 */
	private $review_url;

	/**
	 * Rotating review phrases.
	 *
	 * @var array<int, string>
	 */
	private $review_phrases;

	/**
	 * Constructor.
	 *
	 * @param array<string, mixed> $args {
	 *     Optional. Configuration arguments.
	 *
	 *     @type string      $version    Plugin version. Default from SPARTAN_MCE_VERSION or CMATIC_VERSION.
	 *     @type bool        $is_pro     Whether Pro is active. Default auto-detected.
	 *     @type string|null $api_status API status: 'connected', 'disconnected', or null. Default null.
	 *     @type string      $review_url Review link URL. Default WordPress.org review URL.
	 * }
	 */
	public function __construct( array $args = array() ) {
		$this->version    = $this->resolve_version( $args );
		$this->is_pro     = $this->resolve_pro_status( $args );
		$this->api_status = isset( $args['api_status'] ) && is_string( $args['api_status'] ) ? $args['api_status'] : null;
		$this->review_url = isset( $args['review_url'] ) && is_string( $args['review_url'] ) ? $args['review_url'] : $this->get_default_review_url();

		$this->review_phrases = array(
			__( 'Loving Chimpmatic? Leave a review', 'chimpmatic-lite' ),
			__( 'We run on coffee & 5-star reviews', 'chimpmatic-lite' ),
			__( 'Make a developer smile today', 'chimpmatic-lite' ),
			__( 'Got 10 seconds? Rate us!', 'chimpmatic-lite' ),
			__( 'Fuel our plugin addiction', 'chimpmatic-lite' ),
			__( 'Stars make us code faster', 'chimpmatic-lite' ),
			__( 'Help us stay free & caffeinated', 'chimpmatic-lite' ),
			__( "Love us? Don't keep it a secret", 'chimpmatic-lite' ),
			__( 'Your review = our dopamine', 'chimpmatic-lite' ),
			__( 'Be our hero on WordPress.org', 'chimpmatic-lite' ),
			__( 'Psst... we love 5 stars', 'chimpmatic-lite' ),
			__( 'Worth 5 stars? Let us know', 'chimpmatic-lite' ),
			__( 'Support indie plugins', 'chimpmatic-lite' ),
			__( 'Reviews keep the lights on', 'chimpmatic-lite' ),
			__( 'Spread the Chimpmatic love', 'chimpmatic-lite' ),
			__( 'Got love? Leave stars', 'chimpmatic-lite' ),
			__( 'One click, endless gratitude', 'chimpmatic-lite' ),
			__( 'Help other WP folks find us', 'chimpmatic-lite' ),
			__( 'Like us? Rate us!', 'chimpmatic-lite' ),
			__( 'Your stars = our motivation', 'chimpmatic-lite' ),
		);
	}

	/**
	 * Resolve plugin version from args or constants.
	 *
	 * @param array<string, mixed> $args Configuration arguments.
	 * @return string Plugin version.
	 */
	private function resolve_version( array $args ): string {
		if ( isset( $args['version'] ) && is_string( $args['version'] ) ) {
			return $args['version'];
		}

		// Pro constant takes precedence.
		if ( defined( 'CMATIC_VERSION' ) ) {
			return (string) CMATIC_VERSION;
		}

		// Lite constant.
		if ( defined( 'SPARTAN_MCE_VERSION' ) ) {
			return (string) SPARTAN_MCE_VERSION;
		}

		return '0.0.0';
	}

	/**
	 * Resolve Pro license status from args or function check.
	 *
	 * @param array<string, mixed> $args Configuration arguments.
	 * @return bool Whether Pro is active and licensed.
	 */
	private function resolve_pro_status( array $args ): bool {
		if ( isset( $args['is_pro'] ) ) {
			return (bool) $args['is_pro'];
		}

		// Check if Pro's blessing function exists and returns true.
		if ( function_exists( 'cmatic_is_blessed' ) ) {
			return (bool) cmatic_is_blessed();
		}

		return false;
	}

	/**
	 * Get default WordPress.org review URL.
	 *
	 * @return string Review URL.
	 */
	private function get_default_review_url(): string {
		return 'https://wordpress.org/support/plugin/contact-form-7-mailchimp-extension/reviews/?filter=5';
	}

	/**
	 * Get random review phrase.
	 *
	 * @return string Random review phrase.
	 */
	private function get_review_phrase(): string {
		$index = wp_rand( 0, count( $this->review_phrases ) - 1 );
		return $this->review_phrases[ $index ];
	}

	/**
	 * Render the header HTML.
	 *
	 * @return void
	 */
	public function render(): void {
		$badge_class = $this->is_pro ? 'cmatic-header__badge--pro' : 'cmatic-header__badge--lite';
		$badge_text  = $this->is_pro ? __( 'PRO', 'chimpmatic-lite' ) : __( 'Lite', 'chimpmatic-lite' );
		?>
		<header class="cmatic-header">
			<div class="cmatic-header__inner">
				<div class="cmatic-header__brand">
					<span class="cmatic-header__title"><?php esc_html_e( 'Chimpmatic', 'chimpmatic-lite' ); ?></span>
					<span class="cmatic-header__badge <?php echo esc_attr( $badge_class ); ?>"><?php echo esc_html( $badge_text ); ?></span>
					<span class="cmatic-header__version">v<?php echo esc_html( $this->version ); ?></span>
					<?php $this->render_api_status(); ?>
				</div>
				<div class="cmatic-header__actions">
					<a href="<?php echo esc_url( $this->review_url ); ?>" target="_blank" rel="noopener noreferrer" class="cmatic-header__review">
						<?php echo esc_html( $this->get_review_phrase() ); ?>
						<span class="cmatic-sparkles" aria-label="5 sparkles"></span>
					</a>
				</div>
			</div>
		</header>
		<?php
	}

	/**
	 * Render API status indicator if status is set.
	 *
	 * @return void
	 */
	private function render_api_status(): void {
		if ( null === $this->api_status ) {
			return;
		}

		$is_connected = ( 'connected' === $this->api_status );
		$dot_class    = $is_connected ? 'cmatic-header__status-dot--connected' : 'cmatic-header__status-dot--disconnected';
		$status_text  = $is_connected
			? __( 'API Connected', 'chimpmatic-lite' )
			: __( 'API Inactive', 'chimpmatic-lite' );
		?>
		<div class="cmatic-header__status">
			<span class="cmatic-header__status-dot <?php echo esc_attr( $dot_class ); ?>"></span>
			<span class="cmatic-header__status-text"><?php echo esc_html( $status_text ); ?></span>
		</div>
		<?php
	}

	/**
	 * Output header CSS styles.
	 *
	 * @deprecated Styles are now in assets/css/chimpmatic-lite.css
	 * @return void
	 */
	public static function render_styles(): void {
		// Styles moved to external CSS file: assets/css/chimpmatic-lite.css
		// This method is kept for backward compatibility but does nothing.
	}

	/**
	 * Set API status dynamically.
	 *
	 * @param string|null $status API status: 'connected', 'disconnected', or null.
	 * @return self
	 */
	public function set_api_status( ?string $status ): self {
		$this->api_status = $status;
		return $this;
	}

	/**
	 * Static factory for quick rendering.
	 *
	 * @param array<string, mixed> $args Configuration arguments.
	 * @return void
	 */
	public static function output( array $args = array() ): void {
		$header = new self( $args );
		$header->render();
	}
}
