<?php
/**
 * Deactivation Survey Modal
 *
 * Modern, accessible, portable deactivation feedback system.
 * Uses WordPress REST API (not wp-ajax.php).
 * Vanilla JavaScript (no jQuery).
 * WCAG 2.1 AA accessible.
 * Sends feedback via email to chimpmatic@gmail.com
 *
 * @package ChimpMatic_Lite
 * @since 0.9.28
 * @version 1.3.0
 *
 * Copyright (C) 2025, Renzo Johnson (email: renzo.johnson at gmail.com)
 *
 * For the full copyright and license information, please view the LICENSE
 * file that was distributed with this source code, or visit:
 * https://www.gnu.org/licenses/gpl-2.0.html
 */

defined( 'ABSPATH' ) || exit;

if ( ! class_exists( 'Cmatic_Deactivation_Survey' ) ) {

	/**
	 * Class Cmatic_Deactivation_Survey
	 *
	 * Handles deactivation survey modal display and data collection.
	 * Plugin-agnostic, reusable across multiple plugins.
	 * Sends feedback via email (wp_mail).
	 */
	class Cmatic_Deactivation_Survey {

		/**
		 * Plugin slug (e.g., 'contact-form-7-mailchimp-extension')
		 *
		 * @var string
		 */
		private $plugin_slug;

		/**
		 * Plugin basename (e.g., 'chimpmatic-lite/chimpmatic-lite.php')
		 *
		 * @var string
		 */
		private $plugin_basename;

		/**
		 * Deactivation reasons array
		 *
		 * @var array
		 */
		private $reasons;

		/**
		 * REST API namespace
		 *
		 * @var string
		 */
		private $rest_namespace = 'cmatic/v1';

		/**
		 * REST API endpoint route
		 *
		 * @var string
		 */
		private $rest_route = '/deactivation-feedback';

		/**
		 * Email address for feedback
		 *
		 * @var string
		 */
		private $feedback_email = 'chimpmatic@gmail.com';

		/**
		 * Constructor
		 *
		 * @param array $args {
		 *     Configuration arguments.
		 *
		 *     @type string $plugin_slug     Plugin slug (required)
		 *     @type string $plugin_basename Plugin basename (required)
		 *     @type array  $reasons         Deactivation reasons (required)
		 * }
		 */
		public function __construct( $args ) {
			$this->plugin_slug     = isset( $args['plugin_slug'] ) ? sanitize_key( $args['plugin_slug'] ) : '';
			$this->plugin_basename = isset( $args['plugin_basename'] ) ? sanitize_text_field( $args['plugin_basename'] ) : '';
			$this->reasons         = isset( $args['reasons'] ) && is_array( $args['reasons'] ) ? $args['reasons'] : array();

			// Validate required parameters.
			if ( empty( $this->plugin_slug ) || empty( $this->plugin_basename ) || empty( $this->reasons ) ) {
				return;
			}
		}

		/**
		 * Initialize hooks
		 *
		 * @return void
		 */
		public function init() {
			add_action( 'admin_enqueue_scripts', array( $this, 'enqueue_assets' ) );
			add_action( 'admin_footer', array( $this, 'render_modal' ) );
			add_action( 'rest_api_init', array( $this, 'register_rest_endpoint' ) );
		}

		/**
		 * Enqueue CSS and JavaScript (only on plugins.php)
		 *
		 * @param string $hook Current admin page hook.
		 * @return void
		 */
		public function enqueue_assets( $hook ) {
			if ( 'plugins.php' !== $hook ) {
				return;
			}

			// Enqueue CSS.
			wp_enqueue_style(
				'cmatic-deactivate-modal',
				SPARTAN_MCE_PLUGIN_URL . 'assets/css/chimpmatic-lite-deactivate.css',
				array(),
				SPARTAN_MCE_VERSION
			);

			// Enqueue JavaScript.
			wp_enqueue_script(
				'cmatic-deactivate-modal',
				SPARTAN_MCE_PLUGIN_URL . 'assets/js/chimpmatic-lite-deactivate.js',
				array(),
				SPARTAN_MCE_VERSION,
				true
			);

			// Localize script.
			wp_localize_script(
				'cmatic-deactivate-modal',
				'cmaticDeactivate',
				array(
					'pluginSlug'     => $this->plugin_slug,
					'pluginBasename' => $this->plugin_basename,
					'restUrl'        => rest_url( $this->rest_namespace . $this->rest_route ),
					'restNonce'      => wp_create_nonce( 'wp_rest' ),
					'reasons'        => $this->reasons,
					'strings'        => $this->get_strings(),
					'plugins'        => $this->get_plugin_list(),
				)
			);
		}

		/**
		 * Get list of all plugins for dropdowns
		 *
		 * @return array Array of plugin options with value and label.
		 */
		private function get_plugin_list() {
			if ( ! function_exists( 'get_plugins' ) ) {
				require_once ABSPATH . 'wp-admin/includes/plugin.php';
			}

			$all_plugins    = get_plugins();
			$active_plugins = get_option( 'active_plugins', array() );
			$plugin_options = array();

			foreach ( $all_plugins as $plugin_file => $plugin_data ) {
				// Exclude self from the list.
				if ( $plugin_file !== $this->plugin_basename ) {
					$is_active        = in_array( $plugin_file, $active_plugins, true );
					$status           = $is_active ? ' (Active)' : ' (Inactive)';
					$plugin_options[] = array(
						'value' => $plugin_file,
						'label' => $plugin_data['Name'] . $status,
					);
				}
			}

			return $plugin_options;
		}

		/**
		 * Render modal HTML container (only on plugins.php)
		 *
		 * @return void
		 */
		public function render_modal() {
			$screen = get_current_screen();
			if ( ! $screen || 'plugins' !== $screen->id ) {
				return;
			}

			// Minimal HTML - JavaScript will build the modal content.
			echo '<div id="cmatic-deactivate-modal" class="cmatic-modal" role="dialog" aria-modal="true" aria-labelledby="cmatic-modal-title" aria-describedby="cmatic-modal-description"></div>';
		}

		/**
		 * Register REST API endpoint
		 *
		 * @return void
		 */
		public function register_rest_endpoint() {
			register_rest_route(
				$this->rest_namespace,
				$this->rest_route,
				array(
					'methods'             => 'POST',
					'callback'            => array( $this, 'handle_feedback_submission' ),
					'permission_callback' => array( $this, 'check_permissions' ),
					'args'                => array(
						'reason_id'   => array(
							'required'          => true,
							'type'              => 'integer',
							'sanitize_callback' => 'absint',
							'validate_callback' => function ( $param ) {
								return is_numeric( $param ) && $param > 0;
							},
						),
						'reason_text' => array(
							'required'          => false,
							'type'              => 'string',
							'sanitize_callback' => 'sanitize_text_field',
							'default'           => '',
						),
					),
				)
			);
		}

		/**
		 * Check permissions for REST API endpoint
		 *
		 * @return bool True if user has permission, false otherwise.
		 */
		public function check_permissions() {
			return current_user_can( 'install_plugins' );
		}

		/**
		 * Handle feedback submission (REST API callback)
		 *
		 * @param WP_REST_Request $request REST API request object.
		 * @return WP_REST_Response|WP_Error Response or error.
		 */
		public function handle_feedback_submission( $request ) {
			// Get parameters.
			$reason_id   = $request->get_param( 'reason_id' );
			$reason_text = $request->get_param( 'reason_text' );

			// Truncate text to max length (200 chars for single-line inputs).
			if ( ! empty( $reason_text ) ) {
				$reason_text = substr( $reason_text, 0, 200 );
			}

			// Get activation date from cmatic option.
			$activation_timestamp = 0;
			if ( function_exists( 'mce_get_cmatic' ) ) {
				$activation_timestamp = mce_get_cmatic( 'install.quest', 0 );
			}
			$activation_date = $activation_timestamp ? gmdate( 'Y-m-d H:i:s', $activation_timestamp ) : '';

			// Get active and inactive plugins.
			if ( ! function_exists( 'get_plugins' ) ) {
				require_once ABSPATH . 'wp-admin/includes/plugin.php';
			}
			$all_plugins      = get_plugins();
			$active_plugins   = get_option( 'active_plugins', array() );
			$inactive_plugins = array_diff( array_keys( $all_plugins ), $active_plugins );

			// Get emails sent count.
			$emails_sent = 0;
			if ( function_exists( 'mce_get_cmatic' ) ) {
				$emails_sent = (int) mce_get_cmatic( 'stats.sent', 0 );
			}

			// Check API connection status.
			// If emails were sent successfully, API must be connected.
			$api_connected      = ( $emails_sent > 0 );
			$api_connected_date = $api_connected ? $activation_date : '';

			// Get language.
			$language = get_locale();

			// Get theme info.
			$theme         = wp_get_theme();
			$theme_name    = $theme->get( 'Name' );
			$theme_version = $theme->get( 'Version' );
			$theme_info    = $theme_name . ' ' . $theme_version;

			// Build feedback data.
			$feedback = array(
				'reason_id'          => $reason_id,
				'reason_text'        => $reason_text,
				'activation_date'    => $activation_date,
				'plugin_version'     => SPARTAN_MCE_VERSION,
				'wp_version'         => get_bloginfo( 'version' ),
				'php_version'        => phpversion(),
				'site_url'           => home_url(),
				'timestamp'          => current_time( 'mysql' ),
				'active_plugins'     => $active_plugins,
				'inactive_plugins'   => array_values( $inactive_plugins ),
				'api_connected'      => $api_connected,
				'api_connected_date' => $api_connected_date,
				'emails_sent'        => $emails_sent,
				'language'           => $language,
				'theme'              => $theme_info,
			);

			// Send feedback via email.
			$result = $this->send_feedback_email( $feedback );

			// Always return success (non-blocking deactivation).
			// Errors are logged but don't prevent deactivation.
			return new WP_REST_Response(
				array(
					'success' => true,
					'message' => __( 'Thank you for your feedback!', 'chimpmatic-lite' ),
				),
				200
			);
		}

		/**
		 * Send feedback via email
		 *
		 * Sends deactivation feedback to chimpmatic@gmail.com.
		 * Uses WordPress wp_mail() function.
		 * Non-blocking (always succeeds even if email fails).
		 *
		 * @param array $feedback Feedback data to send.
		 * @return bool True if email sent, false on error.
		 */
		private function send_feedback_email( $feedback ) {
			// Map reason IDs to readable labels.
			$reason_labels = array(
				0 => 'Skipped survey',
				1 => 'I found a better Mailchimp integration',
				2 => 'Missing features I need',
				3 => 'Too complicated to set up',
				4 => "It's a temporary deactivation",
				5 => 'Conflicts with another plugin',
				6 => 'Other reason',
			);

			$reason_label = isset( $reason_labels[ $feedback['reason_id'] ] ) ? $reason_labels[ $feedback['reason_id'] ] : 'Unknown';

			// Calculate days active.
			$days_active = 0;
			if ( ! empty( $feedback['activation_date'] ) ) {
				$activation_timestamp   = strtotime( $feedback['activation_date'] );
				$deactivation_timestamp = strtotime( $feedback['timestamp'] );
				$days_active            = round( ( $deactivation_timestamp - $activation_timestamp ) / DAY_IN_SECONDS );
			}

			// Build email subject with timestamp and reason.
			$domain = wp_parse_url( $feedback['site_url'], PHP_URL_HOST );
			$subject = sprintf(
				'[Deact: %s%s] %s | %s',
				gmdate( 'H' ), // Hour (00-23).
				gmdate( 'i' ), // Minute (00-59).
				$reason_label,
				$domain
			);

			// Build email body.
			$message  = "ChimpMatic Lite - Deactivation Feedback\n";
			$message .= "==========================================\n\n";

			$message .= "DEACTIVATION REASON\n";
			$message .= "-------------------\n";
			$message .= "Reason: {$reason_label}\n";
			if ( ! empty( $feedback['reason_text'] ) ) {
				$message .= "Details: {$feedback['reason_text']}\n";
			}
			$message .= "\n";

			$message .= "SITE INFORMATION\n";
			$message .= "----------------\n";
			$message .= "Site URL: {$feedback['site_url']}\n";
			$message .= 'Activation Date: ' . ( ! empty( $feedback['activation_date'] ) ? $feedback['activation_date'] : 'Unknown' ) . "\n";
			$message .= "Deactivation Date: {$feedback['timestamp']}\n";
			if ( $days_active > 0 ) {
				$message .= "Days Active: {$days_active} days\n";
			}
			$message .= 'API Connected: ' . ( $feedback['api_connected'] ? 'Yes' : 'No' );
			if ( $feedback['api_connected'] && ! empty( $feedback['api_connected_date'] ) ) {
				$message .= " (since {$feedback['api_connected_date']})";
			}
			$message .= "\n";
			$message .= "Emails Sent: {$feedback['emails_sent']}\n";
			$message .= "Language: {$feedback['language']}\n";
			$message .= "Theme: {$feedback['theme']}\n";
			$message .= "\n";

			$message .= "VERSIONS\n";
			$message .= "--------\n";
			$message .= "Plugin Version: {$feedback['plugin_version']}\n";
			$message .= "WordPress Version: {$feedback['wp_version']}\n";
			$message .= "PHP Version: {$feedback['php_version']}\n";
			$message .= "\n";

			$message .= "PLUGINS\n";
			$message .= "-------\n";
			$message .= 'Active Plugins: ' . count( $feedback['active_plugins'] ) . "\n";
			if ( ! empty( $feedback['active_plugins'] ) ) {
				foreach ( $feedback['active_plugins'] as $plugin ) {
					$message .= "  - {$plugin}\n";
				}
			}
			$message .= "\n";

			$message .= 'Inactive Plugins: ' . count( $feedback['inactive_plugins'] ) . "\n";
			if ( ! empty( $feedback['inactive_plugins'] ) ) {
				foreach ( $feedback['inactive_plugins'] as $plugin ) {
					$message .= "  - {$plugin}\n";
				}
			}
			$message .= "\n";

			// Set email headers.
			$headers = array(
				'Content-Type: text/plain; charset=UTF-8',
				'From: Chimpmatic Stats <wordpress@' . wp_parse_url( home_url(), PHP_URL_HOST ) . '>',
			);

			// Send email.
			$sent = wp_mail( $this->feedback_email, $subject, $message, $headers );

			return $sent;
		}

		/**
		 * Get translatable strings for JavaScript
		 *
		 * @return array Localized strings.
		 */
		private function get_strings() {
			return array(
				'title'           => __( 'Quick Feedback', 'chimpmatic-lite' ),
				'description'     => __( 'If you have a moment, please let us know why you are deactivating ChimpMatic Lite:', 'chimpmatic-lite' ),
				'submitButton'    => __( 'Submit & Deactivate', 'chimpmatic-lite' ),
				'skipButton'      => __( 'Skip & Deactivate', 'chimpmatic-lite' ),
				'cancelButton'    => __( 'Cancel', 'chimpmatic-lite' ),
				'thankYou'        => __( 'Thank you for your feedback!', 'chimpmatic-lite' ),
				'deactivating'    => __( 'Deactivating plugin...', 'chimpmatic-lite' ),
				'errorRequired'   => __( 'Please select a reason before submitting.', 'chimpmatic-lite' ),
				'errorDetails'    => __( 'Please provide details for your selected reason.', 'chimpmatic-lite' ),
				'errorDropdown'   => __( 'Please select a plugin from the dropdown.', 'chimpmatic-lite' ),
				'errorSubmission' => __( 'Failed to submit feedback. The plugin will still be deactivated.', 'chimpmatic-lite' ),
				'closeLabel'      => __( 'Close dialog', 'chimpmatic-lite' ),
			);
		}
	}
}
