<?php
/**
 * Chimpmatic Data Container Component
 *
 * Shared data container for passing PHP data to JavaScript without hidden inputs.
 * Follows ClickSyncr pattern: single data element with data-* attributes.
 * Designed for LITE but PRO inherits via class_exists() pattern.
 *
 * @package    ChimpmaticLite
 * @subpackage Admin
 * @since      0.9.38
 * @author     Renzo Johnson <renzo.johnson@gmail.com>
 * @copyright  2013-2025 Chimpmatic LLC
 * @license    GPL-3.0-or-later
 */

// Exit if accessed directly.
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Class Cmatic_Data_Container
 *
 * Renders a data container element with data-* attributes for JavaScript access.
 * Replaces legacy hidden input fields (#mce_txtcomodin, #chm_txcomodin, etc.).
 *
 * Usage:
 * PHP: Cmatic_Data_Container::render( $form_id, $apivalid );
 * JS:  const formId = document.getElementById('cmatic_data').dataset.formId;
 *
 * @since 0.9.38
 */
class Cmatic_Data_Container {

	/**
	 * Container element ID.
	 *
	 * @var string
	 */
	const ELEMENT_ID = 'cmatic_data';

	/**
	 * Render the data container element (legacy standalone mode).
	 *
	 * @param int    $form_id   CF7 form ID.
	 * @param string $apivalid  API validation status ('0' or '1').
	 * @param array  $extra     Optional extra data attributes.
	 * @return void
	 */
	public static function render( int $form_id, string $apivalid = '0', array $extra = array() ): void {
		$attrs_html = self::build_data_attrs( $form_id, $apivalid, $extra );

		// Output hidden container element (display:none but accessible to JS).
		printf(
			'<div id="%s"%s style="display:none;"></div>',
			esc_attr( self::ELEMENT_ID ),
			$attrs_html
		);
	}

	/**
	 * Render opening tag for combined wrapper (ID + class + data attributes).
	 *
	 * Merges #cmatic_data with .cmatic-inner into single element.
	 * Use with render_close() for the closing tag.
	 *
	 * @param int    $form_id   CF7 form ID.
	 * @param string $apivalid  API validation status ('0' or '1').
	 * @param array  $extra     Optional extra data attributes.
	 * @return void
	 */
	public static function render_open( int $form_id, string $apivalid = '0', array $extra = array() ): void {
		$attrs_html = self::build_data_attrs( $form_id, $apivalid, $extra );

		printf(
			'<div id="%s" class="cmatic-inner"%s>',
			esc_attr( self::ELEMENT_ID ),
			$attrs_html
		);
	}

	/**
	 * Render closing tag for combined wrapper.
	 *
	 * @return void
	 */
	public static function render_close(): void {
		echo '</div><!-- #cmatic_data.cmatic-inner -->';
	}

	/**
	 * Build data attributes string.
	 *
	 * @param int    $form_id   CF7 form ID.
	 * @param string $apivalid  API validation status ('0' or '1').
	 * @param array  $extra     Optional extra data attributes.
	 * @return string HTML data attributes string.
	 */
	private static function build_data_attrs( int $form_id, string $apivalid = '0', array $extra = array() ): string {
		$data_attrs = array(
			'form-id'   => $form_id,
			'api-valid' => $apivalid,
		);

		// Merge extra data attributes.
		foreach ( $extra as $key => $value ) {
			// Sanitize key: convert underscores to hyphens for HTML5 data-* convention.
			$key = str_replace( '_', '-', sanitize_key( $key ) );

			// JSON encode arrays/objects, escape strings.
			if ( is_array( $value ) || is_object( $value ) ) {
				$data_attrs[ $key ] = wp_json_encode( $value );
			} else {
				$data_attrs[ $key ] = esc_attr( $value );
			}
		}

		// Build data attributes string.
		$attrs_html = '';
		foreach ( $data_attrs as $key => $value ) {
			$attrs_html .= sprintf( ' data-%s="%s"', esc_attr( $key ), esc_attr( $value ) );
		}

		return $attrs_html;
	}

	/**
	 * Get the element ID for JavaScript reference.
	 *
	 * @return string Element ID.
	 */
	public static function get_element_id(): string {
		return self::ELEMENT_ID;
	}
}
