<?php
/**
 * Chimpmatic Audiences Panel Component
 *
 * Shared audiences dropdown panel for LITE and PRO.
 * Renders the Mailchimp audiences list, dropdown, and Sync Fields button.
 * Designed for LITE but PRO extends via class_exists() pattern.
 *
 * @package    ChimpmaticLite
 * @subpackage Admin
 * @since      0.9.38
 * @author     Renzo Johnson <renzo.johnson@gmail.com>
 * @copyright  2013-2025 Chimpmatic LLC
 * @license    GPL-3.0-or-later
 */

// Exit if accessed directly.
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Class Cmatic_Audiences
 *
 * Renders the audiences dropdown panel with list selection and Sync Fields button.
 * PRO can extend this class for additional features (groups, tags, GDPR).
 *
 * @since 0.9.38
 */
class Cmatic_Audiences {

	/**
	 * Help URL for list ID instructions.
	 *
	 * @var string
	 */
	const HELP_URL = '//chimpmatic.com/help/how-to-get-your-mailchimp-api-key';

	/**
	 * Render the audiences panel.
	 *
	 * Label, select, and button are static. JS only injects <option> elements.
	 *
	 * @param string $apivalid API validation status ('0' or '1').
	 * @param array  $listdata Mailchimp lists data array.
	 * @param array  $cf7_mch  Form settings array.
	 * @return void
	 */
	public static function render( string $apivalid, ?array $listdata, array $cf7_mch ): void {
		$vlist = isset( $cf7_mch['list'] ) ? sanitize_text_field( $cf7_mch['list'] ) : '';
		$count = isset( $listdata['lists'] ) && is_array( $listdata['lists'] ) ? count( $listdata['lists'] ) : 0;

		// Help URL with UTM.
		$help_url = self::HELP_URL;
		if ( function_exists( 'vc_utm' ) ) {
			$help_url .= vc_utm() . 'MC-list-id';
		}

		// Progressive disclosure class.
		$disclosure_class = ( '1' === $apivalid ) ? 'chmp-active' : 'chmp-inactive';

		?>
		<div class="cmatic-audiences cmatic-field-group <?php echo esc_attr( $disclosure_class ); ?>">
			<label for="wpcf7-mailchimp-list" id="cmatic-audiences-label">
				<?php
				if ( '1' === $apivalid && $count > 0 ) {
					printf(
						/* translators: %d: Number of Mailchimp audiences */
						esc_html__( 'Total Mailchimp Audiences: %d', 'chimpmatic-lite' ),
						$count
					);
				} else {
					esc_html_e( 'Mailchimp Audiences', 'chimpmatic-lite' );
				}
				?>
			</label><br />

			<select id="wpcf7-mailchimp-list" name="wpcf7-mailchimp[list]">
				<?php self::render_options( $listdata, $vlist ); ?>
			</select>

			<button type="button" id="mce_fetch_fields" class="button">
				<?php esc_html_e( 'Sync Fields', 'chimpmatic-lite' ); ?>
			</button>

			<small class="description">
				<?php esc_html_e( 'Hit the Connect button to load your lists', 'chimpmatic-lite' ); ?>
				<a href="<?php echo esc_url( $help_url ); ?>" class="helping-field" target="_blank" rel="noopener noreferrer" title="<?php esc_attr_e( 'Get help with MailChimp List ID', 'chimpmatic-lite' ); ?>">
					<?php esc_html_e( 'Learn More', 'chimpmatic-lite' ); ?>
				</a>
			</small>
		</div>
		<?php
	}

	/**
	 * Render only the <option> elements (for AJAX injection into select).
	 *
	 * @param array  $listdata    Mailchimp lists data.
	 * @param string $selected_id Currently selected list ID.
	 * @return void
	 */
	public static function render_options( ?array $listdata, string $selected_id = '' ): void {
		if ( ! isset( $listdata['lists'] ) || ! is_array( $listdata['lists'] ) || empty( $listdata['lists'] ) ) {
			return;
		}

		$i = 0;
		foreach ( $listdata['lists'] as $list ) :
			if ( ! is_array( $list ) || ! isset( $list['id'], $list['name'] ) ) {
				continue;
			}
			$i++;
			$list_id      = sanitize_text_field( $list['id'] );
			$list_name    = sanitize_text_field( $list['name'] );
			$member_count = isset( $list['stats']['member_count'] ) ? (int) $list['stats']['member_count'] : 0;
			$field_count  = isset( $list['stats']['merge_field_count'] ) ? (int) $list['stats']['merge_field_count'] : 0;
			$selected     = selected( $selected_id, $list_id, false );
			?>
			<option value="<?php echo esc_attr( $list_id ); ?>" <?php echo $selected; ?>>
				<?php
				printf(
					'%d:%d %s  %d fields #%s',
					$i,
					$member_count,
					$list_name,
					$field_count,
					$list_id
				);
				?>
			</option>
		<?php endforeach;
	}

	/**
	 * Static factory for quick output.
	 *
	 * @param string $apivalid API validation status.
	 * @param array  $listdata Mailchimp lists data.
	 * @param array  $cf7_mch  Form settings.
	 * @return void
	 */
	public static function output( string $apivalid, ?array $listdata, array $cf7_mch ): void {
		self::render( $apivalid, $listdata, $cf7_mch );
	}
}
