<?php
/**
 * Chimpmatic API Panel Component
 *
 * Shared API key input panel with masking and eye toggle for Lite and Pro.
 * Designed for Lite but extensible for Pro inheritance.
 *
 * @package    ChimpmaticLite
 * @subpackage Admin
 * @since      0.9.37
 * @author     Renzo Johnson <renzo.johnson@gmail.com>
 * @copyright  2013-2025 Chimpmatic LLC
 * @license    GPL-3.0-or-later
 */

// Exit if accessed directly.
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Class Cmatic_Api_Panel
 *
 * Renders the API key input panel with masking, eye toggle, and connect button.
 * Used by both Lite and Pro for consistent UI.
 *
 * @since 0.9.37
 */
class Cmatic_Api_Panel {

	/**
	 * Help URL for API key instructions.
	 *
	 * @var string
	 */
	const HELP_URL = '//chimpmatic.com/help/how-to-get-your-mailchimp-api-key';

	/**
	 * Mask an API key for display.
	 *
	 * @param string $key Raw API key.
	 * @return string Masked key (first 8 + bullets + last 4).
	 */
	public static function mask_key( string $key ): string {
		if ( empty( $key ) || strlen( $key ) < 12 ) {
			return $key;
		}
		$prefix = substr( $key, 0, 8 );
		$suffix = substr( $key, -4 );
		return $prefix . str_repeat( "\u{2022}", 20 ) . $suffix;
	}

	/**
	 * Render the API panel.
	 *
	 * @param array  $cf7_mch   Form settings array containing 'api' key.
	 * @param string $apivalid  API validation status ('1' = valid).
	 * @return void
	 */
	public static function render( array $cf7_mch, string $apivalid = '0' ): void {
		$api_key    = isset( $cf7_mch['api'] ) ? $cf7_mch['api'] : '';
		$masked_key = self::mask_key( $api_key );
		$is_masked  = ! empty( $api_key ) && strlen( $api_key ) >= 12;
		$is_valid   = '1' === $apivalid;

		// Button state based on API validation.
		$btn_value = $is_valid ? 'Connected' : 'Sync Audiences';
		$btn_class = 'button';

		// Help URL with UTM.
		$help_url = self::HELP_URL;
		if ( function_exists( 'vc_utm' ) ) {
			$help_url .= vc_utm() . 'MC-api';
		}

		?>
		<div class="cmatic-field-group">
		<label for="cmatic-api"><?php echo esc_html__( 'MailChimp API Key:', 'chimpmatic-lite' ); ?></label><br />
		<div class="cmatic-api-wrap">
			<input
				type="text"
				id="cmatic-api"
				name="wpcf7-mailchimp[api]"
				class="wide"
				placeholder="<?php echo esc_attr__( 'Enter Your Mailchimp API key Here', 'chimpmatic-lite' ); ?>"
				value="<?php echo esc_attr( $is_masked ? $masked_key : $api_key ); ?>"
				data-real-key="<?php echo esc_attr( $api_key ); ?>"
				data-masked-key="<?php echo esc_attr( $masked_key ); ?>"
				data-is-masked="<?php echo $is_masked ? '1' : '0'; ?>"
			/>
			<button type="button" class="cmatic-eye" title="<?php echo esc_attr__( 'Show/Hide', 'chimpmatic-lite' ); ?>">
				<span class="dashicons <?php echo $is_masked ? 'dashicons-visibility' : 'dashicons-hidden'; ?>"></span>
			</button>
		</div>

		<input
			id="chm_activalist"
			type="button"
			value="<?php echo esc_attr( $btn_value ); ?>"
			class="<?php echo esc_attr( $btn_class ); ?>"
		/>

		<small class="description need-api">
			<a href="<?php echo esc_url( $help_url ); ?>" class="helping-field" target="_blank" rel="noopener noreferrer" title="<?php echo esc_attr__( 'Get help with MailChimp API Key', 'chimpmatic-lite' ); ?>">
				<?php echo esc_html__( 'Find your Mailchimp API here', 'chimpmatic-lite' ); ?>
				<span class="red-icon dashicons dashicons-arrow-right"></span>
				<span class="red-icon dashicons dashicons-arrow-right"></span>
			</a>
		</small>
		</div><!-- .cmatic-field-group -->
		<?php
	}

	/**
	 * Static factory for quick output.
	 *
	 * @param array  $cf7_mch  Form settings.
	 * @param string $apivalid API validation status.
	 * @return void
	 */
	public static function output( array $cf7_mch, string $apivalid = '0' ): void {
		self::render( $cf7_mch, $apivalid );
	}
}
