<?php
/*
 * This file is part of Contact Form 7 Extension For Mailchimp.
 *
 * Copyright (C) 2010-2025, Renzo Johnson (email: renzo.johnson at gmail.com)
 *
 * For the full copyright and license information, please view the LICENSE
 * file that was distributed with this source code, or visit:
 * https://www.gnu.org/licenses/gpl-2.0.html
 */

defined( 'ABSPATH' ) || exit;


function mce_error() {

	if ( ! file_exists( WP_PLUGIN_DIR . '/contact-form-7/wp-contact-form-7.php' ) ) {

		wp_cache_flush();

	} elseif ( ! class_exists( 'WPCF7' ) ) {

		plugin_activation( 'contact-form-7/wp-contact-form-7.php' );

	}
}
add_action( 'admin_notices', 'mce_error' );



function mce_save_date_activation() {
	// Run migration to modern 'cmatic' option structure
	mce_migrate_to_cmatic_option();
}

/**
 * Migrate old mce_* options to modern 'cmatic' structure
 *
 * Consolidates fragmented options into single organized array.
 * Based on WooCommerce/CF7 patterns for future-proof data organization.
 *
 * @since 0.9.28
 */
function mce_migrate_to_cmatic_option() {

	// Check if already migrated (cmatic option exists with version)
	$cmatic = get_option( 'cmatic' );
	if ( $cmatic && isset( $cmatic['version'] ) ) {
		return; // Already migrated
	}

	// Gather old options
	$old_loyalty          = get_option( 'mce_loyalty' );
	$old_install_id       = get_option( 'mce_install_id' );
	$old_sent             = get_option( 'mce_sent', 0 );
	$old_news             = get_option( 'mce_show_update_news', 1 );
	$old_notice           = get_option( 'mce_show_notice', 1 );
	$old_panel_content    = get_option( 'mce_conten_panel_master' );
	$old_panel_title      = get_option( 'mce_conten_tittle_master' );

	// Extract quest timestamp from old loyalty data
	$quest_timestamp = time(); // Default to now
	if ( is_array( $old_loyalty ) && isset( $old_loyalty[0] ) ) {
		$quest_timestamp = $old_loyalty[0]; // Unix timestamp from getdate() array
	}

	// Generate install ID if doesn't exist
	if ( ! $old_install_id ) {
		$old_install_id = substr( md5( home_url() . time() . wp_generate_password( 20, false ) ), 0, 12 );
	}

	// Build modern structure
	$cmatic_data = array(
		'version'  => SPARTAN_MCE_VERSION,
		'install'  => array(
			'id'           => $old_install_id,
			'quest'        => $quest_timestamp,
			'activated_at' => gmdate( 'Y-m-d H:i:s', $quest_timestamp ),
		),
		'stats'    => array(
			'sent' => (int) $old_sent,
		),
		'ui'       => array(
			'news'          => (bool) $old_news,
			'notice_banner' => (bool) $old_notice,
			'welcome_panel' => array(
				'title'   => $old_panel_title ? $old_panel_title : 'Chimpmatic Lite is now ' . SPARTAN_MCE_VERSION . '!',
				'content' => $old_panel_content ? $old_panel_content : '',
			),
		),
		'migrated' => true,
	);

	// Save new consolidated option (autoload: no)
	// Delete existing partial option if it exists, then add the full structure
	delete_option( 'cmatic' );
	add_option( 'cmatic', $cmatic_data, '', 'no' );

	// Delete old fragmented options (cleanup)
	delete_option( 'mce_loyalty' );
	delete_option( 'mce_install_id' );
	delete_option( 'mce_sent' );
	delete_option( 'mce_show_update_news' );
	delete_option( 'mce_show_notice' );
	delete_option( 'mce_conten_panel_master' );
	delete_option( 'mce_conten_tittle_master' );
}

/**
 * Get value from cmatic option
 *
 * Helper function to retrieve values from nested cmatic structure.
 *
 * @param string $key Dot-notation key (e.g., 'install.id', 'stats.sent')
 * @param mixed  $default Default value if not found
 * @return mixed
 */
function mce_get_cmatic( $key, $default = null ) {
	$cmatic = get_option( 'cmatic' );

	if ( ! $cmatic ) {
		return $default;
	}

	// Support dot notation (install.id, stats.sent)
	$keys  = explode( '.', $key );
	$value = $cmatic;

	foreach ( $keys as $k ) {
		if ( ! isset( $value[ $k ] ) ) {
			return $default;
		}
		$value = $value[ $k ];
	}

	return $value;
}

/**
 * Update value in cmatic option
 *
 * Helper function to update values in nested cmatic structure.
 *
 * @param string $key Dot-notation key (e.g., 'stats.sent', 'ui.news')
 * @param mixed  $value Value to set
 * @return bool
 */
function mce_update_cmatic( $key, $value ) {
	$cmatic = get_option( 'cmatic', array() );

	// Support dot notation (stats.sent, ui.news)
	$keys = explode( '.', $key );
	$ref  = &$cmatic;

	foreach ( $keys as $i => $k ) {
		if ( $i === count( $keys ) - 1 ) {
			$ref[ $k ] = $value;
		} else {
			if ( ! isset( $ref[ $k ] ) || ! is_array( $ref[ $k ] ) ) {
				$ref[ $k ] = array();
			}
			$ref = &$ref[ $k ];
		}
	}

	return update_option( 'cmatic', $cmatic );
}



function mce_difer_dateact_date() {
	mce_save_date_activation();

	$quest = mce_get_cmatic( 'install.quest', time() );

	$datetime_ini = new DateTime( 'now' );
	$datetime_fin = new DateTime( '@' . $quest ); // Unix timestamp

	$fechaF = date_diff( $datetime_ini, $datetime_fin );

	if ( $fechaF->y > 0 ) {

		if ( $fechaF->m > 0 ) {

			$differenceFormat = '%y Years %m Months %d Days ';

		} else {

			$differenceFormat = '%y Years %d Days ';

		}
	} else {

		if ( $fechaF->m > 0 ) {

			$differenceFormat = '%m Months %d Days ';

		} else {

			$differenceFormat = '%d Days ';

		}
	}

	$resultf = $fechaF->format( $differenceFormat );

	return $resultf;
}



function mce_diferdays_dateact_date() {
	mce_save_date_activation();

	$quest = mce_get_cmatic( 'install.quest', time() );

	$datetime_ini = new DateTime( 'now' );
	$datetime_fin = new DateTime( '@' . $quest ); // Unix timestamp

	$fechaF = date_diff( $datetime_ini, $datetime_fin );

	$resultf = $fechaF->format( '%a' );
	return $resultf;
}



if ( mce_get_cmatic( 'ui.notice_banner', false ) == 1 ) {

	function mce_show_update_notice() {

		if ( ! current_user_can( 'manage_options' ) ) {
			return;
		}

		// Only show on CF7 pages - check both screen and GET parameter
		$screen      = get_current_screen();
		$is_cf7_page = false;

		// Check if screen object exists and contains 'wpcf7'
		if ( $screen && false !== strpos( $screen->id, 'wpcf7' ) ) {
			$is_cf7_page = true;
		}

		// Also check URL parameter as fallback
		if ( ! $is_cf7_page && isset( $_GET['page'] ) && false !== strpos( $_GET['page'], 'wpcf7' ) ) {
			$is_cf7_page = true;
		}

		// DEBUG: Log to console what page we're on (remove after debugging)
		if ( defined( 'WP_DEBUG' ) && WP_DEBUG ) {
			$screen_id  = $screen ? esc_html( $screen->id ) : 'no-screen';
			$page_param = isset( $_GET['page'] ) ? esc_html( sanitize_text_field( wp_unslash( $_GET['page'] ) ) ) : 'no-page-param';
			echo '<!-- ChimpMatic Debug: Screen ID = ' . $screen_id . ', Page Param = ' . $page_param . ', Is CF7 Page = ' . ( $is_cf7_page ? 'YES' : 'NO' ) . ' -->';
		}

		// Return early if not on CF7 page
		if ( ! $is_cf7_page ) {
			return;
		}

		$class = 'notice is-dismissible vc-notice welcome-panel';

		$message  = '<h2>' . esc_html__( 'MailChimp Extension has been improved!', 'mail-chimp-extension' ) . '</h2>';
		$message .= '<p class="about-description">' . esc_html__( 'We worked hard to make it more reliable, faster, and now with a better Debugger, and more help documents.', 'mail-chimp-extension' ) . '</p>';

		$message .= sprintf( __( '<div class="welcome-panel-column-container"><div class="welcome-panel-column"><h3>Get Started</h3><p>Make sure it works as you expect <br><a class="button button-primary button-hero load-customize" href="%s">Review your settings <span alt="f111" class="dashicons dashicons-admin-generic" style="font-size: 17px;vertical-align: middle;"> </span> </a>', 'mail-chimp-extension' ), MCE_SETT ) . '</p></div>';

		$message .= '<div class="welcome-panel-column"><h3>Next Steps</h3><p>' . __( 'Help me develop the plugin and provide support by <br><a class="donate button button-primary button-hero load-customize" href="' . MCE_DON . '" target="_blank">Donating even a small sum <span alt="f524" class="dashicons dashicons-tickets-alt"> </span></a>', 'mail-chimp-extension' ) . '</p></div></div>';

		global $wp_version;

		if ( version_compare( $wp_version, '4.2' ) < 0 ) {

			$message .= ' | <a id="mce-dismiss-notice" href="javascript:mce_dismiss_notice();">' . __( 'Dismiss this notice.' ) . '</a>';

		}
		echo '<div id="mce-notice" class="' . $class . '"><div class="welcome-panel-content">' . $message . '</div></div>';
	}

	if ( is_multisite() ) {

		add_action( 'network_admin_notices', 'mce_show_update_notice' );

	} else {

		add_action( 'admin_notices', 'mce_show_update_notice' );

	}
}



function mce_help() {

	if ( mce_get_cmatic( 'ui.notice_banner' ) === null ) {

		mce_update_cmatic( 'ui.notice_banner', true );

	}
}



function mce_news_notices() {

	// Only show on CF7 pages - same check as mce_show_update_notice
	$screen      = get_current_screen();
	$is_cf7_page = false;

	// Check if screen object exists and contains 'wpcf7'
	if ( $screen && false !== strpos( $screen->id, 'wpcf7' ) ) {
		$is_cf7_page = true;
	}

	// Also check URL parameter as fallback
	if ( ! $is_cf7_page && isset( $_GET['page'] ) && false !== strpos( $_GET['page'], 'wpcf7' ) ) {
		$is_cf7_page = true;
	}

	// Return early if not on CF7 page
	if ( ! $is_cf7_page ) {
		return;
	}

	$class  = 'notice is-dismissible vc-notice welcome-panel';
	$check  = 0;
	$tittle = '';
	//$message = mce_get_postnotice ($check,$tittle ) ;

	$Defaulttittle = 'Chimpmatic Lite is now ' . SPARTAN_MCE_VERSION . '!';
	$Defaultpanel  = '<p class="about-description">Easier setup to get you up and running in no time. Please <a href="https://chimpmatic.com/contact" target="_blank" rel="noopener noreferrer">lets us know</a> what kind of features you would like to see added <a href="https://chimpmatic.com/contact" target="_blank" rel="noopener noreferrer">HERE</a></p>
<div class="welcome-panel-column-container">
<div class="welcome-panel-columns">
<h3>Get Started</h3>
<p>Make sure it works as you expect <br><a class="button button-primary button-hero load-customize" href="' . get_admin_url() . 'admin.php?page=wpcf7&post=' . cmatic_newest() . '&active-tab=chimpmatic-lite-panel">Review your settings <span alt="f111" class="dashicons dashicons-admin-generic" style="font-size: 17px;vertical-align: middle;"> </span> </a></p>
</div>

</div>';

	// Get welcome panel from cmatic structure
	$panel_content = mce_get_cmatic( 'ui.welcome_panel.content' );
	$panel_title   = mce_get_cmatic( 'ui.welcome_panel.title' );

	// Initialize if not set
	if ( null === $panel_content ) {
		mce_update_cmatic( 'ui.welcome_panel.content', $Defaultpanel );
		mce_update_cmatic( 'ui.welcome_panel.title', $Defaulttittle );
		$banner = $Defaultpanel;
		$tittle = $Defaulttittle;
	} else {
		$grabbanner = trim( $panel_content );
		$grabtittle = trim( $panel_title );

		$banner = ( '' === $grabbanner ) ? $Defaultpanel : $grabbanner;
		$tittle = ( '' === $grabtittle ) ? $Defaulttittle : $grabtittle;
	}

	$tittle2  = '<h2>' . $tittle . '</h2>';
	$message2 = $tittle2 . $banner;

	echo '<div id="mce-notice" class="' . $class . '"><div class="welcome-panel-content">' . $message2 . '</div></div>';

	echo "<script>
        function mce_dismiss_update_news(){
          var data = {
          'action': 'mce_dismiss_update_news',
          };

          jQuery.post(ajaxurl, data, function(response) {
            jQuery('#mce-notice').hide();
          });
        }

        jQuery(document).ready(function(){
          jQuery('body').on('click', '.notice-dismiss', function(){

            mce_dismiss_update_news();
          });
        });
        </script>";
}



function mce_dismiss_update_news() {

	mce_update_cmatic( 'ui.news', false );

	wp_die();
}
add_action( 'wp_ajax_mce_dismiss_update_news', 'mce_dismiss_update_news' );



// Only initialize ui.news if not migrated yet (prevents partial cmatic creation before migration)
if ( !  mce_get_cmatic( 'migrated' ) && ( mce_get_cmatic( 'ui.news' ) === null ) ) {
	mce_update_cmatic( 'ui.news', true );
}

if ( ( mce_get_cmatic( 'ui.news' ) === null ) or mce_get_cmatic( 'ui.news' ) == 1 ) {

	if ( is_multisite() ) {

			add_action( 'network_admin_notices', 'mce_news_notices' );

	} else {

		add_action( 'admin_notices', 'mce_news_notices' );

	}
}



if ( ! function_exists( 'mc_meta_links' ) ) {

	function mc_meta_links( $links, $file ) {

		if ( $file === 'contact-form-7-mailchimp-extension/chimpmatic-lite.php' ) {

			$links[] = '<a href="' . MCE_URL . '" target="_blank" title="Chimpmatic Lite Documentation">Chimpmatic Documentation</a>';
		}
		return $links;
	}

}
add_filter( 'plugin_row_meta', 'mc_meta_links', 10, 2 );



/**
 * Register plugin action links using shared Cmatic_Plugin_Links class.
 *
 * @since 0.9.37
 */
if ( ! function_exists( 'mc_after_setup_theme' ) ) {

	function mc_after_setup_theme() {
		// Use shared class if available (loaded via bootstrap.php).
		if ( class_exists( 'Cmatic_Plugin_Links' ) ) {
			add_filter(
				'plugin_action_links_' . SPARTAN_MCE_PLUGIN_BASENAME,
				array( 'Cmatic_Plugin_Links', 'filter_action_links' )
			);
		}
	}
	add_action( 'after_setup_theme', 'mc_after_setup_theme' );

}

/**
 * Detect existing Mailchimp API keys from other plugins
 *
 * Checks popular Mailchimp plugins for API keys to auto-populate.
 * Priority: MC4WP > MailChimp List Subscribe Form > Easy Forms for Mailchimp
 *
 * @since 0.9.29
 * @return string|false API key if found, false otherwise
 */
if ( ! function_exists( 'cmatic_detect_existing_api_key' ) ) {
	function cmatic_detect_existing_api_key() {
		// 1. MC4WP (Mailchimp for WordPress) - most popular
		$mc4wp = get_option( 'mc4wp' );
		if ( ! empty( $mc4wp['api_key'] ) ) {
			return $mc4wp['api_key'];
		}

		// 2. MailChimp List Subscribe Form (yikes-inc-easy-mailchimp-extender)
		$yikes = get_option( 'yikes-mc-api-key' );
		if ( ! empty( $yikes ) ) {
			return $yikes;
		}

		// 3. Easy Forms for Mailchimp
		$easy_forms = get_option( 'yikes-easy-mailchimp-extender-api-key' );
		if ( ! empty( $easy_forms ) ) {
			return $easy_forms;
		}

		// 4. MailChimp for WooCommerce
		$woo_mc = get_option( 'mailchimp-woocommerce' );
		if ( ! empty( $woo_mc['mailchimp_api_key'] ) ) {
			return $woo_mc['mailchimp_api_key'];
		}

		// 5. Mailchimp Top Bar (by MC4WP)
		$mc4wp_top_bar = get_option( 'mc4wp_top_bar' );
		if ( ! empty( $mc4wp_top_bar['api_key'] ) ) {
			return $mc4wp_top_bar['api_key'];
		}

		return false;
	}
}

/**
 * Import API key from other plugins and save to a form
 *
 * @since 0.9.29
 * @return bool True if key was imported, false otherwise
 */
if ( ! function_exists( 'cmatic_import_api_key_on_activation' ) ) {
	function cmatic_import_api_key_on_activation() {
		$existing_key = cmatic_detect_existing_api_key();

		if ( ! $existing_key ) {
			return false;
		}

		// Get the newest CF7 form to save the API key to
		$form_id = cmatic_newest();
		if ( ! $form_id ) {
			return false;
		}

		// Get existing form settings
		$option_name = 'cf7_mch_' . $form_id;
		$cf7_mch     = get_option( $option_name, array() );

		// Only import if no API key is set for this form
		if ( ! empty( $cf7_mch['api'] ) ) {
			return false;
		}

		// Save the detected API key silently - automagic!
		$cf7_mch['api'] = $existing_key;
		update_option( $option_name, $cf7_mch );

		return true;
	}
}

/**
 * Plugin activation handler
 *
 * @return void
 */
if ( ! function_exists( 'cmatic_on_activation' ) ) {
	function cmatic_on_activation() {
		// Save activation date
		mce_save_date_activation();

		// TODO: Auto-import API key from other Mailchimp plugins (MC4WP, etc.)
		// Commented out - need to decide: save to which form? Global option? On-demand per form?
		// cmatic_import_api_key_on_activation();

		// Trigger telemetry activation event
		do_action( 'csyncr_metrics_on_activation' );

		// Set redirect flag if conditions are met
		if ( cmatic_can_redirect_on_activation() ) {
			add_option( 'cmatic_do_activation_redirect', true );
		}
	}
}

/**
 * Plugin deactivation handler
 *
 * @return void
 */
if ( ! function_exists( 'cmatic_on_deactivation' ) ) {
	function cmatic_on_deactivation() {
		// Trigger telemetry deactivation event
		do_action( 'csyncr_metrics_on_deactivation' );
	}
}

/**
 * Check if plugin should redirect on activation
 *
 * @return bool
 */
if ( ! function_exists( 'cmatic_can_redirect_on_activation' ) ) {
	function cmatic_can_redirect_on_activation() {
		// Don't redirect if bulk activation
		if ( isset( $_GET['activate-multi'] ) ) {
			return false;
		}

		// Don't redirect if network activation
		if ( is_network_admin() ) {
			return false;
		}

		// Don't redirect if AJAX request
		if ( defined( 'DOING_AJAX' ) && DOING_AJAX ) {
			return false;
		}

		// Don't redirect if WP-CLI
		if ( defined( 'WP_CLI' ) && WP_CLI ) {
			return false;
		}

		// Optional: Don't redirect in debug mode (helps developers)
		// TODO: Uncomment this later for production to improve developer experience
		// if ( defined( 'WP_DEBUG' ) && WP_DEBUG ) {
		// 	return false;
		// }

		return true;
	}
}

/**
 * Redirect to plugin settings on activation
 *
 * @return void
 */
if ( ! function_exists( 'cmatic_activation_redirect' ) ) {
	function cmatic_activation_redirect() {
		// Check if redirect flag is set
		if ( get_option( 'cmatic_do_activation_redirect', false ) ) {
			// Delete flag immediately to prevent redirect loops
			delete_option( 'cmatic_do_activation_redirect' );

			// Don't redirect if bulk activation
			if ( isset( $_GET['activate-multi'] ) ) {
				return;
			}

			// Build redirect URL (same as Settings link)
			$latest_form_id = cmatic_newest();
			$redirect_url   = admin_url( 'admin.php?page=wpcf7&post=' . $latest_form_id . '&active-tab=chimpmatic-lite-panel' );

			// Perform safe redirect
			wp_safe_redirect( $redirect_url );
			exit;
		}
	}

	// Hook the redirect handler
	add_action( 'admin_init', 'cmatic_activation_redirect' );
}
