// Chimpmatic - Modern Vanilla JavaScript (no jQuery)

/**
 * Get form ID from data container (modern pattern).
 * Replaces legacy #chm_idformxx hidden input.
 * Uses same Cmatic_Data_Container rendered by LITE.
 *
 * @returns {number} Form ID or 0 if not found.
 */
function getFormId() {
    const dataContainer = document.getElementById('cmatic_data');
    if (dataContainer && dataContainer.dataset.formId) {
        return parseInt(dataContainer.dataset.formId, 10) || 0;
    }
    return 0;
}

/**
 * Get API validation status from data container.
 * Replaces legacy #chm_txcomodin2 hidden input.
 *
 * @returns {string} '0' or '1'.
 */
function getApiValid() {
    const dataContainer = document.getElementById('cmatic_data');
    if (dataContainer && dataContainer.dataset.apiValid) {
        return dataContainer.dataset.apiValid;
    }
    return '0';
}

/**
 * Get API key from input, handling masked display.
 * Uses real key from data-real-key attribute if masked, otherwise input value.
 *
 * @returns {string} The actual API key.
 */
function getApiKey() {
    const apiInput = document.getElementById('cmatic-api');
    if (!apiInput) return '';
    const isMasked = apiInput.dataset.isMasked === '1';
    return (isMasked && apiInput.dataset.realKey)
        ? apiInput.dataset.realKey
        : apiInput.value;
}

/**
 * Update API validation status in data container.
 *
 * @param {string|boolean} value New validation status.
 */
function setApiValid(value) {
    const dataContainer = document.getElementById('cmatic_data');
    if (dataContainer) {
        dataContainer.dataset.apiValid = value ? '1' : '0';
    }
}

// Helper function to get REST API URL
function getRestUrl() {
    // WordPress provides wpApiSettings in admin
    if (typeof wpApiSettings !== 'undefined' && wpApiSettings.root) {
        return wpApiSettings.root + 'chimpmatic/v1/';
    }
    // Fallback to standard WordPress REST API path
    return '/wp-json/chimpmatic/v1/';
}

// Helper function to get AJAX URL (fallback)
function getAjaxUrl() {
    return (typeof ajaxurl !== 'undefined') ? ajaxurl : (typeof chmAjax !== 'undefined' ? chmAjax.ajaxurl : '/wp-admin/admin-ajax.php');
}

// Map AJAX actions to REST endpoints
const actionToEndpoint = {
    'wpcf7_chm_savetool': 'settings/save',
    'wpcf7_chm_savetool_cfg': 'settings/config',
    'wpcf7_chm_loadlistas': 'mailchimp/audiences',
    'wpcf7_chm_loadcampos': 'mailchimp/fields',
    'wpcf7_chm_loadgrupos': 'mailchimp/groups',
    'wpcf7_chm_get_interest': 'mailchimp/interests',
    'wpcf7_chm_exporuser': 'export/users',
    'wpcf7_chm_read_log': 'logs/read'
};

// Helper function to make API requests (REST API with AJAX fallback)
async function chmAjax(action, data = {}) {
    const endpoint = actionToEndpoint[action];

    // Try REST API first if endpoint mapping exists
    if (endpoint) {
        try {
            return await chmRestRequest(endpoint, data);
        } catch (error) {
            console.warn('[Chimpmatic] REST API failed, falling back to admin-ajax:', error.message);
            // Fall through to admin-ajax fallback
        }
    }

    // Fallback to traditional admin-ajax
    return await chmAjaxRequest(action, data);
}

// REST API request
async function chmRestRequest(endpoint, data = {}) {
    const url = getRestUrl() + endpoint;
    console.log('[Chimpmatic REST] 🌐 URL:', url);
    console.log('[Chimpmatic REST] 📤 Endpoint:', endpoint);

    // Log data (mask sensitive info)
    const logData = {...data};
    if (logData.chimpapi) {
        logData.chimpapi = logData.chimpapi.substring(0, 10) + '...';
    }
    console.log('[Chimpmatic REST] 📦 Data:', logData);

    try {
        const response = await fetch(url, {
            method: 'POST',
            headers: {
                'Content-Type': 'application/json',
                'X-WP-Nonce': wpApiSettings?.nonce || ''
            },
            body: JSON.stringify(data),
            credentials: 'same-origin'
        });

        console.log('[Chimpmatic REST] 📥 Response status:', response.status, response.statusText);

        if (!response.ok) {
            throw new Error(`HTTP error! status: ${response.status}`);
        }

        const result = await response.json();
        console.log('[Chimpmatic REST] ✅ Response received');

        // Extract HTML or data from REST response
        return result.html || result.data || '';
    } catch (error) {
        console.error('[Chimpmatic REST] ❌ Error:', error);
        throw error;
    }
}

// Traditional admin-ajax request (fallback)
async function chmAjaxRequest(action, data = {}) {
    const url = getAjaxUrl();
    console.log('[Chimpmatic AJAX] 🌐 URL:', url);
    console.log('[Chimpmatic AJAX] 🎬 Action:', action);

    const formData = new FormData();

    formData.append('action', action);
    for (const [key, value] of Object.entries(data)) {
        if (Array.isArray(value)) {
            value.forEach((item, index) => {
                if (Array.isArray(item)) {
                    item.forEach((subItem, subIndex) => {
                        formData.append(`${key}[${index}][${subIndex}]`, subItem);
                    });
                } else {
                    formData.append(`${key}[${index}]`, item);
                }
            });
        } else {
            formData.append(key, value);
        }
    }

    // Log formData contents
    console.log('[Chimpmatic AJAX] 📤 Sending FormData:');
    for (let [key, value] of formData.entries()) {
        if (key === 'chimpapi') {
            console.log(`  ${key}: ${value.substring(0, 10)}...`);
        } else {
            console.log(`  ${key}: ${value}`);
        }
    }

    try {
        console.log('[Chimpmatic AJAX] ⏳ Fetching...');
        const response = await fetch(url, {
            method: 'POST',
            body: formData,
            credentials: 'same-origin'
        });

        console.log('[Chimpmatic AJAX] 📥 Response status:', response.status, response.statusText);

        if (!response.ok) {
            throw new Error(`HTTP error! status: ${response.status}`);
        }

        const responseText = await response.text();
        console.log('[Chimpmatic AJAX] ✅ Response received, length:', responseText.length);

        return responseText;
    } catch (error) {
        console.error('[Chimpmatic AJAX] ❌ Error:', error);
        throw error;
    }
}

// Spinner helper removed - using button text states instead (ClickSyncr pattern)

// Save Tool Configuration
document.addEventListener('click', function(event) {
    if (event.target && event.target.id === 'chm_submitme') {
        event.preventDefault();

        const data = {
            tool_key: document.getElementById('wpcf7-mailchimp-tool_key')?.value || '',
            chm_idformxx: getFormId(),
            uemail: document.getElementById('wpcf7-mailchimp-uemail')?.value || '',
            prod_id: document.getElementById('wpcf7-mailchimp-prod-id')?.value || ''
        };

        chmAjax('wpcf7_chm_savetool', data)
            .then(response => {
                const panel = document.getElementById('chm_panel_principal');
                if (panel) panel.innerHTML = response;
            })
            .catch(error => {
                alert('Error: ' + error.message);
            });
    }
});

// Save Tool Config
document.addEventListener('click', function(event) {
    if (event.target && event.target.id === 'chm_submitme_cfg') {
        event.preventDefault();

        const data = {
            tool_key: document.getElementById('wpcf7-mailchimp-tool_key')?.value || '',
            uemail: document.getElementById('wpcf7-mailchimp-uemail')?.value || '',
            prod_id: document.getElementById('wpcf7-mailchimp-prod-id')?.value || ''
        };

        chmAjax('wpcf7_chm_savetool_cfg', data)
            .then(response => {
                const liveElements = document.querySelectorAll('.chimp-live');
                liveElements.forEach(el => el.innerHTML = response);
            })
            .catch(error => {
                alert('Error: ' + error.message);
            });
    }
});

// Connect and Fetch Your Audiences (Load Lists)
document.addEventListener('click', function(event) {
    if (event.target && event.target.id === 'chm_activalist') {
        console.log('[Chimpmatic] 🚀 Connect and Fetch button clicked');
        event.preventDefault();

        const button = event.target;

        // Prevent double-clicks
        if (button.disabled) {
            return;
        }

        // Find the API key input in the panel (same container as the button)
        const panel = button.closest('.cmatic-field-group') || button.closest('.mce-custom-fields');
        console.log('[Chimpmatic] 🎯 Panel:', panel);

        let apiKeyElement = null;
        if (panel) {
            apiKeyElement = panel.querySelector('#cmatic-api');
            console.log('[Chimpmatic] 🔍 API Key element (from panel):', apiKeyElement);
        }

        // Fallback to global search
        if (!apiKeyElement) {
            apiKeyElement = document.getElementById('cmatic-api');
            console.log('[Chimpmatic] 🔍 API Key element (global):', apiKeyElement);
        }

        const apiKey = apiKeyElement?.value || '';
        console.log('[Chimpmatic] 🔑 API Key:', apiKey ? apiKey.substring(0, 10) + '...' : 'EMPTY');
        console.log('[Chimpmatic] 🔑 API Key length:', apiKey.length);
        console.log('[Chimpmatic] 🔑 API Key full (for debug):', apiKey);

        if (!apiKey || apiKey.trim() === '') {
            console.error('[Chimpmatic] ❌ No API key entered');
            console.error('[Chimpmatic] ❌ Element value:', apiKeyElement ? `"${apiKeyElement.value}"` : 'Element not found');
            console.error('[Chimpmatic] ❌ Element:', apiKeyElement);
            alert('Please enter a MailChimp API Key first.');
            return;
        }

        const data = {
            tool_key: document.getElementById('wpcf7-mailchimp-tool_key')?.value || '',
            chm_idformxx: getFormId(),
            chimpapi: apiKey
        };

        console.log('[Chimpmatic] 📦 Request data:', {
            action: 'wpcf7_chm_loadlistas',
            tool_key: data.tool_key,
            form_id: data.chm_idformxx,
            api_key_length: apiKey.length
        });

        // Modern button states: Sync Audiences → Syncing... → Synced → Connected
        const originalText = button.value;
        button.value = 'Syncing...';
        button.disabled = true;
        console.log('[Chimpmatic] ⏳ Button state: Syncing...');

        chmAjax('wpcf7_chm_loadlistas', data)
            .then(response => {
                console.log('[Chimpmatic] ✅ AJAX response received');
                console.log('[Chimpmatic] 📄 Response length:', response.length, 'characters');
                console.log('[Chimpmatic] 📄 Response preview:', response.substring(0, 200));

                const listPanel = document.getElementById('chm_panel_listamail');
                console.log('[Chimpmatic] 🎯 List panel element:', listPanel ? 'FOUND' : 'NOT FOUND');

                if (listPanel) {
                    console.log('[Chimpmatic] 📝 Setting panel innerHTML...');
                    listPanel.innerHTML = response;
                    console.log('[Chimpmatic] ✅ Panel innerHTML updated');
                }

                // Check API validation status from data container
                const valor = getApiValid();

                console.log('[Chimpmatic] 🔍 Validation value from data container:', valor);

                let attrclass = '';
                let chm_valid = '';

                if (valor === '1') {
                    console.log('[Chimpmatic] ✅ API validation SUCCESS');
                    attrclass = 'spt-response-out spt-valid';
                    chm_valid = '<h3 class="title">Chimpmatic<span><span class="chmm valid">API Connected</span></span></h3>';

                    // Show "Synced" briefly, then "Connected"
                    button.value = 'Synced';
                    button.disabled = false;
                    console.log('[Chimpmatic] ⏳ Button state: Synced');
                    setTimeout(() => {
                        button.value = 'Connected';
                        console.log('[Chimpmatic] ✅ Button state: Connected (final)');

                        // AUTO-CHAIN: Trigger fields/groups fetch (ClickSyncr pattern)
                        const fieldsBtn = document.getElementById('chm_selgetcampos');
                        if (fieldsBtn && !fieldsBtn.disabled) {
                            console.log('[Chimpmatic] 🔗 Auto-chaining: Triggering Sync Fields and Groups');
                            fieldsBtn.click();
                        }
                    }, 800);
                } else {
                    console.log('[Chimpmatic] ❌ API validation FAILED - valor:', valor);
                    attrclass = 'spt-response-out';
                    chm_valid = '<h3 class="title">Chimpmatic<span><span class="chmm invalid">API Inactive</span></span></h3>';

                    // Reset button for retry
                    button.value = originalText;
                    button.disabled = false;
                    console.log('[Chimpmatic] ⏳ Button state: Reset for retry');

                    const configPanel = document.getElementById('chm_panel_configcampos');
                    if (configPanel) {
                        console.log('[Chimpmatic] 🧹 Clearing config panel');
                        configPanel.innerHTML = '<span> </span>';
                    }
                }

                if (listPanel) {
                    console.log('[Chimpmatic] 🎨 Setting panel class:', attrclass);
                    listPanel.className = attrclass;
                }

                const validPanel = document.getElementById('chm_apivalid');
                console.log('[Chimpmatic] 🎯 Valid panel element:', validPanel ? 'FOUND' : 'NOT FOUND');

                if (validPanel) {
                    console.log('[Chimpmatic] ✅ Updating validation panel');
                    validPanel.innerHTML = chm_valid;
                }

                console.log('[Chimpmatic] 🏁 Process complete!');
            })
            .catch(error => {
                console.error('[Chimpmatic] ❌ AJAX ERROR:', error);
                console.error('[Chimpmatic] 📋 Error details:', {
                    message: error.message,
                    stack: error.stack
                });
                // Reset button on error
                button.value = originalText;
                button.disabled = false;
                alert('Error loading audiences: ' + error.message);
            });
    }
});

// Auto-connect on paste - trigger API connection when user pastes API key
document.addEventListener('paste', function(event) {
    const apiInput = document.getElementById('cmatic-api');

    // Only trigger if paste is into the API key input
    if (event.target !== apiInput) {
        return;
    }

    console.log('[Chimpmatic] 📋 Paste detected in API key field');

    // Small delay to let the paste complete and populate the input value
    setTimeout(function() {
        const apiKey = apiInput.value.trim();

        // Validate it looks like an API key (32 chars + dash + datacenter)
        // Format: xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx-us1
        if (apiKey && apiKey.length >= 30 && apiKey.includes('-')) {
            console.log('[Chimpmatic] 🔑 Valid API key format detected, auto-triggering connect');

            const connectBtn = document.getElementById('chm_activalist');
            if (connectBtn && !connectBtn.disabled) {
                connectBtn.click();
            }
        } else {
            console.log('[Chimpmatic] ⚠️ Pasted value does not look like API key, skipping auto-connect');
        }
    }, 100);
});

// Auto-refresh fields when audience dropdown changes
document.addEventListener('change', function(event) {
    if (event.target && event.target.id === 'wpcf7-mailchimp-list') {
        console.log('[Chimpmatic] 📋 Audience dropdown changed, auto-refreshing fields');
        // Try both button IDs: legacy #chm_selgetcampos and new #mce_fetch_fields
        const fieldsBtn = document.getElementById('chm_selgetcampos') || document.getElementById('mce_fetch_fields');
        if (fieldsBtn && !fieldsBtn.disabled) {
            fieldsBtn.click();
        }
    }
});

// Fetch Your Fields and Groups
// Listen for both legacy #chm_selgetcampos and new #mce_fetch_fields buttons
document.addEventListener('click', function(event) {
    if (event.target && (event.target.id === 'chm_selgetcampos' || event.target.id === 'mce_fetch_fields')) {
        event.preventDefault();

        const button = event.target;

        // Prevent double-clicks
        if (button.disabled) {
            return;
        }

        const data = {
            tool_key: document.getElementById('wpcf7-mailchimp-tool_key')?.value || '',
            chm_idformxx: getFormId(),
            chm_listid: document.getElementById('wpcf7-mailchimp-list')?.value || '',
            chimpapi: getApiKey()
        };

        // Modern button states: Sync Fields and Groups → Syncing... → Synced! → Sync Fields and Groups
        // Handle both <input> (.value) and <button> (.textContent) elements
        const isInputButton = button.tagName === 'INPUT';
        const originalText = isInputButton ? button.value : button.textContent;
        if (isInputButton) {
            button.value = 'Syncing...';
        } else {
            button.textContent = 'Syncing...';
        }
        button.disabled = true;
        console.log('[Chimpmatic] ⏳ Fields button state: Syncing...');

        chmAjax('wpcf7_chm_loadcampos', data)
            .then(response => {

                console.log('[Chimpmatic] Fetch response length:', response.length);
                console.log('[Chimpmatic] Response contains "Groups for":', response.includes('Groups for'));
                console.log('[Chimpmatic] Response contains "Tags for":', response.includes('Tags for'));

                const genPanel = document.getElementById('chm_panel_gencamposygrupos');

                console.log('[Chimpmatic] genPanel found:', !!genPanel);
                console.log('[Chimpmatic] genPanel display before:', genPanel ? window.getComputedStyle(genPanel).display : 'N/A');

                if (genPanel) {
                    genPanel.innerHTML = response;
                    genPanel.className = 'spt-response-out';

                    console.log('[Chimpmatic] genPanel display after:', window.getComputedStyle(genPanel).display);
                    console.log('[Chimpmatic] genPanel children count:', genPanel.children.length);

                    // Check if groups panel exists (unique ID)
                    const groupsPanel = genPanel.querySelector('#cmatic-fields');
                    console.log('[Chimpmatic] Groups panel (#cmatic-fields) found:', !!groupsPanel);
                    if (groupsPanel) {
                        console.log('[Chimpmatic] Groups panel display:', window.getComputedStyle(groupsPanel).display);
                        console.log('[Chimpmatic] Groups panel parent display:', window.getComputedStyle(groupsPanel.parentElement).display);
                        console.log('[Chimpmatic] Groups panel innerHTML length:', groupsPanel.innerHTML.length);
                        console.log('[Chimpmatic] Groups panel HTML preview:', groupsPanel.innerHTML.substring(0, 300));
                    }

                    // Check if tags panel exists
                    const tagsPanel = genPanel.querySelector('#chm_panel_camposformatags');
                    console.log('[Chimpmatic] Tags panel (#chm_panel_camposformatags) found:', !!tagsPanel);
                    if (tagsPanel) {
                        console.log('[Chimpmatic] Tags panel display:', window.getComputedStyle(tagsPanel).display);
                        console.log('[Chimpmatic] Tags panel innerHTML length:', tagsPanel.innerHTML.length);
                        console.log('[Chimpmatic] Tags panel HTML preview:', tagsPanel.innerHTML.substring(0, 500));
                    }

                    const listPanel = document.getElementById('chm_panel_listamail');
                    const attrclass = listPanel ? listPanel.className : '';

                    if (attrclass === 'spt-response-out') {
                        genPanel.className = 'spt-response-out';
                    } else {
                        genPanel.className = 'spt-response-out spt-valid';
                    }

                    // Apply fuzzy matching to auto-map fields after AJAX loads
                    // Use setTimeout to ensure DOM is fully updated after innerHTML injection
                    setTimeout(function() {
                        if (typeof applyFuzzyMatchingPro === 'function') {
                            console.log('[Chimpmatic] Triggering fuzzy matching after DOM update...');
                            applyFuzzyMatchingPro();
                        }
                        // Tag chips use event delegation - no init needed
                    }, 100);
                }

                // Show "Synced!" briefly, then reset to original text
                if (isInputButton) {
                    button.value = 'Synced!';
                } else {
                    button.textContent = 'Synced!';
                }
                console.log('[Chimpmatic] ✅ Fields button state: Synced!');
                setTimeout(() => {
                    if (isInputButton) {
                        button.value = originalText;
                    } else {
                        button.textContent = originalText;
                    }
                    button.disabled = false;
                    console.log('[Chimpmatic] 🔄 Fields button state: Reset to original');
                }, 800);
            })
            .catch(error => {
                // Reset button on error
                if (isInputButton) {
                    button.value = originalText;
                } else {
                    button.textContent = originalText;
                }
                button.disabled = false;
                alert('Error: ' + error.message);
            });
    }
});

// Load Groups
document.addEventListener('click', function(event) {
    if (event.target && event.target.id === 'chm_activagroups') {
        event.preventDefault();

        const data = {
            tool_key: document.getElementById('wpcf7-mailchimp-tool_key')?.value || '',
            chm_idformxx: getFormId(),
            chm_listid: document.getElementById('wpcf7-mailchimp-list')?.value || '',
            chimpapi: getApiKey()
        };

        chmAjax('wpcf7_chm_loadgrupos', data)
            .then(response => {
                const groupPanel = document.getElementById('chm_panel_listgroup');
                if (groupPanel) {
                    groupPanel.innerHTML = response;
                    groupPanel.className = 'spt-response-out';

                    const listPanel = document.getElementById('chm_panel_listamail');
                    const attrclass = listPanel ? listPanel.className : '';

                    if (attrclass === 'spt-response-out') {
                        groupPanel.className = 'spt-response-out';
                    } else {
                        groupPanel.className = 'spt-response-out spt-valid';
                    }
                }
            })
            .catch(error => {
                alert('Error: ' + error.message);
            });
    }
});

// Export Users
document.addEventListener('click', function(event) {
    if (event.target && event.target.id === 'chm_userexport') {
        event.preventDefault();

        const valuesChecked = [];
        let icont = 1;

        document.querySelectorAll("input[type='checkbox'][name='usercheck']:checked").forEach(checkbox => {
            const idListInput = document.getElementById(`wpcf7-mailchimp-idlistexport${icont}`);

            valuesChecked.push([
                checkbox.value,
                idListInput ? idListInput.value : ''
            ]);

            icont++;
        });

        const data = {
            tool_key: document.getElementById('wpcf7-mailchimp-tool_key')?.value || '',
            chm_idformxx: getFormId(),
            chimpapi: getApiKey(),
            cadseluser: valuesChecked
        };

        chmAjax('wpcf7_chm_exporuser', data)
            .then(response => {
                const exportPanel = document.getElementById('chm_panel_exporuser');
                if (exportPanel) {
                    exportPanel.innerHTML = response;
                }
            })
            .catch(error => {
                alert('Error: ' + error.message);
            });
    }
});

// Get Interest (Groups - Arbitrary)
document.addEventListener('change', function(event) {
    if (event.target && event.target.classList.contains('chimp-gg-arbirary')) {
        event.preventDefault();

        const itag = event.target.getAttribute('data-tag');
        const xchk = event.target.checked ? 1 : 0;
        const ggKeyInput = document.getElementById(`wpcf7-mailchimp-ggCustomKey${itag}`);

        console.log('[Chimpmatic Interest] 🔘 Checkbox toggled, tag:', itag, 'checked:', xchk);

        const data = {
            valcheck: xchk,
            chm_idformxx: getFormId(),
            chm_listid: document.getElementById('wpcf7-mailchimp-list')?.value || '',
            chimpapi: getApiKey(),
            indtag: itag,
            ggid: ggKeyInput ? ggKeyInput.value : ''
        };

        console.log('[Chimpmatic Interest] 📦 Request data:', {
            valcheck: data.valcheck,
            form_id: data.chm_idformxx,
            list_id: data.chm_listid,
            tag_index: data.indtag,
            category_id: data.ggid,
            api_key_length: data.chimpapi ? data.chimpapi.length : 0
        });

        console.log('[Chimpmatic Interest] ⏳ Sending AJAX request...');

        chmAjax('wpcf7_chm_get_interest', data)
            .then(response => {
                console.log('[Chimpmatic Interest] ✅ Response received, length:', response.length);
                console.log('[Chimpmatic Interest] 📄 Response preview:', response.substring(0, 200));

                const selectElement = document.getElementById(`gg-select${itag}`);
                console.log('[Chimpmatic Interest] 🎯 Target element #gg-select' + itag + ':', selectElement ? 'FOUND' : 'NOT FOUND');

                if (selectElement) {
                    selectElement.innerHTML = response;
                    console.log('[Chimpmatic Interest] ✅ Dropdown updated successfully');
                }
            })
            .catch(error => {
                console.error('[Chimpmatic Interest] ❌ Error:', error);
                console.error('[Chimpmatic Interest] 📋 Error details:', {
                    message: error.message,
                    stack: error.stack
                });
                alert('Error loading interests: ' + error.message);
            });
    }
});

// Helper function to toggle panels and update button text
function togglePanel(panelSelector, buttonElement, showText, hideText) {
    const panel = typeof panelSelector === 'string' ?
        (panelSelector.startsWith('.') ? document.querySelector(panelSelector) : document.getElementById(panelSelector)) :
        panelSelector;

    if (panel) {
        const isHidden = panel.style.display === 'none' || !panel.style.display || window.getComputedStyle(panel).display === 'none';
        panel.style.display = isHidden ? 'block' : 'none';

        if (buttonElement && showText && hideText) {
            buttonElement.textContent = isHidden ? hideText : showText;
        }
    }
}

// Toggle Advanced Settings
document.addEventListener('click', function(event) {
    if (event.target && event.target.classList.contains('mce-trigger')) {
        event.preventDefault();
        togglePanel('.mce-container.mce-support', event.target, 'Show Advanced Settings', 'Hide Advanced Settings');
    }
});

// Toggle Export WP Users
document.addEventListener('click', function(event) {
    if (event.target && event.target.classList.contains('cme-trigger-exp')) {
        event.preventDefault();
        togglePanel('mce-export', event.target, 'Export WP Users', 'Hide Export');
    }
});

// Toggle System Information
document.addEventListener('click', function(event) {
    if (event.target && event.target.classList.contains('cme-trigger-sys')) {
        event.preventDefault();
        togglePanel('toggle-sys', event.target, 'Get System Information', 'Hide System Info');
    }
});

// Toggle Debug Logger
document.addEventListener('click', function(event) {
    if (event.target && event.target.classList.contains('cme-trigger-log')) {
        event.preventDefault();
        togglePanel('log-dev', event.target, 'View Debug Logger', 'Hide Debug Logger');
    }
});

// On page load, ensure all panels are hidden and buttons show correct text
document.addEventListener('DOMContentLoaded', function() {
    // Force hide all toggle panels
    const panels = [
        document.querySelector('.mce-container.mce-support'),
        document.getElementById('mce-export'),
        document.getElementById('toggle-sys'),
        document.getElementById('log-dev')
    ];

    panels.forEach(panel => {
        if (panel) {
            panel.style.display = 'none';
        }
    });

    // Set correct button text
    const advBtn = document.querySelector('.mce-trigger');
    if (advBtn) advBtn.textContent = 'Show Advanced Settings';

    const expBtn = document.querySelector('.cme-trigger-exp');
    if (expBtn) expBtn.textContent = 'Export WP Users';

    const sysBtn = document.querySelector('.cme-trigger-sys');
    if (sysBtn) sysBtn.textContent = 'Get System Information';

    const logBtn = document.querySelector('.cme-trigger-log');
    if (logBtn) logBtn.textContent = 'View Debug Logger';

    // Restore API button text based on data container validation status
    const connectBtn = document.getElementById('chm_activalist');
    const isApiValid = getApiValid() === '1';

    if (connectBtn && isApiValid && connectBtn.value !== 'Connected') {
        // API is valid in DB but button wasn't rendered correctly - fix text
        connectBtn.value = 'Connected';
        console.log('[Chimpmatic] 🔧 Restored button text: Connected (from data container)');
    }
});

// Handle log file selection
document.addEventListener('change', function(event) {
    if (event.target && event.target.id === 'chm-log-selector') {
        const logFile = event.target.value;
        const contentDiv = document.getElementById('chm-log-content');

        if (!logFile || !contentDiv) {
            if (contentDiv) {
                contentDiv.innerHTML = '<p>Select a log file above.</p>';
            }
            return;
        }

        // Show loading message
        contentDiv.innerHTML = '<p>Loading log file...</p>';

        // Fetch log content via REST API
        const data = {
            log_file: logFile,
            chm_idformxx: getFormId()
        };

        // Use fetch directly for log reading (returns plain text, not JSON)
        const restUrl = (typeof wpApiSettings !== 'undefined' && wpApiSettings.root) ?
            wpApiSettings.root + 'chimpmatic/v1/logs/read' :
            '/wp-json/chimpmatic/v1/logs/read';

        fetch(restUrl, {
            method: 'POST',
            headers: {
                'Content-Type': 'application/json',
                'X-WP-Nonce': wpApiSettings?.nonce || ''
            },
            body: JSON.stringify(data),
            credentials: 'same-origin'
        })
        .then(response => {
            if (!response.ok) {
                return response.text().then(text => {
                    try {
                        const json = JSON.parse(text);
                        throw new Error(json.message || 'HTTP error! status: ' + response.status);
                    } catch (e) {
                        throw new Error(text || 'HTTP error! status: ' + response.status);
                    }
                });
            }
            return response.text();
        })
        .then(text => {
            // Try to parse as JSON first (WordPress wraps it)
            try {
                const json = JSON.parse(text);
                const content = json.data || json;
                if (content && typeof content === 'string' && content.trim()) {
                    contentDiv.innerHTML = '<pre style="background: #f5f5f5; padding: 15px; border: 1px solid #ddd; border-radius: 4px; max-height: 600px; overflow-y: auto; white-space: pre-wrap; word-wrap: break-word; font-family: monospace; font-size: 12px;">' +
                        escapeHtml(content) +
                        '</pre>';
                } else {
                    contentDiv.innerHTML = '<p style="color: #d63638;">Log file is empty.</p>';
                }
            } catch (e) {
                // Not JSON, treat as plain text
                if (text && text.trim()) {
                    contentDiv.innerHTML = '<pre style="background: #f5f5f5; padding: 15px; border: 1px solid #ddd; border-radius: 4px; max-height: 600px; overflow-y: auto; white-space: pre-wrap; word-wrap: break-word; font-family: monospace; font-size: 12px;">' +
                        escapeHtml(text) +
                        '</pre>';
                } else {
                    contentDiv.innerHTML = '<p style="color: #d63638;">Log file is empty.</p>';
                }
            }
        })
        .catch(error => {
            console.error('[Chimpmatic] Log error:', error);
            contentDiv.innerHTML = '<p style="color: #d63638;">Error loading log file: ' + escapeHtml(error.message) + '</p>';
        });
    }
});

// Handle delete logs button
document.addEventListener('click', function(event) {
    if (event.target && event.target.id === 'chm-delete-logs-btn') {
        const button = event.target;
        const originalText = button.textContent;

        // Change button state to "Deleting Logs..."
        button.textContent = 'Deleting Logs...';
        button.disabled = true;

        const restUrl = (typeof wpApiSettings !== 'undefined' && wpApiSettings.root) ?
            wpApiSettings.root + 'chimpmatic/v1/logs/delete' :
            '/wp-json/chimpmatic/v1/logs/delete';

        fetch(restUrl, {
            method: 'POST',
            headers: {
                'Content-Type': 'application/json',
                'X-WP-Nonce': wpApiSettings?.nonce || ''
            },
            credentials: 'same-origin'
        })
        .then(response => response.json())
        .then(data => {
            // Change button state to "Logs Deleted"
            button.textContent = 'Logs Deleted';

            // Clear the log selector and content
            const selector = document.getElementById('chm-log-selector');
            const contentDiv = document.getElementById('chm-log-content');

            if (selector) {
                selector.innerHTML = '<option value="">-- Select Log File --</option>';
                selector.value = '';
            }

            if (contentDiv) {
                contentDiv.innerHTML = '<p>All log files deleted. ' + (data.message || '') + '</p>';
            }

            // Reset button after 2 seconds
            setTimeout(function() {
                button.textContent = originalText;
                button.disabled = false;
            }, 2000);
        })
        .catch(error => {
            console.error('[Chimpmatic] Delete logs error:', error);
            button.textContent = 'Delete Failed';

            // Reset button after 2 seconds
            setTimeout(function() {
                button.textContent = originalText;
                button.disabled = false;
            }, 2000);
        });
    }
});

// Helper function to escape HTML
function escapeHtml(text) {
    const div = document.createElement('div');
    div.textContent = text;
    return div.innerHTML;
}

/**
 * Fuzzy Match: Find best matching CF7 tag for MailChimp merge field (PRO)
 *
 * Two-tier matching strategy (inspired by ClickSyncr):
 * 1. Keyword-based matching (high confidence) - semantic mappings
 * 2. Substring fallback - if field contains tag or vice versa
 *
 * @param {string} mergeTag  - MailChimp merge tag (e.g., "FNAME", "EMAIL")
 * @param {string} fieldName - MailChimp field name (e.g., "First Name")
 * @param {Array}  cf7Tags   - Available CF7 form tags [{name: "your-name"}, ...]
 * @returns {string|null} Best matching CF7 tag name or null
 */
function findBestMatchPro(mergeTag, fieldName, cf7Tags) {
    if (!mergeTag || !cf7Tags || cf7Tags.length === 0) return null;

    const normalize = (str) => String(str).toLowerCase().replace(/[^a-z0-9]/g, '');
    const normalizedTag = normalize(mergeTag);
    const normalizedName = normalize(fieldName);

    // High-confidence keyword mappings: MailChimp merge tags → CF7 field patterns
    // Supports English + Spanish keywords
    const keywordMappings = {
        // Email variations
        email: ['email', 'mail', 'correo'],
        emailaddress: ['email', 'mail'],
        // Name variations
        fname: ['name', 'firstname', 'first', 'nombre', 'your-name'],
        firstname: ['name', 'firstname', 'first', 'nombre'],
        lname: ['lastname', 'last', 'apellido', 'surname'],
        lastname: ['lastname', 'last', 'apellido'],
        name: ['name', 'nombre', 'your-name'],
        fullname: ['name', 'fullname', 'nombre'],
        // Phone variations
        phone: ['phone', 'tel', 'telefono', 'mobile', 'cell'],
        mobilephone: ['phone', 'tel', 'mobile', 'cell'],
        // Address variations
        address: ['address', 'direccion', 'street'],
        address1: ['address', 'address1', 'street'],
        address2: ['address2', 'apt', 'suite'],
        city: ['city', 'ciudad'],
        state: ['state', 'province', 'region', 'estado'],
        zip: ['zip', 'postal', 'postcode'],
        country: ['country', 'pais'],
        // Other common fields
        company: ['company', 'organization', 'empresa', 'org'],
        website: ['website', 'url', 'web', 'sitio'],
        birthday: ['birthday', 'birth', 'dob', 'cumpleanos'],
        message: ['message', 'comments', 'mensaje', 'nota', 'your-message']
    };

    // Tier 1: Keyword-based matching (most reliable)
    for (const [mcKeyword, cf7Keywords] of Object.entries(keywordMappings)) {
        // Check if MailChimp field matches this keyword
        if (normalizedTag.includes(mcKeyword) || normalizedName.includes(mcKeyword)) {
            // Find CF7 tag containing any of the mapped keywords
            for (const cf7Keyword of cf7Keywords) {
                const match = cf7Tags.find(tag => {
                    const tagName = normalize(tag.name || tag);
                    return tagName.includes(cf7Keyword);
                });
                if (match) {
                    console.log(`[Chimpmatic PRO] Fuzzy match: ${mergeTag} → ${match.name || match} (keyword: ${mcKeyword})`);
                    return match.name || match;
                }
            }
        }
    }

    // Tier 2: Substring fallback (less confident)
    for (const tag of cf7Tags) {
        const tagName = normalize(tag.name || tag);

        // Check if merge tag contains CF7 tag or vice versa
        if (normalizedTag.includes(tagName) || tagName.includes(normalizedTag)) {
            console.log(`[Chimpmatic PRO] Fuzzy match (substring): ${mergeTag} → ${tag.name || tag}`);
            return tag.name || tag;
        }

        // Check if field name contains CF7 tag or vice versa
        if (normalizedName.includes(tagName) || tagName.includes(normalizedName)) {
            console.log(`[Chimpmatic PRO] Fuzzy match (substring-name): ${mergeTag} → ${tag.name || tag}`);
            return tag.name || tag;
        }
    }

    console.log(`[Chimpmatic PRO] No fuzzy match found for: ${mergeTag}`);
    return null;
}

/**
 * Apply fuzzy matching to PRO field dropdowns after AJAX loads fields
 * Targets the #chm_panel_camposforma container with CustomValue dropdowns
 */
function applyFuzzyMatchingPro() {
    console.log('[Chimpmatic PRO] Applying fuzzy matching to field dropdowns...');

    // Debug: Check if the panel exists
    const genPanel = document.getElementById('chm_panel_gencamposygrupos');
    const camposPanel = document.getElementById('chm_panel_camposforma');
    console.log('[Chimpmatic PRO] #chm_panel_gencamposygrupos exists:', !!genPanel);
    console.log('[Chimpmatic PRO] #chm_panel_camposforma exists:', !!camposPanel);

    // Find all field rows in PRO panel - look in multiple possible locations
    let fieldRows = document.querySelectorAll('#chm_panel_camposforma .mcee-container');
    console.log('[Chimpmatic PRO] Field rows in #chm_panel_camposforma:', fieldRows.length);

    // Fallback: try broader search within the gen panel
    if (fieldRows.length === 0) {
        fieldRows = document.querySelectorAll('#chm_panel_gencamposygrupos .mcee-container');
        console.log('[Chimpmatic PRO] Field rows in #chm_panel_gencamposygrupos:', fieldRows.length);
    }

    // Last resort: any .mcee-container
    if (fieldRows.length === 0) {
        fieldRows = document.querySelectorAll('.mcee-container');
        console.log('[Chimpmatic PRO] Field rows anywhere (.mcee-container):', fieldRows.length);
    }

    if (!fieldRows || fieldRows.length === 0) {
        console.log('[Chimpmatic PRO] No field rows found for fuzzy matching');
        return;
    }

    console.log('[Chimpmatic PRO] Found', fieldRows.length, 'field rows');

    // Collect ALL unique CF7 tags from ALL dropdowns (since PRO type-filters each dropdown)
    const cf7TagsSet = new Set();
    const allDropdowns = document.querySelectorAll('[id^="wpcf7-mailchimp-CustomValue"]');
    console.log('[Chimpmatic PRO] Total dropdowns found:', allDropdowns.length);

    allDropdowns.forEach(dropdown => {
        Array.from(dropdown.options).forEach(option => {
            if (option.value && option.value.trim() !== '' && option.value !== ' ') {
                cf7TagsSet.add(option.value);
            }
        });
    });

    const cf7Tags = Array.from(cf7TagsSet).map(name => ({ name }));
    console.log('[Chimpmatic PRO] Unique CF7 tags collected:', cf7Tags.length);

    if (cf7Tags.length === 0) {
        console.log('[Chimpmatic PRO] No CF7 tags found for fuzzy matching');
        return;
    }

    console.log('[Chimpmatic PRO] CF7 tags available:', cf7Tags.map(t => t.name));

    fieldRows.forEach((row, index) => {
        // Get the merge tag from hidden input (CustomKey)
        const keyInput = row.querySelector('[id^="wpcf7-mailchimp-CustomKey"]');
        const dropdown = row.querySelector('[id^="wpcf7-mailchimp-CustomValue"]');

        console.log(`[Chimpmatic PRO] Row ${index + 1}: keyInput=${!!keyInput}, dropdown=${!!dropdown}`);
        if (keyInput) console.log(`[Chimpmatic PRO] Row ${index + 1}: keyInput.id=${keyInput.id}, value=${keyInput.value}`);
        if (dropdown) console.log(`[Chimpmatic PRO] Row ${index + 1}: dropdown.id=${dropdown.id}, value="${dropdown.value}"`);

        if (!keyInput || !dropdown) return;

        // Skip if already has a saved value (not empty)
        if (dropdown.value && dropdown.value.trim() !== '' && dropdown.value !== ' ') {
            console.log(`[Chimpmatic PRO] Field ${index + 1} already mapped to: ${dropdown.value}`);
            return;
        }

        // Get available options for THIS dropdown (PRO type-filters each dropdown)
        const dropdownOptions = [];
        Array.from(dropdown.options).forEach(option => {
            if (option.value && option.value.trim() !== '' && option.value !== ' ') {
                dropdownOptions.push({ name: option.value });
            }
        });
        console.log(`[Chimpmatic PRO] Row ${index + 1} available options:`, dropdownOptions.map(o => o.name));

        if (dropdownOptions.length === 0) {
            console.log(`[Chimpmatic PRO] Row ${index + 1} has no options to match`);
            return;
        }

        // Get merge tag and field name from the row
        const mergeTag = keyInput.value; // e.g., "EMAIL", "FNAME"
        const label = row.querySelector('label');
        const fieldName = label ? label.textContent.split('*|')[0].trim() : mergeTag;

        console.log(`[Chimpmatic PRO] Trying fuzzy match for: mergeTag="${mergeTag}", fieldName="${fieldName}"`);

        // Try fuzzy matching against THIS dropdown's options
        const bestMatch = findBestMatchPro(mergeTag, fieldName, dropdownOptions);
        if (bestMatch) {
            dropdown.value = bestMatch;
            console.log(`[Chimpmatic PRO] Auto-mapped ${mergeTag} → ${bestMatch}`);
        }
    });

    console.log('[Chimpmatic PRO] Fuzzy matching complete');
}

/**
 * Tags Chip UI - Toggle selected class on click
 * Uses event delegation for dynamically loaded content (PRO tags panel)
 * Note: Since chip is a <label>, clicking it auto-toggles the checkbox.
 * We just need to sync the 'selected' class with checkbox state.
 */
(function() {
    document.addEventListener('change', function(e) {
        // Only handle checkbox changes inside tag chips
        if (e.target.type !== 'checkbox') return;
        const chip = e.target.closest('.cmatic-tag-chip');
        if (!chip) return;

        chip.classList.toggle('selected', e.target.checked);
        console.log('[Chimpmatic Tags] Toggled:', e.target.name, '→', e.target.checked);
    });
})();

// License Activation Handler
document.addEventListener('DOMContentLoaded', function() {
    const activationForm = document.getElementById('chimpmatic-activation-form');
    const deactivateBtn = document.getElementById('chimpmatic-deactivate-btn');

    // Helper to show feedback messages
    function showFeedback(message, type = 'info') {
        const feedbackDiv = document.getElementById('chimpmatic-license-feedback');
        const messageP = document.getElementById('chimpmatic-license-feedback-message');

        if (!feedbackDiv || !messageP) return;

        // Set message
        messageP.textContent = message;

        // Set styling based on type
        if (type === 'success') {
            feedbackDiv.style.backgroundColor = '#d4edda';
            feedbackDiv.style.borderColor = '#c3e6cb';
            feedbackDiv.style.color = '#155724';
            feedbackDiv.style.border = '1px solid #c3e6cb';
        } else if (type === 'error') {
            feedbackDiv.style.backgroundColor = '#f8d7da';
            feedbackDiv.style.borderColor = '#f5c6cb';
            feedbackDiv.style.color = '#721c24';
            feedbackDiv.style.border = '1px solid #f5c6cb';
        } else {
            feedbackDiv.style.backgroundColor = '#d1ecf1';
            feedbackDiv.style.borderColor = '#bee5eb';
            feedbackDiv.style.color = '#0c5460';
            feedbackDiv.style.border = '1px solid #bee5eb';
        }

        feedbackDiv.style.display = 'block';
    }

    function hideFeedback() {
        const feedbackDiv = document.getElementById('chimpmatic-license-feedback');
        if (feedbackDiv) {
            feedbackDiv.style.display = 'none';
        }
    }

    // Handle activation form submission
    if (activationForm) {
        console.log('[Chimpmatic License] Activation form found, adding event listener');
        activationForm.addEventListener('submit', async function(e) {
            e.preventDefault();
            console.log('[Chimpmatic License] Form submitted');

            const licenseKey = document.getElementById('license_key').value.trim();
            const productId = document.getElementById('product_id').value.trim();
            const activateBtn = document.getElementById('chimpmatic-activate-btn');

            console.log('[Chimpmatic License] License Key:', licenseKey.substring(0, 10) + '...');
            console.log('[Chimpmatic License] Product ID:', productId);

            if (!licenseKey || !productId) {
                console.log('[Chimpmatic License] Missing license key or product ID');
                showFeedback('Please enter both license key and product ID', 'error');
                return;
            }

            // Disable button and show loading state
            activateBtn.disabled = true;
            activateBtn.textContent = 'Activating...';
            hideFeedback();

            try {
                // Step 1: Show checking status
                showFeedback('Checking activation status...', 'info');
                await new Promise(resolve => setTimeout(resolve, 500));

                // Step 2: Make activation request (which auto-deactivates if needed)
                const restUrl = (typeof wpApiSettings !== 'undefined' && wpApiSettings.root) ?
                    wpApiSettings.root + 'chimpmatic/v1/license/activate' :
                    '/wp-json/chimpmatic/v1/license/activate';

                console.log('[Chimpmatic License] REST URL:', restUrl);
                console.log('[Chimpmatic License] wpApiSettings:', wpApiSettings);

                const response = await fetch(restUrl, {
                    method: 'POST',
                    headers: {
                        'Content-Type': 'application/json',
                        'X-WP-Nonce': wpApiSettings?.nonce || ''
                    },
                    body: JSON.stringify({
                        license_key: licenseKey,
                        product_id: productId
                    }),
                    credentials: 'same-origin'
                });

                console.log('[Chimpmatic License] Response status:', response.status);

                const result = await response.json();
                console.log('[Chimpmatic License] Response data:', result);

                if (result.success) {
                    console.log('[Chimpmatic License] Activation successful!');
                    // Show deactivation message if previous activation was removed
                    if (result.was_deactivated) {
                        console.log('[Chimpmatic License] Previous activation was deactivated');
                        showFeedback('Site already activated... deactivating previous activation...', 'info');
                        await new Promise(resolve => setTimeout(resolve, 800));
                    }

                    // Show activating message
                    showFeedback('Activating license...', 'info');
                    await new Promise(resolve => setTimeout(resolve, 500));

                    // Show success
                    showFeedback('Success! License activated.', 'success');
                    await new Promise(resolve => setTimeout(resolve, 1000));

                    // Reload page to show activated state
                    console.log('[Chimpmatic License] Reloading page...');
                    window.location.reload();
                } else {
                    console.log('[Chimpmatic License] Activation failed:', result.message);
                    showFeedback(result.message || 'Activation failed', 'error');
                    activateBtn.disabled = false;
                    activateBtn.textContent = 'Activate License';
                }
            } catch (error) {
                console.error('[Chimpmatic License] Error:', error);
                showFeedback('Error: ' + error.message, 'error');
                activateBtn.disabled = false;
                activateBtn.textContent = 'Activate License';
            }
        });
    }

    // Handle deactivation button click
    if (deactivateBtn) {
        deactivateBtn.addEventListener('click', async function(e) {
            // Confirmation is handled by onclick attribute
            if (!e.target.onclick || e.target.onclick.call(e.target) === false) {
                return;
            }

            const btn = e.target;
            btn.disabled = true;
            btn.textContent = 'Deactivating...';

            try {
                const restUrl = (typeof wpApiSettings !== 'undefined' && wpApiSettings.root) ?
                    wpApiSettings.root + 'chimpmatic/v1/license/deactivate' :
                    '/wp-json/chimpmatic/v1/license/deactivate';

                const response = await fetch(restUrl, {
                    method: 'POST',
                    headers: {
                        'Content-Type': 'application/json',
                        'X-WP-Nonce': wpApiSettings?.nonce || ''
                    },
                    credentials: 'same-origin'
                });

                const result = await response.json();

                if (result.success) {
                    // Reload page to show deactivated state
                    window.location.reload();
                } else {
                    alert('Deactivation failed: ' + (result.message || 'Unknown error'));
                    btn.disabled = false;
                    btn.textContent = 'Deactivate License For This Site';
                }
            } catch (error) {
                console.error('[Chimpmatic License] Error:', error);
                alert('Error: ' + error.message);
                btn.disabled = false;
                btn.textContent = 'Deactivate License For This Site';
            }
        });
    }
});

// ========================================================================
// CF7 Integration Page License Activation/Deactivation
// ========================================================================
(function() {
    'use strict';

    const activateBtn = document.getElementById('cmatic-cf7-activate-btn');
    const deactivateBtn = document.getElementById('cmatic-cf7-deactivate-btn');
    const feedbackDiv = document.getElementById('cmatic-cf7-license-feedback');

    // Try multiple methods to find the form
    let form = activateBtn ? activateBtn.closest('form') : (deactivateBtn ? deactivateBtn.closest('form') : null);

    // Fallback: If closest() doesn't work, traverse manually
    if (!form && activateBtn) {
        let element = activateBtn.parentElement;
        while (element && element.tagName !== 'FORM') {
            element = element.parentElement;
        }
        form = element;
    }

    console.log('[Chimpmatic Activation] Form:', form);
    console.log('[Chimpmatic Activation] Activate button:', activateBtn);
    console.log('[Chimpmatic Activation] Deactivate button:', deactivateBtn);

    // Show feedback message
    function showFeedback(message, type = 'info') {
        if (!feedbackDiv) return;

        feedbackDiv.textContent = message;
        feedbackDiv.style.display = 'block';

        // Set colors based on type
        if (type === 'success') {
            feedbackDiv.style.backgroundColor = '#d4edda';
            feedbackDiv.style.color = '#155724';
            feedbackDiv.style.border = '1px solid #c3e6cb';
        } else if (type === 'error') {
            feedbackDiv.style.backgroundColor = '#f8d7da';
            feedbackDiv.style.color = '#721c24';
            feedbackDiv.style.border = '1px solid #f5c6cb';
        } else {
            feedbackDiv.style.backgroundColor = '#d1ecf1';
            feedbackDiv.style.color = '#0c5460';
            feedbackDiv.style.border = '1px solid #bee5eb';
        }
    }

    function hideFeedback() {
        if (feedbackDiv) {
            feedbackDiv.style.display = 'none';
        }
    }

    // Handle activation
    // CF7 strips the <form> tag, so we attach directly to the button
    if (activateBtn) {
        console.log('[Chimpmatic Activation] Setting up button click handler (CF7 strips form tags)');
        activateBtn.addEventListener('click', async function(e) {
            e.preventDefault();
            console.log('[Chimpmatic Activation] Activate button clicked - activation started');

            const licenseKey = document.getElementById('license_key').value.trim();
            const productId = document.getElementById('product_id').value.trim();

            console.log('[Chimpmatic Activation] License key length:', licenseKey.length);
            console.log('[Chimpmatic Activation] Product ID:', productId);

            if (!licenseKey || !productId) {
                console.warn('[Chimpmatic Activation] Missing license key or product ID');
                showFeedback('Please enter both License Key and Product ID', 'error');
                return;
            }

            activateBtn.disabled = true;
            activateBtn.textContent = 'Activating...';
            hideFeedback();

            try {
                const restUrl = (typeof wpApiSettings !== 'undefined' && wpApiSettings.root) ?
                    wpApiSettings.root + 'chimpmatic/v1/license/activate' :
                    '/wp-json/chimpmatic/v1/license/activate';

                console.log('[Chimpmatic Activation] REST URL:', restUrl);
                console.log('[Chimpmatic Activation] Nonce available:', wpApiSettings?.nonce ? 'YES' : 'NO');

                const requestBody = {
                    license_key: licenseKey,
                    product_id: productId
                };
                console.log('[Chimpmatic Activation] Request body:', requestBody);

                const response = await fetch(restUrl, {
                    method: 'POST',
                    headers: {
                        'Content-Type': 'application/json',
                        'X-WP-Nonce': wpApiSettings?.nonce || ''
                    },
                    credentials: 'same-origin',
                    body: JSON.stringify(requestBody)
                });

                console.log('[Chimpmatic Activation] Response status:', response.status);
                const result = await response.json();
                console.log('[Chimpmatic Activation] Response data:', result);

                if (result.success) {
                    console.log('[Chimpmatic Activation] ✅ Activation successful!');
                    showFeedback('✅ License activated successfully!', 'success');

                    // Reload page after 1 second to show activated state
                    setTimeout(function() {
                        console.log('[Chimpmatic Activation] Reloading page...');
                        window.location.reload();
                    }, 1000);
                } else {
                    console.error('[Chimpmatic Activation] ❌ Activation failed:', result.message || result);
                    showFeedback(result.message || 'Activation failed', 'error');
                    activateBtn.disabled = false;
                    activateBtn.textContent = 'Activate License';
                }
            } catch (error) {
                console.error('[Chimpmatic Activation] ❌ Network error:', error);
                showFeedback('Network error: ' + error.message, 'error');
                activateBtn.disabled = false;
                activateBtn.textContent = 'Activate License';
            }
        });
    } else {
        console.warn('[Chimpmatic Activation] Activate button not found - handler NOT attached');
        console.log('[Chimpmatic Activation] Activate button:', activateBtn);
    }

    // Handle deactivation - single click (warning text already displayed in UI)
    if (deactivateBtn) {
        console.log('[Chimpmatic Deactivation] Setting up deactivate button handler');

        deactivateBtn.addEventListener('click', async function(e) {
            e.preventDefault();
            console.log('[Chimpmatic Deactivation] Deactivate button clicked - starting deactivation');

            deactivateBtn.disabled = true;
            deactivateBtn.textContent = 'Deactivating...';
            hideFeedback();

            try {
                const restUrl = (typeof wpApiSettings !== 'undefined' && wpApiSettings.root) ?
                    wpApiSettings.root + 'chimpmatic/v1/license/deactivate' :
                    '/wp-json/chimpmatic/v1/license/deactivate';

                console.log('[Chimpmatic Deactivation] REST URL:', restUrl);
                console.log('[Chimpmatic Deactivation] Nonce available:', wpApiSettings?.nonce ? 'YES' : 'NO');

                const response = await fetch(restUrl, {
                    method: 'POST',
                    headers: {
                        'Content-Type': 'application/json',
                        'X-WP-Nonce': wpApiSettings?.nonce || ''
                    },
                    credentials: 'same-origin'
                });

                console.log('[Chimpmatic Deactivation] Response status:', response.status);
                const result = await response.json();
                console.log('[Chimpmatic Deactivation] Response data:', result);

                if (result.success) {
                    console.log('[Chimpmatic Deactivation] ✅ Deactivation successful!');
                    showFeedback('✅ License deactivated successfully!', 'success');

                    // Reload page after 1 second to show clean activation form
                    setTimeout(function() {
                        console.log('[Chimpmatic Deactivation] Reloading page...');
                        window.location.reload();
                    }, 1000);
                } else {
                    console.error('[Chimpmatic Deactivation] ❌ Deactivation failed:', result.message || result);
                    showFeedback(result.message || 'Deactivation failed', 'error');
                    deactivateBtn.disabled = false;
                    deactivateBtn.textContent = 'Deactivate License';
                }
            } catch (error) {
                console.error('[Chimpmatic Deactivation] ❌ Network error:', error);
                showFeedback('Network error: ' + error.message, 'error');
                deactivateBtn.disabled = false;
                deactivateBtn.textContent = 'Deactivate License';
            }
        });
    } else {
        console.log('[Chimpmatic Deactivation] Deactivate button not found - handler NOT attached');
    }
})();

/**
 * ========================================
 * Dependency Update Handler
 * Used by PRO plugin to update LITE/CF7
 * ========================================
 */
(function() {
    'use strict';

    console.log('[Chimpmatic Update] Initializing dependency update handler...');

    // Handle "Update Now" button click
    const updateBtn = document.querySelector('.cmatic-update-dependencies-btn');
    if (updateBtn) {
        console.log('[Chimpmatic Update] Update button found - attaching handler');

        updateBtn.addEventListener('click', function(e) {
            console.log('[Chimpmatic Update] Update Now button clicked');

            // Disable button and show loading state
            updateBtn.disabled = true;
            updateBtn.textContent = 'Updating... ⏳';
            updateBtn.style.opacity = '0.6';

            // Disable dismiss button
            const dismissBtn = document.querySelector('.cmatic-dismiss-notice-btn');
            if (dismissBtn) {
                dismissBtn.disabled = true;
                dismissBtn.style.opacity = '0.6';
            }

            // Disable the notice dismiss (X) button
            const noticeDiv = updateBtn.closest('.notice');
            if (noticeDiv) {
                const dismissX = noticeDiv.querySelector('.notice-dismiss');
                if (dismissX) {
                    dismissX.style.pointerEvents = 'none';
                    dismissX.style.opacity = '0.3';
                }
            }

            console.log('[Chimpmatic Update] Button disabled, proceeding with update...');
            // Link will proceed naturally (form submission or href navigation)
        });
    } else {
        console.log('[Chimpmatic Update] Update button not found - handler NOT attached');
    }

    // Handle "Dismiss" button click (AJAX dismissal)
    const dismissBtn = document.querySelector('.cmatic-dismiss-notice-btn');
    if (dismissBtn) {
        console.log('[Chimpmatic Update] Dismiss button found - attaching handler');

        dismissBtn.addEventListener('click', function(e) {
            e.preventDefault();
            console.log('[Chimpmatic Update] Dismiss button clicked');

            const nonce = dismissBtn.getAttribute('data-nonce');
            if (!nonce) {
                console.error('[Chimpmatic Update] No nonce found on dismiss button');
                return;
            }

            // Disable button
            dismissBtn.disabled = true;
            dismissBtn.textContent = 'Dismissing...';

            // Send AJAX request
            const ajaxUrl = typeof cmaticUpdateHelper !== 'undefined' && cmaticUpdateHelper.ajaxUrl ?
                cmaticUpdateHelper.ajaxUrl : '/wp-admin/admin-ajax.php';

            console.log('[Chimpmatic Update] Sending dismiss request to:', ajaxUrl);

            fetch(ajaxUrl, {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/x-www-form-urlencoded',
                },
                credentials: 'same-origin',
                body: new URLSearchParams({
                    action: 'cmatic_dismiss_update_notice',
                    nonce: nonce
                })
            })
            .then(response => response.json())
            .then(result => {
                console.log('[Chimpmatic Update] Dismiss response:', result);

                if (result.success) {
                    console.log('[Chimpmatic Update] ✅ Notice dismissed successfully');

                    // Hide the notice with fade effect
                    const noticeDiv = dismissBtn.closest('.notice');
                    if (noticeDiv) {
                        noticeDiv.style.transition = 'opacity 0.3s';
                        noticeDiv.style.opacity = '0';
                        setTimeout(function() {
                            noticeDiv.remove();
                        }, 300);
                    }
                } else {
                    console.error('[Chimpmatic Update] ❌ Dismiss failed:', result);
                    dismissBtn.disabled = false;
                    dismissBtn.textContent = 'Dismiss';
                }
            })
            .catch(error => {
                console.error('[Chimpmatic Update] ❌ Dismiss error:', error);
                dismissBtn.disabled = false;
                dismissBtn.textContent = 'Dismiss';
            });
        });
    } else {
        console.log('[Chimpmatic Update] Dismiss button not found - handler NOT attached');
    }

    console.log('[Chimpmatic Update] Handler initialization complete');
})();
