/*
 * This file is part of Contact Form 7 Extension For Mailchimp.
 *
 * Copyright (C) 2010-2025, Renzo Johnson (email: renzo.johnson at gmail.com)
 *
 * For the full copyright and license information, please view the LICENSE
 * file that was distributed with this source code, or visit:
 * https://www.gnu.org/licenses/gpl-2.0.html
 */

'use strict';

/**
 * ChimpMatic Lite - Modern Vanilla JavaScript REST API
 * Handles only REST API calls for fetching lists and rendering fields
 */

// Browser Console Logger - Intercepts all console methods and logs to server
(function() {
  // Check if global object exists
  if (typeof chimpmaticLite === 'undefined') {
    console.error('[ChimpMatic Lite] Global object not found. REST API calls will fail.');
    return;
  }

  // Check if logging is enabled - if not, skip initialization
  if (!chimpmaticLite.loggingEnabled) {
    return;
  }

  // Store original console methods
  const originalConsole = {
    log: console.log,
    info: console.info,
    warn: console.warn,
    error: console.error,
    debug: console.debug
  };

  // Send log to server via REST API
  async function sendLogToServer(level, message, ...args) {
    // Format the message with additional arguments
    let formattedMessage = message;
    let dataString = '';

    if (args.length > 0) {
      try {
        dataString = args.map(arg => {
          if (typeof arg === 'object') {
            return JSON.stringify(arg, null, 2);
          }
          return String(arg);
        }).join(' | ');
        formattedMessage += ' ' + dataString;
      } catch (e) {
        dataString = '[Unable to stringify arguments]';
      }
    }

    try {
      await fetch(`${chimpmaticLite.restUrl}logs/browser`, {
        method: 'POST',
        headers: {
          'Content-Type': 'application/json',
          'X-WP-Nonce': chimpmaticLite.restNonce
        },
        body: JSON.stringify({
          level: level,
          message: formattedMessage,
          data: dataString
        })
      });
    } catch (error) {
      // Silently fail to avoid infinite loops
      originalConsole.error('[ChimpMatic Lite] Failed to send log to server:', error);
    }
  }

  // Override console methods
  ['log', 'info', 'warn', 'error', 'debug'].forEach(level => {
    console[level] = function(...args) {
      // Call original console method
      originalConsole[level].apply(console, args);

      // Send to server in real-time
      const message = args[0] ? String(args[0]) : '';
      const additionalArgs = args.slice(1);
      sendLogToServer(level, message, ...additionalArgs);
    };
  });

  console.log('[ChimpMatic Lite] Browser console logger initialized. All logs will be sent to debug.log.');
})();

// Main Application
document.addEventListener('DOMContentLoaded', function() {
  console.log('[ChimpMatic Lite] Initializing REST API integration...');
  console.log('[ChimpMatic Lite] Config:', chimpmaticLite);

  /**
   * Detect if PRO's field panel exists (PRO handles its own field injection).
   * When PRO panel exists, LITE skips field-related functionality to avoid conflicts.
   */
  const isProFieldPanelActive = !!document.getElementById('chm_panel_gencamposygrupos');
  if (isProFieldPanelActive) {
    console.log('[ChimpMatic Lite] PRO field panel detected - LITE will skip field handling');
  }

  /**
   * Get form ID from data container (modern pattern).
   * Replaces legacy #mce_txtcomodin hidden input.
   * PRO inherits this pattern via shared Cmatic_Data_Container class.
   *
   * @returns {number} Form ID or 0 if not found.
   */
  function getFormId() {
    const dataContainer = document.getElementById('cmatic_data');
    if (dataContainer && dataContainer.dataset.formId) {
      return parseInt(dataContainer.dataset.formId, 10) || 0;
    }
    return 0;
  }

  /**
   * Get API validation status from data container.
   *
   * @returns {string} '0' or '1'.
   */
  function getApiValid() {
    const dataContainer = document.getElementById('cmatic_data');
    if (dataContainer && dataContainer.dataset.apiValid) {
      return dataContainer.dataset.apiValid;
    }
    return '0';
  }

  /**
   * REST API: Fetch Mailchimp Lists
   */
  async function fetchMailchimpLists(formId, apiKey) {
    console.log('[ChimpMatic Lite] Fetching Mailchimp lists...');

    try {
      const response = await fetch(`${chimpmaticLite.restUrl}lists`, {
        method: 'POST',
        headers: {
          'Content-Type': 'application/json',
          'X-WP-Nonce': chimpmaticLite.restNonce
        },
        body: JSON.stringify({
          form_id: formId,
          api_key: apiKey
        })
      });

      const data = await response.json();
      console.log('[ChimpMatic Lite] API Response:', data);

      if (!response.ok) {
        throw new Error(data.message || 'Failed to fetch lists');
      }

      return data;
    } catch (error) {
      console.error('[ChimpMatic Lite] Fetch lists error:', error);
      throw error;
    }
  }

  /**
   * REST API: Set Auto-Update Setting
   */
  async function setAutoUpdate(enabled) {
    console.log('[ChimpMatic Lite] Setting auto-update:', enabled);

    try {
      const response = await fetch(`${chimpmaticLite.restUrl}settings/autoupdate`, {
        method: 'POST',
        headers: {
          'Content-Type': 'application/json',
          'X-WP-Nonce': chimpmaticLite.restNonce
        },
        body: JSON.stringify({
          enabled: enabled
        })
      });

      const data = await response.json();
      console.log('[ChimpMatic Lite] Auto-update response:', data);

      if (!response.ok) {
        throw new Error(data.message || 'Failed to update setting');
      }

      return data;
    } catch (error) {
      console.error('[ChimpMatic Lite] Auto-update error:', error);
      throw error;
    }
  }

  /**
   * REST API: Get Debug Log
   */
  async function getDebugLog() {
    console.log('[ChimpMatic Lite] Fetching debug log...');

    try {
      const response = await fetch(`${chimpmaticLite.restUrl}logs`, {
        method: 'GET',
        headers: {
          'X-WP-Nonce': chimpmaticLite.restNonce
        }
      });

      const data = await response.json();
      console.log('[ChimpMatic Lite] Debug log response:', data);

      if (!response.ok) {
        throw new Error(data.message || 'Failed to fetch log');
      }

      return data;
    } catch (error) {
      console.error('[ChimpMatic Lite] Debug log error:', error);
      throw error;
    }
  }

  /**
   * REST API: Clear Debug Log
   */
  async function clearDebugLog() {
    console.log('[ChimpMatic Lite] Clearing debug log...');

    try {
      const response = await fetch(`${chimpmaticLite.restUrl}logs/clear`, {
        method: 'POST',
        headers: {
          'Content-Type': 'application/json',
          'X-WP-Nonce': chimpmaticLite.restNonce
        }
      });

      const data = await response.json();
      console.log('[ChimpMatic Lite] Clear log response:', data);

      if (!response.ok) {
        throw new Error(data.message || 'Failed to clear log');
      }

      return data;
    } catch (error) {
      console.error('[ChimpMatic Lite] Clear log error:', error);
      throw error;
    }
  }

  /**
   * Fuzzy Match: Find best matching CF7 tag for MailChimp merge field
   *
   * Two-tier matching strategy (inspired by ClickSyncr):
   * 1. Keyword-based matching (high confidence) - semantic mappings
   * 2. Substring fallback - if field contains tag or vice versa
   *
   * @param {string} mergeTag  - MailChimp merge tag (e.g., "FNAME", "EMAIL")
   * @param {string} fieldName - MailChimp field name (e.g., "First Name")
   * @param {Array}  cf7Tags   - Available CF7 form tags [{name: "your-name", basetype: "text"}, ...]
   * @returns {string|null} Best matching CF7 tag name or null
   */
  function findBestMatch(mergeTag, fieldName, cf7Tags) {
    if (!mergeTag || !cf7Tags || cf7Tags.length === 0) return null;

    const normalize = (str) => String(str).toLowerCase().replace(/[^a-z0-9]/g, '');
    const normalizedTag = normalize(mergeTag);
    const normalizedName = normalize(fieldName);

    // High-confidence keyword mappings: MailChimp merge tags → CF7 field patterns
    // Supports English + Spanish keywords
    const keywordMappings = {
      // Email variations
      email: ['email', 'mail', 'correo'],
      emailaddress: ['email', 'mail'],
      // Name variations
      fname: ['name', 'firstname', 'first', 'nombre', 'your-name'],
      firstname: ['name', 'firstname', 'first', 'nombre'],
      lname: ['lastname', 'last', 'apellido', 'surname'],
      lastname: ['lastname', 'last', 'apellido'],
      name: ['name', 'nombre', 'your-name'],
      fullname: ['name', 'fullname', 'nombre'],
      // Phone variations
      phone: ['phone', 'tel', 'telefono', 'mobile', 'cell'],
      mobilephone: ['phone', 'tel', 'mobile', 'cell'],
      // Address variations
      address: ['address', 'direccion', 'street'],
      address1: ['address', 'address1', 'street'],
      address2: ['address2', 'apt', 'suite'],
      city: ['city', 'ciudad'],
      state: ['state', 'province', 'region', 'estado'],
      zip: ['zip', 'postal', 'postcode'],
      country: ['country', 'pais'],
      // Other common fields
      company: ['company', 'organization', 'empresa', 'org'],
      website: ['website', 'url', 'web', 'sitio'],
      birthday: ['birthday', 'birth', 'dob', 'cumpleanos'],
      message: ['message', 'comments', 'mensaje', 'nota', 'your-message']
    };

    // Tier 1: Keyword-based matching (most reliable)
    for (const [mcKeyword, cf7Keywords] of Object.entries(keywordMappings)) {
      // Check if MailChimp field matches this keyword
      if (normalizedTag.includes(mcKeyword) || normalizedName.includes(mcKeyword)) {
        // Find CF7 tag containing any of the mapped keywords
        for (const cf7Keyword of cf7Keywords) {
          const match = cf7Tags.find(tag => {
            const tagName = normalize(tag.name || tag);
            return tagName.includes(cf7Keyword);
          });
          if (match) {
            console.log(`[ChimpMatic Lite] Fuzzy match: ${mergeTag} → ${match.name || match} (keyword: ${mcKeyword})`);
            return match.name || match;
          }
        }
      }
    }

    // Tier 2: Substring fallback (less confident)
    for (const tag of cf7Tags) {
      const tagName = normalize(tag.name || tag);

      // Check if merge tag contains CF7 tag or vice versa
      if (normalizedTag.includes(tagName) || tagName.includes(normalizedTag)) {
        console.log(`[ChimpMatic Lite] Fuzzy match (substring): ${mergeTag} → ${tag.name || tag}`);
        return tag.name || tag;
      }

      // Check if field name contains CF7 tag or vice versa
      if (normalizedName.includes(tagName) || tagName.includes(normalizedName)) {
        console.log(`[ChimpMatic Lite] Fuzzy match (substring-name): ${mergeTag} → ${tag.name || tag}`);
        return tag.name || tag;
      }
    }

    console.log(`[ChimpMatic Lite] No fuzzy match found for: ${mergeTag}`);
    return null;
  }

  /**
   * Apply fuzzy matching to field dropdowns after fields are rendered
   *
   * @param {Array} mergeFields - MailChimp merge fields from API
   */
  function applyFuzzyMatching(mergeFields) {
    // Get CF7 form tags from the page (extract from existing dropdowns)
    const cf7Tags = [];
    const sampleDropdown = document.getElementById('wpcf7-mailchimp-field3');
    if (sampleDropdown) {
      Array.from(sampleDropdown.options).forEach(option => {
        if (option.value && option.value.trim() !== '' && option.value !== ' ') {
          cf7Tags.push({ name: option.value });
        }
      });
    }

    if (cf7Tags.length === 0) {
      console.log('[ChimpMatic Lite] No CF7 tags found for fuzzy matching');
      return;
    }

    console.log('[ChimpMatic Lite] CF7 tags available for fuzzy matching:', cf7Tags.map(t => t.name));

    // Map field IDs to merge field indices (field3-field8 = indices 0-5)
    const fieldMappings = [
      { id: 'field3', index: 0 },
      { id: 'field4', index: 1 },
      { id: 'field5', index: 2 },
      { id: 'field6', index: 3 },
      { id: 'field7', index: 4 },
      { id: 'field8', index: 5 }
    ];

    fieldMappings.forEach(mapping => {
      const mergeField = mergeFields[mapping.index];
      if (!mergeField) return;

      const dropdown = document.getElementById(`wpcf7-mailchimp-${mapping.id}`);
      if (!dropdown) return;

      // Skip if already has a saved value (not empty)
      if (dropdown.value && dropdown.value.trim() !== '' && dropdown.value !== ' ') {
        console.log(`[ChimpMatic Lite] ${mapping.id} already mapped to: ${dropdown.value}`);
        return;
      }

      // Try fuzzy matching
      const bestMatch = findBestMatch(mergeField.tag, mergeField.name, cf7Tags);
      if (bestMatch) {
        dropdown.value = bestMatch;
        console.log(`[ChimpMatic Lite] Auto-mapped ${mergeField.tag} → ${bestMatch}`);
      }
    });
  }

  /**
   * UI: Update Field Labels with Merge Field Names
   * Only show/update fields that have data from the API
   * Also updates the dropdown text to show actual usable field count
   */
  function updateFieldLabels(mergeFields) {
    console.log('[ChimpMatic Lite] Updating field labels with merge fields...');
    console.log('[ChimpMatic Lite] Number of fields received:', mergeFields.length);

    // Map of field IDs to update (6 fields: field3 through field8)
    const fieldMappings = [
      { id: 'field3', index: 0 },
      { id: 'field4', index: 1 },
      { id: 'field5', index: 2 },
      { id: 'field6', index: 3 },
      { id: 'field7', index: 4 },
      { id: 'field8', index: 5 }
    ];

    fieldMappings.forEach(mapping => {
      const label = document.querySelector(`label[for="wpcf7-mailchimp-${mapping.id}"]`);
      const container = label ? label.closest('.mcee-container') : null;

      if (mergeFields[mapping.index]) {
        // Field exists in API response - show and update it
        const field = mergeFields[mapping.index];

        if (label) {
          // EMAIL field is always required
          const requiredBadge = field.tag === 'EMAIL' ? '<span class="mce-required">Required</span>' : '';
          label.innerHTML = `${field.name} - *|${field.tag}|* <span class="mce-type">${field.type}</span> ${requiredBadge}`;
          console.log(`[ChimpMatic Lite] Updated ${mapping.id} label to: ${field.name} (${field.tag})`);
        }

        // Show the container if it was hidden
        if (container) {
          container.style.display = '';
        }

      } else {
        // No field data for this position - hide the container
        if (container) {
          container.style.display = 'none';
          console.log(`[ChimpMatic Lite] Hiding ${mapping.id} (no field data)`);
        }
      }
    });

    // Field count is already displayed in the dropdown from Mailchimp stats
    // No need to update it dynamically

    // Apply fuzzy matching after labels are updated
    // This auto-maps MailChimp fields to CF7 tags based on semantic matching
    applyFuzzyMatching(mergeFields);
  }

  /**
   * UI: Render Lists Dropdown
   * Injects only <option> elements into #wpcf7-mailchimp-list select.
   * Label and button are static (rendered by PHP).
   */
  function renderListsDropdown(listsData, currentSelection) {
    const { api_valid, lists, total } = listsData;

    // Update data container with new API validation status.
    const dataContainer = document.getElementById('cmatic_data');
    if (dataContainer) {
      dataContainer.dataset.apiValid = api_valid ? '1' : '0';
    }

    // Update label text with count.
    const label = document.getElementById('cmatic-audiences-label');
    if (label) {
      label.textContent = api_valid && total > 0
        ? `Total Mailchimp Audiences: ${total}`
        : 'Mailchimp Audiences';
    }

    // Build only <option> elements.
    let optionsHtml = '';

    if (api_valid && total > 0) {
      // AUTO-SELECT: If no current selection, auto-select FIRST audience.
      let selectedAudience = currentSelection;
      if (!selectedAudience && lists.length > 0) {
        selectedAudience = lists[0].id;
        console.log('[ChimpMatic Lite] Auto-selecting first audience:', selectedAudience);
      }

      lists.forEach((list, index) => {
        const selected = selectedAudience === list.id ? ' selected' : '';
        const optionText = `${index + 1}:${list.member_count} ${list.name}  ${list.field_count} fields #${list.id}`;
        console.log('[ChimpMatic Lite] Audience option:', optionText);
        optionsHtml += `<option value="${list.id}"${selected}>${optionText}</option>`;
      });
    }

    return optionsHtml;
  }

  /**
   * Update version info in submitdiv after API validation
   *
   * Updates the API status indicator in the CF7 Status metabox (#submitdiv)
   * to match the validation state shown in the main panel.
   * Updates the status text to show "API Connected" or "API Inactive".
   *
   * @param {boolean} isValid - True if API is connected, false if inactive.
   */
  function updateSubmitdivApiStatus(isValid) {
    const versionInfo = document.getElementById('chimpmatic-version-info');
    if (!versionInfo) {
      return; // Submitdiv not present (shouldn't happen, but defensive)
    }

    const statusText = versionInfo.querySelector('.chmm');
    if (!statusText) {
      return; // Status element should be present
    }

    if (isValid) {
      statusText.classList.remove('invalid');
      statusText.classList.add('valid');
      statusText.textContent = 'API Connected';
    } else {
      statusText.classList.remove('valid');
      statusText.classList.add('invalid');
      statusText.textContent = 'API Inactive';
    }
  }

  /**
   * Update Lite badge status in H2 header with color-coded indicator
   *
   * Updates the .cm-lite badge with smooth color transitions:
   * - Neutral (gray): Not connected yet
   * - Green: API connected successfully
   * - Red: API validation failed
   *
   * @param {string} status - 'neutral', 'connected', or 'error'
   */
  function updateLiteBadgeStatus(status) {
    const liteBadge = document.querySelector('.cm-lite');
    if (!liteBadge) {
      return; // Badge not present
    }

    // Remove all status classes
    liteBadge.classList.remove('cm-status-neutral', 'cm-status-connected', 'cm-status-error');

    // Add appropriate status class (triggers CSS transition + pulse animation)
    if (status === 'connected') {
      liteBadge.classList.add('cm-status-connected');
    } else if (status === 'error') {
      liteBadge.classList.add('cm-status-error');
    } else {
      liteBadge.classList.add('cm-status-neutral');
    }
  }

  /**
   * Event Handler: Fetch Mailchimp Lists Button
   */
  const fetchListsButton = document.getElementById('chm_activalist');
  if (fetchListsButton) {
    fetchListsButton.addEventListener('click', async function(event) {
      event.preventDefault();
      console.log('[ChimpMatic Lite] Fetch lists button clicked.');

      const apiKeyInput = document.getElementById('cmatic-api');
      const selectElement = document.getElementById('wpcf7-mailchimp-list');

      if (!apiKeyInput || !selectElement) {
        console.error('[ChimpMatic Lite] Required elements not found.');
        return;
      }

      const formId = getFormId();
      const isMasked = apiKeyInput.dataset.isMasked === '1';
      const apiKey = (isMasked && apiKeyInput.dataset.realKey)
        ? apiKeyInput.dataset.realKey
        : apiKeyInput.value.trim();

      if (!formId || !apiKey) {
        alert('Please enter a valid API key.');

        // Reset UI to disconnected state
        updateSubmitdivApiStatus(false);
        updateLiteBadgeStatus('neutral');

        // Hide all active sections
        document.querySelectorAll('.chmp-active').forEach(el => {
          el.classList.remove('chmp-active');
          el.classList.add('chmp-inactive');
        });

        // Show new user section
        const newUserSection = document.getElementById('chmp-new-user');
        if (newUserSection) {
          newUserSection.classList.remove('chmp-inactive');
          newUserSection.classList.add('chmp-active');
        }

        return;
      }

      // Save original button text and update to loading state
      const originalText = fetchListsButton.value || fetchListsButton.textContent;
      fetchListsButton.disabled = true;
      if (fetchListsButton.tagName === 'INPUT') {
        fetchListsButton.value = 'Syncing Audiences...';
      } else {
        fetchListsButton.textContent = 'Syncing Audiences...';
      }

      try {
        const data = await fetchMailchimpLists(formId, apiKey);

        console.log('[ChimpMatic Lite] Lists fetched successfully. Total:', data.total);

        // Two-step flow: NO field updates here. Fields are fetched separately when user selects a list.

        // Update UI with lists - inject options into select
        const currentSelection = selectElement.value || '';
        selectElement.innerHTML = renderListsDropdown(data, currentSelection);

        // Re-attach event listeners after DOM update
        attachFetchFieldsListeners();

        // AUTO-FETCH: If audience is now selected (auto-selected), fetch fields immediately
        const newListDropdown = document.getElementById('wpcf7-mailchimp-list');
        if (newListDropdown && newListDropdown.value) {
          console.log('[ChimpMatic Lite] Auto-fetching fields for selected audience:', newListDropdown.value);
          // When PRO is active, dispatch change event to trigger PRO's handler
          // Otherwise call LITE's fetch function directly
          if (isProFieldPanelActive) {
            console.log('[ChimpMatic Lite] PRO active - dispatching change event to trigger PRO field fetch');
            newListDropdown.dispatchEvent(new Event('change', { bubbles: true }));
          } else {
            fetchFieldsForSelectedList();
          }
        }

        // Update UI based on API validation
        if (data.api_valid) {
          console.log('[ChimpMatic Lite] API Key is valid.');
          updateSubmitdivApiStatus(true);
          updateLiteBadgeStatus('connected');

          // Show active sections
          document.querySelectorAll('.chmp-inactive').forEach(el => {
            el.classList.remove('chmp-inactive');
            el.classList.add('chmp-active');
          });

          // Hide new user section
          const newUserSection = document.getElementById('chmp-new-user');
          if (newUserSection) {
            newUserSection.classList.remove('chmp-active');
            newUserSection.classList.add('chmp-inactive');
          }
        } else {
          console.log('[ChimpMatic Lite] API Key is invalid.');
          updateSubmitdivApiStatus(false);
          updateLiteBadgeStatus('error');

          // Hide active sections
          document.querySelectorAll('.chmp-active').forEach(el => {
            el.classList.remove('chmp-active');
            el.classList.add('chmp-inactive');
          });

          // Show new user section
          const newUserSection = document.getElementById('chmp-new-user');
          if (newUserSection) {
            newUserSection.classList.remove('chmp-inactive');
            newUserSection.classList.add('chmp-active');
          }
        }

        // Show success state briefly
        if (fetchListsButton.tagName === 'INPUT') {
          fetchListsButton.value = 'Synced ✓';
        } else {
          fetchListsButton.textContent = 'Synced ✓';
        }

        // Reset to original state after 1 second
        setTimeout(() => {
          if (fetchListsButton.tagName === 'INPUT') {
            fetchListsButton.value = originalText;
          } else {
            fetchListsButton.textContent = originalText;
          }
          fetchListsButton.disabled = false;
        }, 1000);

      } catch (error) {
        // Reset button on error
        if (fetchListsButton.tagName === 'INPUT') {
          fetchListsButton.value = originalText;
        } else {
          fetchListsButton.textContent = originalText;
        }
        fetchListsButton.disabled = false;

        alert(chimpmaticLite.i18n.error);
      }
    });
  }

  /**
   * Auto-Magic: Fetch audiences when API key is pasted or typed
   */
  const apiKeyInput = document.getElementById('cmatic-api');
  if (apiKeyInput && fetchListsButton) {

    // Validate Mailchimp API key format (32 chars + dash + us1-us21)
    function isValidApiKey(key) {
      if (key.length !== 36) return false;
      if (key.charAt(32) !== '-') return false;

      const hash = key.substring(0, 32);
      if (!/^[a-f0-9]{32}$/i.test(hash)) return false;

      const dc = key.substring(33).toLowerCase();
      const validDCs = ['us1','us2','us3','us4','us5','us6','us7','us8','us9','us10',
                        'us11','us12','us13','us14','us15','us16','us17','us18','us19',
                        'us20','us21'];

      return validDCs.includes(dc);
    }

    // Debounce utility
    function debounce(func, wait) {
      let timeout;
      return function(...args) {
        clearTimeout(timeout);
        timeout = setTimeout(() => func.apply(this, args), wait);
      };
    }

    // Auto-fetch on paste (instant)
    apiKeyInput.addEventListener('paste', function() {
      setTimeout(() => {
        const apiKey = apiKeyInput.value.trim();
        if (isValidApiKey(apiKey)) {
          fetchListsButton.click();
        }
      }, 50);
    });

    // Reset Lite badge to neutral when user starts typing new API key
    apiKeyInput.addEventListener('input', function() {
      updateLiteBadgeStatus('neutral');
    });

    // Auto-fetch on typing (debounced 500ms)
    apiKeyInput.addEventListener('input', debounce(function() {
      const apiKey = apiKeyInput.value.trim();
      if (isValidApiKey(apiKey)) {
        fetchListsButton.click();
      } else if (apiKey === '') {
        // API key cleared - reset database AND UI
        console.log('[ChimpMatic Lite] API key cleared - resetting database and UI');

        const formId = getFormId();
        if (formId) {
          // Call REST API to clear database
          fetch(`${chimpmaticLite.restUrl}settings/reset`, {
            method: 'POST',
            headers: {
              'Content-Type': 'application/json',
              'X-WP-Nonce': chimpmaticLite.restNonce
            },
            body: JSON.stringify({ form_id: formId })
          })
          .then(response => response.json())
          .then(data => {
            console.log('[ChimpMatic Lite] Database cleared:', data);

            // Clear UI after database is cleared - clear select options and reset label
            const selectElement = document.getElementById('wpcf7-mailchimp-list');
            if (selectElement) {
              selectElement.innerHTML = '';
            }
            const label = document.getElementById('cmatic-audiences-label');
            if (label) {
              label.textContent = 'Mailchimp Audiences';
            }

            const fieldsContainer = document.getElementById('cmatic-fields');
            if (fieldsContainer) {
              fieldsContainer.innerHTML = '';
            }

            // Reset badge status
            updateLiteBadgeStatus('neutral');

            document.querySelectorAll('.chmp-active').forEach(el => {
              el.classList.remove('chmp-active');
              el.classList.add('chmp-inactive');
            });

            const newUserSection = document.getElementById('chmp-new-user');
            if (newUserSection) {
              newUserSection.classList.remove('chmp-inactive');
              newUserSection.classList.add('chmp-active');
            }
          })
          .catch(error => {
            console.error('[ChimpMatic Lite] Failed to clear database:', error);
          });
        }
      }
    }, 500));
  }

  /**
   * Event Handler: Auto-Update Toggle
   */
  const autoUpdateCheckbox = document.getElementById('chimpmatic-update');
  if (autoUpdateCheckbox) {
    autoUpdateCheckbox.addEventListener('change', async function(event) {
      const isChecked = this.checked;
      console.log('[ChimpMatic Lite] Auto-update toggle changed:', isChecked);

      try {
        await setAutoUpdate(isChecked);
      } catch (error) {
        console.error('[ChimpMatic Lite] Failed to save auto-update setting.');
        // Revert checkbox on error
        this.checked = !isChecked;
      }
    });
  }

  /**
   * Event Handler: View Debug Logs Button
   * Toggles debug log panel and hides advanced settings if visible
   * NOTE: Skip if PRO's log panel exists (PRO has its own handler in chimpmatic.js)
   */
  const debugLogButton = document.querySelector('.cme-trigger-log');
  const proLogPanel = document.getElementById('log-dev'); // PRO uses #log-dev
  if (debugLogButton && !proLogPanel) {
    debugLogButton.addEventListener('click', async function(event) {
      event.preventDefault();
      console.log('[ChimpMatic Lite] Debug log button clicked.');

      const logPanelContainer = document.getElementById('eventlog-sys');
      const logPanel = document.getElementById('log_panel');
      const advancedSettings = document.querySelector('.vc-advanced-settings');
      const advancedToggleButton = document.querySelector('.vc-view-advanced');

      if (!logPanelContainer || !logPanel) {
        console.error('[ChimpMatic Lite] Log panel elements not found.');
        return;
      }

      // Check if log panel is currently visible
      const isLogVisible = window.getComputedStyle(logPanelContainer).display !== 'none';

      if (isLogVisible) {
        // Hide log panel with fade-out
        console.log('[ChimpMatic Lite] Hiding debug log panel...');
        logPanelContainer.style.transition = 'opacity 0.5s ease-out';
        logPanelContainer.style.opacity = '0';

        setTimeout(() => {
          logPanelContainer.style.display = 'none';
          logPanelContainer.style.removeProperty('opacity');
          logPanelContainer.style.removeProperty('transition');
          console.log('[ChimpMatic Lite] Debug log panel hidden.');
        }, 500);

        // Update button text
        this.textContent = 'View Debug Logs';

      } else {
        // Check if advanced settings is visible and hide it first
        if (advancedSettings) {
          const isAdvancedVisible = window.getComputedStyle(advancedSettings).display !== 'none';

          if (isAdvancedVisible) {
            console.log('[ChimpMatic Lite] Hiding advanced settings before showing debug log...');
            advancedSettings.style.transition = 'opacity 0.5s ease-out';
            advancedSettings.style.opacity = '0';

            setTimeout(() => {
              advancedSettings.style.display = 'none';
              advancedSettings.style.removeProperty('opacity');
              advancedSettings.style.removeProperty('transition');
              console.log('[ChimpMatic Lite] Advanced settings hidden.');
            }, 500);

            // Update advanced settings button text
            if (advancedToggleButton) {
              advancedToggleButton.textContent = 'Show Advanced Settings';
            }
          }
        }

        // Show log panel with loading state
        console.log('[ChimpMatic Lite] Loading debug log...');
        logPanel.textContent = chimpmaticLite.i18n.loading;

        // Show log panel with fade-in
        logPanelContainer.style.opacity = '0';
        logPanelContainer.style.display = 'block';
        logPanelContainer.style.transition = 'opacity 0.5s ease-in';

        setTimeout(() => {
          logPanelContainer.style.opacity = '1';
        }, 50);

        // Update button text
        this.textContent = 'Hide Debug Logs';

        // Fetch logs from REST API
        try {
          const data = await getDebugLog();

          if (data.success) {
            logPanel.textContent = data.content;
            console.log('[ChimpMatic Lite] Debug log loaded successfully.');
          } else {
            logPanel.textContent = 'Error: ' + (data.message || 'Unknown error');
            console.error('[ChimpMatic Lite] Failed to load debug log:', data.message);
          }
        } catch (error) {
          logPanel.textContent = chimpmaticLite.i18n.error;
          console.error('[ChimpMatic Lite] Debug log API error:', error);
        }
      }
    });
  }

  /**
   * Event Handler: Toggle Advanced Settings
   * Shows/hides .vc-advanced-settings when .vc-view-advanced is clicked
   * Hides debug log panel if visible (mutually exclusive)
   */
  const advancedToggleButton = document.querySelector('.vc-view-advanced');
  if (advancedToggleButton) {
    advancedToggleButton.addEventListener('click', function(event) {
      event.preventDefault();
      console.log('[ChimpMatic Lite] Advanced settings toggle clicked.');

      const advancedSettings = document.querySelector('.vc-advanced-settings');
      const logPanelContainer = document.getElementById('eventlog-sys');
      const debugLogButton = document.querySelector('.cme-trigger-log');

      if (!advancedSettings) {
        console.error('[ChimpMatic Lite] .vc-advanced-settings element not found.');
        return;
      }

      // Check current display state
      const isVisible = window.getComputedStyle(advancedSettings).display !== 'none';

      if (isVisible) {
        // Hide with fade-out
        console.log('[ChimpMatic Lite] Hiding advanced settings with fade-out...');
        advancedSettings.style.transition = 'opacity 0.5s ease-out';
        advancedSettings.style.opacity = '0';

        setTimeout(() => {
          advancedSettings.style.display = 'none';
          advancedSettings.style.removeProperty('opacity');
          advancedSettings.style.removeProperty('transition');
          console.log('[ChimpMatic Lite] Advanced settings hidden.');
        }, 500); // Match transition duration

        // Update button text
        this.textContent = 'Show Advanced Settings';
      } else {
        // Check if debug log panel is visible and hide it first
        if (logPanelContainer) {
          const isLogVisible = window.getComputedStyle(logPanelContainer).display !== 'none';

          if (isLogVisible) {
            console.log('[ChimpMatic Lite] Hiding debug log panel before showing advanced settings...');
            logPanelContainer.style.transition = 'opacity 0.5s ease-out';
            logPanelContainer.style.opacity = '0';

            setTimeout(() => {
              logPanelContainer.style.display = 'none';
              logPanelContainer.style.removeProperty('opacity');
              logPanelContainer.style.removeProperty('transition');
              console.log('[ChimpMatic Lite] Debug log panel hidden.');
            }, 500);

            // Update debug log button text
            if (debugLogButton) {
              debugLogButton.textContent = 'View Debug Logs';
            }
          }
        }

        // Show with fade-in
        console.log('[ChimpMatic Lite] Showing advanced settings with fade-in...');
        advancedSettings.style.opacity = '0';
        advancedSettings.style.display = 'block';
        advancedSettings.style.transition = 'opacity 0.5s ease-in';

        setTimeout(() => {
          advancedSettings.style.opacity = '1';
          console.log('[ChimpMatic Lite] Advanced settings visible.');
        }, 50); // Small delay to ensure transition takes effect

        // Update button text
        this.textContent = 'Hide Advanced Settings';
      }
    });
  }

  /**
   * Event Handler: Clear Debug Logs Button
   * Clears the debug.log file completely with animated button states
   */
  const clearLogsButton = document.querySelector('.vc-clear-logs');
  if (clearLogsButton) {
    clearLogsButton.addEventListener('click', async function(event) {
      event.preventDefault();
      console.log('[ChimpMatic Lite] Clear logs button clicked.');

      const logPanel = document.getElementById('log_panel');
      const originalText = this.textContent;

      // Disable button during operation
      this.disabled = true;

      // Change to "Clearing Logs..."
      this.textContent = 'Clearing Logs...';
      console.log('[ChimpMatic Lite] Button state: Clearing Logs...');

      try {
        const data = await clearDebugLog();

        if (data.success && data.cleared) {
          // Change to "Cleared"
          this.textContent = 'Cleared';
          console.log('[ChimpMatic Lite] Debug log cleared successfully.');

          // Update log panel to show it's cleared
          if (logPanel) {
            logPanel.textContent = 'Debug log cleared.';
          }

          // After 2 seconds, revert to "Clear Logs"
          setTimeout(() => {
            this.textContent = 'Clear Logs';
            this.disabled = false;
            console.log('[ChimpMatic Lite] Button state reset to: Clear Logs');
          }, 2000);

        } else {
          // File didn't exist or wasn't cleared
          this.textContent = 'Cleared';
          console.log('[ChimpMatic Lite] ' + data.message);

          if (logPanel) {
            logPanel.textContent = data.message || 'Debug log was already empty.';
          }

          setTimeout(() => {
            this.textContent = 'Clear Logs';
            this.disabled = false;
          }, 2000);
        }

      } catch (error) {
        // Error occurred
        this.textContent = 'Clearing Log Error';
        console.error('[ChimpMatic Lite] Failed to clear debug log:', error);

        // After 3 seconds, revert to "Clear Logs"
        setTimeout(() => {
          this.textContent = 'Clear Logs';
          this.disabled = false;
          console.log('[ChimpMatic Lite] Button state reset after error.');
        }, 3000);
      }
    });
  }

  /**
   * Immediate Element Relocation: Move .mce-move promotional banner
   * Executes immediately when ChimpMatic Lite panel loads
   */
  function relocatePromotionalBanner() {
    console.log('[ChimpMatic Lite] Attempting to relocate .mce-move element...');

    const moveElement = document.querySelector('.mce-move');
    const submitDiv = document.getElementById('submitdiv');
    const postboxContainer = document.querySelector('.postbox-container');

    // Validate all required elements exist
    if (!moveElement) {
      console.warn('[ChimpMatic Lite] .mce-move element not found. Relocation aborted.');
      return;
    }

    if (!submitDiv) {
      console.warn('[ChimpMatic Lite] #submitdiv element not found. Relocation aborted.');
      return;
    }

    if (!postboxContainer) {
      console.warn('[ChimpMatic Lite] .postbox-container element not found. Relocation aborted.');
      return;
    }

    console.log('[ChimpMatic Lite] All elements found. Relocating .mce-move after #submitdiv...');

    // Insert .mce-move after #submitdiv inside .postbox-container
    if (submitDiv.nextSibling) {
      postboxContainer.insertBefore(moveElement, submitDiv.nextSibling);
    } else {
      postboxContainer.appendChild(moveElement);
    }

    console.log('[ChimpMatic Lite] Element relocated successfully. Applying fade-in animation...');

    // Apply fade-in animation
    moveElement.style.opacity = '0';
    moveElement.style.display = 'block';
    moveElement.style.transition = 'opacity 0.5s ease-in';

    // Trigger fade-in
    setTimeout(() => {
      moveElement.style.opacity = '1';
      console.log('[ChimpMatic Lite] Promotional banner fade-in complete.');
    }, 50); // Small delay to ensure transition takes effect
  }

  // Execute relocation immediately when panel loads
  relocatePromotionalBanner();

  /**
   * Event Handler: Fetch Fields Button (NEW TWO-STEP FLOW)
   * Fetches merge fields for the selected list only
   */
  async function fetchFieldsForSelectedList() {
    // If PRO panel is active, PRO handles field fetching via its own AJAX
    if (isProFieldPanelActive) {
      console.log('[ChimpMatic Lite] PRO active - skipping LITE field fetch');
      return;
    }

    const listDropdown = document.getElementById('wpcf7-mailchimp-list');
    const fetchFieldsButton = document.getElementById('mce_fetch_fields');
    // Spinner removed - using button text states instead
    const formId = getFormId();

    if (!listDropdown || !listDropdown.value || !formId) {
      console.error('[ChimpMatic Lite] Cannot fetch fields: missing list or form ID.');
      return;
    }

    const listId = listDropdown.value;

    console.log('[ChimpMatic Lite] Fetching fields for list:', listId);

    // Save original button text and update to loading state
    const originalText = fetchFieldsButton ? (fetchFieldsButton.value || fetchFieldsButton.textContent) : 'Sync Fields';
    if (fetchFieldsButton) {
      fetchFieldsButton.disabled = true;
      if (fetchFieldsButton.tagName === 'INPUT') {
        fetchFieldsButton.value = 'Syncing Fields...';
      } else {
        fetchFieldsButton.textContent = 'Syncing Fields...';
      }
    }

    try {
      const response = await fetch(`${chimpmaticLite.restUrl}merge-fields`, {
        method: 'POST',
        headers: {
          'Content-Type': 'application/json',
          'X-WP-Nonce': chimpmaticLite.restNonce
        },
        body: JSON.stringify({
          form_id: formId,
          list_id: listId
        })
      });

      const data = await response.json();
      console.log('[ChimpMatic Lite] Fields response:', data);

      if (data.success && data.merge_fields) {
        // Update field labels with new merge fields
        updateFieldLabels(data.merge_fields);
        console.log('[ChimpMatic Lite] Field labels updated successfully.');

        // Show success state briefly
        if (fetchFieldsButton) {
          if (fetchFieldsButton.tagName === 'INPUT') {
            fetchFieldsButton.value = 'Synced ✓';
          } else {
            fetchFieldsButton.textContent = 'Synced ✓';
          }

          // Reset to original state after 1 second
          setTimeout(() => {
            if (fetchFieldsButton.tagName === 'INPUT') {
              fetchFieldsButton.value = originalText;
            } else {
              fetchFieldsButton.textContent = originalText;
            }
            fetchFieldsButton.disabled = false;
          }, 1000);
        }

      } else {
        console.error('[ChimpMatic Lite] Failed to fetch fields:', data);
        alert('Failed to load fields. Please try again.');

        // Reset button on error
        if (fetchFieldsButton) {
          if (fetchFieldsButton.tagName === 'INPUT') {
            fetchFieldsButton.value = originalText;
          } else {
            fetchFieldsButton.textContent = originalText;
          }
          fetchFieldsButton.disabled = false;
        }
      }

    } catch (error) {
      console.error('[ChimpMatic Lite] Fetch fields error:', error);
      alert('Error loading fields. Check console for details.');

      // Reset button on error
      if (fetchFieldsButton) {
        if (fetchFieldsButton.tagName === 'INPUT') {
          fetchFieldsButton.value = originalText;
        } else {
          fetchFieldsButton.textContent = originalText;
        }
        fetchFieldsButton.disabled = false;
      }
    }
  }

  /**
   * Attach event listeners for Fetch Fields functionality
   * Called after DOM updates to re-attach listeners
   */
  function attachFetchFieldsListeners() {
    // Event Handler: Audience Dropdown Change (OPTION C - HYBRID)
    const listDropdown = document.getElementById('wpcf7-mailchimp-list');
    if (listDropdown) {
      // Remove old listener if exists (prevent duplicates)
      listDropdown.removeEventListener('change', handleListChange);
      // Add listener
      listDropdown.addEventListener('change', handleListChange);
    }

    // Event Handler: Manual Fetch Fields Button Click
    const fetchFieldsButton = document.getElementById('mce_fetch_fields');
    if (fetchFieldsButton) {
      // Remove old listener if exists (prevent duplicates)
      fetchFieldsButton.removeEventListener('click', handleFetchFieldsClick);
      // Add listener
      fetchFieldsButton.addEventListener('click', handleFetchFieldsClick);
    }
  }

  /**
   * Handle list dropdown change
   */
  function handleListChange(e) {
    const selectedList = e.target.value;
    const fetchFieldsButton = document.getElementById('mce_fetch_fields');

    console.log('[ChimpMatic Lite] Audience changed to:', selectedList);

    // If PRO is active, let PRO handle everything
    if (isProFieldPanelActive) {
      console.log('[ChimpMatic Lite] PRO active - skipping LITE audience change handler');
      return;
    }

    if (selectedList) {
      // Enable fetch fields button
      if (fetchFieldsButton) {
        fetchFieldsButton.disabled = false;
      }

      // AUTO-CLEAR all field mappings (field3-field8)
      console.log('[ChimpMatic Lite] Auto-clearing field mappings...');
      for (let i = 3; i <= 8; i++) {
        const dropdown = document.getElementById(`wpcf7-mailchimp-field${i}`);
        if (dropdown) {
          dropdown.value = ' '; // Set to empty/choose option
        }
      }

      // AUTO-FETCH fields for new audience (Option C - Hybrid)
      console.log('[ChimpMatic Lite] Auto-fetching fields for selected audience...');
      fetchFieldsForSelectedList();

    } else {
      // No list selected, disable button
      if (fetchFieldsButton) {
        fetchFieldsButton.disabled = true;
      }
    }
  }

  /**
   * Handle fetch fields button click
   */
  async function handleFetchFieldsClick(event) {
    // If PRO panel is active, let PRO handle the click (don't preventDefault)
    if (isProFieldPanelActive) {
      console.log('[ChimpMatic Lite] PRO active - delegating field fetch to PRO');
      return; // Don't preventDefault, let PRO's handler run
    }
    event.preventDefault();
    console.log('[ChimpMatic Lite] Fetch Fields button clicked manually.');
    await fetchFieldsForSelectedList();
  }

  // Attach listeners on initial page load
  attachFetchFieldsListeners();

  // AUTO-SELECT: On page load, if dropdown exists but no selection, auto-select first
  const initialListDropdown = document.getElementById('wpcf7-mailchimp-list');
  const initialFetchButton = document.getElementById('mce_fetch_fields');
  if (initialListDropdown && initialListDropdown.options.length > 0) {
    console.log('[ChimpMatic Lite] Page load check - Dropdown value:', initialListDropdown.value);
    console.log('[ChimpMatic Lite] Page load check - Button disabled:', initialFetchButton ? initialFetchButton.disabled : 'no button');

    // Check if nothing is selected OR if value is empty
    if (!initialListDropdown.value || initialListDropdown.value === '' || initialListDropdown.value === ' ') {
      // Auto-select first option by setting its value explicitly
      const firstOptionValue = initialListDropdown.options[0].value;
      initialListDropdown.value = firstOptionValue;
      console.log('[ChimpMatic Lite] Page load: Auto-selected first audience:', firstOptionValue);

      // Enable button
      if (initialFetchButton) {
        initialFetchButton.disabled = false;
        console.log('[ChimpMatic Lite] Page load: Enabled Sync Fields button');
      }
    } else {
      console.log('[ChimpMatic Lite] Page load: Audience already selected:', initialListDropdown.value);
      // Still ensure button is enabled if value exists
      if (initialFetchButton && initialFetchButton.disabled) {
        initialFetchButton.disabled = false;
        console.log('[ChimpMatic Lite] Page load: Enabled button for existing selection');
      }
    }
  }

  // Load saved merge fields on page load
  if (chimpmaticLite.mergeFields && chimpmaticLite.mergeFields.length > 0) {
    console.log('[ChimpMatic Lite] Loading saved merge fields on page load:', chimpmaticLite.mergeFields);
    updateFieldLabels(chimpmaticLite.mergeFields);
  }

  /**
   * License Reset Button Handler
   *
   * Simple button to clear cached license data.
   * Fixes "zombie activation" bug where license data keeps regenerating.
   */
  function initLicenseResetButton() {
    const button = document.getElementById('cmatic-license-reset-btn');
    const messageDiv = document.getElementById('cmatic-license-reset-message');

    if (!button || !messageDiv) {
      return; // Elements not found, exit gracefully
    }

    button.addEventListener('click', async function(e) {
      e.preventDefault();

      // Disable button and change text
      button.disabled = true;
      button.textContent = 'Resetting...';
      messageDiv.innerHTML = '';

      try {
        const response = await fetch(chimpmaticLite.licenseResetUrl, {
          method: 'POST',
          headers: {
            'Content-Type': 'application/json',
            'X-WP-Nonce': chimpmaticLite.nonce
          },
          credentials: 'same-origin'
        });

        const data = await response.json();

        if (data.success) {
          // Success state
          button.textContent = 'Done Resetting';
          messageDiv.innerHTML = '<span style="color: #46b450;">✓ ' + escapeHtml(data.message) + '</span><br>' +
            '<small style="color: #666;">Deleted ' + data.deleted_counts.options + ' options and ' +
            data.deleted_counts.transients + ' transients</small>';

          // Reset button after 3 seconds
          setTimeout(function() {
            button.textContent = 'Reset License Data';
            button.disabled = false;
            messageDiv.innerHTML = '';
          }, 3000);
        } else {
          // Error state
          button.textContent = 'Reset License Data';
          button.disabled = false;
          messageDiv.innerHTML = '<span style="color: #dc3232;">✗ Error: ' +
            escapeHtml(data.message || 'Unknown error occurred') + '</span>';

          // Clear error message after 5 seconds
          setTimeout(function() {
            messageDiv.innerHTML = '';
          }, 5000);
        }
      } catch (error) {
        // Network/fetch error
        button.textContent = 'Reset License Data';
        button.disabled = false;
        messageDiv.innerHTML = '<span style="color: #dc3232;">✗ Network error: ' +
          escapeHtml(error.message) + '</span>';

        // Clear error message after 5 seconds
        setTimeout(function() {
          messageDiv.innerHTML = '';
        }, 5000);
      }
    });
  }

  // Simple HTML escape helper function
  function escapeHtml(text) {
    const div = document.createElement('div');
    div.textContent = text;
    return div.innerHTML;
  }

  // Initialize license reset button
  initLicenseResetButton();

  console.log('[ChimpMatic Lite] REST API integration complete.');
});

/**
 * API Key Eye Toggle - Value masking (not type toggle)
 */
document.addEventListener('DOMContentLoaded', function() {
	const eye = document.querySelector('.cmatic-eye');
	const input = document.getElementById('cmatic-api');
	if (!eye || !input) {
		console.log('[ChimpMatic] Eye toggle: elements not found');
		return;
	}

	console.log('[ChimpMatic] Eye toggle initialized', {
		realKey: input.dataset.realKey,
		maskedKey: input.dataset.maskedKey,
		isMasked: input.dataset.isMasked,
		currentValue: input.value
	});

	eye.addEventListener('click', function(e) {
		e.preventDefault();
		const icon = this.querySelector('.dashicons');
		const isMasked = input.dataset.isMasked === '1';

		console.log('[ChimpMatic] Eye clicked, isMasked:', isMasked);

		if (isMasked) {
			// Show real key
			input.value = input.dataset.realKey;
			input.dataset.isMasked = '0';
			icon.classList.remove('dashicons-visibility');
			icon.classList.add('dashicons-hidden');
		} else {
			// Show masked key
			input.value = input.dataset.maskedKey;
			input.dataset.isMasked = '1';
			icon.classList.remove('dashicons-hidden');
			icon.classList.add('dashicons-visibility');
		}
	});

	// CRITICAL: Restore real API key before form submission
	// When masked, the input value contains bullets which would corrupt the saved data
	const form = input.closest('form');
	if (form) {
		form.addEventListener('submit', function() {
			const isMasked = input.dataset.isMasked === '1';
			if (isMasked && input.dataset.realKey) {
				input.value = input.dataset.realKey;
				console.log('[ChimpMatic] Form submit: Restored real API key');
			}
		});
	}
});
