/*
 * ChimpMatic Lite - Admin Notices Handler
 * Vanilla JavaScript for dismissing admin notices via REST API
 *
 * Copyright (C) 2010-2025, Renzo Johnson (email: renzo.johnson at gmail.com)
 */

'use strict';

(function() {
  // Wait for DOM to be ready
  document.addEventListener('DOMContentLoaded', function() {

    // Check if global object exists
    if (typeof chimpmaticNotices === 'undefined') {
      console.error('[ChimpMatic Lite] Notices object not found.');
      return;
    }

    console.log('[ChimpMatic Lite] Admin notices handler initialized.');

    // Handle dismiss button click
    document.addEventListener('click', function(event) {
      // Check if clicked element is the dismiss button inside #mce-notice
      if (event.target.classList.contains('notice-dismiss')) {
        const noticeElement = event.target.closest('#mce-notice');

        if (noticeElement) {
          event.preventDefault();
          console.log('[ChimpMatic Lite] Dismiss notice clicked.');

          // Call REST API to dismiss notice
          dismissNotice(noticeElement);
        }
      }
    });

    /**
     * Dismiss notice via REST API
     */
    async function dismissNotice(noticeElement) {
      console.log('[ChimpMatic Lite] Dismissing notice via REST API...');

      try {
        const response = await fetch(`${chimpmaticNotices.restUrl}/notices/dismiss`, {
          method: 'POST',
          headers: {
            'Content-Type': 'application/json',
            'X-WP-Nonce': chimpmaticNotices.restNonce
          }
        });

        const data = await response.json();
        console.log('[ChimpMatic Lite] Dismiss notice response:', data);

        if (response.ok && data.success) {
          // Hide the notice with fade-out
          noticeElement.style.transition = 'opacity 0.3s ease-out';
          noticeElement.style.opacity = '0';

          setTimeout(() => {
            noticeElement.style.display = 'none';
            console.log('[ChimpMatic Lite] Notice dismissed and hidden.');
          }, 300);

        } else {
          console.error('[ChimpMatic Lite] Failed to dismiss notice:', data.message);
        }

      } catch (error) {
        console.error('[ChimpMatic Lite] Dismiss notice error:', error);
      }
    }
  });
})();
